{
// WORDCLS.PAS (C) 1995 W. Raike
//              ALL RIGHTS RESERVED.
//   NOTICE:    This file consists of trade secrets that are the property
//              of William Raike.  The contents may not be used or disclosed
//              without express written permission of the owner.

//   Revised 5/10/95 to include private FSelection member
//   alloc/dealloc. in constructor/destructor, and Selection method.

//   Added to Ver 1.1 TOLEAutoClient distribution package - 31/12/95.
}
unit WordCls;
{*******************************************************************}
{
  Class TWordObj that "mirrors" (some of) the properties and methods
  exposed by the Microsoft Word for Windows 6.0 OLE Automation server.
  This class models only a few of the hundreds of WordBasic commands,
  for illustrative purposes.  For details of WordBasic syntax, refer
  to the online help supplied with Microsoft Word and also to the file
  POSITION.TXT supplied by Microsoft with the Microsoft Office Developers'
  Kit (ODK).  This latter file correctly describes the order of the
  arguments for WordBasic commands/macros, sometimes shown incorrectly
  in the online help.

  Also works with Word 7.0 for Windows 95.
  ***NOTE***  This implementation works only for English-language
              versions of Word.  If your copy of Word is for a
              different language, you will need to change the
              strings containing names of the WordBasic commands
              to correspond to those supported by your version of Word.
}
{*******************************************************************}

interface

uses
    SysUtils, OleAuto;

type
  TWordObj = class(TOleObject)
  private
    FSelection : PChar;
  public
    constructor CreateObject(progID : String); override;
    destructor Release; override;

    procedure AllCaps;
    procedure AppShow;
    procedure Cancel;
    function CountFoundFiles : Integer;
    procedure EditSelectAll;
    procedure FileClose(CloseMode : Integer);
    procedure FileFind(Name, SearchPath : String);
    procedure FileNew(TemplateName : String);
    procedure FileOpen(FileName : String);
    procedure FileSaveAs(FileName : String);
    function FoundFileName(index : Integer) : String;
    procedure Insert(pszText : PChar);
    function Selection : PChar;
    procedure SetTab(TabPos : String);
    procedure ToolsMacro(MacroName : String);
    procedure ToolsOptionsEdit;
  end;

implementation

constructor TWordObj.CreateObject(progID : String);
begin
  { Override the usual CreateObject constructor in order to
    allocate space to hold the text returned by the WordBasic
    Selection$() function. }
  inherited CreateObject(progID);
  { The maximum length of the Selection$() string in WordBasic is 65280. }
  try
    FSelection := StrAlloc(65281);
    FSelection[0] := #0;
  except
    FSelection := nil;  { Swallow exception if memory is unavailable. }
  end;
end;

destructor TWordObj.Release;
begin
  { Override destructor to deallocate memory allocated in constructor. }
  if Assigned(FSelection) then
    StrDispose(FSelection);
  inherited Release;
end;

procedure TWordObj.AllCaps;
begin
  CallOleProc('AllCaps');
end;

procedure TWordObj.AppShow;
begin
  CallOleProc('AppShow');
end;

procedure TWordObj.Cancel;
begin
  CallOleProc('Cancel');
end;

function TWordObj.CountFoundFiles : Integer;
var
  Count : Integer;
begin
  CallOleFunction('CountFoundFiles', 'Integer', Count);
  Result := Count;
end;

procedure TWordObj.EditSelectAll;
begin
  CallOleProc('EditSelectAll');
end;

procedure TWordObj.FileClose(CloseMode : Integer);
begin
  SetOleMethodArg('Integer', CloseMode);
  CallOleProc('FileClose');
end;

procedure TWordObj.FileFind(Name, SearchPath : String);
const
  pszEmpty : PChar = #0;
var
  nZero : Integer;
  pszName, pszSearchPath : PChar;
begin
  pszName := StrAlloc(256);
  pszSearchPath := StrAlloc(256);
  StrPLCopy(pszName, Name, 255);
  StrPLCopy(pszSearchPath, SearchPath, 255);
  nZero := 0;

  SetOleMethodArg('Integer', nZero);  {.Delete}
  SetOleMethodArg('Integer', nZero);  {.Add}
  SetOleMethodArg('Integer', nZero);  {.SelectedFile}
  SetOleMethodArg('Integer', nZero);  {.ListBy}
  SetOleMethodArg('Integer', nZero);  {.SortBy}
  SetOleMethodArg('Integer', nZero);  {.View}
  SetOleMethodArg('PChar', pszEmpty);  {.DateCreatedTo}
  SetOleMethodArg('PChar', pszEmpty);  {.DateCreatedFrom}
  SetOleMethodArg('PChar', pszEmpty);  {.SavedBy}
  SetOleMethodArg('PChar', pszEmpty);  {.DateSavedTo}
  SetOleMethodArg('PChar', pszEmpty);  {.DateSavedFrom}
  SetOleMethodArg('Integer', nZero);  {.PatternMatch}
  SetOleMethodArg('PChar', pszEmpty);  {.Text}
  SetOleMethodArg('Integer', nZero);  {.MatchCase}
  SetOleMethodArg('Integer', nZero);  {.Options}
  SetOleMethodArg('PChar', pszEmpty);  {.Subject}
  SetOleMethodArg('PChar', pszEmpty);  {.Keywords}
  SetOleMethodArg('PChar', pszEmpty);  {.Author}
  SetOleMethodArg('PChar', pszEmpty);  {.Title}
  SetOleMethodArg('Integer', nZero);  {.SubDir}
  SetOleMethodArg('PChar', pszName);  {.Name}
  SetOleMethodArg('PChar', pszSearchPath); {.SearchPath}
  SetOleMethodArg('PChar', pszEmpty);  {.SearchName}

  CallOleProc('FileFind');

  StrDispose(pszName);
  StrDispose(pszSearchPath);
end;

procedure TWordObj.FileNew(TemplateName : String);
var
  pszName : PChar;
begin
  pszName := StrAlloc(256);
  StrPLCopy(pszName, TemplateName, 255);
  SetOleMethodArg('PChar', pszName);
  CallOleProc('FileNew');
  StrDispose(pszName);
end;

procedure TWordObj.FileOpen(FileName : String);
var
  pszName : PChar;
begin
  pszName := StrAlloc(256);
  StrPLCopy(pszName, FileName, 255);
  SetOleMethodArg('PChar', pszName);
  CallOleProc('FileOpen');
  StrDispose(pszName);
end;

procedure TWordObj.FileSaveAs(FileName : String);
var
  pszName : PChar;
begin
  pszName := StrAlloc(256);
  StrPLCopy(pszName, FileName, 255);
  SetOleMethodArg('PChar', pszName);
  CallOleProc('FileSaveAs');
  StrDispose(pszName);
end;

function TWordObj.FoundFileName(index : Integer) : String;
var
  pszName : PChar;
begin
  if index <= 0 then Exit;
  pszName := StrAlloc(256);
  SetOleMethodArg('Integer', index);
  CallOleFunction('FoundFileName$', 'PChar', pszName);
  Result := StrPas(pszName);
  StrDispose(pszName);
end;

procedure TWordObj.Insert(pszText : PChar);
begin
  SetOleMethodArg('PChar', pszText);
  CallOleProc('Insert');
end;

function TWordObj.Selection : PChar;
begin
  if Assigned(FSelection) then
  begin
    CallOleFunction('Selection$', 'PChar', FSelection);
    Result := FSelection;
  end
  else
    Result := nil;
end;

procedure TWordObj.SetTab(TabPos : String);
const
  pszPos : PChar = 'zzzzzzzzzz';
  pszDefTabs : PChar = '1.0';
var
  n0, n1 : Integer;
begin
  n0 := 0;
  n1 := 1;
  StrPLCopy(pszPos, TabPos, 6);

  SetOleMethodArg('Integer', n0); { .ClearAll }
  SetOleMethodArg('Integer', n0); { .Clear }
  SetOleMethodArg('Integer', n1); { .Set }
  SetOleMethodArg('Integer', n0); { .Leader }
  SetOleMethodArg('Integer', n0); { .Align }
  SetOleMethodArg('PChar', pszDefTabs);   { .DefTabs }
  SetOleMethodArg('PChar', pszPos);   { .Position }
  CallOLEProc('FormatTabs');
end;

procedure TWordObj.ToolsMacro(MacroName : String);
var
  mode : Integer;
  pszName : PChar;
begin
  mode := 1;
  pszName := StrAlloc(256);
  SetOleMethodArg('Integer', mode); { This arg *MUST* be present. }
  StrPLCopy(pszName, MacroName, 255);
  SetOleMethodArg('PChar', pszName);
  CallOleProc('ToolsMacro');
  StrDispose(pszName);
end;

procedure TWordObj.ToolsOptionsEdit;
const
  pszMSW : PChar = 'Microsoft Word';
var
  x : Integer;
begin
  { It's necessary to supply ALL the arguments to ToolsOptionsEdit. }
  {  We always set the arguments in reverse order: i.e., right to left. }
  {
      For this WordBasic command, all the options indicate check-box status
      values (0 = unchecked) in the Word 6 Tools|Options\Edit dialog.
  }
  SetOleMethodArg('PChar', pszMSW); { .PictureEditor }

  x := 0; { Unchecked. }
  SetOleMethodArg('Integer', x);  { .AllowAccentedUppercase }
  SetOleMethodArg('Integer', x);  { .SmartCutPaste }
  SetOleMethodArg('Integer', x);  { .Overtype }
  SetOleMethodArg('Integer', x);  { .InsForPaste }
  SetOleMethodArg('Integer', x);  { .AutoWordSelection }

  x := 1; { Checked. }
  SetOleMethodArg('Integer', x);  { .DragAndDrop }
  SetOleMethodArg('Integer', x);  { .ReplaceSelection }
  CallOleProc('ToolsOptionsEdit');
end;

end.
