#include <iostream.h>
#include <string.h>
#include "vlong.h"

// the main routine and functions show an example of the use of "vlong"
// this type represents an arbitrarily long integer


// IsPrime returns true (1) if n is prime and false (0) if it is not
// it makes a table of the first 100 primes and tries division by
// each of them and then uses Miller's test for primality using
// the first 20 primes as bases

int IsPrime(vlong& n)
{
  long primes[100];
  long odd;
  vlong odd2;
  long sample;
  int i,j;
  vlong b, n1, nm1, r;

  // make prime table
  if (n<2) return 0;

  primes[0] = 2;
  primes[1] = 3;
  j = 2;
  sample = 3;
  while (j<100)
  {
	 sample += 2;
	 for (i=0; i<j; i++)
	 {
		odd = sample%primes[i];
		if (!odd) break;
	 }
	 if (odd)
	 {
		primes[j++]=sample;
	 }
  }

 // see if n is prime by comparing to the table and by trial division
  for(i=0; i<100; i++)
  {
	 if (n == primes[i]) return 1;	// is prime
	 if (!(n%primes[i])) return 0;	// not prime
  }

  // Miller's test ... note that the use of arithmetic operators % and /
  // works fine on the arbitrarily long numbers

  for (i=0; i<20; i++)	// try for each prime
  {
	 nm1 = n1 = n - 1;	// n1 = n-1
	 b = primes[i];
	 r = 1;  // residual
	 odd2 = 0;				// indicates was even
	 while (!odd2 && (r==1))
	 {
		r = PowerMod(b, n1, n);
		odd2 = n1%2;
		n1 = n1/2;
	 }
	 if ((r != 1) && (r != nm1)) return 0;
  }
  return 1;
}


// function calculates x such that x*y mod modulus = 1
// that is it returns the inverse of y to a particular modulus
// again note how the arbitrarily long numbers are treated in
// the same way as ordinary integers

vlong Inverse(const vlong& y, const vlong& modulus)
  {
	 vlong a, b, b0, r, s, s1, s2, t, t1, t2, q;
	 vlong inv;

	 a = y;
	 b = modulus;
	 b0 = b;
	 s2 = 1; s1 = 0; t2 = 0; t1 = 1;
	 while (b>0)
	 {
		q = a/b; r = a%b;
		s = s2 - q*s1;
		t = t2 - q*t1;
		a = b;
		b = r;
		s2 = s1;
		t2 = t1;
		s1 = s;
		t1 = t;
	 }
	 if (a != 1) inv = 0;
	 else inv = s2%b0;
	 if (inv<0) inv += b0;
	 return inv;
  }


// this example shows the use of vlong arbitrarily long integers to
// encrypt and decrypt text
// it is just a demonstration vehicle for vlong - the usefullness of
// RSA ciphers is lost by using a prime rather than the product of primes
// as the modulus, thus the decrypt key is easily derived from the encrypt
// key

// when "too long" an integer comes about in calculations, an exception of
// type char* is thrown - catching this exception is shown in the code
// in reality, the enhanced version of the vlong library allows integers
// as large as 10 to the 8000 so this exception is unlikely, however it
// may appear in the more limited unregistered version which only allows
// numbers as large as 10 to the 100 or so

int main()
{
  vlong p, a, b, c, d, e;

  char text[256];
  char ctext[256] = "";
  char ptext[256] = "";

  char* intextptr;
  char* outtextptr;

 try
 {
  cout << "Using vlong library version " << vlong_version() << "\n\n";

  cout << "Enter a random integer of many digits.  The program will find\n";
  cout << " the next prime number.\n\n";
  cout << "sample for prime? ";
  cin >> p;

  while (!IsPrime(p)) p++;

  cout << p << " is prime.\n\n";

  cout << "Enter a random integer of somewhat less digits.  The program\n";
  cout << " will find the next integer with a suitable inverse to form\n";
  cout << " an encrypt/decrypt pair.\n\n";
  cout << "encode? ";
  cin >> e;

  for(;;)
  {
	 d = Inverse(e,p-1);
	 if (d!=0) break;
	 e++;
  }
  cout << "inverse of " << e << " is " << d << "\n\n";

  cout << "Enter a phrase or sentence.  The program will change it into\n";
  cout << " vlong integers using Import and then encrypt by raising the\n";
  cout << " numbers to the encode power mod prime.  The encoded numbers\n";
  cout << " are displayed and then decrypted by raising them to the\n";
  cout << " decode power mod prime. The result is turned back into \n";
  cout << " characters using Export and displayed.  Note that the \n";
  cout << " encrypted character string in ctext could be displayed but \n";
  cout << " many of the characters would not be valid ASCII characters. \n";
  cout << "\ntext sample? ";
  cin.getline(text,256);
  cout << "\n";

  // CharLength returns the largest number of characters the contents of
  // which would always be a smaller number than the prime p
  // A character string blocksize+1 long would be large enough to hold
  // any number p or smaller
  int blocksize = CharLength(p);

  int linelength = strlen(text);	// the length of the input line

  intextptr = text;
  outtextptr = ctext;
  int blocks = 0;
  while (intextptr<text+linelength)
  {
	 Import(intextptr, &a, blocksize);	// turn text into vlong integer a
	 b = PowerMod(a,e,p);               // a to the e mod p
	 cout << a << " encodes to \n" << b << "\n";
	 Export(b, outtextptr,blocksize+1);
	 intextptr += blocksize;
	 outtextptr += blocksize + 1;       // the cipher text must be 1 character
	 blocks++;									// longer in each block
  }

  cout << "\n";

  intextptr = ctext;
  outtextptr = ptext;
  for (int i=0; i<blocks; i++)
  {
	 Import(intextptr, &a, blocksize+1);
	 b = PowerMod(a,d,p);
	 cout << a << " decodes to \n" << b << "\n";
	 Export(b, outtextptr,blocksize);
	 intextptr += blocksize + 1;
	 outtextptr += blocksize;
  }

  cout << "\ndecodes to : " << ptext << "\n";

 }
 catch (char *string)	// this displays any string exceptions resulting
 {                      // from too big a number
  cout << "\n" << string << "!!!\n\n";
 }
  return 1;
}