/*
  WinStdio V2.1 - Simple stdio facilties for Windows
  Copyright 1989 - 1996 I. H. Ting (I.H.Ting@wlv.ac.uk)

  WinStdio provides simple console-based (stdio) input output in Windows.
  At the moment, only the following calls are supported:

  puts(), printf(), and gets().

  Please feel free to use and distribute the WinStdio source code.
  Just respect my copyright. No waranty is implied or expressed.
  I will only guarantee that the code will fail occasionally!
*/
#include <limits.h>
#include <windows.h>
#include <windowsx.h>
#include "WinStdio.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#ifndef _WIN32
#include "toolhelp.h"  /* remember to link with toolhelp.lib */
#endif

#define IDC_EDIT 300
#define MAX_LINES 100
#define DEL_EXTRA_LINES 20
#define WINSTDIO_CLASS_NAME "WinStdioClass"
#define BUFFER_SIZE 1024


void MyYield(HWND hWnd);

struct WinStdioInfo {
  HWND hEditWnd;
  BOOL bWaitingForInput;
  BOOL bAppQuiting;
  int iLastCaretPos; 
};


static BOOL bWinStdioRegistered=FALSE;
static WNDPROC lpOldEditProc;
static HWND hDefaultStdioWnd=NULL;



static HINSTANCE GetCurrentInstance(void)
{
#ifdef _WIN32
  return GetModuleHandle(NULL);
#else
  static TASKENTRY te;
  te.dwSize = sizeof(TASKENTRY);
  TaskFindHandle(&te,GetCurrentTask());
  return te.hInst;
#endif
}

static char * GetCurrentModuleName(void)
{
#ifdef _WIN32
  GetModuleFileName(NULL,workBuffer, WORK_BUFFER_SIZE);
  return workBuffer;
#else
  static TASKENTRY te;
  te.dwSize = sizeof(TASKENTRY);
  TaskFindHandle(&te,GetCurrentTask());
  return te.szModule;
#endif
}



static HWND VerifyDefaultWindow(void)
{
  if (!bWinStdioRegistered){
#ifndef _WIN32
    if(FindWindow(WINSTDIO_CLASS_NAME,NULL))
      bWinStdioRegistered=TRUE;
    else
#endif
      RegisterWinStdioClass(GetCurrentInstance());
  }
  if(hDefaultStdioWnd==NULL)
    hDefaultStdioWnd=CreateStdioWindow(GetCurrentModuleName(),GetCurrentInstance(),SW_SHOW);
  return hDefaultStdioWnd;
}


static DWORD MyEdit_GetSel(HWND hEdit,int *piStart,int *piEnd)
{
    DWORD dwSel;
#ifdef _WIN32
  dwSel = (DWORD)SendMessage(hEdit,EM_GETSEL,(WPARAM)(LPDWORD)piStart,(LPARAM)(LPDWORD)piEnd);
#else
    dwSel = Edit_GetSel(hEdit);
    if(piStart)
        *piStart = LOWORD(dwSel);
    if(piEnd)
        *piEnd = HIWORD(dwSel);
#endif
    return dwSel;
}

static void ClearOldLines(HWND hWnd, unsigned numLines,BOOL bRedraw)
{
  int iDelToPos;
  unsigned uiTextLength;
  struct WinStdioInfo *lpInfo;

  if(IsWindow(hWnd)){
    lpInfo = (struct WinStdioInfo *)GetWindowLong(hWnd,0);
    if(lpInfo->bAppQuiting)
      return;
  }
  else 
    return;
  iDelToPos = Edit_LineIndex(lpInfo->hEditWnd, numLines);
  if(iDelToPos == -1){
    uiTextLength = GetWindowTextLength(lpInfo->hEditWnd);
    iDelToPos = uiTextLength/2;
  }
  SetWindowRedraw(lpInfo->hEditWnd, bRedraw);
  Edit_SetSel(lpInfo->hEditWnd, 0,iDelToPos);
  Edit_ReplaceSel(lpInfo->hEditWnd,"");
  uiTextLength = GetWindowTextLength(lpInfo->hEditWnd);
  Edit_SetSel(lpInfo->hEditWnd, uiTextLength,uiTextLength);
}



void StdioOnCommand (HWND hwnd, int id, HWND hwndCtl, UINT codeNotify) 
{
  switch(id){
    case IDC_EDIT:
      if (codeNotify == EN_ERRSPACE || codeNotify == EN_MAXTEXT) {
        ClearOldLines(hwnd, DEL_EXTRA_LINES,TRUE);
      }
      break;
  }
  FORWARD_WM_COMMAND(hwnd, id, hwndCtl, codeNotify,DefWindowProc);
}
  

EXPORT_STD_CALLBACK(LRESULT) MainWndProc(HWND hWnd,UINT message,WPARAM wParam,LPARAM lParam)
{
    struct WinStdioInfo *lpInfo;
    RECT rect;

    switch (message) {
      case WM_COMMAND :
        return HANDLE_WM_COMMAND(hWnd, wParam,lParam,StdioOnCommand);
      case WM_SETFOCUS:
        lpInfo = (struct WinStdioInfo *)GetWindowLong(hWnd,0);
        SetFocus (lpInfo->hEditWnd);
        return 0;
      case WM_SIZE:
        lpInfo = (struct WinStdioInfo *)GetWindowLong(hWnd,0);
        MoveWindow(lpInfo->hEditWnd, 0, 0, LOWORD(lParam), HIWORD(lParam), TRUE);
        Edit_GetRect(lpInfo->hEditWnd, &rect);
        rect.left +=4;
        Edit_SetRect(lpInfo->hEditWnd, &rect);
        return 0;
      case WM_DESTROY:
        lpInfo = (struct WinStdioInfo *)GetWindowLong(hWnd,0);
        free(lpInfo);
        if(hDefaultStdioWnd==hWnd)
          hDefaultStdioWnd=NULL;
        return 0;
    }
    return (DefWindowProc(hWnd, message, wParam, lParam));
}


 
/* New edit control proc. Handles specific keys otherwise pass it to old proc */
EXPORT_STD_CALLBACK(LRESULT) MyEditProc(HWND hWnd,UINT message,WPARAM wParam,LPARAM lParam)
{
  struct WinStdioInfo *lpInfo;
  int iTextLength;
  int iKeyState;
  int iStartChar, iEndChar,iTemp;

  lpInfo = (struct WinStdioInfo *)GetWindowLong(GetParent(hWnd),0);
  
  if(lpInfo->bWaitingForInput){
    switch(message){
      case WM_KEYDOWN:
        switch(wParam){
          case VK_RETURN:
          iKeyState=GetKeyState(VK_CONTROL) & (~1); /*throw away the toggle bit*/
          if(! iKeyState){
            iTextLength = GetWindowTextLength(lpInfo->hEditWnd);
            Edit_SetSel(lpInfo->hEditWnd,iTextLength,iTextLength);
          }
          break;
        }
        break;             
      case WM_KEYUP:
        switch(wParam){
          case VK_RETURN:
          iKeyState=GetKeyState(VK_CONTROL) & (~1); //throw away the toggle bit
          if(! iKeyState)
              lpInfo->bWaitingForInput=FALSE;
          break;
        }
        break;
      case WM_CHAR:
        MyEdit_GetSel(hWnd,&iStartChar,&iEndChar );
        /*move any selection to a valid range */
        if(lpInfo->iLastCaretPos >iStartChar)
          iStartChar = lpInfo->iLastCaretPos;
        if(lpInfo->iLastCaretPos >iEndChar)
          iEndChar = lpInfo->iLastCaretPos;
        /*normalise the start and end of selection */
        if(iStartChar >iEndChar){
          iTemp = iStartChar;
          iStartChar = iEndChar;
          iEndChar = iTemp;
        }  
        if(wParam== VK_BACK && lpInfo->iLastCaretPos == iStartChar){
          /* can't delete stuff printed previously */
          MessageBeep(MB_ICONEXCLAMATION);
          return 0;
        }
        if(lpInfo->iLastCaretPos == iStartChar){
          /*move the selection back to valid position */
          Edit_SetSel(hWnd,iStartChar,iEndChar); 
        }
        break;
    }
  }
  return CallWindowProc(lpOldEditProc,hWnd,message,wParam, lParam);
}


static void GetDecentWindowRect(RECT *pRect)
{
  int iScreenWidth = GetSystemMetrics(SM_CXFULLSCREEN);
  int iScreenHeight = GetSystemMetrics(SM_CYFULLSCREEN);
  
  pRect->left = iScreenWidth/6;
  pRect->top = iScreenHeight/6;
  pRect->right = 5*iScreenWidth/6;
  pRect->bottom = 5*iScreenHeight/6;
  
}
               
               
/* creates the necessary windows */
HWND CreateStdioWindow(LPSTR szWindowName, HANDLE hInstance,int  nCmdShow)
{
    HWND hMainWnd=NULL, hEditWnd=NULL;                  
    RECT Rect;
    struct WinStdioInfo *lpInfoStruct=NULL;

    if(!bWinStdioRegistered)
      RegisterWinStdioClass(hInstance);
    GetDecentWindowRect(&Rect);
    hMainWnd = CreateWindowEx(0,WINSTDIO_CLASS_NAME,szWindowName,WS_OVERLAPPEDWINDOW,
        Rect.left,Rect.top,Rect.right-Rect.left,Rect.bottom-Rect.top,NULL,NULL,
        hInstance,NULL);

    if (!hMainWnd)
        return (FALSE);

    GetClientRect(hMainWnd, (LPRECT) &Rect);

    /* Create a child edit window */
    hEditWnd = CreateWindowEx(0,"Edit",NULL,WS_CHILD|WS_VISIBLE|ES_MULTILINE|WS_VSCROLL
      |WS_HSCROLL|ES_AUTOHSCROLL|ES_AUTOVSCROLL,0,0,
      (Rect.right-Rect.left),(Rect.bottom-Rect.top), hMainWnd,
      (HMENU)IDC_EDIT, /* Child control i.d. HMENU cast just to stop warnings*/
      hInstance,NULL);

    if (!hEditWnd) {
      DestroyWindow(hMainWnd);
      return (NULL);
    }
    if(hDefaultStdioWnd==NULL)
      hDefaultStdioWnd=hMainWnd;
    lpInfoStruct=(struct WinStdioInfo *)malloc(sizeof(struct WinStdioInfo));
    lpInfoStruct->hEditWnd=hEditWnd;
    lpInfoStruct->bWaitingForInput=FALSE;
    lpInfoStruct->bAppQuiting=FALSE;
    lpInfoStruct->iLastCaretPos=0;

    SetWindowLong(hMainWnd, 0, (LONG)lpInfoStruct);
    
    /* subclass the edit control procedure using SubclassWindow from windowsx.h*/              
    lpOldEditProc=SubclassWindow(hEditWnd, MyEditProc);
    /* set the edit control to READONLY state */
    Edit_SetReadOnly(hEditWnd, TRUE);
    SetWindowFont(hEditWnd, GetStockFont(ANSI_FIXED_FONT), TRUE);                      
    Edit_SetTabStops(hEditWnd, 0, NULL);    
    Edit_LimitText(hEditWnd, INT_MAX);
    ShowWindow(hMainWnd, nCmdShow);
    
    UpdateWindow(hMainWnd);
    return (hMainWnd);
}


/* registers our window class */
BOOL RegisterWinStdioClass(HANDLE hInstance)
{
    BOOL bRetVal=FALSE;
    WNDCLASS  wc;

    wc.style = CS_GLOBALCLASS;
    wc.lpfnWndProc = MainWndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = sizeof(LONG);
    wc.hInstance = hInstance;
    wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject(GRAY_BRUSH); 
    wc.lpszMenuName =  NULL;
    wc.lpszClassName = WINSTDIO_CLASS_NAME;
    bRetVal=RegisterClass(&wc);
    bWinStdioRegistered=TRUE;
    return bRetVal;
}



/* PeekMessage loop to let Windows paint the various windows */
void MyYield(HWND hWnd)
{
  MSG msg;
  struct WinStdioInfo *lpInfo;

  if(IsWindow(hWnd)){
    lpInfo = (struct WinStdioInfo *)GetWindowLong(hWnd,0);
    if(lpInfo->bAppQuiting)
      return;
  }
  else
    return;

  while (IsWindow(hWnd)&& PeekMessage(&msg, NULL, 0, 0,PM_REMOVE)) {
    if (msg.message == WM_QUIT){
      lpInfo->bAppQuiting=TRUE;
      PostQuitMessage(0);
    }
    TranslateMessage(&msg);
    DispatchMessage(&msg);
  }
}



int WinPuts(HWND hWnd,LPCSTR lpString,BOOL bAddReturn)
{
  int nLines, iEndSel,iTextLength;
  unsigned idx1=0,idx2=0;
  char lastChar='\0';
  char buffer[BUFFER_SIZE];
  struct WinStdioInfo *lpInfo;

  if(IsWindow(hWnd)){
    lpInfo = (struct WinStdioInfo *)GetWindowLong(hWnd,0);
    if(lpInfo->bAppQuiting)
      return 0;
  }
  else
    return 0;

  while(lpString[idx1] && idx2 < BUFFER_SIZE){
    buffer[idx2]=lpString[idx1];
    if(buffer[idx2]=='\n' && lastChar != '\r'){
      buffer[idx2]='\r';
      idx2++;
      buffer[idx2]='\n';
    }
    lastChar=buffer[idx2];
    idx1++;
    idx2++;
  }
  if(bAddReturn){
    buffer[idx2]='\r';
    idx2++;
    buffer[idx2]='\n';
    idx2++;
  }
  buffer[idx2]='\0';


  nLines = Edit_GetLineCount(lpInfo->hEditWnd);
  if (nLines > MAX_LINES){
    SetWindowRedraw(lpInfo->hEditWnd, FALSE);
    iEndSel = Edit_LineIndex(lpInfo->hEditWnd,nLines - MAX_LINES);
    Edit_SetSel(lpInfo->hEditWnd,0,iEndSel);
    Edit_ReplaceSel(lpInfo->hEditWnd,"");
    SetWindowRedraw(lpInfo->hEditWnd, TRUE);
  }

  iTextLength = GetWindowTextLength(lpInfo->hEditWnd);
  Edit_SetSel(lpInfo->hEditWnd,iTextLength,iTextLength);
  Edit_ReplaceSel(lpInfo->hEditWnd,buffer);
  return idx2;
}

int puts( const char *string )
{
  if (!VerifyDefaultWindow())
    return EOF;
  return WinPuts(hDefaultStdioWnd,string,1);
}

/* Our replacement Windows printf. relies on WinPuts*/
int WinPrintf(HWND hWnd, const char *fmt,...)
{
  int ret;
  char *sBuffer = malloc(1024*sizeof(char));
  va_list arg_ptr;
  va_start(arg_ptr, fmt);
  vsprintf(sBuffer,fmt,arg_ptr);
  va_end(arg_ptr);
  ret = WinPuts(hWnd, sBuffer,0);
  free(sBuffer);
  return(ret);
}

int printf(const char *fmt,...)
{
  int ret;
  char *sBuffer = malloc(1024*sizeof(char));
  va_list arg_ptr;

  if(!VerifyDefaultWindow())
    return -1;
  va_start(arg_ptr, fmt);
  vsprintf(sBuffer,fmt,arg_ptr);
  va_end(arg_ptr);
  ret = WinPuts(hDefaultStdioWnd, sBuffer,0);
  free(sBuffer);
  return(ret);
}




char *WinGets(HWND hWnd, char *buffer,BOOL bWantNewline)
{
  int i;
  LPSTR pEditText, pBuffer;
  int iTextLength;
  struct WinStdioInfo *lpInfo;

  if(IsWindow(hWnd)){
    lpInfo = (struct WinStdioInfo *)GetWindowLong(hWnd,0);
    if(lpInfo->bAppQuiting)
      return 0;
  }
  else{
    buffer[0]='\0';
    return 0;
  }

  buffer[0]='\0';

  /* Move the caret to the end of the edit control */
  iTextLength = GetWindowTextLength(lpInfo->hEditWnd);
  Edit_SetSel(lpInfo->hEditWnd, iTextLength,iTextLength);

  /*Remember where it is sitting*/
  lpInfo->iLastCaretPos = iTextLength;
  /*Turn off the READONLY state to allow input */
  Edit_SetReadOnly(lpInfo->hEditWnd,FALSE);

  /* Wait until input has finsihed ie. RETURN key */
  lpInfo->bWaitingForInput=TRUE;
  while(IsWindow(hWnd) && lpInfo->bWaitingForInput && !lpInfo->bAppQuiting){
    MyYield(hWnd);
  }

  if(IsWindow(hWnd)){
    /*Turn READONLY state back on to disallow input */
    Edit_SetReadOnly(lpInfo->hEditWnd,TRUE);

    /* find out where we are now */
    iTextLength = GetWindowTextLength(lpInfo->hEditWnd);
    pEditText = (char*)malloc(iTextLength+2);
    GetWindowText(lpInfo->hEditWnd,pEditText,iTextLength+1);
    pBuffer=buffer;
    for(i=lpInfo->iLastCaretPos; i<=iTextLength; i++){
      if((pEditText[i] =='\r'|| pEditText[i] =='\n') && !bWantNewline)
        *pBuffer = '\0';
      else
        *pBuffer = pEditText[i];
      pBuffer++;
    }
    *pBuffer = '\0';    
  free(pEditText);
  }
  else{
    buffer[0]='\0';
    /* 
      if somebody killed the window whilst waiting an input, 
      you might want to quit the program here by calling exit().
    */
  }
  return buffer;
}




char *gets(char *buffer)
{
  if (!VerifyDefaultWindow())
    return NULL;
  return WinGets(hDefaultStdioWnd,buffer,0);
}



int WinStdioYield(void)
{
  if (!IsWindow(hDefaultStdioWnd))
    return 0;
  MyYield(hDefaultStdioWnd);
  return 1;
}
