program NEWSIE ;

 (*****************************************************************)
 (* This software is provided as-is, free of charge and           *)
 (* includes Turbo Pascal 6.0 source code.  The source code was   *)
 (* written by John Parlin and has been contributed  to  the      *)
 (* public  domain  in the  interest of furthering the            *)
 (* program's development but  also  to  encourage others to      *)
 (* create software.                                              *)
 (*                                                               *)
 (* If you  add  to this program,  please  redistribute  the      *)
 (* source  code as freeware to others so that they may use the   *)
 (* enhancements or add even more to the program.                 *)
 (*****************************************************************)

uses dos,crt ;

const
   version      = '1.00 (freeware)' ;
   copyright    = 'Copyright 1996 John Parlin' ;
   copyright2   = 'All rights reserved' ;

type
   str2         = string[2] ;

var
   infile       : text ;
   outfile      : text ;
   line         : string ;
   banner       : string ;
   editor       : string ;
   mouseclick   : boolean ;
   mousetext    : string ;


 (************************************************************)
 (*                STRING HANDLING ROUTINES                  *)
 (************************************************************)

function ChangeCase(instr:string) : string ;

 (* changes a string to uppercase alpha *)

var i : byte ;
begin
   for i := 1 to length(instr) do instr[i] := upcase(instr[i]);
   ChangeCase:=instr
end ;


function trimright(instr:string) : string ;

 (* trims trailing spaces from the end of a string *)

begin
   while (length(instr) > 0) and (instr[length(instr)] = ' ') do
      instr[0] := pred(instr[0]) ;
   trimright := instr
end ;


function trimleft(instr:string) : string ;

 (* trims leading spaces from the beginning of a string *)

begin
   while (length(instr) > 0) and (instr[1] = ' ') do
      instr := copy(instr,2,length(instr)-1) ;
   trimleft := instr ;
end ;


function fixbyte(n:word) : str2 ;

 (* converts a word to a string[2] with a leading 0 in front of  *)
 (* numbers lower than 10 (i.e. 1 becomes '01')                  *)

var
   temp    : str2 ;
   solve   : str2 ;
begin
   str(n,temp) ;
   case n of
      0..9 : solve := concat('0',temp) ;
      else solve := temp ;
   end ;
   fixbyte := solve ;
end ;


function itoa(i:longint) : string ;

 (* converts a byte, integer, or long integer into a string *)

var
   s  : string ;
begin
   str(i,s) ;
   itoa := s ;
end ;


 (************************************************************)
 (*                RIPScript Related Routines                *)
 (************************************************************)

function mega(n:word) : STRING ;

 (* returns the meganumber conversion of 'n' as a string *)
 (* mega numbers are used in most RipScript statements   *)

const
   digits : array [0..35] of char = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ' ;
var
   i,s,d   : integer ;
begin
   if n < 37 then begin
      mega := '0' + digits[n] ;
      exit ;
   end ;
   i := n div 36 ;                    { how many times does 36 go into 'n' }
   s := 36 * i ;                                           { sum of 36 * i }
   d := n - s ;                                  { difference of 'n' - sum }
   mega := digits[i] + digits[d] ;
end ;


 (************************************************************)
 (*                   Other Program Routines                 *)
 (************************************************************)

procedure get_config ;

 (* get program settings from NEWSIE.CFG and parse them out *)

var
   configfile  : text ;
   tstr,tstr2  : string ;
begin
   assign(configfile,'NEWSIE.CFG') ;
   {$I-} reset(configfile) ; {$I+}
   if ioresult <> 0 then begin
      writeln ;
      writeln(' NEWSIE ERROR: Configuration file NEWSIE.CFG') ;
      writeln('                file not found! See NEWSIE.DOC.') ;
      halt ;
   end ;
   writeln(' Reading NEWSIE.CFG') ;

   (* give the settings variables an initial value *)

   line := '' ;
   editor := '' ;
   banner := '' ;
   mouseclick := false ;
   mousetext := '^M' ;

   (* now read the config file settings *)

   while not eof(configfile) do begin
      readln(configfile,line) ;
      if pos('=',line) > 0 then begin
         tstr := changecase(copy(line,1,pos('=',line)-1)) ;
         tstr2 := copy(line,pos('=',line)+1,50) ;
         tstr2 := trimleft(trimright(tstr2)) ;
         if tstr = 'BANNER' then begin
            if length(tstr2) > 18 then begin
               writeln(#7,' BANNER > 18 characters!  TRUNCATING...') ;
               delay(1000) ;
               tstr2 := copy(tstr2,1,18) ;
            end ;
            banner := trimleft(trimright(tstr2)) ;
         end else if tstr = 'EDITOR' then begin
            editor := trimleft(trimright(tstr2)) ;
         end else if tstr = 'MOUSECLICK' then begin
            mouseclick := (changecase(tstr2) = 'YES') ;
         end else if tstr = 'MOUSETEXT' then begin
            mousetext := changecase(tstr2) ;
         end ;
      end ;
   end ;
end ;


procedure write_banner ;

 (* creates the banner that goes onto the newspaper graphic *)

var
   j,k    : integer ;
begin
   writeln(' Creating banner') ;
   j := length(banner) ;
   j := j * 28 ;                                        {width in points}
   k := 640 - j ;                              {total width - used width}
   k := k div 2 ;                      {what's leftover gets cut in half}
   k := trunc(k) ;
   if k mod 28 <> 0 then begin              {if k is NOT divisible by 28}
      repeat                                   {decrement it until it is}
         dec(k) ;                            {this is to be certain that}
      until k mod 28 = 0 ;          {centering of the large font is done}
   end ;                                                      {correctly}

   (* The following RIP sequences are a combination of RipScript *)
   (* statements and program settings from NEWSIE.CFG.  One RIP  *)
   (* statement is written per-line.                             *)

   writeln(outfile,'!|Y04000700') ;
   writeln(outfile,'!|c01') ;
   writeln(outfile,'!|@'+mega(k)+'00'+banner) ;
   writeln(outfile,'!|c00') ;
   writeln(outfile,'!|L051WHD1W') ;
   writeln(outfile,'!|L050GHD0G') ;
   writeln(outfile,'!|Y02000300') ;
   writeln(outfile,'!|@GE0810 cents') ;
   writeln(outfile,'!|@0K08NEWSIE v1.00') ;
   writeln(outfile,'!|c07') ;
   writeln(outfile,'!|=000NLJ01') ;
   writeln(outfile,'!|LHI00HI84') ;
   writeln(outfile,'!|L0084HI84') ;
   writeln(outfile,'!|c0F') ;
   writeln(outfile,'!|L0000HI00') ;
   writeln(outfile,'!|L00000084') ;
   writeln(outfile,'!|Y02000400') ;
   writeln(outfile,'!|c00') ;
end ;


procedure write_date ;

 (* this procedure adds today's date on the newspaper *)

const
   months : array [1..12] of string[10] =
            ('January','February','March','April','May','June',
             'July','August','September','October','November','December') ;
var
   yr,mn,dt,dy : word ;
   tstr        : string ;
begin
   getdate(yr,mn,dt,dy) ;
   tstr := months[mn] + ' ' + fixbyte(dt) + ', ' + itoa(yr) ;
   writeln(outfile,'!|@0A20'+tstr) ;
end ;


procedure write_editor ;

 (* this adds the "Editor in Chief" name to the newspaper *)

var
   tstr   : string ;
   j,k    : integer ;
begin
   tstr := 'Editor in Chief: '+editor ;
   j := length(tstr) * 7 ;
   k := 640 - j ;
   writeln(outfile,'!|@'+mega(k)+'20'+tstr) ;
   writeln(outfile,'!|=040GUT03') ;
   writeln(outfile,'!|L052GHD2G') ;
   writeln(outfile,'!|=000NLJ03') ;
   writeln(outfile,'!|L5U2G5U88') ;
   writeln(outfile,'!|LBO2GBO88') ;
   writeln(outfile,'!|w000L270N12') ;
   writeln(outfile,'!|c00') ;
end ;


procedure open_output_file ;

 (* Assigns and opens the output text file and draws the *)
 (* initial newspaper.                                   *)

begin
   assign(outfile,'NEWSIE.rip') ;
   rewrite(outfile) ;
   writeln(outfile,'!|K') ;
   writeln(outfile,'!|*') ;
   writeln(outfile,'!|w0010271610|W00|S0107|B0000HI84|c00|=000GUT03|c0F|=000NLJ01') ;
   writeln(outfile,'!|=000NLJ03|=000NLJ03|LHI00HI84|c07|LHL03HL84|c0B|LHO06HO84') ;
   writeln(outfile,'!|c07|LHQ09HQ83|c00') ;
end ;


procedure import_text ;

 (* Assigns/opens the import (input) text file NEWSIE.TXT.  Also   *)
 (* parses headlines and story text and places onto the newspaper. *)

var
   ts       : string ;
   col      : integer ;
   row      : integer ;
   column   : integer ;
   i        : integer ;
   lastline : string ;
   headline : boolean ;
   line2    : string ;
begin
   assign(infile,'NEWSIE.txt') ;
   {$I-} reset(infile) ; {$I+}
   if ioresult <> 0 then begin
      writeln ;
      writeln(#7,' NEWSIE ERROR:   NEWSIE.TXT not found during') ;
      writeln(   '                  import. See NEWSIE.DOC.') ;
      halt(99) ;
   end ;
   writeln(' Importing NEWSIE.TXT') ;
   col := 10 ;                             {keeps track of current column}
   row := 88 ;                                {keeps track of current row}
   column := 1 ;            {keeps track of newspaper story column (1..3)}
   i := 0 ;                  {this just counts the lines in the text file}
   headline := false ;
   lastline := '' ;
   while not eof(infile) do begin
      readln(infile,line) ;
      line := trimleft(trimright(line)) ;
      inc(i) ;
      if line = '' then begin
         lastline := line ;
         case column of
            1 : ts := mega(10) ;
            2 : ts := mega(220) ;
            3 : ts := mega(430) ;
         end ;
         writeln(outfile,'!|Y02000400') ;
         writeln(outfile,'!|@'+ts+mega(row)+line) ;
         row := row + 8 ;
         if row >= 280 then begin
            inc(column) ;
            if column > 3 then begin
               writeln(#7,' Input file too large!') ;
               delay(5000) ;
               exit ;
            end ;
            row := 88 ;
         end ;
      end else begin
         if line[1] = '' then begin
            line2 := copy(line,2,33) ;
            headline := true ;
         end else headline := false ;
         if headline then begin
            if (280 - row) < 48 then begin
               inc(column) ;
               if column > 3 then begin
                  writeln(#7,' Input file too large!') ;
                  delay(5000) ;
                  exit ;
               end ;
               row := 88 ;
            end ;
            if length(line2) > 19 then line2 := copy(line2,1,19) ;
            case column of
               1 : ts := mega(10) ;
               2 : ts := mega(220) ;
               3 : ts := mega(430) ;
            end ;
            writeln(outfile,'!|Y01000200') ;
            if (i <> 1) and (lastline[1] <> '') then inc(row,8) ;
            writeln(outfile,'!|@'+ts+mega(row)+line2) ;
            inc(row,24) ;
         end else begin
            if length(line) > 33 then line := copy(line,1,33) ;
            case column of
               1 : ts := mega(10) ;
               2 : ts := mega(220) ;
               3 : ts := mega(430) ;
            end ;
            if lastline[1] = '' then writeln(outfile,'!|Y02000400') ;
            writeln(outfile,'!|@'+ts+mega(row)+line) ;
            row := row + 8 ;
         end ;
         if row >= 280 then begin
            inc(column) ;
            if column > 3 then begin
               writeln(#7,' Input file too large!') ;
               delay(5000) ;
               exit ;
            end ;
            row := 88 ;
         end ;
      end ;
      lastline := line ;
   end ; {while}
   close(infile) ;
end ;


procedure make_mouse ;

 (* adds RIP mouse region making the entire newspaper "click-able" *)

begin
   writeln(' Adding RIP mouse region') ;
   writeln(outfile,'!|1M000000HI800100000'+mousetext) ;
end ;


begin                                    (* * * M A I N   B O D Y * * *)
   filemode := 66 ;
   textcolor(7) ;
   writeln ;
   writeln('NEWSIE '+version) ;
   writeln(copyright) ;
   writeln(copyright2) ;
   writeln ;
   get_config ;
   open_output_file ;
   write_banner ;
   write_date ;
   write_editor ;
   import_text ;
   if mouseclick then make_mouse ;
   writeln(outfile,'!|#') ;
   writeln(outfile,'!|#') ;
   writeln(outfile,'!|#') ;
   flush(outfile) ;
   close(outfile) ;
   writeln(' RIP graphic news/bulletin file NEWSIE.RIP has been created') ;
   writeln(' Bye-bye now') ;
   writeln ;
end.
