#ifdef _plan9_
#include <u.h>
#include <libc.h>
#include <stdio.h>
#else
#include <limits.h>
#include <math.h>
#include <string.h>
#endif
#include "config.h"
#include "complex.h"
#include "formulas.h"
#include "zoom.h"

int coloringmode;
int incoloringmode;

char *incolorname[] =
{
    "maxiter",
    "zmag",
};

char *outcolorname[] =
{
    "iter",
    "iter+real",
    "iter+imag",
    "iter+real/imag",
    "iter+real+imag+real/imag",
    "binary decomposition"
};


#define OUTPUT() if(iter>=MAXITER)\
		return(incoloringmode?incolor_output(zre,zim,iter,MAXITER):iter); else \
		return(!coloringmode?iter:color_output(zre,zim,iter,MAXITER))

int INLINE
 color_output(number_t zre, number_t zim, int iter, int maxiter)
{
    switch (coloringmode) {
    case 1:			/* real */
	return (iter + zre);
	break;
    case 2:			/* imag */
	return (iter + zim);
	break;
    case 3:			/* real / imag */
	if (myfabs(zim) < 0.00001)
	    return 0;
	return (iter + zre / zim);
	break;
    case 4:			/* all of the above */
	if (myfabs(zim) < 0.00001)
	    return 0;
	return (iter + zre + zim + zre / zim);
	break;
    case 5:
	if (zim < 0)
	    return (iter);
	else
	    return (MAXITER - iter);
	break;
    default:
	break;
    }
    return iter;
}

int INLINE
 incolor_output(number_t zre, number_t zim, int iter, int maxiter)
{
    switch (incoloringmode) {
    case 1:			/* zmag */
	return ((zre * zre + zim * zim) * (number_t) (maxiter >> 1) + 1);
	break;
    default:
	break;
    }
    return iter;
}




static int mand_calc(register number_t cre, register number_t cim, register number_t pre, register number_t pim)
{
    register number_t rp = 0, ip = 0;
    register int iter = MAXITER & (~(int) 3);
    register number_t zre, zim;
    zre = cre;
    zim = cim;
    while ((iter) && (rp + ip < 4)) {
	ip = (zim * zim);
	zim = (zim * zre) * 2 + pim;
	rp = (zre * zre);
	zre = rp - ip + pre;
	iter--;
	/*
	   ip = (zim * zim);
	   zim = (zim * zre) * 2 + pim;
	   rp = (zre * zre);
	   zre = rp - ip + pre;
	   if (rp + ip > 4)
	   break;
	   ip = (zim * zim);
	   zim = (zim * zre) * 2 + pim;
	   rp = (zre * zre);
	   zre = rp - ip + pre;
	   if (rp + ip > 4) {
	   iter -= 1;
	   break;
	   }
	   ip = (zim * zim);
	   zim = (zim * zre) * 2 + pim;
	   rp = (zre * zre);
	   zre = rp - ip + pre;
	   if (rp + ip > 4) {
	   iter -= 2;
	   break;
	   }
	   ip = (zim * zim);
	   zim = (zim * zre) * 2 + pim;
	   rp = (zre * zre);
	   zre = rp - ip + pre;
	   if (rp + ip > 4) {
	   iter -= 3;
	   break;
	   }
	   iter -= 4; */

    }
    iter = MAXITER - iter;
    OUTPUT();
}

static int mand3_calc(register number_t cre, register number_t cim, register number_t pre, register number_t pim)
{
    register number_t rp = 0, ip = 0;
    register int iter = MAXITER;
    register number_t zre, zim;

    zre = cre;
    zim = cim;
    rp = zre * zre;
    ip = zim * zim;
    while ((iter) && (rp + ip < 4)) {
	rp = zre * (rp - 3 * ip);
	zim = zim * (3 * zre * zre - ip) + pim;
	zre = rp + pre;
	rp = zre * zre;
	ip = zim * zim;
	iter--;
    }
    iter = MAXITER - iter;
    OUTPUT();
}

static int mand4_calc(register number_t cre, register number_t cim, register number_t pre, register number_t pim)
{
    register number_t rp = 0, ip = 0;
    register int iter = MAXITER;
    register number_t zre, zim;

    zre = cre;
    zim = cim;
    rp = zre * zre;
    ip = zim * zim;
    while ((iter) && (rp + ip < 4)) {
	rp = rp * rp - 6 * rp * ip + ip * ip + pre;
	zim = 4 * zre * zre * zre * zim - 4 * zre * ip * zim + pim;
	zre = rp;
	rp = zre * zre;
	ip = zim * zim;
	iter--;
    }
    iter = MAXITER - iter;
    OUTPUT();
}

static int mand5_calc(register number_t cre, register number_t cim, register number_t pre, register number_t pim)
{
    register number_t rp = 0, ip = 0, t;
    register int iter = MAXITER;
    register number_t zre, zim;

    zre = cre;
    zim = cim;
    while ((iter) && (rp + ip < 4)) {
	c_pow4(zre, zim, rp, ip);
	c_mul(zre, zim, rp, ip, t, zim);
	zre = t + pre;
	zim += pim;
	rp = zre * zre;
	ip = zim * zim;
	iter--;
    }
    iter = MAXITER - iter;
    OUTPUT();
}

static int mand6_calc(register number_t cre, register number_t cim, register number_t pre, register number_t pim)
{
    register number_t rp = 0, ip = 0, t;
    register int iter = MAXITER;
    register number_t zre, zim;

    zre = cre;
    zim = cim;
    while ((iter) && (rp + ip < 4)) {
	c_pow3(zre, zim, rp, ip);
	c_pow3(rp, ip, t, zim);
	zre = t + pre;
	zim += pim;
	rp = zre * zre;
	ip = zim * zim;
	iter--;
    }
    iter = MAXITER - iter;
    OUTPUT();
}

static int barnsley1_calc(register number_t cre, register number_t cim, register number_t pre, register number_t pim)
{
    register number_t rp = 0;
    register int iter = MAXITER;
    register number_t zre, zim;

    zre = cre;
    zim = cim;
    while ((iter) && (zre * zre + zim * zim < 4)) {
	if (zre >= 0) {
	    c_mul(zre - 1, zim, pre, pim, rp, zim);

	} else {
	    c_mul(zre + 1, zim, pre, pim, rp, zim);
	}
	zre = rp;
	iter--;
    }
    iter = MAXITER - iter;
    OUTPUT();
}

static int newton_calc(register number_t cre, register number_t cim, register number_t pre, register number_t pim)
{
    register number_t rp, ip;
    register int iter = MAXITER;
    register number_t zre, zim, zre1 = 10.0, zim1 = 10.0;
    register number_t tr, ti;

    zre = cre;
    zim = cim;
    while (iter && distance(zre1, zim1, zre, zim) > 1E-6) {
	iter--;
	zre1 = tr = zre;
	zim1 = ti = zim;
	c_pow2(zre, zim, tr, ti);
	c_mul(tr, ti, zre, zim, rp, ip);
	if (myabs(tr) + myabs(ti) < 0.0000001)
	    zre = INT_MAX, zim = INT_MAX;
	else {
	    c_div(2 * rp + 1, 2 * ip, 3 * tr, 3 * ti, zre, zim);
	}
    }
    iter = MAXITER - iter;
    OUTPUT();
}


static int phoenix_calc(register number_t cre, register number_t cim, register number_t pre, register number_t pim)
{
    register int iter = MAXITER;
    register number_t zre, zim, zpr, zpm, rp = 0.0, ip = 0.0;

    zre = cre;
    zim = cim;
    zpr = zre * zre, zpm = zim * zim;
    while (iter && (zpr + zpm < 4)) {
	zpr = zpr - zpm + pre + pim * rp;
	zpm = 2 * zre * zim + pim * ip;
	rp = zre, ip = zim;
	zre = zpr;
	zim = zpm;
	zpr = zre * zre, zpm = zim * zim;
	iter--;
    }
    iter = MAXITER - iter;
    OUTPUT();
}

static int octo_calc(register number_t cre, register number_t cim, register number_t pre, register number_t pim)
{
    register number_t rp, ip, tr, ti;
    register int iter = MAXITER;
    register number_t zre = cre, zim = cim, zpr = 0, zpm = 0;

    while (iter && ((zpr * zpr + zpm * zpm) < 4)) {
	rp = zre;
	ip = zim;
	c_pow3(zre, zim, tr, ti);
	c_add(tr, ti, zpr, zpm, zre, zim);
	zpr = rp;
	zpm = ip;
	iter--;
    }
    iter = MAXITER - iter;
    OUTPUT();
    return (iter);
}

/*FIXME: symetry for wont work */
symetry sym6[] =
{
    {0, 1.73205080758},
    {0, -1.73205080758}
};

symetry sym8[] =
{
    {0, 1},
    {0, -1}
};

symetry sym16[] =
{
    {0, 1},
    {0, -1},
    {0, 0.414214},
    {0, -0.414214},
    {0, 2.414214},
    {0, -2.414214}
};

struct formula formulas[] =
{
    {
	mand_calc,
	{"Mandelbrot", "Julia"},
	{0.5, -2.0, 1.25, -1.25},
	1, 0.0, 0.0,
	{
	    {INT_MAX, 0, 0, NULL},
	    {INT_MAX, 0, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL}
	},
	{
	    {INT_MAX, 0, 0, NULL},
	    {INT_MAX, 0, 0, NULL}
	},
    },
    {
	mand3_calc,
	{"Mandelbrot^3", "Julia^3"},
	{1.25, -1.25, 1.25, -1.25},
	1, 0.0, 0.0,
	{
	    {0, 0, 0, NULL},
	    {INT_MAX, 0, 0, NULL},
	    {0, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {0, INT_MAX, 0, NULL}
	},
	{
	    {0, 0, 0, NULL},
	    {0, 0, 0, NULL}
	},
    },
    {
	mand4_calc,
	{"Mandelbrot^4", "Julia^4"},
	{1.25, -1.25, 1.25, -1.25},
	1, 0.0, 0.0,
	{
	    {INT_MAX, 0, 2, sym6},
	    {INT_MAX, 0, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL}
	},
	{
	    {INT_MAX, 0, 2, sym6},
	    {INT_MAX, 0, 2, sym6}
	}
    },
    {
	mand5_calc,
	{"Mandelbrot^5", "Julia^5"},
	{1.25, -1.25, 1.25, -1.25},
	1, 0.0, 0.0,
	{
	    {0, 0, 2, sym8},
	    {INT_MAX, 0, 0, NULL},
	    {0, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {0, INT_MAX, 0, NULL}
	},
	{
	    {0, 0, 2, sym8},
	    {0, 0, 2, sym8}
	},
    },
    {
	mand6_calc,
	{"Mandelbrot^6", "Julia^6"},
	{1.25, -1.25, 1.25, -1.25},
	1, 0.0, 0.0,
	{
	    {0, 0, 6, sym16},
	    {INT_MAX, 0, 0, NULL},
	    {0, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {0, INT_MAX, 0, NULL}
	},
	{
	    {0, 0, 6, sym16},
	    {0, 0, 6, sym16}
	},
    },
    {
	octo_calc,
	{"Octal", "Octal"},
	{1.25, -1.25, 1.25, -1.25},
	0, 0.0, 0.0,
	{
	    {0, 0, 6, sym16},
	    {INT_MAX, 0, 0, NULL},
	    {0, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {0, INT_MAX, 0, NULL}
	},
	{
	    {0, 0, 6, sym16},
	    {0, 0, 6, sym16}
	},
    },
    {
	newton_calc,
	{"Newton", "Newton"},
	{1.25, -1.25, 1.25, -1.25},
	1, 0.0, 0.0,
	{
	    {INT_MAX, 0, 2, sym6},
	    {INT_MAX, 0, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL}
	},
	{
	    {INT_MAX, 0, 2, sym6},
	    {INT_MAX, 0, 2, sym6}
	},
    },
    {
	barnsley1_calc,
	{"Barnsley1 Mandelbrot", "Barnsley1"},
	{1.25, -1.25, 1.25, -1.25},
	0, -0.6, 1.1,
	{
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL}
	},
	{
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL}
	},
    },
    {
	phoenix_calc,
	{"MandPhoenix", "Phoenix"},
	{1.25, -1.25, 1.25, -1.25},
	0, 0.56667000000000001, -0.5,
	{
	    {INT_MAX, 0, 0, NULL},
	    {INT_MAX, 0, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL},
	    {INT_MAX, INT_MAX, 0, NULL}
	},
	{
	    {INT_MAX, 0, 0, NULL},
	    {INT_MAX, 0, 0, NULL}
	},
    },
};

struct formula *currentformula = formulas;
CONST int nformulas = sizeof(formulas) / sizeof(struct formula);
