/*
	This is a DMapEdit source code module.  Though it is copyrighted, you
	may modify it and use it for your own personal use, meaning that new
	modified code and anything derived from it (such as exe files) doesn't
	get distributed to anyone, unless you get my permission first.  Code
	from this file, or code based on ideas from this file may be used with
	other programs, provided that you give credit for it in the source code,
	documentation, and 'about' windows or screens, if one exists, for the
	programs using it.  Giving credit means something like this:

	Code from DMapEdit was used in this program

							  or

	Some code for this program was based on ideas presented in DMapEdit

	Whatever.  Just be sure to mention "DMapEdit" in such a way that it's
	self-evident how it was useful to the new program, and be sure to have
	"DMapEdit is a trademark of Jason Hoffoss" in the docs.  That's all..
*/

#include <stdio.h>
#include <alloc.h>
#include <stdarg.h>
#include "dme.h"
#include "dme2.h"

extern int mem_trace;
extern char path[];

uint freesize, largest, core;
ulong farfreesize, farlargest, farcore;

void *get_mem(uint size, char *name)
{
	void *ptr;

	if (heapcheck() == -1)
		heaperr("get_mem/", name);
	if (!size)
		return 0;

alloc:
	ptr = malloc(size);
	if (!ptr)
		fatal_mem_error(name, size);
	if (mem_trace)
		mem_log("alloc %s(%u)", name, size);
	return ptr;
}

void *resize_mem(void *old, uint size, char *name)
{
	void *ptr;

	if (heapcheck() == -1)
		heaperr("resize_mem/", name);
	check_if_used(old, name);

alloc:
	ptr = farrealloc(old, size);
	if (!ptr)
		fatal_mem_error(name, size);
	if (mem_trace)
		mem_log("realloc %s(%u)", name, size);
	return ptr;
}

void huge *get_farmem(ulong size, char *name)
{
	void huge *ptr;

	if (farheapcheck() == -1)
		farheaperr("get_farmem/", name);
	if (!size)
		return 0L;

alloc:
	while (!(ptr = farmalloc(size)))
	{
		if (free_patch())
			fatal_farmem_error(name, size);
	}
	if (mem_trace)
		mem_log("faralloc %s(%lu)", name, size);
	return ptr;
}

void huge *resize_farmem(void huge *old, ulong size, char *name)
{
	void huge *ptr;

	if (farheapcheck() == -1)
		farheaperr("resize_farmem/", name);
	check_if_used_far(old, name);

alloc:
	while (!(ptr = farrealloc(old, size)))
	{
		if (free_patch())
			fatal_farmem_error(name, size);
	}
	if (mem_trace)
		mem_log("farrealloc %s(%lu)", name, size);
	return ptr;
}

void free_mem(void *ptr, char *name)
{
	if (!ptr)
	{
		mem_log("%s=@null", name);
		fatal_error("Call to free_mem with %s pointing to null", name);
	}

	if (heapcheck() == -1)
		heaperr("free_mem/", name);
	check_if_used(ptr, name);

	free(ptr);
	if (mem_trace)
		mem_log("free %s", name);
	return;
}

void free_farmem(void huge *ptr, char *name)
{
	if (!ptr)
	{
		mem_log("%s=@null", name);
		fatal_error("Call to free_farmem with %s pointing to null", name);
	}

	if (farheapcheck() == -1)
		farheaperr("free_farmem/", name);
	check_if_used_far(ptr, name);

	farfree(ptr);
	if (mem_trace)
		mem_log("farfree %s", name);
	return;
}

void check_if_used(void *ptr, char *name)
{
	struct heapinfo info;

	info.ptr = 0;
	while (heapwalk(&info) == 2)
	{
		if (info.ptr == ptr)
		{
			if (info.in_use)
				return;
			break;
		}
	}

	mem_log("%s is free", name);
	fatal_error("%s's memory block is free!", name);
}

void check_if_used_far(void huge *ptr, char *name)
{
	struct farheapinfo info;

	info.ptr = 0L;
	while (farheapwalk(&info) == 2)
	{
		if (info.ptr == ptr)
		{
			if (info.in_use)
				return;
			break;
		}
	}

	mem_log("%s is free", name);
	fatal_error("%s's memory block is free!", name);
}

void sizeof_mem_block(void *ptr, char *name)
{
	struct heapinfo info;

	if (heapcheck() == -1)
		heaperr("sizeof_mem_block/", name);

	info.ptr = 0;
	while (heapwalk(&info) == 2)
	{
		if (info.ptr == ptr)
		{
			if (!info.in_use)
				fatal_error("%s's memory block is free!", name);

			mem_log("%s=%u@%p\n", name, info.size, ptr);
			return;
		}
	}
	fatal_error("%s's pointer was not found!", name);
}

void sizeof_farmem_block(void far *ptr, char *name)
{
	struct farheapinfo info;

	if (farheapcheck() == -1)
		farheaperr("sizeof_farmem_block/", name);

	info.ptr = 0L;
	while (farheapwalk(&info) == 2)
	{
		if (info.ptr == ptr)
		{
			if (!info.in_use)
				fatal_error("%s's far memory block is free!", name);

			mem_log("%s=%luL@%Fp\n", name, info.size, ptr);
			return;
		}
	}
	return;
}

void check_mem(void)
{
	struct heapinfo info;

	if (heapcheck() == -1)
		heaperr("check_mem", "");

	freesize = largest = core = coreleft();
	info.ptr = 0;
	while (heapwalk(&info) == 2)
	{
		if (!info.in_use)
		{
			freesize += info.size;
			if (info.size > largest)
				largest = info.size;
		}
	}

	mem_log("mem=%u %u %u\n", freesize, largest, core);
	return;
}

void check_farmem(void)
{
	struct farheapinfo info;

	if (farheapcheck() == -1)
		farheaperr("check_farmem", "");

	farfreesize = farlargest = farcore = farcoreleft();
	info.ptr = 0L;
	while (farheapwalk(&info) == 2)
	{
		if (!info.in_use)
		{
			farfreesize += info.size;
			if (info.size > farlargest)
				farlargest = info.size;
		}
	}

	mem_log("farmem=%lu %lu %lu\n", farfreesize, farlargest, farcore);
	return;
}

void mem_log(char *msg, ...)
{
	va_list args;
	FILE *file;

	if (file = fopen("memory.log", "a"))
	{
		va_start(args, msg);
		vfprintf(file, msg, args);
		va_end(args);
		fprintf(file, "\n");
		fclose(file);
	}
	return;
}

void statistics(void)
{
	char msg[4096];

	check_mem();
	check_farmem();
	sprintf(msg, "Statistics:\t\n"
		"  Things: %5u (%lu)\n"
		"Linedefs: %5u (%lu)\n"
		"Sidedefs: %5u (%lu)\n"
		"Vertexes: %5u (%lu)\n"
		"          %5u\n"
		"    Segs: %5u (%lu)\n"
		"Ssectors: %5u (%lu)\n"
		"   Nodes: %5u (%lu)\n"
		" Sectors: %5u (%lu)\n"
		"  Reject: %5u\n"
		"Blockmap: %5u\n"
		"          %u x %u\n\n"
		"Near mem: %u\n"
		"^   (%u/%u)\n"
		" Far mem: %lu\n"
		"^   (%lu/%lu)\n",
		t_size, (ulong) t_max * sizeof(struct t_struct),
		l_size, (ulong) l_max * sizeof(struct l_struct),
		s_size, (ulong) s_max * sizeof(struct s_struct),
		v_size, (ulong) v_max * sizeof(struct v_struct), max_vertex,
		seg_size, (ulong) seg_max * sizeof(struct seg_struct),
		ss_size, (ulong) ss_max * sizeof(struct ss_struct),
		n_size, (ulong) n_max * sizeof(struct n_struct),
		sec_size, (ulong) sec_max * sizeof(struct sec_struct),
		r_size, b_size, blockmap->xsize, blockmap->ysize,
		freesize, largest, core,
		farfreesize, farlargest, farcore);

	window_text(msg, 1);
	while (!mouse_check())
		if (keypress)
			break;

	return;
}

void heaperr(char *func, char *name)
{
	mem_log("heap is corrupt (%s%s)", func, name);
	fatal_error("Heap is corrupt!!!");
}

void farheaperr(char *func, char *name)
{
	mem_log("farheap is corrupt (%s%s)", func, name);
	fatal_error("FarHeap is corrupt!!!");
}

void fatal_mem_error(char *name, uint size)
{
	struct heapinfo info;
	uint freesize, largest;

	if (heapcheck() == -1)
		heaperr("fatal_mem_error/", name);

	freesize = largest = core = coreleft();
	info.ptr = 0;
	while (heapwalk(&info) == 2)
	{
		if (!info.in_use)
		{
			freesize += info.size;
			if (info.size > largest)
				largest = info.size;
		}
	}

	mem_log("out of mem for %s(%u)\n"
		"mem=%u %u %u", name, size, freesize, largest, core);

	info.ptr = 0;
	mem_log("   Size   Status\n   ----   ------");
	while(heapwalk(&info) == _HEAPOK)
	{
		sprintf(path, "%7u    %s", info.size, info.in_use ? "used" : "free");
		mem_log(path);
	}

	fatal_error("Out of near memory!\n"
		"Could not allocate %u bytes for: %s\n"
		"%u free bytes of memory left\n"
		"Largest free block is %u bytes in size\n"
		"Coreleft = %u bytes",
		size, name, freesize, largest, core);
}

void fatal_farmem_error(char *name, ulong size)
{
	struct farheapinfo info;
	ulong freesize, largest;

	if (farheapcheck() == -1)
		farheaperr("fatal_farmem_error/", name);

	freesize = largest = farcore = farcoreleft();
	info.ptr = 0L;
	while (farheapwalk(&info) == 2)
	{
		if (!info.in_use)
		{
			freesize += info.size;
			if (info.size > largest)
				largest = info.size;
		}
	}

	mem_log("out of farmem for %s(%lu)\n"
		"mem=%lu %lu %lu", name, size, freesize, largest, farcore);

	fatal_error("Out of far memory!\n"
		"Could not allocate %lu bytes for: %s\n"
		"%lu free bytes of memory left\n"
		"Largest free block is %lu bytes in size\n"
		"Farcoreleft = %lu bytes",
		size, name, freesize, largest, farcore);
}
