/*
	This is a DMapEdit source code module.  Though it is copyrighted, you
	may modify it and use it for your own personal use, meaning that new
	modified code and anything derived from it (such as exe files) doesn't
	get distributed to anyone, unless you get my permission first.  Code
	from this file, or code based on ideas from this file may be used with
	other programs, provided that you give credit for it in the source code,
	documentation, and 'about' windows or screens, if one exists, for the
	programs using it.  Giving credit means something like this:

	Code from DMapEdit was used in this program

							  or

	Some code for this program was based on ideas presented in DMapEdit

	Whatever.  Just be sure to mention "DMapEdit" in such a way that it's
	self-evident how it was useful to the new program, and be sure to have
	"DMapEdit is a trademark of Jason Hoffoss" in the docs.  That's all..
*/

#include <stdio.h>
#include <string.h>
#include "dme.h"
#include "dme2.h"

int entry_num;

int open_wad(char *name, char *fmode)
{
	fp = fopen(name, fmode);
	if (!fp)
		return -1;

	if (verify_wad())
		fatal_error("WAD file corrupt");

	fread(&entry_count, 4, 1, fp);
	dir_size = entry_count;
	fread(&dir_ptr, 4, 1, fp);
	fseek(fp, dir_ptr, SEEK_SET);
	return entry_num = 0;
}

int open_wad_seek(char *fmode, char *entry)
{
	if (editing_pwad)
	{
		if (open_wad(pwad, fmode))
			fatal_error("open_wad_seek() can't find PWAD");
		if (!wad_seek(entry))
			return 0;
	}
	open_wad(iwad, fmode);
	return wad_seek(entry);
}

int open_wad_seekto(char *fmode, char *entry)
{
	if (open_wad_seek(fmode, entry))
		return -1;
	fseek(fp, entry_ptr, SEEK_SET);
	return 0;
}

int open_iwad_seek(char *fmode, char *entry)
{
	open_wad(iwad, fmode);
	return wad_seek(entry);
}

/* open pwad or iwad and position file pointer to map header */

void open_wad_seek_map(char *fmode, int which)
{
	static char header[5];

	sprintf(header, "E%dM%d", episode, mission);

	if (which)
		which = open_iwad_seek(fmode, header);
	else
		which = open_wad_seek(fmode, header);

	if (which)
		fatal_error("\"%s\" not found in IWAD", header);

	return;
}

int wad_seek(char *entry)
{
	struct
	{
		long ptr;
		long len;
		char fname[8];
	} dir;

	while (entry_count--)
	{
		if (!fread(&dir, sizeof(dir), 1, fp))
			dir_error();
		if (!cmp_entry(entry, dir.fname))
		{
			entry_ptr = dir.ptr;
			entry_len = dir.len;
			return 0;
		}
		entry_num++;
	}
	return -1;
}

void load_pwad_startup(void)
{
	char maps[27];
	int i, temp;

	struct
	{
		long ptr;
		long len;
		char fname[8];
	} dir;

	if (!editing_pwad)
	{
		load_wad_map(1);
		return;
	}

	for (i=0; i<27; i++)
		maps[i] = 0;

	if (open_wad(pwad, "rb"))
		fatal_error("Can't open \"%s\"", pwad);

	for (i=0; i<dir_size; i++)
	{
		if (!fread(&dir, sizeof(dir), 1, fp))
			dir_error();
		if (dir.len)
			continue;
		if (dir.fname[0] != 'E')
			continue;
		if (dir.fname[2] != 'M')
			continue;
		if (dir.fname[4])
			continue; /* at this point, we know it's a map header */
		temp = (dir.fname[1] - '1') * 9 + (dir.fname[3] - '1');
		if (temp > 26)
			dir_error(); /* just in case, cause you never know */
		maps[temp] = 1;
	}
	fclose(fp);

	if (!maps[episode*9 + mission])
		for (i=0; i<27; i++)
			if (maps[i])
			{
				episode = i / 9 + 1;
				mission = i % 9 + 1;
		}
	load_wad_map(0);
	return;
}

void load_wad_map(int which) /* load map info from wad file */
{
	int i;

	struct {
		long ptr;
		long len;
		char fname[8];
	} dir[10];

	clear_map();
	window_text("Loading map from WAD file, please wait..\n", 1);
	open_wad_seek_map("rb", which);
	fread(dir, 16, 10, fp);
	for (i=0; i<10; i++)
		if (cmp_entry(dir[i].fname, subfiles[i]))
			dir_error();

	things = get_farmem(dir[0].len, "Things");
	t_size = t_max = dir[0].len / sizeof(struct t_struct);
	fseek(fp, dir[0].ptr, SEEK_SET);
	if (far_read(things, dir[0].len))
		rd_error();

	linedefs = get_farmem(dir[1].len, "Linedefs");
	l_size = l_max = dir[1].len / sizeof(struct l_struct);
	fseek(fp, dir[1].ptr, SEEK_SET);
	if (far_read(linedefs, dir[1].len))
		rd_error();

	sidedefs = get_farmem(dir[2].len, "Sidedefs");
	s_size = s_max = dir[2].len / sizeof(struct s_struct);
	fseek(fp, dir[2].ptr, SEEK_SET);
	if (far_read(sidedefs, dir[2].len))
		rd_error();

	vertexes = get_farmem(dir[3].len, "Vertexes");
	v_size = v_max = dir[3].len / sizeof(struct v_struct);
	fseek(fp, dir[3].ptr, SEEK_SET);
	if (far_read(vertexes, dir[3].len))
		rd_error();

	segs = get_farmem(dir[4].len, "Segments");
	seg_size = seg_max = dir[4].len / sizeof(struct seg_struct);
	fseek(fp, dir[4].ptr, SEEK_SET);
	if (far_read(segs, dir[4].len))
		rd_error();

	ssectors = get_farmem(dir[5].len, "Sub Sectors");
	ss_size = ss_max = dir[5].len / sizeof(struct ss_struct);
	fseek(fp, dir[5].ptr, SEEK_SET);
	if (far_read(ssectors, dir[5].len))
		rd_error();

	nodes = get_farmem(dir[6].len, "Nodes");
	n_size = n_max = dir[6].len / sizeof(struct n_struct);
	fseek(fp, dir[6].ptr, SEEK_SET);
	if (far_read(nodes, dir[6].len))
		rd_error();

	sectors = get_farmem(dir[7].len, "Sectors");
	sec_size = sec_max = dir[7].len / sizeof(struct sec_struct);
	fseek(fp, dir[7].ptr, SEEK_SET);
	if (far_read(sectors, dir[7].len))
		rd_error();

	reject = get_farmem(r_size = dir[8].len, "Reject bitmap");
	fseek(fp, dir[8].ptr, SEEK_SET);
	if (far_read(reject, r_size))
		rd_error();

	blockmap = get_farmem(b_size = dir[9].len, "Blockmap");
	fseek(fp, dir[9].ptr, SEEK_SET);
	if (far_read(blockmap, b_size))
		rd_error();

	fclose(fp);
	center_map();
	return;
}

void save_to_wad(int get_name) /* save map info to a pwad file */
{
	char header[5], name[9];
	int i;

	struct {
		long ptr;
		long len;
		char fname[8];
	} dir[10];

	fix_wadname(pwad);
	strcat(pwad, ".WAD");
	if (get_name || !editing_pwad)
		if (get_wadname("Save", "to"))
			return;

	if (open_wad(pwad, "rb+"))
	{
		make_new_wad(); /* it's a new pwad */
		return;
	}

	sprintf(header, "E%dM%d", episode, mission);
	if (wad_seek(header))
	{
		remake_wad(1); /* it's a new map in existing pwad */
		return;
	}

	if (fread(dir, 16, 10, fp) != 10)
		rd_error();
	for (i=0; i<10; i++)
		if (cmp_entry(dir[i].fname, subfiles[i]))
			dir_error();

	if ((long) t_size * sizeof(things[0]) == dir[0].len &&
		(long) l_size * sizeof(linedefs[0]) == dir[1].len &&
		(long) s_size * sizeof(sidedefs[0]) == dir[2].len &&
		(long) v_size * sizeof(vertexes[0]) == dir[3].len &&
		(long) seg_size * sizeof(segs[0]) == dir[4].len &&
		(long) ss_size * sizeof(ssectors[0]) == dir[5].len &&
		(long) n_size * sizeof(nodes[0]) == dir[6].len &&
		(long) sec_size * sizeof(sectors[0]) == dir[7].len &&
		r_size == dir[8].len &&
		b_size == dir[9].len) /* size hasn't changed, just overwrite old */
	{
		window_text("Saving map to WAD file, please wait..\n", 1);
		fseek(fp, dir[0].ptr, SEEK_SET);
		if (far_write(things, dir[0].len))
			wr_error();

		fseek(fp, dir[1].ptr, SEEK_SET);
		if (far_write(linedefs, dir[1].len))
			wr_error();

		fseek(fp, dir[2].ptr, SEEK_SET);
		if (far_write(sidedefs, dir[2].len))
			wr_error();

		fseek(fp, dir[3].ptr, SEEK_SET);
		if (far_write(vertexes, dir[3].len))
			wr_error();

		fseek(fp, dir[4].ptr, SEEK_SET);
		if (far_write(segs, dir[4].len))
			wr_error();

		fseek(fp, dir[5].ptr, SEEK_SET);
		if (far_write(ssectors, dir[5].len))
			wr_error();

		fseek(fp, dir[6].ptr, SEEK_SET);
		if (far_write(nodes, dir[6].len))
			wr_error();

		fseek(fp, dir[7].ptr, SEEK_SET);
		if (far_write(sectors, dir[7].len))
			wr_error();

		fseek(fp, dir[8].ptr, SEEK_SET);
		if (far_write(reject, dir[8].len))
			wr_error();

		fseek(fp, dir[9].ptr, SEEK_SET);
		if (far_write(blockmap, dir[9].len))
			wr_error();

		rewind(fp);
		fputc('P', fp); /* this informs doom that this wad is modified */
		fclose(fp);
		draw_map();
		return;
	}
	remake_wad(0);
	return;
}

void make_new_wad(void)
{
	char buffer[256], *ident = "PWAD";
	int i, j;
	long offset;

	struct {
		long ptr;
		long len;
		char fname[8];
	} dir[11];

	sprintf(buffer,
		"\"%s\" does not exist!  Do you want to\n"
		"create this new WAD file?\n\n"
		"[ Yes ]\t", pwad);
	window_text(buffer, 1);
	i = window_check();
	draw_map();
	await_release();
	if (i != -2)
		return;

	window_text("Creating new WAD file, please wait..\n", 1);

	fp = fopen(pwad, "wb");
	if (!fp)
		fatal_error("Can't create PWAD");

	dir[1].len = (long) t_size * sizeof(things[0]);
	dir[2].len = (long) l_size * sizeof(linedefs[0]);
	dir[3].len = (long) s_size * sizeof(sidedefs[0]);
	dir[4].len = (long) v_size * sizeof(vertexes[0]);
	dir[5].len = (long) seg_size * sizeof(segs[0]);
	dir[6].len = (long) ss_size * sizeof(ssectors[0]);
	dir[7].len = (long) n_size * sizeof(nodes[0]);
	dir[8].len = (long) sec_size * sizeof(sectors[0]);
	dir[9].len = r_size;
	dir[10].len = b_size;
	offset = 12L;

	sprintf(buffer, "E%dM%d", episode, mission);
	buffer[5] = buffer[6] = buffer[7] = 0;
	for (i=0; i<8; i++)
		dir[0].fname[i] = buffer[i];
	dir[0].ptr = dir[0].len = 0L;
	for (i=1; i<11; i++)
	{
		for (j=0; j<8; j++)
			if (!(dir[i].fname[j] = subfiles[i-1][j]))
				break;
		while (j<8)
			dir[i].fname[j++] = 0;
		dir[i].ptr = offset;
		offset += dir[i].len;
	}

	if (!fwrite(ident, 4, 1, fp))
		goto error;
	if (fwrite_long(11L)) /* 11 entries in this new WAD file */
		goto error;
	if (!fwrite(&offset, 4, 1, fp))
		goto error;

	if (far_write(things, dir[1].len))
		goto error;
	if (far_write(linedefs, dir[2].len))
		goto error;
	if (far_write(sidedefs, dir[3].len))
		goto error;
	if (far_write(vertexes, dir[4].len))
		goto error;
	if (far_write(segs, dir[5].len))
		goto error;
	if (far_write(ssectors, dir[6].len))
		goto error;
	if (far_write(nodes, dir[7].len))
		goto error;
	if (far_write(sectors, dir[8].len))
		goto error;
	if (far_write(reject, r_size))
		goto error;
	if (far_write(blockmap, b_size))
		goto error;

	if (fwrite(dir, 16, 11, fp) != 11) /* write out directory structure */
		goto error;
	fclose(fp);
	draw_map();
	return;

error:
	fclose(fp);
	error("Failed on write to \"%s\"", pwad);
	draw_map();
	return;
}

int remake_wad(int add)
{
	char buffer[4096], name2[13], *ident = "PWAD";
	int i, j, cur_entry, xx, yy;
	long len, new_ptr;

	struct {
		long ptr;
		long len;
		char fname[8];
	} far *dir, temp;

	window_text("Rebuilding WAD file, please wait..\n\n\n\n", 1);
	yy = win.top + 19;
	xx = draw_time_graph(yy);

	fseek(fp, dir_ptr, SEEK_SET); /* position to start of dir structure */
	len = dir_size;
	dir = get_farmem(dir_size * 16, "Wad directory");
	for (i=0; i<len; i++) /* and get it */
	{
		if (!fread(&temp, 16, 1, fp))
		{
			free_farmem(dir, "Wad directory");
			dir_error();
		}
		dir[i] = temp;
	}
	if (add)
	{
   	entry_num = dir_size;
		dir_size += 11;
		sprintf(buffer, "E%dM%d", episode, mission);
		buffer[5] = buffer[6] = buffer[7] = 0;
		for (i=0; i<8; i++)
			dir[len].fname[i] = buffer[i];
		dir[len].ptr = dir[len].len = 0L;
		for (i=1; i<11; i++)
			for (j=0; j<8; j++)
				dir[len+i].fname[j] = subfiles[i-1][j];
	}

	fp2 = fopen("wad.dme", "wb"); /* make destination wad */
	if (!fp2)
	{
		error("Can't create backup WAD file");
		goto gen_error;
	}

	if (!fwrite(ident, 4, 1, fp2))
		goto write_error;
	if (!fwrite(&dir_size, 4, 1, fp2))
		goto write_error;
	if (!fwrite(&dir_ptr, 4, 1, fp2))
		goto write_error;

	new_ptr = 12; /* current file offset */
	for (cur_entry=0; cur_entry<dir_size; cur_entry++)
	{
		for (i=0; i<8; i++)
			if (!(buffer[i] = dir[cur_entry].fname[i]))
				break;
		while (i<8)
			buffer[i++] = 0;
		buffer[8] = 0;
		erase_text(win.left + 108, win.top + 34, 8);
		text(win.left + 140 - strlen(buffer)*4, win.top + 34, buffer);

		if (cur_entry == entry_num)
		{
			cur_entry++;

			len = (long) t_size * sizeof(things[0]);
			if (write_wad_entry("THINGS", things, len, dir[cur_entry].fname))
				goto gen_error;
			dir[cur_entry].ptr = new_ptr;
			dir[cur_entry].len = len;
			new_ptr += len;
			cur_entry++;

			len = (long) l_size * sizeof(linedefs[0]);
			if (write_wad_entry("LINEDEFS", linedefs, len, dir[cur_entry].fname))
				goto gen_error;
			dir[cur_entry].ptr = new_ptr;
			dir[cur_entry].len = len;
			new_ptr += len;
			cur_entry++;

			len = (long) s_size * sizeof(sidedefs[0]);
			if (write_wad_entry("SIDEDEFS", sidedefs, len, dir[cur_entry].fname))
				goto gen_error;
			dir[cur_entry].ptr = new_ptr;
			dir[cur_entry].len = len;
			new_ptr += len;
			cur_entry++;

			len = (long) v_size * sizeof(vertexes[0]);
			if (write_wad_entry("VERTEXES", vertexes, len, dir[cur_entry].fname))
				goto gen_error;
			dir[cur_entry].ptr = new_ptr;
			dir[cur_entry].len = len;
			new_ptr += len;
			cur_entry++;

			len = (long) seg_size * sizeof(segs[0]);
			if (write_wad_entry("SEGS", segs, len, dir[cur_entry].fname))
				goto gen_error;
			dir[cur_entry].ptr = new_ptr;
			dir[cur_entry].len = len;
			new_ptr += len;
			cur_entry++;

			len = (long) ss_size * sizeof(ssectors[0]);
			if (write_wad_entry("SSECTORS", ssectors, len, dir[cur_entry].fname))
				goto gen_error;
			dir[cur_entry].ptr = new_ptr;
			dir[cur_entry].len = len;
			new_ptr += len;
			cur_entry++;

			len = (long) n_size * sizeof(nodes[0]);
			if (write_wad_entry("NODES", nodes, len, dir[cur_entry].fname))
				goto gen_error;
			dir[cur_entry].ptr = new_ptr;
			dir[cur_entry].len = len;
			new_ptr += len;
			cur_entry++;

			len = (long) sec_size * sizeof(sectors[0]);
			if (write_wad_entry("SECTORS", sectors, len, dir[cur_entry].fname))
				goto gen_error;
			dir[cur_entry].ptr = new_ptr;
			dir[cur_entry].len = len;
			new_ptr += len;
			cur_entry++;

			if (write_wad_entry("REJECT", reject, (long) r_size,
				dir[cur_entry].fname))
					goto gen_error;
			dir[cur_entry].ptr = new_ptr;
			dir[cur_entry].len = r_size;
			new_ptr += r_size;
			cur_entry++;

			if (write_wad_entry("BLOCKMAP", blockmap, (long) b_size,
				dir[cur_entry].fname))
				goto gen_error;
			dir[cur_entry].ptr = new_ptr;
			dir[cur_entry].len = b_size;
			new_ptr += b_size;
			continue;
		}
		if (!dir[cur_entry].ptr || !dir[cur_entry].len)
			continue; /* not a real subfile; only a marker */
		fseek(fp, dir[cur_entry].ptr, SEEK_SET);
		len = dir[cur_entry].len;

		while (len > 4096) /* transfer subfile to new wad */
		{
			if (fread(buffer, 1, 4096, fp) != 4096)
			{
				free_farmem(dir, "Wad directory");
				rd_error();
			}
			if (fwrite(buffer, 1, 4096, fp2) != 4096)
				goto write_error;
			len -= 4096;
		}

		if (fread(buffer, 1, len, fp) != len)
		{
			free_farmem(dir, "Wad directory");
			rd_error();
		}
		if (fwrite(buffer, 1, len, fp2) != len)
			goto write_error;
		dir[cur_entry].ptr = new_ptr;
		new_ptr += dir[cur_entry].len;
		time_graph(cur_entry, dir_size, xx, yy);
	}
	fclose(fp);
	for (i=0; i<dir_size; i++)
	{
		temp = dir[i];
		if (!fwrite(&temp, 16, 1, fp2))
			goto write_error;
	}
	fclose(fp2);

	fp = fopen("wad.dme", "rb+");
	fseek(fp, 8L, SEEK_SET);
	if (!fwrite(&new_ptr, 4, 1, fp))
		goto write_error;
	fclose(fp);
	free_farmem(dir, "Wad directory");

	if (backup)
	{
		strcpy(buffer, pwad);
		strcpy(buffer+(strlen(buffer)-3), "bak"); /* change extention */
		unlink(buffer);
		rename(pwad, buffer);
	} else {
		unlink(pwad);
	}
	rename("wad.dme", pwad);
	unlink("wad.dme");

	draw_map();
	return 0;

write_error:
	error("Failed on write to backup WAD file");

gen_error:
	free_farmem(dir, "Wad directory");
	return -1;
}

void find_doom_wad(void)
{
	char header[5];
	int found, i, j, len;
	long dir_ptr;

	struct
	{
		long ptr;
		long len;
		char fname[8];
	} dir;

	len = strlen(iwad);
	while (len--)
		if (iwad[len] == '/')
			iwad[len] = '\\';

	if ((len = strlen(iwad)) && iwad[len-1] != ':')
	{
		if (iwad[len-1] != '\\')
		{
			if ((fp = fopen(iwad, "rb")) != NULL)
			{
				if (verify_wad())
					fatal_error("WAD file corrupt");
				return;
			}
			iwad[len++] = '\\';
			iwad[len] = 0;
		}
	}

	iwad[51] = 0;
	strcat(iwad, "doom.wad");
	if ((fp = fopen(iwad, "rb")) == NULL)
	{
		strcpy(iwad+(strlen(iwad)-4), "1.wad");
		if ((fp = fopen(iwad, "rb")) == NULL)
			fatal_error("Can't open Doom WAD file");
	}

	if (verify_wad())
		fatal_error("WAD file corrupt");

	fread(&entry_count, 4, 1, fp);
	dir_size = entry_count;
	fread(&dir_ptr, 4, 1, fp);
	fseek(fp, dir_ptr, SEEK_SET);

	sw_doom = 0;
	for (i=1; i<4; i++) /* to be totally sure, why not check them all? */
		for (j=1; j<10; j++) /* note we assume all headers are in order */
		{
			sprintf(header, "E%dM%d", i, j);
			if (wad_seek(header))
				sw_doom = 1;
		}
	fclose(fp);
	return;
}

int far_read(char huge *farptr, long len) /* read to a far location */
{
	char buffer[4096];
	int i;

	if (!len) return 0;
	while (len > 4096)
	{
		if (fread(buffer, 1, 4096, fp) != 4096)
			return -1;
		for (i=0; i<4096; i++, farptr++)
			*farptr = buffer[i];
		len -= 4096;
	}

	if (fread(buffer, 1, len, fp) != len)
		return -1;
	for (i=0; i<len; i++, farptr++)
		*farptr = buffer[i];
	return 0;
}

long fread_long(void)
{
	long integer;

	if (!fread(&integer, sizeof(integer), 1, fp))
		return -1;
	return integer;
}

int fwrite_long(long integer)
{
	if (!fwrite(&integer, 4, 1, fp))
		return -1;
	return 0;
}

int cmp_entry(char *look_for, char far *entry) /* compare wad entry name */
{ /* strcpy won't work, because if entry is 8 chars long, it won't end */
	int i; /* with a zero */

	for (i=0; i<8; i++)
	{
		if (!*look_for) return 0;
		if (*look_for >= 'a' && *look_for <= 'z')
			*look_for -= 32;
		if (*entry >= 'a' && *entry <= 'z')
			*entry -= 32;

		if (*look_for++ != *entry++) return 1;
	}
	return 0;
}

int verify_wad(void) /* make sure this is a wad file */
{
	char str[4];

	fread(&str, 1, 4, fp);
	if (str[0] != 'I' && str[0] != 'P' && str[0] != 'T') return 1;
	if (str[1] != 'W') return 2;
	if (str[2] != 'A') return 3;
	if (str[3] != 'D') return 4;
	return 0;
}
