;;; pm-win.el --- code specific to the OS/2 Presentation Manager

;; Copyright (C) 1993 Eberhard Mattes

;; Author: Eberhard Mattes <mattes@azu.informatik.uni-stuttgart.de>
;; Keywords: emx

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

(if (not (eq window-system 'pm))
    (error "Loading pm-win.el but not compiled for PM"))

(require 'frame)
(require 'mouse)
(require 'faces)
(require 'lucid)

(emacs-priority 'foreground-server)
(setq min-skip-run 10)

(setq pm-color-alist
      '(
        ;;
        ;; OS/2 colors
        ;;
        ("black"         . [  0   0   0])
        ("blue"          . [  0   0 255])
        ("green"         . [  0 255   0])
        ("cyan"          . [  0 255 255])
        ("red"           . [255   0   0])
        ("pink"          . [255   0 255])
        ("yellow"        . [255 255   0])
        ("white"         . [255 255 255])
        ("darkblue"      . [  0   0 127])
        ("darkgreen"     . [  0 127   0])
        ("darkcyan"      . [  0 127 127])
        ("darkred"       . [127   0   0])
        ("darkpink"      . [127   0 127])
        ("brown"         . [127 127   0])
        ("darkgray"      . [ 85  85  85])
        ("palegray"      . [127 127 127])
        ;;
        ;; Some X colors
        ;;
        ("gray"          . [170 170 170]) ; region
        ("darkseagreen2" . [180 238 180]) ; highlight
        ("seagreen2"     . [ 78 238 148])
        ("forestgreen"   . [34  139  34])
        ("royalblue"     . [65  105 225])
        ("skyblue"       . [135 206 235])
        ("lightblue"     . [173 216 230])
        ("purple"        . [160  32 240])
        ("firebrick"     . [178  34  34])
        ("magenta"       . [255   0 255])
        ("darkgoldenrod" . [184 134  11])
        ("olivedrab2"    . [179 238  58])
        ("moccasin"      . [255 228 181])
        ("plum1"         . [255 187 255])
        ("orange"        . [255 165   0])
        ("goldenrod"     . [218 165  32])
        ("tan"           . [210 180 140])
        ("grey40"        . [102 102 102])
        ("grey50"        . [127 127 127])
        ))

(setq command-switch-alist
      (append '(("-d"    .      pm-handle-display)
		("-name" .      pm-handle-switch)
		("-T"    .      pm-handle-switch)
		("-fn"   .      pm-handle-switch)
		("-font" .      pm-handle-switch)
		("-fg" .	pm-handle-switch)
		("-foreground".	pm-handle-switch)
		("-bg" .	pm-handle-switch)
		("-background".	pm-handle-switch))
              command-switch-alist))

(defconst pm-switch-definitions
  '(("-name" name)
    ("-T" name)
    ("-fn" font)
    ("-font" font)
    ("-fg" foreground-color)
    ("-foreground" foreground-color)
    ("-bg" background-color)
    ("-background" background-color)))

(pm-open-connection)

(setq suspend-hook
      '(lambda ()
	 (error "Suspending an Emacs running under PM isn't a good idea")))

;; Note: does *not* take an argument
(defun pm-handle-display (switch))

;; Handler for switches of the form "-switch"
(defun pm-handle-switch (switch)
  (let ((aelt (assoc switch pm-switch-definitions)))
    (if aelt
        (setq default-frame-alist
              (cons (cons (nth 1 aelt)
                          (car pm-invocation-args))
                    default-frame-alist)
              pm-invocation-args (cdr pm-invocation-args)))))

(defun pm-handle-args (args)
  "Here the PM-related command line arguments are processed."
  (setq pm-invocation-args args
	args nil)
  (while pm-invocation-args
    (let* ((this-switch (car pm-invocation-args))
	   (aelt (assoc this-switch command-switch-alist)))
      (setq pm-invocation-args (cdr pm-invocation-args))
      (if aelt
	  (funcall (cdr aelt) this-switch)
	(setq args (cons this-switch args)))))
  (setq args (nreverse args)))

(setq command-line-args (pm-handle-args command-line-args))

(defun pm-paste ()
  "Paste text from clipboard into buffer at point."
  (interactive "*")
  (insert (pm-get-clipboard)))

(defun pm-copy (beg end)
  "Copy the region to the clipboard.\n\
Note that several programs cannot handle more than 64KB of text in the\n\
clipboard.  Therefore you should avoid copying more than 64KB of text."
  (interactive "r")
  (pm-put-clipboard (buffer-substring beg end)))

(defun pm-cut (beg end)
  "Kill between point and mark and place the killed text in the clipboard.\n\
Note that several programs cannot handle more than 64KB of text in the\n\
clipboard.  Therefore you should avoid killing more than 64KB of text."
  (interactive "*r")
  (pm-copy beg end)
  (delete-region beg end))

(defun pm-close-frame ()
  "Close the current frame or kill Emacs, if this is the last frame."
  (interactive)
  (if (= (length (frame-list)) 1)
      (save-buffers-kill-emacs)
    (delete-frame)))

(setq split-window-keep-point t)
(setq frame-creation-function 'x-create-frame-with-faces)

(define-key global-map [S-left]      'scroll-right)
(define-key global-map [S-right]     'scroll-left)
(define-key global-map [S-insert]    'pm-paste)
(define-key global-map [S-delete]    'pm-cut)
(define-key global-map [C-insert]    'pm-copy)
(define-key global-map [close-frame] 'pm-close-frame)

; This is easier to type than C-_ (doesn't need the SHIFT key)
(define-key global-map [?\C--]       'undo)

; Put this on a more convenient key
(define-key global-map [?\C-.]       'repeat-complex-command)

; Disable extended scan codes, re-enable C-@
(define-key global-map "\C-@" 'set-mark-command)

(defun pm-defined-colors ()
  "Return a list of colors."
  (let ((all-colors pm-color-alist)
        defined-colors this-color)
    (while all-colors
      (setq this-color (car all-colors)
	    all-colors (cdr all-colors))
      (setq defined-colors (cons (car this-color) defined-colors)))
    defined-colors))

;;
;; Font menu
;;
(setq x-fixed-font-alist
  '("Font menu"
    ("Small"
     ("6pt" "6.Small"))
    ("System Monospaced"
     ("8pt" "8.System Monospaced")
     ("10pt" "10.System Monospaced"))
    ("System VIO"
     ("2pt" "2.System VIO")
     ("3pt" "3.System VIO")
     ("4pt" "4.System VIO")
     ("5pt" "5.System VIO")
     ("6pt" "6.System VIO")
     ("7pt" "7.System VIO")
     ("8pt" "8.System VIO")
     ("9pt" "9.System VIO")
     ("10pt" "10.System VIO")
     ("11pt" "11.System VIO")
     ("12pt" "12.System VIO")
     ("13pt" "13.System VIO")
     ("14pt" "14.System VIO")
     ("15pt" "15.System VIO")
     ("16pt" "16.System VIO")
     ("17pt" "17.System VIO")
     ("18pt" "18.System VIO"))
    ("Courier"
     ("8pt" "8.Courier")
     ("10pt" "10.Courier")
     ("12pt" "12.Courier"))
    ))

;;
;; Update a submenu of the menubar.  This is called asynchronously.
;; pm-menu-bar-menu MUST be called, otherwise the Presentation Manager
;; will hang.
;;
(defun pm-menu-bar-update (frame number)
  (condition-case err
      (let ((menu (nth number (frame-menu-bar-items frame)))
            object)
        (if menu
            (setq object
                  (key-binding (vector 'menu-bar (car menu)))))
        (cond ((keymapp object)
               (pm-menu-bar-menu frame (car menu) object))
              ((eq object 'mouse-menu-bar-buffers)
               (pm-menu-bar-buffers frame menu))
              (t (pm-menu-bar-menu frame nil nil))))
    (error (pm-menu-bar-menu frame nil nil)
           (message "%s" err))))

;;
;; This function is called to create the PM menu for
;; mouse-menu-bar-menu.
;;
(defun pm-menu-bar-buffers (frame arg)
  (let ((buffers (nreverse (buffer-list)))
        (menu (make-sparse-keymap))
        (index 1)
        entry elt sym)
    ;; If requested, list only the N most recently selected buffers.
    (if (and (integerp buffers-menu-max-size)
	     (> buffers-menu-max-size 1))
	(if (> (length buffers) buffers-menu-max-size)
	    (setcdr (nthcdr buffers-menu-max-size buffers) nil)))
    (define-key global-map (vector 'menu-bar 'pm-mbb) menu)
    (define-key menu [pm-mbb-list] '("List All Buffers" . list-buffers))
    (while buffers
      (setq elt (car buffers))
      (if (string-match "^ " (buffer-name elt))
          nil
        (setq entry (format "%-14s   %s"
                            (buffer-name elt)
                            (or (buffer-file-name elt) "")))
        (setq sym  (intern (format "pm-mbb-%d" index)))
        (fset sym (list 'lambda nil "Select a buffer." '(interactive)
                        (list 'switch-to-buffer elt)))
        (define-key menu (vector sym)
          (cons entry sym))
        (setq index (1+ index)))
      (setq buffers (cdr buffers)))
    (pm-menu-bar-menu frame 'pm-mbb menu)))

;;
;; Stolen from menu-bar.el (menu-bar-mode)
;;
(defvar pm-menu-bar-mode nil)

(defun pm-menu-bar-mode (flag)
  "Toggle display of a PM menu bar on each frame.
This command applies to all frames that exist and frames to be
created in the future.
With a numeric argument, if the argument is negative,
turn off menu bars; otherwise, turn on menu bars."
  (interactive "P")
  (setq pm-menu-bar-mode (if (null flag) (not pm-menu-bar-mode)
                           (or (not (numberp flag)) (>= flag 0))))
  (let ((parameter (assq 'pm-menu-bar default-frame-alist)))
    (if (consp parameter)
	(setcdr parameter pm-menu-bar-mode)
      (setq default-frame-alist
	    (cons (cons 'pm-menu-bar pm-menu-bar-mode)
		  default-frame-alist))))
  (let ((frames (frame-list)))
    (while frames
      ;; Turn menu bar on or off in existing frames.
      ;; (Except for minibuffer-only frames.)
      (or (eq 'only (cdr (assq 'minibuffer (frame-parameters (car frames)))))
	  (modify-frame-parameters
	   (car frames)
	   (list (cons 'pm-menu-bar pm-menu-bar-mode))))
      (setq frames (cdr frames)))))

(require 'menu-bar)

(defun pm-edit-menu ()
  "Modify the Edit menu of the menu bar to use the clipboard."
  (define-key menu-bar-edit-menu [paste] '("Paste" . pm-paste))
  (define-key menu-bar-edit-menu [copy] '("Copy" . pm-copy))
  (define-key menu-bar-edit-menu [cut] '("Cut" . pm-cut))
  (define-key menu-bar-edit-menu [choose-selection] nil)
  (put 'pm-copy 'menu-enable 'mark-active)
  (put 'pm-cut 'menu-enable 'mark-active)
  (put 'pm-paste 'menu-enable '(pm-clipboard-ready-p))
  nil)

;;
;; Make the Spell choice of the Edit menu work with the PM menu bar
;;
;; Copied from ispell.el
;;
(setq ispell-menu-map (make-sparse-keymap "Spell"))
(define-key ispell-menu-map [reload-ispell]
  '("Reload Dictionary" . reload-ispell))
(define-key ispell-menu-map [ispell-next]
  '("Continue Check" . ispell-next))
(define-key ispell-menu-map [ispell-region]
  '("Check Region" . ispell-region))
(define-key ispell-menu-map [ispell-buffer]
  '("Check Buffer" . ispell))
(define-key ispell-menu-map [ispell-word]
  '("Check Word" . ispell-word))

(define-key menu-bar-edit-menu [spell]
  (cons "@Spell..." ispell-menu-map))

(defun x-get-resource (attribute class &optional component subclass)
  nil)

(defalias 'x-create-frame 'pm-create-frame)
(defalias 'x-defined-color 'pm-defined-color)
(defalias 'x-display-color-p 'pm-display-color-p)
(defalias 'x-display-planes 'pm-display-planes)
(defalias 'x-defined-colors 'pm-defined-colors)
(defalias 'x-list-fonts 'pm-list-fonts)
