// File    : DPMIEXCP.H
// Author  : Eric Woodruff,  CIS ID: 72134,1150
// Updated : Mon 03/06/95 20:39:31
// Note    : Copyright 1995, Eric Woodruff, All rights reserved
// Compiler: Borland 4.xx
//
// This is the class declaration for the custom exception handler.
// The C version also uses this header file (conditional compilation).
//

#if !defined(__DPMIEXCP_H)
#define __DPMIEXCP_H

// A structure to hold the separate parts of a selector:offset pair.
// The integers will be sized appropriately (16-bits or 32-bits).
struct DPMISelectorOffset
{
    int Offset;
    int Selector;
};

#if !defined(__FLAT__)

// 16-bit exception handler data structure

// This is how the stack looks upon entry to the custom handler.  It receives
// a pointer to this structure.  Modifications to it will be made directly to
// the stack image.  Thus, altering the FaultCSIP so that it points to
// a "Clean up and Exit function" you can attempt a more graceful exit
// from the application.
struct DPMIExceptionData
{
    short DI;               // PUSHA pushes these.
    short SI;
    short BP;
    short originalSP;       // PUSHA pushes original SP which can be ignored.
    short BX;
    short DX;
    short CX;
    short AX;

    short DS;               // Saved by the calling dispatch function.
    short ES;

    short ExceptionNumber;  // Pushed by the entry point so the dispatch code
                            // knows which function pointer to use.

    // Return address to the exception handler caller.
    // NEVER MODIFY THE RETURN ADDRESS STORED HERE!!  It is here for
    // completeness and proper structure layout only!
    union {
        void *CS_IP;
        struct DPMISelectorOffset IP_CS;
    } ReturnCSIP;

    short ErrorCode;        // Error code value for faults

    union {
        void *CS_IP;        // Pointer to the faulting instruction
        struct DPMISelectorOffset IP_CS;
    } FaultCSIP;

    short Flags;            // CPU flags

    union {
        void *SS_SP;        // Stack pointer
        struct DPMISelectorOffset SP_SS;
    } StackSSSP;
};

#else

// 32-bit exception handler data structure

// This is how the stack looks upon entry to the custom handler.  It receives
// a pointer to this structure.  Modifications to it will be made directly to
// the stack image.  Thus, altering the FaultCSEIP so that it points to
// a "Clean up and Exit function" you can attempt a more graceful exit
// from the application.
struct DPMIExceptionData
{
    int EDI;                // PUSHAD pushes these.
    int ESI;
    int EBP;
    int originalESP;        // PUSHAD pushes original ESP which can be ignored.
    int EBX;
    int EDX;
    int ECX;
    int EAX;

    int DS;                 // Saved by the calling dispatch function.
    int ES;

    int ExceptionNumber;    // Pushed by the entry point so the dispatch code
                            // knows which function pointer to use.

    // Return address to the exception handler caller.
    // NEVER MODIFY THE RETURN ADDRESS STORED HERE!!  It is here for
    // completeness and proper structure layout only!
    union {
        void *CS_EIP;
        struct DPMISelectorOffset EIP_CS;
    } ReturnCSEIP;

    int ErrorCode;          // Error code value for faults

    union {
        void *CS_EIP;       // Pointer to the faulting instruction
        struct DPMISelectorOffset EIP_CS;
    } FaultCSEIP;

    int Flags;              // CPU flags

    union {
        void *SS_ESP;       // Stack pointer
        struct DPMISelectorOffset ESP_SS;
    } StackSSESP;

    int CR2;                // 80386 Control Register 2.  For page faults,
                            // it contains the linear address that caused
                            // the exception.
};

#endif

// Typedef for a custom exception handler function and the recovery function.
typedef int (*DPMICustomHandler)(struct DPMIExceptionData *);

// Typedef for the "clean up and exit" function
typedef void (*ExceptionCleanUp)(int , struct DPMIExceptionData *);

// Protoype for the default recovery function.  Note that it has the same
// form as the exception handler function.  That's only because it is passed
// the exception data structure and also returns an 'int'.  Since exception
// recovery may change from application to application, all you need to do
// is point to the new recovery function instead of altering the exception
// handler code.
int DefaultRecovery(struct DPMIExceptionData *ex);

// Prototype for the default clean up function.
void DefaultCleanUp(int ExitType, struct DPMIExceptionData *ex);

#if !defined(__WINDOWS_H)
#include <windows.h>            // For HGLOBAL
#endif

#if !defined(_HUGE)
    #if defined(__DPMI16__)
        #define _HUGE   huge
    #else
        #define _HUGE
    #endif
#endif

#ifdef __cplusplus

//
// DPMI Exception Handler Class
//
class DPMIExceptionHandler
{
private:
    // An array of pointers to the custom handler functions for
    // exceptions 00h through 10h.  Note that the 32-bit version is
    // doubled in size because we have to store two 32-bit values (CS:EIP)
    // per handler.  CS is 16 bits, but we can't store 48 bit pointers and
    // must use the full 64 bits.
#if !defined(__FLAT__)
    static DPMICustomHandler customFuncs[17];
#else
    static DPMICustomHandler customFuncs[34];
#endif

    // Flag to signal when the exceptions have been trapped.
    static int exceptionsTrapped;

    // Handle and pointer to the custom handler's stack.  The constructor
    // allocates memory for it with GlobalAlloc().  A different stack is
    // used because the SS:(E)SP upon entry may not be valid.
    static HGLOBAL     stackHandle;
    static char _HUGE *stackPointer;

    // An area to store the exception data.  Due to the nature of
    // 32-bit protected mode, there is no guarantee that DS=ES=SS when
    // the exception dispatch handler receives control.  To make life
    // simple, it returns control to an intermediate routine after
    // copying the exception data into this static structure.
    static DPMIExceptionData exceptionInfo;

public:
    // Flag to signal that the exception handler has already been
    // called and that any further exceptions should be routed to
    // the original handler (i.e. the exception handler has caused
    // an exception).
    // This can be checked to see if the exception handler has been triggered.
    // It is useful for event handlers and other such code that may do
    // "background" processing while waiting for an event.  If an exception
    // occurs, it is probably unsafe to call the background code.  This
    // can be checked to see if an exception has occurred and, if so,
    // skip the processing.
    static int inExceptionHandler;

    // A pointer to the recovery function.
    static DPMICustomHandler RecoveryFunction;

    // A pointer to the "clean up and exit" function.
    static ExceptionCleanUp CleanUpFunction;

    DPMIExceptionHandler(void);
    ~DPMIExceptionHandler(void);

    // These trap and release the exceptions.  If you shell to DOS or
    // spawn other applications, be sure to release the exceptions
    // prior to the call and trap them again afterwards!
    static void trapExceptions(void);
    static void releaseExceptions(void);

    // Set a custom exception handler.
    static int setExceptionHandler(int Exception, DPMICustomHandler Function);

    // Get the current address of the custom exception handler.
    static const DPMICustomHandler getExceptionHandler(int Exception);
};

extern "C" {
    // A small function to obtain the limit for a segment selector.
    int GetSegmentLimit(int Selector, int *Limit);
}

#else   // ifdef __cplusplus

//
// DPMI Exception Handler Functions - C Version
//
    // Flag to signal that the exception handler has already been
    // triggered and further exceptions should go to the original handler.
    extern int inExceptionHandler;

    // A pointer to the recovery function.
    extern DPMICustomHandler RecoveryFunction;

    // A pointer to the "clean up and exit" function.
    extern ExceptionCleanUp CleanUpFunction;

    void DPMIExceptionHandlerCtor(void);
    void DPMIExceptionHandlerDtor(void);

    // These trap and release the exceptions.  If you shell to DOS or
    // spawn other applications, be sure to release the exceptions
    // prior to the call and trap them again afterwards!
    void trapExceptions(void);
    void releaseExceptions(void);

    // Set a custom exception handler.
    int setExceptionHandler(int Exception, DPMICustomHandler Function);

    // Get the current address of the custom exception handler.
    const DPMICustomHandler getExceptionHandler(int Exception);

    // A small function to obtain the limit for a segment selector.
    int GetSegmentLimit(int Selector, int *Limit);

#endif  // ifdef __cplusplus

#endif  // __DPMIEXCP_H
