// File    : EXCPHND.CPP
// Author  : Eric Woodruff,  CIS ID: 72134,1150
// Updated : Mon 03/06/95 20:39:31
// Note    : Copyright 1995, Eric Woodruff, All rights reserved
// Compiler: Borland 4.xx
//
// C++ version of the custom exception handler (no UI library)
//

#include <conio.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <excphnd.h>

// Replace the standard library abort() function in an attempt to
// exit cleanly.
void abort(void)
{
    puts("\n\nAn attempt was made to abnormally terminate the program.");
    puts("This may be due to a low memory condition or an exception.\n");
    puts("Press any key to clean up and shutdown...\n\n");
    getch();

    // Force a graceful exit.
    DPMIExceptionHandler::releaseExceptions();

    // Act like an abort function() by returning 3.  However, instead of
    // calling _exit() it calls exit() (no leading underscore).  Buffers
    // are flushed, files are closed, global objects are destroyed, and
    // functions registered with atexit() are called.
    exit(3);
}

// A generic exception handler for use with any application.
int AppExceptionHandler(DPMIExceptionData *ex)
{
    char *Options, choice = ' ';

    // Display the exception data and get a list of valid responses.
    Options = DisplayExceptionData(ex);

    while(!strchr(Options, choice))
    {
        choice = toupper(getch());

        if(strchr(Options, choice))
            switch(choice)
            {
                // Attempt to recover from the exception.
                case 'A':
                    if(!DPMIExceptionHandler::RecoveryFunction(ex))
                    {
                        puts("\nUnable to recover from the exception."
                            "  Please exit the application.");
                        choice = ' ';
                    }
                    break;

                // Clean up and exit via a user specified function.
                case 'E':
                    DPMIExceptionHandler::CleanUpFunction(1, ex);

                    // Always release the exception handlers prior to exit.
                    DPMIExceptionHandler::releaseExceptions();
                    break;

                // Clean up and exit via the original handler.
                case 'H':
                    DPMIExceptionHandler::CleanUpFunction(0, ex);

                    // Always release the exception handlers prior to exit.
                    DPMIExceptionHandler::releaseExceptions();
                    break;
            }
    }

    if(choice == 'H')
        return 0;       // Exit via old handler.

    // Exit without going through the old handler or return after recovery.
    return 1;
}

#if !defined(__FLAT__)

// 16-bit exception handler dialog functions.

// ***************************************************************************

char *DisplayExceptionData(DPMIExceptionData *ex)
{
    static char  *ExName[] = {
        "Divide Error",
        "Debug",
        "NMI",
        "Breakpoint",
        "Overflow (INTO)",
        "Bounds Check (BOUND)",
        "Invalid Op-Code",
        "Coprocessor Unavailable",
        "Double Fault",
        "Coprocessor Segment Overrun",
        "Invalid Task State Segment",
        "Segment Not Present",
        "Stack Fault",
        "General Protection Fault (GPF)",
        "Page Fault",
        "???",                  // 0x0F is marked as reserved.
        "Coprocessor Error"
    };

    static char Options[3];     // Valid options.

    unsigned char *CSIP;

    int ExType, CSLimit, CSGood, DSLimit, DSGood, ESLimit, ESGood,
        SSLimit, SSGood, i, j;

    puts("--------------------------------------------------------------------");
    printf("Exception #%02X ", ex->ExceptionNumber);

    switch(ex->ExceptionNumber)
    {
        case 0x08:
        case 0x09:
        case 0x0F:
            printf("(Abort) - %s\n", ExName[ex->ExceptionNumber]);
            ExType = 0;
            break;

        case 0x00:
        case 0x05:
        case 0x06:
        case 0x07:
        case 0x0A:
        case 0x0B:
        case 0x0C:
        case 0x0D:
        case 0x0E:
        case 0x10:
            printf("(Fault) - %s\n", ExName[ex->ExceptionNumber]);
            ExType = 1;
            break;

        case 0x01:      // 0x01 may also be a fault
        case 0x02:
        case 0x03:
        case 0x04:
            printf("(Trap) - %s\n", ExName[ex->ExceptionNumber]);
            ExType = 2;
            break;

        default:
            puts("(???) - ???");       // Not one I've expected.
            ExType = 0;
            break;
    }

    printf("\nAX=%04X  BX=%04X  CX=%04X  DX=%04X"
           "\nSI=%04X  DI=%04X  BP=%04X  SP=%04X\n",
        ex->AX, ex->BX, ex->CX, ex->DX,
        ex->SI, ex->DI, ex->BP, ex->StackSSSP.SP_SS.Offset);

    // Determine the segment limits and their validity.
    CSGood = GetSegmentLimit(ex->FaultCSIP.IP_CS.Selector, &CSLimit);
    DSGood = GetSegmentLimit(ex->DS, &DSLimit);
    ESGood = GetSegmentLimit(ex->ES, &ESLimit);
    SSGood = GetSegmentLimit(ex->StackSSSP.SP_SS.Selector, &SSLimit);

    printf("\nCS=%04X   Limit=", ex->FaultCSIP.IP_CS.Selector);
    if(CSGood)
        printf("%04X", CSLimit);
    else
        printf("Invalid segment");

    printf("\nDS=%04X   Limit=", ex->DS);
    if(DSGood)
        printf("%04X", DSLimit);
    else
        printf("Invalid segment");

    printf("\nES=%04X   Limit=", ex->ES);
    if(ESGood)
        printf("%04X", ESLimit);
    else
        printf("Invalid segment");

    printf("\nSS=%04X   Limit=", ex->StackSSSP.SP_SS.Selector);
    if(SSGood)
        printf("%04X", SSLimit);
    else
        printf("Invalid segment");

    // Get the fault or trap bytes at CS:IP.
    if(ExType == 1 || ExType == 2)
    {
        printf("\n\nFault Information (CS:IP):  %04X:%04X -",
            ex->FaultCSIP.IP_CS.Selector, ex->FaultCSIP.IP_CS.Offset);

        CSIP = (unsigned char *)ex->FaultCSIP.CS_IP;

        for(short i = 0; i < 8; i++, CSIP++)
            printf(" %02X", *CSIP);
    }

    printf("\n\nFlags: C - P - A - Z S T I D O ioplN\n       ");
    for(i = j = 1; i < 16; i++, j <<= 1)
        printf("%c ", (ex->Flags & j) ? '1' : '0');

    puts("\n");

    // Error code info is only valid for faults 0A-0D.  Error code is
    // always zero for 08.
    if(ex->ExceptionNumber > 9 && ex->ExceptionNumber < 0x0E)
    {
        // The upper 14 bits are the segment selector involved.
        printf("\nSegment Selector (Faults 0A-0D)=%04X\n",
            ex->ErrorCode & 0xFFFC);

        // If error code is zero for a fault, it could be due
        // to a NULL selector (doesn't apply to 0C).
        if(!ex->ErrorCode && ex->ExceptionNumber != 0x0C)
            puts("Possible NULL selector\n");
        else
        {
            // Was it caused by an external event?
            if(ex->ErrorCode & 0x01)
                puts("Caused by an external event");

            // Was it for a gate descriptor in the IDT?
            if(ex->ErrorCode & 0x02)
                puts("Refers to an IDT gate descriptor\n");
            else
                if(ex->ErrorCode & 0x04)    // Was it for the LDT?
                    puts("Refers to the LDT\n");
                else
                    puts("Refers to the GDT\n");
        }
    }
    else
        if(ex->ExceptionNumber == 0x0E)
        {
            // Page fault exception.  The error code is different for
            // a page fault.

            puts("\nPage Fault (0E) Information");

            // Was it due to a page-level protection violation?
            // If not, it was due to a not-present page.
            if(ex->ErrorCode & 0x01)
                printf("Page-level protection violation ");
            else
                printf("Not-present page ");

            // Was it due to a write access?
            // If not, it was due to a read access.
            if(ex->ErrorCode & 0x02)
                puts("caused by a write access");
            else
                puts("caused by a read access");

            // Did the fault occur while the CPU was in user mode?
            // If not, it occurred in supervisor mode.
            if(ex->ErrorCode & 0x04)
                puts("Processor was in user mode\n");
            else
                puts("Processor was in supervisor mode\n");
        }

    Options[0] = '\x0';

    // Can't recover on an Abort exception.  The system is probably unstable.
    // Also disable it if there is no recovery function defined.
    if(ExType && DPMIExceptionHandler::RecoveryFunction)
    {
        printf("<A>ttempt Recovery  |  ");
        strcpy(Options, "A");
    }

    printf("<E>xit Application  ");
    strcat(Options, "E");

    // Overflow exceptions don't seem to work properly if you exit
    // via the old handler.  Checking the code at the original vector
    // shows HLT instructions (?).  Recovery is easy enough, so it's best
    // to either ignore it or just exit the application.
    //
    // Chaining to the original vector will not work for exception 5
    // because it will reflect it as an interrupt (INT 5 = Print Screen)
    // and it gets stuck in an endless loop.  A regular press of the
    // Print Screen key will *NOT* invoke the exception handler though.
    //
    if(ex->ExceptionNumber != 4 && ex->ExceptionNumber != 5)
    {
        printf("|  Exit via old <H>andler");
        strcat(Options, "H");
    }

    puts("\n--------------------------------------------------------------------");

    return Options;
}

// ***************************************************************************

#else

// 32-bit exception handler.

char *DisplayExceptionData(DPMIExceptionData *ex)
{
    static char  *ExName[] = {
        "Divide Error",
        "Debug",
        "NMI",
        "Breakpoint",
        "Overflow (INTO)",
        "Bounds Check (BOUND)",
        "Invalid Op-Code",
        "Coprocessor Unavailable",
        "Double Fault",
        "Coprocessor Segment Overrun",
        "Invalid Task State Segment",
        "Segment Not Present",
        "Stack Fault",
        "General Protection Fault (GPF)",
        "Page Fault",
        "???",                  // 0x0F is marked as reserved.
        "Coprocessor Error"
    };

    static char Options[3];     // Valid options.

    unsigned char tempData[10];

    int ExType, CSLimit, CSGood, DSLimit, DSGood, ESLimit, ESGood,
        SSLimit, SSGood, i, j;

    puts("--------------------------------------------------------------------");
    printf("Exception #%02X ", ex->ExceptionNumber);

    switch(ex->ExceptionNumber)
    {
        case 0x08:
        case 0x09:
        case 0x0F:
            printf("(Abort) - %s\n", ExName[ex->ExceptionNumber]);
            ExType = 0;
            break;

        case 0x00:
        case 0x05:
        case 0x06:
        case 0x07:
        case 0x0A:
        case 0x0B:
        case 0x0C:
        case 0x0D:
        case 0x0E:
        case 0x10:
            printf("(Fault) - %s\n", ExName[ex->ExceptionNumber]);
            ExType = 1;
            break;

        case 0x01:      // 0x01 may also be a fault
        case 0x02:
        case 0x03:
        case 0x04:
            printf("(Trap) - %s\n", ExName[ex->ExceptionNumber]);
            ExType = 2;
            break;

        default:
            puts("(???) - ???");       // Not one I've expected.
            ExType = 0;
            break;
    }

    printf("\nEAX=%08X  EBX=%08X  ECX=%08x  EDX=%08X"
           "\nESI=%08X  EDI=%08X  EBP=%08X  ESP=%08X",
           ex->EAX, ex->EBX, ex->ECX, ex->EDX,
           ex->ESI, ex->EDI, ex->EBP, ex->StackSSESP.ESP_SS.Offset);

    // Determine the segment limits and their validity.
    CSGood = GetSegmentLimit(ex->FaultCSEIP.EIP_CS.Selector, &CSLimit);
    DSGood = GetSegmentLimit(ex->DS, &DSLimit);
    ESGood = GetSegmentLimit(ex->ES, &ESLimit);
    SSGood = GetSegmentLimit(ex->StackSSESP.ESP_SS.Selector, &SSLimit);

    printf("\nCS=%04X   Limit=", ex->FaultCSEIP.EIP_CS.Selector);
    if(CSGood)
        printf("%08X", CSLimit);
    else
        printf("Invalid segment");

    printf("\nDS=%04X   Limit=", ex->DS);
    if(DSGood)
        printf("%08X", DSLimit);
    else
        printf("Invalid segment");

    printf("\nES=%04X   Limit=", ex->ES);
    if(ESGood)
        printf("%08X", ESLimit);
    else
        printf("Invalid segment");

    printf("\nSS=%04X   Limit=", ex->StackSSESP.ESP_SS.Selector);
    if(SSGood)
        printf("%08X", SSLimit);
    else
        printf("Invalid segment");

    // Get the fault or trap bytes at CS:IP.
    if(ExType == 1 || ExType == 2)
    {
        // If the exception didn't happen in our code, the selectors
        // might not be the same.  Move the eight bytes to a temporary
        // buffer before formatting them.

        // The BCC32 generated assembler file for this won't compile so
        // __emit__() statements are used instead.

        _EBX = ex->FaultCSEIP.EIP_CS.Selector;
        _ESI = ex->FaultCSEIP.EIP_CS.Offset;
        _EDI = (unsigned long)&tempData[0];

        __emit__(0x50);     // push     eax    BCC32 generates code that uses
                            //                 EAX and it doesn't know about
                            //                 the emitted lodsd's below.

        __emit__(0x1E);     // push    ds
        __emit__(0x8E);     // mov     ds, bx
        __emit__(0xDB);

        __emit__(0x16);     // push    ss
        __emit__(0x07);     // pop     es

        __emit__(0xAD);     // lodsd
        __emit__(0xAB);     // stosd
        __emit__(0xAD);     // lodsd
        __emit__(0xAB);     // stosd

        __emit__(0x1F);     // pop     ds
        __emit__(0x58);     // pop     eax

        printf("\n\nFault Information (CS:EIP):  %04X:%08X -",
            ex->FaultCSEIP.EIP_CS.Selector, ex->FaultCSEIP.EIP_CS.Offset);

        for(short i = 0; i < 8; i++)
            printf(" %02X", tempData[i]);
    }

    printf("\n\nFlags: C - P - A - Z S T I D O ioplN - R V\n       ");
    for(i = j = 1; i < 19; i++, j <<= 1)
        printf("%c ", (ex->Flags & j) ? '1' : '0');

    puts("\n");

    // Error code info is only valid for faults 0A-0D.  Error code is
    // always zero for 08.
    if(ex->ExceptionNumber > 9 && ex->ExceptionNumber < 0x0E)
    {
        // The upper 14 bits are the segment selector involved.
        printf("\nSegment Selector (Faults 0A-0D)=%04X\n",
            ex->ErrorCode & 0xFFFC);

        // If error code is zero for a fault, it could be due
        // to a NULL selector (doesn't apply to 0C).
        if(!ex->ErrorCode && ex->ExceptionNumber != 0x0C)
            puts("Possible NULL selector\n");
        else
        {
            // Was it caused by an external event?
            if(ex->ErrorCode & 0x01)
                puts("Caused by an external event");

            // Was it for a gate descriptor in the IDT?
            if(ex->ErrorCode & 0x02)
                puts("IDT gate descriptor\n");
            else
                if(ex->ErrorCode & 0x04)    // Was it for the LDT?
                    puts("Refers to LDT\n");
                else
                    puts("Refers to GDT\n");
        }
    }
    else
        if(ex->ExceptionNumber == 0x0E)
        {
            // Page fault exception.  The error code is different for
            // a page fault.

            puts("Page Fault (0E) Information");

            // Was it due to a page-level protection violation?
            // If not, it was due to a not-present page.
            if(ex->ErrorCode & 0x01)
                printf("Page-level protection violation ");
            else
                printf("Not-present page ");

            // Was it due to a write access?
            // If not, it was due to a read access.
            if(ex->ErrorCode & 0x02)
                printf("caused by a write access");
            else
                printf("caused by a read access");

            // CR2 contains the linear address that caused the page fault.
            printf(" at address %08X\n", ex->CR2);

            // Did the fault occur while the CPU was in user mode?
            // If not, it occurred in supervisor mode.
            if(ex->ErrorCode & 0x04)
                puts("Processor was in user mode\n");
            else
                puts("Processor was in supervisor mode\n");
        }

    Options[0] = '\x0';

    // Can't recover on an Abort exception.  The system is probably unstable.
    // Also disable it if there is no recovery function defined.
    if(ExType && DPMIExceptionHandler::RecoveryFunction)
    {
        printf("<A>ttempt Recovery  |  ");
        strcpy(Options, "A");
    }

    printf("<E>xit Application  ");
    strcat(Options, "E");

    // Overflow exceptions don't seem to work properly if you exit
    // via the old handler.  Checking the code at the original vector
    // shows HLT instructions (?).  Recovery is easy enough, so it's best
    // to either ignore it or just exit the application.
    //
    // Chaining to the original vector will not work for exception 5
    // because it will reflect it as an interrupt (INT 5 = Print Screen)
    // and it gets stuck in an endless loop.  A regular press of the
    // Print Screen key will *NOT* invoke the exception handler though.
    //
    if(ex->ExceptionNumber != 4 && ex->ExceptionNumber != 5)
    {
        printf("|  Exit via old <H>andler");
        strcat(Options, "H");
    }

    puts("\n--------------------------------------------------------------------");

    return Options;
}

#endif
