// File    : TVEXCDLG.CPP
// Author  : Eric Woodruff,  CIS ID: 72134,1150
// Updated : Mon 03/06/95 20:39:31
// Note    : Copyright 1995, Eric Woodruff, All rights reserved
// Compiler: Borland C++ 4.xx
//
// This file contains the generic 16-bit and 32-bit exception handler code
// for Turbo Vision.  Compiler definitions determine which one will be
// produced.  It also contains the TVExceptionDialog class functions.
//

#include <conio.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define Uses_MsgBox
#define Uses_TApplication
#define Uses_TButton
#define Uses_TCheckBoxes
#define Uses_TDialog
#define Uses_TInputLine
#define Uses_TKeys
#define Uses_TLabel
#define Uses_TMultiCheckBoxes
#define Uses_TProgram
#define Uses_TRadioButtons
#define Uses_TSItem
#include <tv.h>

#include <tvexcdlg.h>

// Replace the standard library abort() function in an attempt to
// exit cleanly.
void abort(void)
{
    TEvent anEvent;

    puts("\n\nAn attempt was made to abnormally terminate the program.");
    puts("This may be due to a low memory condition or an exception.\n");
    puts("Press any key to clean up and shutdown...\n\n");
    getch();

    // Force a graceful exit.
    anEvent.what = evCommand;
    anEvent.message.command = cmQuit;

    // If it is modal (running), shut down the application.
    // This gives the user's application a chance to unhook interrupts, etc.
    if(TProgram::application->state & sfModal)
        TProgram::application->handleEvent(anEvent);

    // Always release the exception handlers prior to exit.
    DPMIExceptionHandler::releaseExceptions();

    // Act like an abort function() by returning 3.  However, instead of
    // calling _exit() it calls exit() (no leading underscore).  Buffers
    // are flushed, files are closed, global objects are destroyed, and
    // functions registered with atexit() are called.
    exit(3);
}

// A generic exception handler for use with TV.
int TVExceptionHandler(DPMIExceptionData *ex)
{
    // If there isn't enough memory, exit via the original handler.
    if(lowMemory())
    {
        puts("\n\n  AN EXCEPTION HAS OCCURRED BUT THERE IS NOT ENOUGH "
            "MEMORY FOR THE DIALOG BOX\n\n  Exiting via original vector\n\n");

        // Attempt to clean-up first though.
        DPMIExceptionHandler::CleanUpFunction(0, ex);

        // Always release the exception handlers prior to exit.
        DPMIExceptionHandler::releaseExceptions();
        return 0;
    }

    // Execute the exception dialog box.
    TVExceptionDialog *dlg = new TVExceptionDialog(ex);

    ushort result = TProgram::application->executeDialog(dlg, NULL);

    if(result == cmExitViaOldHandler)
        return 0;

    // Exit without going through the old handler or return after recovery.
    return 1;
}

// ****************************************************************************

#if !defined(__FLAT__)

// 16-bit exception handler dialog functions.

// ***************************************************************************

TVExceptionDialog::TVExceptionDialog(DPMIExceptionData *excp) :
    TDialog(TRect(1, 1, 78, 22), NULL),
    TWindowInit( TVExceptionDialog::initFrame ),
    ex(excp)
{
    static char  *ExName[] = {
        "Divide Error",
        "Debug",
        "NMI",
        "Breakpoint",
        "Overflow (INTO)",
        "Bounds Check (BOUND)",
        "Invalid Op-Code",
        "Coprocessor Unavailable",
        "Double Fault",
        "Coprocessor Segment Overrun",
        "Invalid Task State Segment",
        "Segment Not Present",
        "Stack Fault",
        "General Protection Fault (GPF)",
        "Page Fault",
        "???",                  // 0x0F is marked as reserved.
        "Coprocessor Error"
    };

    char  *p, temp[80];
    uchar *CSIP;
    int    ExType, CSLimit, CSGood, DSLimit, DSGood, ESLimit, ESGood,
           SSLimit, SSGood, i, j;

    // Turn off the close button on the frame and center the dialog.
    flags &= ~wfClose;
    options |= ofCentered;

    sprintf(title, "Exception #%02X ", ex->ExceptionNumber);

    switch(ex->ExceptionNumber)
    {
        case 0x08:
        case 0x09:
        case 0x0F:
            strcat(title, "(Abort) - ");
            strcat(title, ExName[ex->ExceptionNumber]);
            ExType = 0;
            break;

        case 0x00:
        case 0x05:
        case 0x06:
        case 0x07:
        case 0x0A:
        case 0x0B:
        case 0x0C:
        case 0x0D:
        case 0x0E:
        case 0x10:
            strcat(title, "(Fault) - ");
            strcat(title, ExName[ex->ExceptionNumber]);
            ExType = 1;
            break;

        case 0x01:      // 0x01 may also be a fault
        case 0x02:
        case 0x03:
        case 0x04:
            strcat(title, "(Trap) - ");
            strcat(title, ExName[ex->ExceptionNumber]);
            ExType = 2;
            break;

        default:
            strcat(title, "(???) - ???");       // Not one I've expected.
            ExType = 0;
            break;
    }

    sprintf(temp, " AX=%04X        BX=%04X        CX=%04X        DX=%04X    ",
        ex->AX, ex->BX, ex->CX, ex->DX);
    insert(new TStaticText(TRect(10, 2, 67, 3), temp));

    sprintf(temp, " SI=%04X        DI=%04X        BP=%04X        SP=%04X    ",
        ex->SI, ex->DI, ex->BP, ex->StackSSSP.SP_SS.Offset);
    insert(new TStaticText(TRect(10, 3, 67, 4), temp));

    // Determine the segment limits and their validity.
    CSGood = GetSegmentLimit(ex->FaultCSIP.IP_CS.Selector, &CSLimit);
    DSGood = GetSegmentLimit(ex->DS, &DSLimit);
    ESGood = GetSegmentLimit(ex->ES, &ESLimit);
    SSGood = GetSegmentLimit(ex->StackSSSP.SP_SS.Selector, &SSLimit);

    sprintf(temp, "CS=%04X   Limit=", ex->FaultCSIP.IP_CS.Selector);
    if(CSGood)
        sprintf(temp + strlen(temp), "%04X", CSLimit);
    else
        strcat(temp, "Invalid segment");
    insert(new TStaticText(TRect(3, 5, 34, 6), temp));

    sprintf(temp, "DS=%04X   Limit=", ex->DS);
    if(DSGood)
        sprintf(temp + strlen(temp), "%04X", DSLimit);
    else
        strcat(temp, "Invalid segment");
    insert(new TStaticText(TRect(3, 6, 34, 7), temp));

    sprintf(temp, "ES=%04X   Limit=", ex->ES);
    if(ESGood)
        sprintf(temp + strlen(temp), "%04X", ESLimit);
    else
        strcat(temp, "Invalid segment");
    insert(new TStaticText(TRect(3, 7, 34, 8), temp));

    sprintf(temp, "SS=%04X   Limit=", ex->StackSSSP.SP_SS.Selector);
    if(SSGood)
        sprintf(temp + strlen(temp), "%04X", SSLimit);
    else
        strcat(temp, "Invalid segment");
    insert(new TStaticText(TRect(3, 8, 34, 9), temp));

    // Get the fault or trap bytes at CS:IP.
    if(ExType == 1 || ExType == 2)
    {
        sprintf(temp, "Fault Information (CS:IP)\n%04X:%04X -",
            ex->FaultCSIP.IP_CS.Selector, ex->FaultCSIP.IP_CS.Offset);

        CSIP = (uchar *)ex->FaultCSIP.CS_IP;
        p = temp + strlen(temp);

        for(i = 0; i < 8; i++, CSIP++)
        {
            sprintf(p, " %02X", *CSIP);
            p += strlen(p);
        }
        insert(new TStaticText(TRect(35, 5, 74, 7), temp));
    }

    strcpy(temp, "Flags: C - P - A - Z S T I D O ioplN\n      ");
    p = temp + strlen(temp);

    for(i = j = 1; i < 16; i++, j <<= 1)
    {
        sprintf(p, " %c", (ex->Flags & j) ? '1' : '0');
        p += strlen(p);
    }
    insert(new TStaticText(TRect(17, 10, 59, 12), temp));

    // Error code info is only valid for faults 0A-0D.  Error code is
    // always zero for 08.
    if(ex->ExceptionNumber > 9 && ex->ExceptionNumber < 0x0E)
    {
        // The upper 14 bits are the segment selector involved.
        sprintf(temp, "Segment Selector (Faults 0A-0D)=%04X",
            ex->ErrorCode & 0xFFFC);

        insert(new TStaticText(TRect(18, 13, 57, 14), temp));

        // If error code is zero for a fault, it could be due
        // to a NULL selector (doesn't apply to 0C).
        if(!ex->ErrorCode && ex->ExceptionNumber != 0x0C)
            insert(new TStaticText(TRect(18, 14, 40, 15),
                "Possible NULL selector"));
        else
        {
            // Was it caused by an external event?
            if(ex->ErrorCode & 0x01)
                insert(new TStaticText(TRect(18, 15, 45, 16),
                    "Caused by an external event"));

            // Was it for a gate descriptor in the IDT?
            if(ex->ErrorCode & 0x02)
                strcpy(temp, "Refers to an IDT gate descriptor");
            else
                if(ex->ErrorCode & 0x04)    // Was it for the LDT?
                    strcpy(temp, "Refers to the LDT");
                else
                    strcpy(temp, "Refers to the GDT");  // If not, the GDT

            insert(new TStaticText(TRect(18, 16, 55, 17), temp));
        }
    }
    else
        if(ex->ExceptionNumber == 0x0E)
        {
            // Page fault exception.  The error code is different for
            // a page fault.

            // Was it due to a page-level protection violation?
            // If not, it was due to a not-present page.
            if(ex->ErrorCode & 0x01)
                insert(new TStaticText(TRect(21, 13, 52, 14),
                    "Page level protection violation"));
            else
                insert(new TStaticText(TRect(21, 13, 52, 14),
                    "Not-Present page"));

            // Was it due to a write access?
            // If not, it was due to a read access.
            if(ex->ErrorCode & 0x02)
                insert(new TStaticText(TRect(21, 14, 50, 15),
                    "Caused by a write access"));
            else
                insert(new TStaticText(TRect(21, 14, 50, 15),
                    "Caused by a read access"));

            // Did the fault occur while the CPU was in user mode?
            // If not, it occurred in supervisor mode.
            if(ex->ErrorCode & 0x04)
                insert(new TStaticText(TRect(21, 15, 53, 16),
                    "Processor was in user mode"));
            else
                insert(new TStaticText(TRect(21, 15, 53, 16),
                    "Processor was in supervisor mode"));
        }

    // Make sure the buttons are enabled by default.
    enableCommand(cmAttemptRecovery);
    enableCommand(cmExitApplication);
    enableCommand(cmExitViaOldHandler);

    insert(new TButton(TRect(2, 18, 24, 20), "~A~ttempt to recover",
        cmAttemptRecovery, bfNormal));

    // Can't recover on an Abort exception.  The system is probably unstable.
    // Also disable it if there is no recovery function defined.
    if(!ExType || !DPMIExceptionHandler::RecoveryFunction)
        disableCommand(cmAttemptRecovery);

    insert(new TButton(TRect(26, 18, 48, 20), "E~x~it Application",
        cmExitApplication, bfNormal));

    insert(new TButton(TRect(50, 18, 74, 20), "Exit ~v~ia old handler",
        cmExitViaOldHandler, bfNormal));

    // Overflow exceptions don't seem to work properly if you exit
    // via the old handler.  Checking the code at the original vector
    // shows HLT instructions (?).  Recovery is easy enough, so it's best
    // to either ignore it or just exit the application.
    //
    // Chaining to the original vector will not work for exception 5
    // because it will reflect it as an interrupt (INT 5 = Print Screen)
    // and it gets stuck in an endless loop.  A regular press of the
    // Print Screen key will *NOT* invoke the exception handler though.
    //
    if(ex->ExceptionNumber == 4 || ex->ExceptionNumber == 5)
        disableCommand(cmExitViaOldHandler);

    // Default to Exit Application.
    selectNext(True);
}

// ***************************************************************************

#else

// 32-bit exception handler.

TVExceptionDialog::TVExceptionDialog(DPMIExceptionData *excp) :
    TDialog(TRect(1, 1, 78, 22), NULL),
    TWindowInit( TVExceptionDialog::initFrame ),
    ex(excp)
{
    static char  *ExName[] = {
        "Divide Error",
        "Debug",
        "NMI",
        "Breakpoint",
        "Overflow (INTO)",
        "Bounds Check (BOUND)",
        "Invalid Op-Code",
        "Coprocessor Unavailable",
        "Double Fault",
        "Coprocessor Segment Overrun",
        "Invalid Task State Segment",
        "Segment Not Present",
        "Stack Fault",
        "General Protection Fault (GPF)",
        "Page Fault",
        "???",                  // 0x0F is marked as reserved.
        "Coprocessor Error"
    };

    char  *p, temp[80];
    uchar  tempData[10];
    int    ExType, CSLimit, CSGood, DSLimit, DSGood, ESLimit, ESGood,
           SSLimit, SSGood, i, j;

    // Turn off the close button on the frame.
    flags &= ~wfClose;
    options |= ofCentered;

    sprintf(title, "Exception #%02X ", ex->ExceptionNumber);

    switch(ex->ExceptionNumber)
    {
        case 0x08:
        case 0x09:
        case 0x0F:
            strcat(title, "(Abort) - ");
            strcat(title, ExName[ex->ExceptionNumber]);
            ExType = 0;
            break;

        case 0x00:
        case 0x05:
        case 0x06:
        case 0x07:
        case 0x0A:
        case 0x0B:
        case 0x0C:
        case 0x0D:
        case 0x0E:
        case 0x10:
            strcat(title, "(Fault) - ");
            strcat(title, ExName[ex->ExceptionNumber]);
            ExType = 1;
            break;

        case 0x01:      // 0x01 may also be a fault
        case 0x02:
        case 0x03:
        case 0x04:
            strcat(title, "(Trap) - ");
            strcat(title, ExName[ex->ExceptionNumber]);
            ExType = 2;
            break;

        default:
            strcat(title, "(???) - ???");       // Not one I've expected.
            ExType = 0;
            break;
    }

    sprintf(temp, "EAX=%08X   EBX=%08X   ECX=%08X   EDX=%08X",
        ex->EAX, ex->EBX, ex->ECX, ex->EDX);
    insert(new TStaticText(TRect(10, 2, 67, 3), temp));

    sprintf(temp, "ESI=%08X   EDI=%08X   EBP=%08X   ESP=%08X",
        ex->ESI, ex->EDI, ex->EBP, ex->StackSSESP.ESP_SS.Offset);
    insert(new TStaticText(TRect(10, 3, 67, 4), temp));

    // Determine the segment limits and their validity.
    CSGood = GetSegmentLimit(ex->FaultCSEIP.EIP_CS.Selector, &CSLimit);
    DSGood = GetSegmentLimit(ex->DS, &DSLimit);
    ESGood = GetSegmentLimit(ex->ES, &ESLimit);
    SSGood = GetSegmentLimit(ex->StackSSESP.ESP_SS.Selector, &SSLimit);

    sprintf(temp, "CS=%04X   Limit=", ex->FaultCSEIP.EIP_CS.Selector);
    if(CSGood)
        sprintf(temp + strlen(temp), "%08X", CSLimit);
    else
        strcat(temp, "Invalid segment");
    insert(new TStaticText(TRect(3, 5, 34, 6), temp));

    sprintf(temp, "DS=%04X   Limit=", ex->DS);
    if(DSGood)
        sprintf(temp + strlen(temp), "%08X", DSLimit);
    else
        strcat(temp, "Invalid segment");
    insert(new TStaticText(TRect(3, 6, 34, 7), temp));

    sprintf(temp, "ES=%04X   Limit=", ex->ES);
    if(ESGood)
        sprintf(temp + strlen(temp), "%08X", ESLimit);
    else
        strcat(temp, "Invalid segment");
    insert(new TStaticText(TRect(3, 7, 34, 8), temp));

    sprintf(temp, "SS=%04X   Limit=", ex->StackSSESP.ESP_SS.Selector);
    if(SSGood)
        sprintf(temp + strlen(temp), "%08X", SSLimit);
    else
        strcat(temp, "Invalid segment");
    insert(new TStaticText(TRect(3, 8, 34, 9), temp));

    // Get the fault or trap bytes at CS:EIP.
    if(ExType == 1 || ExType == 2)
    {
        // If the exception didn't happen in our code, the selectors
        // might not be the same.  Move the eight bytes to a temporary
        // buffer before formatting them.

        // The BCC32 generated assembler file for this won't compile so
        // __emit__() statements are used instead.

        _EBX = ex->FaultCSEIP.EIP_CS.Selector;
        _ESI = ex->FaultCSEIP.EIP_CS.Offset;
        _EDI = (ulong)&tempData[0];

        __emit__(0x50);     // push     eax    BCC32 generates code that uses
                            //                 EAX and it doesn't know about
                            //                 the emitted lodsd's below.

        __emit__(0x1E);     // push    ds
        __emit__(0x8E);     // mov     ds, bx
        __emit__(0xDB);

        __emit__(0x16);     // push    ss
        __emit__(0x07);     // pop     es

        __emit__(0xAD);     // lodsd
        __emit__(0xAB);     // stosd
        __emit__(0xAD);     // lodsd
        __emit__(0xAB);     // stosd

        __emit__(0x1F);     // pop     ds
        __emit__(0x58);     // pop     eax

        sprintf(temp, "Fault Information (CS:EIP)\n%04X:%08X -",
            ex->FaultCSEIP.EIP_CS.Selector, ex->FaultCSEIP.EIP_CS.Offset);

        p = temp + strlen(temp);

        for(short i = 0; i < 8; i++)
        {
            sprintf(p, " %02X", tempData[i]);
            p += strlen(p);
        }
        insert(new TStaticText(TRect(35, 5, 74, 7), temp));
    }

    strcpy(temp, "Flags: C - P - A - Z S T I D O ioplN - R V\n      ");
    p = temp + strlen(temp);

    for(i = j = 1; i < 19; i++, j <<= 1)
    {
        sprintf(p, " %c", (ex->Flags & j) ? '1' : '0');
        p += strlen(p);
    }
    insert(new TStaticText(TRect(17, 10, 59, 12), temp));

    // Error code info is only valid for faults 0A-0D.  Error code is
    // always zero for 08.
    if(ex->ExceptionNumber > 9 && ex->ExceptionNumber < 0x0E)
    {
        // The upper 14 bits are the segment selector involved.
        sprintf(temp, "Segment Selector (Faults 0A-0D)=%04X",
            ex->ErrorCode & 0xFFFC);

        insert(new TStaticText(TRect(18, 13, 57, 14), temp));

        // If error code is zero for a fault, it could be due
        // to a NULL selector (doesn't apply to 0C).
        if(!ex->ErrorCode && ex->ExceptionNumber != 0x0C)
            insert(new TStaticText(TRect(18, 14, 40, 15),
                "Possible NULL selector"));
        else
        {
            // Was it caused by an external event?
            if(ex->ErrorCode & 0x01)
                insert(new TStaticText(TRect(18, 15, 45, 16),
                    "Caused by an external event"));

            // Was it for a gate descriptor in the IDT?
            if(ex->ErrorCode & 0x02)
                strcpy(temp, "Refers to an IDT gate descriptor");
            else
                if(ex->ErrorCode & 0x04)    // Was it for the LDT?
                    strcpy(temp, "Refers to the LDT");
                else
                    strcpy(temp, "Refers to the GDT");  // If not, the GDT

            insert(new TStaticText(TRect(18, 16, 55, 17), temp));
        }
    }
    else
        if(ex->ExceptionNumber == 0x0E)
        {
            // Page fault exception.  The error code is different for
            // a page fault.

            // Was it due to a page-level protection violation?
            // If not, it was due to a not-present page.
            if(ex->ErrorCode & 0x01)
                insert(new TStaticText(TRect(21, 13, 52, 14),
                    "Page level protection violation"));
            else
                insert(new TStaticText(TRect(21, 13, 52, 14),
                    "Not Present page"));

            // Was it due to a write access?
            // If not, it was due to a read access.
            if(ex->ErrorCode & 0x02)
                insert(new TStaticText(TRect(21, 14, 50, 15),
                    "Caused by a write access"));
            else
                insert(new TStaticText(TRect(21, 14, 50, 15),
                    "Caused by a read access"));

            // Did the fault occur while the CPU was in user mode?
            // If not, it occurred in supervisor mode.
            if(ex->ErrorCode & 0x04)
                insert(new TStaticText(TRect(21, 15, 53, 16),
                    "Processor was in user mode"));
            else
                insert(new TStaticText(TRect(21, 15, 53, 16),
                    "Processor was in supervisor mode"));

            // CR2 contains the linear address that caused the page fault.
            sprintf(temp, "Referenced linear address %08X", ex->CR2);
            insert(new TStaticText(TRect(21, 16, 55, 17), temp));
        }

    // Make sure the buttons are enabled by default.
    enableCommand(cmAttemptRecovery);
    enableCommand(cmExitApplication);
    enableCommand(cmExitViaOldHandler);

    insert(new TButton(TRect(2, 18, 24, 20), "~A~ttempt to recover",
        cmAttemptRecovery, bfNormal));

    // Can't recover on an Abort exception.  The system is probably unstable.
    // Also disable it if there is no recovery function defined.
    if(!ExType || !DPMIExceptionHandler::RecoveryFunction)
        disableCommand(cmAttemptRecovery);

    insert(new TButton(TRect(26, 18, 48, 20), "E~x~it Application",
        cmExitApplication, bfNormal));

    insert(new TButton(TRect(50, 18, 74, 20), "Exit ~v~ia old handler",
        cmExitViaOldHandler, bfNormal));

    // Overflow exceptions don't seem to work properly if you exit
    // via the old handler.  Checking the code at the original vector
    // shows HLT instructions (?).  Recovery is easy enough, so it's best
    // to either ignore it or just exit the application.
    //
    // Chaining to the original vector will not work for exception 5
    // because it will reflect it as an interrupt (INT 5 = Print Screen)
    // and it gets stuck in an endless loop.  A regular press of the
    // Print Screen key will *NOT* invoke the exception handler though.
    //
    if(ex->ExceptionNumber == 4 || ex->ExceptionNumber == 5)
        disableCommand(cmExitViaOldHandler);

    // Default to Exit Application.
    selectNext(True);
}

#endif

// ***************************************************************************

#pragma argsused
const char *TVExceptionDialog::getTitle(short maxSize)
{
    return title;
}

void TVExceptionDialog::handleEvent(TEvent &event)
{
    // Ignore ESC, it will simply cause the dialog box to be redisplayed.
    if(event.what == evKeyboard && event.keyDown.keyCode == kbEsc)
        clearEvent(event);

    TDialog::handleEvent(event);

    if(event.what == evCommand && (state & sfModal))
        switch(event.message.command)
        {
            // Attempt to recover from the exception.
            case cmAttemptRecovery:
                if(DPMIExceptionHandler::RecoveryFunction(ex))
                    endModal(event.message.command);
                else
                {
                    messageBox("Unable to recover from the exception.",
                        mfError | mfCancelButton);
                    disableCommand(cmAttemptRecovery);
                    selectNext(False);
                }
                clearEvent(event);
                break;

            // Clean up and exit via a user specified function.
            case cmExitApplication:
                // Hide the exception dialog box prior to calling
                // the clean-up function.  If not, it redisplays part
                // of the desktop when the dialog box terminates.
                // It doesn't hurt anything, it just doesn't look
                // neat and tidy.
                hide();
                DPMIExceptionHandler::CleanUpFunction(1, ex);
                endModal(event.message.command);
                clearEvent(event);

                // Always release the exception handlers prior to exit.
                DPMIExceptionHandler::releaseExceptions();
                break;

            // Clean up and exit via the original handler.
            case cmExitViaOldHandler:
                hide();
                DPMIExceptionHandler::CleanUpFunction(0, ex);
                endModal(event.message.command);
                clearEvent(event);

                // Always release the exception handlers prior to exit.
                DPMIExceptionHandler::releaseExceptions();
                break;
        }
}
