#ifndef DEF_DPMI_H
#define DEF_DPMI_H

#ifndef _Windows
#error DPMI requires Windows or PowerPack!
#endif

#ifdef __FLAT__
#define DPMIFAR
#else
#define DPMIFAR   __far
#endif

#ifdef __cplusplus
extern "C" {
#endif

// PMRegs is used for calling interrupts, functions, etc.
typedef struct tagPMByteRegs
{
   unsigned short di, res1, si, res2, bp, res3;
   unsigned long res4;
   unsigned char bl, bh;
   unsigned short res5;
   unsigned char dl, dh;
   unsigned short res6;
   unsigned char cl, ch;
   unsigned short res7;
   unsigned char al, ah;
   unsigned short res8;
   unsigned short flags, es, ds, fs, gs, ip, cs, sp, ss;
} PMByteRegs;

typedef struct tagPMWordRegs
{
   unsigned short di, res1, si, res2, bp, res3;
   unsigned long res4;
   unsigned short bx, res5;
   unsigned short dx, res6;
   unsigned short cx, res7;
   unsigned short ax, res8;
   unsigned short flags, es, ds, fs, gs, ip, cs, sp, ss;
} PMWordRegs;

typedef struct tagPMDWordRegs
{
   unsigned long edi, esi, ebp, res1, ebx, edx, ecx, eax;
   unsigned short flags, es, ds, fs, gs, ip, cs, sp, ss;
} PMDWordRegs;

typedef union tagPMRegs
{
   // Access registers as bytes.
   PMByteRegs h;
   // Access registers as words.
   PMWordRegs w;
   // Access registers as dwords.
   PMDWordRegs x;
} PMRegs;
typedef PMRegs DPMIFAR* LPPMRegs;

// Types.
// Selectors and segments.
typedef unsigned short PMSelector;  // Protected mode selector.
typedef unsigned short RMSegment;   // Real mode segment.
// 32-bit linear address.
typedef unsigned long LinearAddress;
// Handle used with some DPMI functions.
typedef unsigned long PMHandle;

// Segment:offset real mode address.
typedef unsigned long RMAddress;
typedef RMAddress DPMIFAR* LPRMAddress;
#define RM_OFFSET(addr) ((unsigned short)(RMAddress)(addr))
#define RM_SEGMENT(addr) ((unsigned short)((RMAddress)(addr) >> 16))

// Version info.
typedef struct tagDPMIVersion
{
   unsigned char major;
   unsigned char minor;
   unsigned short flags;
   unsigned char cpu;
   unsigned char masterPIC;
   unsigned char slavePIC;
} DPMIVersion;
typedef DPMIVersion DPMIFAR* LPDPMIVersion;
enum version_flags {
        version_isdpmi32 = 1,
        version_hasRealModeInterrupts = 2,
        version_hasVirtualMemory = 4
};

// Memory info.
typedef struct tagDPMIMemInfo
{
   unsigned long maxFreeBlock;
   unsigned long maxUnlockedPage;
   unsigned long maxLockedPage;
   unsigned long linearPages;
   unsigned long totalUnlockedPages;
   unsigned long totalFreePages;
   unsigned long totalPhysicalPages;
   unsigned long freeLinearPages;
   unsigned long partitionSize;
   char reserved[12];
} DPMIMemInfo;
typedef DPMIMemInfo DPMIFAR* LPDPMIMemInfo;

typedef struct tagDPMIInfo
{
   unsigned char supports32Bit;
   unsigned char cpu;
   unsigned char major;
   unsigned char minor;
} DPMIInfo;
typedef DPMIInfo DPMIFAR* LPDPMIInfo;

typedef unsigned short PMRights;
enum pmrights_types {
   rights_Accessed         = 0x0001,
   rights_ReadWrite        = 0x0002,
   rights_Conforming       = 0x0004,
   rights_Code             = 0x0008,
   rights_Reserved1        = 0x0010,
   rights_Level            = 0x0060,
   rights_Present          = 0x0080,
   rights32                = 0xFF00,
   rights32_Reserved2      = 0x0F00,
   rights32_Avl            = 0x1000,
   rights32_Reserved3      = 0x2000,
   rights32_Default32      = 0x4000,
   rights32_PageGranular   = 0x8000
};

typedef unsigned char Descriptor[8];
typedef unsigned char DPMIFAR* LPDescriptor;
typedef const unsigned char DPMIFAR* LPCDescriptor;


#ifdef __FLAT__
typedef struct tagPMFarAddress  // 48-bit protected mode pointer.
{
   unsigned long offset;
   PMSelector selector;
} PMFarAddress;
#else
typedef PMAddress PMFarAddress;
#endif
typedef PMFarAddress DPMIFAR* LPPMFarAddress;

// Real mode memory info for xxxDosMem() functions.
typedef unsigned long DosMemory;
typedef DosMemory DPMIFAR* LPDosMemory;
#define DM_SEGMENT(addr) ((RMSegment)((DosMemory)(addr)))
#define DM_SELECTOR(addr) ((PMSelector)((DosMemory)(addr) >> 16))

typedef unsigned short WatchPoint;  // Watchpoint handle.

enum watchpoint_types {
   wp_execute,
   wp_write,
   wp_read_write
};

   // Mode Detection.
int DPMIFAR dpmi_present( LPDPMIInfo );
int DPMIFAR dpmi_running(void);

   // LDT Descriptor Management Services.
PMSelector DPMIFAR dpmi_allocateSelector( unsigned short count );
int DPMIFAR dpmi_freeSelector( PMSelector sel );
PMSelector DPMIFAR dpmi_segmentToSelector( RMSegment seg );
unsigned short DPMIFAR dpmi_getSelectorInc(void);
LinearAddress DPMIFAR dpmi_getSelectorLinearAddress( PMSelector sel );
int DPMIFAR dpmi_setSelectorLinearAddress( PMSelector sel, LinearAddress addr );
unsigned long DPMIFAR dpmi_getSelectorLimit( PMSelector sel );
int DPMIFAR dpmi_setSelectorLimit( PMSelector sel, unsigned long len );
PMRights DPMIFAR dpmi_getSelectorRights( PMSelector sel );
int DPMIFAR dpmi_setSelectorRights( PMSelector sel, PMRights rights );
PMSelector DPMIFAR dpmi_aliasCodeToData( PMSelector codeSel );
int DPMIFAR dpmi_getDescriptor( PMSelector sel, LPDescriptor buf );
int DPMIFAR dpmi_setDescriptor( PMSelector sel, LPCDescriptor buf );
int DPMIFAR dpmi_allocateSpecificSelector( PMSelector sel );

   // DOS Memory Management Services.
DosMemory DPMIFAR dpmi_allocateDosMem( unsigned long len );
int DPMIFAR dpmi_freeDosMem( PMSelector sel );
int DPMIFAR dpmi_reallocateDosMem( PMSelector sel, unsigned long len );

   // Interrupt Services
RMAddress DPMIFAR dpmi_getRMVect( unsigned char intNo );
int DPMIFAR dpmi_setRMVect( unsigned char intNo, RMAddress addr );
PMFarAddress DPMIFAR dpmi_getPMVect( unsigned char intNo );
int DPMIFAR dpmi_setPMVect( unsigned char intNo, PMFarAddress addr );
PMFarAddress DPMIFAR dpmi_getPMHandler( unsigned char faultNo );
int DPMIFAR dpmi_setPMHandler( unsigned char faultNo, PMFarAddress addr );

   // Translation Services
int DPMIFAR dpmi_intr( unsigned char intNo, LPPMRegs regs );
int DPMIFAR dpmi_callFar( RMAddress addr, LPPMRegs regs );
int DPMIFAR dpmi_callIntr( RMAddress addr, LPPMRegs regs );
RMAddress DPMIFAR dpmi_allocateCallBack( PMFarAddress proc, LPPMRegs regs );
int DPMIFAR dpmi_freeCallBack( RMAddress addr );
PMFarAddress DPMIFAR dpmi_getPMSaveStateProc(void);
RMAddress DPMIFAR dpmi_getRMSaveStateProc(void);
unsigned short DPMIFAR dpmi_getSaveStateSize(void);
PMFarAddress DPMIFAR dpmi_getPMSwitchProc(void);
RMAddress DPMIFAR dpmi_getRMSwitchProc(void);
void DPMIFAR dpmi_getVersion( LPDPMIVersion ver );

   // Memory Management Services
int DPMIFAR dpmi_getMemInfo( LPDPMIMemInfo info );
PMHandle DPMIFAR dpmi_allocateLinearMem( unsigned long len,
                                         LinearAddress DPMIFAR*addr );
int DPMIFAR dpmi_freeLinearMem( PMHandle handle );
PMHandle DPMIFAR dpmi_reallocateLinearMem( PMHandle handle, unsigned long len,
                                           LPRMAddress addr );

   // Page Locking Services
int DPMIFAR dpmi_lockLinearMem( LinearAddress addr, unsigned long len );
int DPMIFAR dpmi_unlockLinearMem( LinearAddress addr, unsigned long len );
int DPMIFAR dpmi_markRModePageable( RMAddress addr, unsigned long len );
int DPMIFAR dpmi_relockRModeRegion( RMAddress addr, unsigned long len );
unsigned long DPMIFAR dpmi_getPageSize();

   // Demand Paging Performance Tuning Services
int DPMIFAR dpmi_markLinearMem( LinearAddress addr, unsigned long len );
int DPMIFAR dpmi_discardLinearMem( LinearAddress addr, unsigned long len );

   // Physical Address Mapping
LinearAddress DPMIFAR dpmi_mapPhysicalMem( LinearAddress addr, unsigned long len );

   // Virtual Interrupt State Services.
int DPMIFAR dpmi_disableIntr(void);
int DPMIFAR dpmi_enableIntr(void);
int DPMIFAR dpmi_getIntr(void);

   // Debug Register Support.
WatchPoint DPMIFAR dpmi_setWatchPoint( LinearAddress addr, int nrBytes, int type );
int DPMIFAR dpmi_clearWatchPoint( WatchPoint );
int DPMIFAR dpmi_watchPointExecuted( WatchPoint );
int DPMIFAR dpmi_resetWatchPoint( WatchPoint );

   // Goodies.
#ifdef __FLAT__
void copy32( PMFarAddress dest, PMFarAddress source, unsigned long len );
void copy32aux( unsigned long destOfs, PMSelector destSel, unsigned long srcOfs,
                PMSelector srcSel, unsigned long len );
#endif

#ifdef __cplusplus
}
#endif

#endif   // DEF_DPMI_H
