/*
**  -- win_io.c --
**
**  Implements non-overlapped screen I/O
*/

#define FALSE 0
#define TRUE !FALSE
#define BYTE unsigned char

#include <stdio.h>
#include <dos.h>
#include "dos_io.h"
#include "win_io.h"
#include "ascii.h"

#define NBR_WINS 3

#define CR 0x0d
#define LF 0x0a

#define NORMAL  0x07
#define INVERSE 0x70

static BYTE IsReady = FALSE;
static BYTE Attribute[NBR_WINS];
static BYTE Enabled[NBR_WINS];
static BYTE CurRow[NBR_WINS];
static BYTE CurCol[NBR_WINS];
static BYTE TopRow[NBR_WINS];
static BYTE BotRow[NBR_WINS];
static BYTE LeftCol[NBR_WINS];
static BYTE RghtCol[NBR_WINS];


/*** PRIVATE ***/


static void Initialize()
{int i;
 if(!IsReady)
  {IsReady = TRUE;
   for(i=0;i<NBR_WINS;i++) Enabled[i] = FALSE;
  }
}


static int Check(int Win)
{if(!IsReady) Initialize();
 if((Win<0)||(Win>=NBR_WINS)) return WIN_RANGE;
 if(Enabled[Win]) return WIN_AOK;
 else return WIN_DISABLED;
}


static void NewRow(int Win)
{if(++CurRow[Win]>BotRow[Win])
   {Scroll(TopRow[Win],LeftCol[Win],BotRow[Win],RghtCol[Win],1,Attribute[Win]);
    CurRow[Win] = BotRow[Win];
   }
}


static void Advance(int Win)
{if(++CurCol[Win]>RghtCol[Win])
  {CurCol[Win] = LeftCol[Win];
   NewRow(Win);
  }
 Position(CurRow[Win],CurCol[Win]);
}


static int WriteChar(int Win,char C)
{
 switch(C)
   {case LF:
      NewRow(Win);
      /* fall thru */;
    case CR:
      CurCol[Win] = LeftCol[Win];
      break;
    case BS:
      if(CurCol[Win] > LeftCol[Win])
        {/* back up */
         Position(CurRow[Win],--CurCol[Win]);
         AttrWrite((BYTE)' ', (BYTE)Attribute[Win]);
        }
      break;
    default:
      if((C<' ')||(C>'~')) C = ' ';
      AttrWrite((BYTE)C, (BYTE)Attribute[Win]);
      Advance(Win);
      break;
   }
return WIN_AOK;
}


/*** PUBLIC ***/


/* set attribute to normal */

int WinNormal(int Win)
{int rc;
 if((rc=Check(Win))!=WIN_AOK) return rc;
 Attribute[Win] = NORMAL;
 return WIN_AOK;
}


/* set attribute to inverse */

int WinInverse(int Win)
{int rc;
 if((rc=Check(Win))!=WIN_AOK) return rc;
 Attribute[Win] = INVERSE;
 return WIN_AOK;
}


/* create window */


int WinCreate(int Win,int r1,int c1,int r2,int c2)
{if(!IsReady) Initialize();
 if((Win<0)||(Win>=NBR_WINS)) return WIN_RANGE;
 if(Enabled[Win]) return WIN_ENABLED;
 Enabled[Win] = TRUE;
 Attribute[Win] = NORMAL;
 if(r1>r2) return FALSE;
 if(r2>24) return FALSE;
 if(c1>c2) return FALSE;
 if(c2>79) return FALSE;
 TopRow[Win] = r1;
 BotRow[Win] = r2;
 LeftCol[Win] = c1;
 RghtCol[Win] = c2;
 CurRow[Win] = r1;
 CurCol[Win] = c1;
 Position((BYTE)r1,(BYTE)c1);
 return WIN_AOK;
}

/* write char to window */

int WinPutChar(int Win,char C)
{int rc;
 if((rc=Check(Win))!=WIN_AOK) return rc;
 Position(CurRow[Win],CurCol[Win]);
 return WriteChar(Win,C);
}

/* write string to window */

int WinPutString(int Win,char *String)
{int rc;
 if((rc=Check(Win))!=WIN_AOK) return rc;
 Position(CurRow[Win],CurCol[Win]);
 while(*String!='\0') WriteChar(Win,*String++);
 return WIN_AOK;
}

/* clear window */

int WinClear(int Win)
{int rc;
 if((rc=Check(Win))!=WIN_AOK) return rc;
 Scroll(TopRow[Win],LeftCol[Win],BotRow[Win],RghtCol[Win],0,Attribute[Win]);
 CurRow[Win] = TopRow[Win];
 CurCol[Win] = LeftCol[Win];
 return WIN_AOK;
}

/* get string from window */

int WinGetString(int Win, char *String, int Length)
{int  i, rc;
 char c;
 if((rc=Check(Win))!=WIN_AOK) return rc;
 Position(CurRow[Win],CurCol[Win]);
 /* input text from user */
 i = 0;
 while(1)
     {c = ReadKbd();
      switch(c)
        {case LF:
           /* ignore LFs */
           break;
         case CR:
           /* done */
           String[i] = '\0';
           return WIN_AOK;
         case ESC:
         case CAN:
           /* aborting */
           *String = '\0';
           return WIN_AOK;
         case BS:
           if(i>0)
             {/* back up */
              i--;
              Position(CurRow[Win],--CurCol[Win]);
              AttrWrite((BYTE)' ', (BYTE)Attribute[Win]);
             }
           break;
         default:
           /* save character & display on status line */
           if((c>=' ')&&(c<='~'))
              {String[i++] = c;
               AttrWrite((BYTE)c, (BYTE)Attribute[Win]);
               Advance(Win);
               if(i==Length)
                  {/* user string done */
                   String[i] = '\0';
                   return WIN_AOK;
                  }
              }
           break;
        } /* end case */
     } /* end while */
}

/* set cursor position in window */

int WinSetPos(int Win,int Row, int Col)
{int rc;
 if((rc=Check(Win))!=WIN_AOK) return rc;
 if(TopRow[Win]  + Row > BotRow[Win])  return WIN_RANGE;
 if(LeftCol[Win] + Col > RghtCol[Win]) return WIN_RANGE;
 CurRow[Win] = TopRow[Win]+Row;
 CurCol[Win] = LeftCol[Win]+Col;
 Position(CurRow[Win],CurCol[Win]);
 return WIN_AOK;
}

/* get window cursor position */

int WinGetPos(int Win,int *RowP,int *ColP)
{int rc;
 if((rc=Check(Win))!=WIN_AOK) return rc;
 Position(CurRow[Win],CurCol[Win]);
 if(RowP) *RowP = CurRow[Win] - TopRow[Win];
 if(ColP) *ColP = CurCol[Win] - LeftCol[Win];
 return WIN_AOK;
}