{SECTION ..TEXTobjs }
UNIT TEXTobjs;

INTERFACE

uses PbMISC, PbDATA;

{
Description : All of the STRA/TFILE based objects

Author      : Howard Richoux
Date        : 6/18/94
Last revised: 7/12/94  Changed HOLD Nums to Integer and made limit 15,000
              7/13/94  Changed STRA indexes to Word (subtle .done problem)
              7/13/94  added sepchar to HOLD, default ';'
              7/14/94  added loadsectlist
              8/1/94   info.loadremove(fname,term : string)  - trims trailing comments
              8/13/94  added FindAndLoadINFOfile   for .def files
              10/7/94  added Function STRA.bytecount : longint;
              10/17/94 added Sort Pre & Post processing
              11/29/94 added STRA.findsection method
              12/9/94  added text file encryption
              2/23/95  added UnCompressStr to STRA_object.fetchN

Application : IBM PC and compatibles, done in Turbo Pascal 7
Status      : Placed in the Public Domain by HNR Software 2/18/1994
Published in: none
}




{SECTION .STR_object }

{Basic variable length string on Heap. }

type  STR_object = OBJECT
             strptr: StringPtr_type;                       { pointer to string on heap }
             Procedure   init;                        { gets heap space     }
             Function    store (st: String): boolean; { Stores the string   }
             Function    fetch: String;               { Fetches the string  }
             Procedure   dump;                        { debug write         }
             procedure   dispose;                     { releases heap space }
             end;


{SECTION .STRA_object }

{ Array of variable length strings on Heap.  Limits set at init time.
   functions well as memory resident text file.  Used in nearly everything.
   Limit 16,380  }

const STRA_BigArrayMax = 16380;  { close to 16384 }
type  STRA_NdxType  = word;
type  STRA_BigArray = array[1..STRA_BigArrayMax] of STR_object;

type  STRA_object = OBJECT
             arrayptr     : ^STRA_BigArray;
             arrayused    : STRA_NdxType;
             arraymax     : STRA_NdxType;
             arraysorted  : boolean;        { if sorted and not modified }
             modified     : boolean;        { set on stores to force re-sort }
             encryptflag  : boolean;        { set on save to force encryption }
             encrypted    : boolean;        { set on load if encrypted }
             workcount    : longint;        { temporary - counts swaps in sort, tests in search}

             Procedure init      (max : STRA_NdxType);
             Function  append    (st : string)                   : boolean; { put string in next available slot }
             Function  appendpush(st : string)                   : boolean; { delete first string, shift rest down & append }
             Function  insertN   (n : STRA_NdxType;st : string)  : boolean; { Insert string at nth position }
             Function  deleteN   (n : STRA_NdxType)              : boolean; { Delete nth string & shift rest down }
             Function  storeN    (n : STRA_NdxType; st : string) : boolean; { Store stirng into nth position }
             Function  fetchN    (n : STRA_NdxType) : string;               { Fetch string in nth position }
             Function  count     : STRA_NdxType;                            { returns number of slots used }
             Function  bytecount : longint;                                 { returns a total of all stored strings }
             Function  sorted    : boolean;                                 { returns whether sorted }
             Procedure listpage  (f,n,w : STRA_NdxType);                    { mini dump for text windows }
             Procedure dump;                                                { for debugging }
             Procedure clear;                                               { empties array }
             Procedure save        (fname : string);                        { Saves to text file }
             Procedure load        (fname : string);                        { Loads from text file }
             Procedure loadsection (fname,sectiontag,sectionname : string); { from text file }
             Procedure loadsectlist(fname,sectiontag : string);             { just load the section names }
             Procedure swap(i,j : STRA_NdxType);                            { for sort }
             Function  checksort : boolean;
             Procedure SortPreProcess;                                      { adds key uniqueness }
             Procedure SortPostProcess;                                     { removes it }
             Procedure sort;                                                { shell sort}

             Function  search (st : string; mode : byte) : STRA_NdxType;    { linear search }
             Function  bfind  (st : string) : STRA_NdxType;                 { if sorted - binary }
             Function  lfind  (st : string) : STRA_NdxType;                 { if NOT sorted - linear }
             Function  find   (st : string) : STRA_NdxType;                 { exact match, sorted or not }
             Function  findsection(secttag,sectname : string) : STRA_NdxType;
                              { scan for match/ 0 if not }
             Procedure done;
             end;



{SECTION .HOLD_object }

{ Two run time sized arrays on the Heap.  One is a STRA of "keys",
    the other an array of longints (record pointers or text file offsets).
    "Key"s can be searched and sorted.
    Limit 16,380 }

type  HOLD_NumType     = longint;
type  HOLD_NdxType     = word;
type  HOLD_BigIndex    = array[1..STRA_BigArrayMax] of HOLD_NumType;


type  HOLD_object = OBJECT(STRA_object)
          ArrNum     : ^HOLD_BigIndex;
          ArrHighVal : HOLD_NumType;
          MaxEntries : HOLD_NdxType;
          sepchar    : char;      { separator between key and value ';' }

          comment    : string[80];

          Procedure init     (n : HOLD_NdxType);
          Function  append   (                st :string;     Num :HOLD_NumType): Boolean;
          Function  storeN   (n : HOLD_NdxType;    st :string;     Num :HOLD_NumType): Boolean;
          Function  fetchN   (n : HOLD_NdxType;var st :string; var Num :HOLD_NumType): Boolean;
          Function  fetchNumN(n : HOLD_NdxType)   : HOLD_NumType;
          Function  fetchStrN(n : HOLD_NdxType)   : string;

          Function  findstr  (st  : string)    : HOLD_NdxType;
          Function  findnum  (Num : HOLD_NumType)   : HOLD_NdxType;

          Function  count                    : HOLD_NdxType;
          Function  HighNum                  : HOLD_NumType;
          Procedure swap     (i,j : HOLD_NdxType);
          Procedure sort;
          Procedure dump;
          Procedure dumpN    (n : HOLD_NdxType);
          Procedure save     (fname : string);
          Procedure load     (fname : string);
          Procedure done;
          end;



{SECTION .INFO_object }

{ Two STRA arrays of strings, one treated as a "key", the other a "value".
  Key Array is sorted at load and can be searched.  Limit 16,380
  Key MUST be unique, duplicates will store on top of each other.
  }

type  INFO_object = object
          infoheader             : STR_object;
          keystring,keyvalue     : STRA_object;
          UpShiftKey             : boolean;
          sepchar                : char;               { separator between key and value ';' }

          Procedure init    (max : STRA_NdxType);
          Procedure setsepchar(sep  : char);
          Function  count        : STRA_NdxType;
          Function  sorted       : boolean;
          Function  arraymaxsize : STRA_NdxType;

          Function  storeheader(s : string) : boolean;                      { special string }
          Function  fetchheader : string;                                   { special string }
          Function  fetchkeyN  (n : STRA_NdxType) : string;                 { fetch nth key}
          Function  fetchN     (n : STRA_NdxType) : string;                 { fetch nth item}
          Function  storeN     (n : STRA_NdxType; ks,kv : string) : boolean;{ store nth item}
          Function  append     (ks,kv : string) : boolean;                  { append to eof }

          Function  store     (ks,kv : string) : boolean;
          Function  decodeandstore(str : string) : boolean;   {does the splitting}
          Function  decodeandappend(str : string) : boolean;   {does the splitting}
          Function  fetch     (ks : string)    : string;  { fetch by key }
          Function  find      (ks : string)    : STRA_NdxType; { get the position }

          Procedure save      (fname : string);
          Procedure load      (fname : string);
          Procedure loadunique(fname : string; tch : char); {avoids checking each record}
          Procedure loadremove(fname : string; tch : char); {can trim trailing comments }
          Procedure swap      (i,j  : STRA_NdxType);
          Procedure sort;
          Procedure dump;
          Procedure clear;
          Procedure done;
          end;

type OUTINFO_proc = Procedure( var O : INFO_object);  { for XBF scanning }


{SECTION .LOCATION_object }
{ an INFO_object to hold directory names and shortcut names.  Used in
   GO & FETCH }

type LOCATION_object = object(INFO_object)
          sortflag : boolean;        { sort entries alphabetically }
          saveflag : boolean;        { save loc file on exit }
          newdir   : string[50];
          olddir   : string[50];

          Procedure init(max : STRA_NdxType);                  { init object, max entries }
          Procedure load(fn : string);                         { load previously saved INFO file }
          Function  DirNstr(n : STRA_NdxType) : string;        { fetch Nth directory }
          Function  DirTagstr(tag : string) : string;          { fetch directory by name }
          Procedure gotoDir(dir : string);                     { goto directory (CD \...) }
          Procedure gotoTag(tag : string);                     { goto directory by tag (CD \...) }
          Procedure done;                                      { cleanup object, dispose of heap }
          end;


{SECTION .TFILE_object }

{ TextFILE object.  Handy for reading and creating DOS text files. }

type  TFILE_object = OBJECT
          Fil      : TEXT;
          filename : string[60];
          opened   : boolean;
          err      : integer;
          linenum  : longint;
          PosCurr  : longint;
          encrypt  : boolean;   { set before writing }
          encrypted : boolean;  { set on each line read }

          procedure init      (fn : string; create : boolean); { Open / Create text file }
          procedure initAppend(fn : string);                   { Open for Append }
          Procedure open      (fn : string; create : boolean); { - internal open - }
          Procedure seek      (l : longint);                   { random positioning }
          function  currentposition : longint;                 { current position in file }
          Function  fetchnext(var s : string) : boolean;       { return the next text line }
          Function  append    (s : string)    : boolean;       { append text to EOF }
          Procedure OUT       (s : string);                    { append text to EOF }
          Procedure clearfile;                                 { erase all text, leave file }
          Procedure refreshfile;                               { close and open to flush buffers }
          Function  error    : boolean;                        { returns error number }
          Procedure close;                                     { close file }
          procedure done;                                      { close up object }
          Function  IOResultErrChk : boolean;                  { check for I/O errors }
          end;



{SECTION .zIMPLEMENTATION }
IMPLEMENTATION

{PAGE}
{SECTION *STR_object }
Procedure STR_object.init;
     begin
     strptr := NIL;
     end;


procedure STR_object.dispose;
var  nbytes: Word;
      begin
      IF strptr <> NIL then
          begin
          nbytes := Length(strptr^) + 1;
          FreeMem (strptr, nbytes);
          strptr := NIL;
          end;
      end;


Function STR_object.store (st: String): boolean;
var  nbytes: Word;
     begin
     if strptr <> NIL then dispose;
     nbytes := Length (st) + 1;
     IF MaxAvail < nbytes then  store := False
     else begin
          GetMem (strptr, nbytes);
          Move (st, strptr^, nbytes);
          store := True;
          end;
      end;


Function STR_object.fetch: String;
      begin
      IF strptr = NIL then
           fetch := ''
      ELSE fetch := strptr^;
      end;


Procedure STR_object.dump;
      begin
      writeln('STR_object dump: ','{',seg(strptr):5,':',ofs(strptr):4,'}',
                '   ',length(strptr^),'  ',strptr^);
      end;





{SECTION  *STRA_object }
Procedure STRA_object.init(max : STRA_NdxType);
var l : longint;
    i : STRA_NdxType;
     begin
     if max > STRA_BigArrayMax then max := STRA_BigArrayMax;
     arrayptr    := NIL;
     arraymax    := 0;
     arrayused   := 0;
     workcount   := 0;
     encryptflag := false;
     encrypted   := false;
     arraysorted := false;
     l := sizeof(STR_object) * max;
     if memavail > l then
          begin
          getmem(arrayptr,l);
          arraymax := max;
          arrayused := 0;
          for i := 1 to arraymax do arrayptr^[i].init;
          modified    := false;
          end;
     end;


procedure STRA_object.done;
var l : longint;
    i : STRA_NdxType;
    ok : boolean;
     begin
     l := sizeof(STR_object) * arraymax;
     IF (arrayptr <> NIL) and (l > 0) then
          begin
          for i := 1 to arraymax do arrayptr^[i].dispose;
          FreeMem (arrayptr,l);
          arrayptr := NIL;
          end;
     arrayused := 0;
     arraysorted := false;
     end;



Procedure STRA_object.clear;
var i  : STRA_NdxType;
    ok : boolean;
     begin
     encrypted   := false;
     if arrayused < 1 then exit;
     if arrayptr <> NIL then
          begin
          for i := 1 to arrayused do arrayptr^[i].dispose;
          arrayused := 0;
          modified  := false;
          end;
     end;


Function  STRA_object.Count : STRA_NdxType;
     begin
     Count := arrayused;
     end;


Function  STRA_object.bytecount : longint;
var i : STRA_NdxType;
    z : longint;
     begin
     z := 0;
     for i := 1 to count do z := z + length(fetchN(i));
     bytecount := z;
     end;


Function  STRA_object.sorted : boolean;
     begin
     sorted := arraysorted;
     end;


Function STRA_object.append(st : string) : boolean;
var OK : boolean;
     begin
     OK := false;
     if (arrayused < arraymax) and (MaxAvail > (length(st)+10)) then
          begin
          inc(arrayused);
          OK := arrayptr^[arrayused].Store(st);
          arraysorted := false;
          modified    := true;
          end;
     append := OK;
     end;



Function STRA_object.appendpush(st : string) : boolean;
var OK : boolean;
     begin
     OK := true;
     if (arrayused = arraymax) then ok := deleteN(1);
     if OK then OK := STRA_object.storeN(arraymax,st);
     appendpush := OK;
     end;



Function STRA_object.storeN (n : STRA_NdxType; st : string): boolean;
var OK : boolean;
     begin
     OK := false;
     if (n > 0) and (n <= arraymax) and (MaxAvail > (length(st)+10)) then
          begin
          if n > arrayused then arrayused := n;
          OK := arrayptr^[n].Store(st);
          modified    := true;
          arraysorted := false;
          end;
     storeN := OK;
     end;


Function STRA_object.fetchN(n : STRA_NdxType) : string;
var s : string;
     begin
     s := '';
     if (n > 0) and (n <= arrayused) then
          begin
          s := arrayptr^[n].fetch;
          end;
     fetchN := UnCompressStr(s);
     end;


Function  STRA_object.insertN(n : STRA_NdxType;st : string):boolean;
{ append the item to the array, then bubble down to position }
var ok : boolean;
    i  : STRA_NdxType;
     begin
     ok := STRA_object.append(st);
     if ok then
          begin
          modified := true;
          if (n+1) < count then
               begin
               for i := count-1 downto n+1 do swap(i+1,i);
               end;
          end;
     insertN := ok;
     end;


Function  STRA_object.deleteN(n : STRA_NdxType):boolean;
{ for now, just bubble the item to the end, replace with
    null string and decrement the count - this leaves some
    heap garbage which I will ignore for now }

var ok : boolean;
    i  : STRA_NdxType;
     begin
     if n <= count then
          begin
          if (n+1) < count then
               begin
               for i := n to count-1 do swap(i+1,i);
               end;
          ok := STRA_object.storeN(count,'');
          dec(arrayused);
          modified := true;
          end;
     deleteN := ok;
     end;



Procedure STRA_object.dump;
var i  : STRA_NdxType;
     begin
     if arrayused < 1 then exit;
     for i := 1 to arrayused do
          begin
          writeln(i:4,' [',arrayptr^[i].fetch,']  ');
          end;
     writeln('');
     end;


Procedure STRA_object.listpage(f,n,w : STRA_NdxType);
var i  : STRA_NdxType;
     begin
     if (f > arrayused) or (arrayused < 1) then exit;
     i := f;
     if i < 1 then i := 1;
     while (i < (f+n)) do
          begin
          writeln(leftstr(arrayptr^[i].fetch,w-1));
          inc(i);
          end;
     end;


Procedure STRA_object.save(fname : string);
var i  : STRA_NdxType;
    OK : boolean;
    TEXTF : TFILE_object;
     begin
     if arrayused < 1 then exit;
     TEXTF.init(fname,true);
     TEXTF.encrypt := encryptflag;
     for i := 1 to arrayused do
          begin
          ok := TEXTF.append(STRA_object.fetchN(i));
          end;
     TEXTF.done;
     end;


Procedure STRA_object.load(fname : string);
var s : string;
    OK : boolean;
    TEXTF : TFILE_object;
     begin
     TEXTF.init(fname,false);
     ok := TEXTF.opened;
     while ok do
          begin
          ok := TEXTF.fetchnext(s);
          if ok then
               begin
               ok := STRA_object.append(s);
               encrypted := TEXTF.encrypted;
               end;
          end;
     modified := false;
     TEXTF.done;
     end;


Procedure STRA_object.loadsection(fname,sectiontag,sectionname : string);
var secttag,sectname  : string[40];
    sectlen   : STRA_NdxType;
    ok, found : boolean;
    s         : string;
    TEXTF     : TFILE_object;
     begin
     found := false;
     secttag  := UpcaseStr(sectiontag);
     sectname := UpcaseStr(sectionname);
     trim(sectname);
     sectlen  := length(sectname);
     TEXTF.init(fname,false);
     ok := TEXTF.opened;
     while ok do
          begin
          ok := TEXTF.fetchnext(s);
          if ok then
               begin
               if secttag = leftstr(UpCaseStr(s),length(secttag)) then
                     begin
                     if found then
                          begin
                          found := false;
                          ok := false;
                          end
                     else begin
                          delete(s,1,length(secttag));
                          RemoveLeading(s,' ');
                          if leftstr(UpCaseStr(s),sectlen) = sectname then
                                found := true;
                          end;
                     end
               else if found then ok := STRA_object.append(s);
               end;
          end;
     modified := false;
     TEXTF.done;
     end;


Procedure STRA_object.loadsectlist(fname,sectiontag : string); { just load all of the section names }
var secttag   : string[40];
    ok, found : boolean;
    s,s1      : string;
    TEXTF     : TFILE_object;
     begin
     found := false;
     secttag  := UpcaseStr(sectiontag);
     TEXTF.init(fname,false);
     ok := TEXTF.opened;
     while ok do
          begin
          ok := TEXTF.fetchnext(s);
          if ok then
               begin
               if secttag = leftstr(UpCaseStr(s),length(secttag)) then
                     begin
                     delete(s,1,length(secttag));
                     RemoveLeading(s,' ');
                     s1 := GetLeftStr(s,' ');
                     ok := STRA_object.append(s1);
                     end;
               end;
          end;
     TEXTF.done;
     end;


Function  STRA_object.checksort : boolean;
var ok : boolean;
    i  : STRA_NdxType;
     begin
     ok := true;
     i := 0;
     if count > 1 then
          begin
          while ok and (i < count-1) do
               begin
               inc(i);
               if arrayptr^[i].strptr^ >= arrayptr^[i+1].strptr^ then
                     ok := false;
               end;
          end;
     checksort := ok;
     end;


Procedure STRA_object.SortPreProcess;
     { preserve original order with duplicate entries - important for HOLD_object }
var i : integer;
     begin
     for i := 1 to count do
           storeN(i,fetchN(i)+chr(0)+integerstr(i,5));
     end;


Procedure STRA_object.SortPostProcess;
     { undo the pre-process }
var i : integer;
     begin
     for i := 1 to count do
           storeN(i,LeftStrCh(fetchN(i),chr(0)));
     end;


{$R-}
{ a long time ago I had subtle problems with range check, I havent
  had the time to remove this and trace the problem - no problems last 3 years }

Procedure STRA_object.swap(i,j : STRA_NdxType);
var sptr : StringPtr_type;
     begin
     inc(workcount);
     sptr := arrayptr^[i].strptr;
     arrayptr^[i].strptr := arrayptr^[j].strptr;
     arrayptr^[j].strptr := sptr;
     modified := true;
     end;


procedure STRA_object.sort;
var Gap,I,J,N  : INTEGER;   { !?! 7/13/94 - algorithm needs it }
    s1,s2      : StringPtr_type;
     begin
     workcount := 0;
     if arraysorted then exit;
     SortPreProcess;
     N   := STRA_object.count;
     Gap := N div 2;
     while (Gap > 0) do
         begin
         I := Gap;
         while (I < N) do
              begin
              J := I - Gap;
              s1 := arrayptr^[J+Gap+1].strptr;
              s2 := arrayptr^[J+1].strptr;
              while (J >= 0) and (s1^ < s2^) do
                   begin
                   STRA_object.swap(J+1,J+Gap+1);
                   dec(J,Gap);
                   s1 := arrayptr^[J+Gap+1].strptr;
                   s2 := arrayptr^[J+1].strptr;
                   end;
              inc(I);
              end;
         Gap:=Gap div 2;
         end;
     SortPostProcess;
     modified := true;
     arraysorted := true;
     end;
{$R+}



Function STRA_object.search(st : string; mode : byte) : STRA_NdxType;
var n : STRA_NdxType;
    found : boolean;
    s     : string;
     begin
     workcount := 0;
     n := 0;
     s := UpCaseStr(st);
     if (arrayused > 0) then
          begin
          found := false;
          while (n < arrayused) and not found do
               begin
               inc(n); inc(workcount);
               found := CompareStrs(s,arrayptr^[n].fetch,mode,BOTHcasemode);
               end;
          end;
     if not found then n := 0;
     search := n;
     end;


Function STRA_object.bfind(st : string) : STRA_NdxType;  {exact match}
var i,n,p : STRA_NdxType;
     begin
     workcount := 0;
     p := 0;
     n := arrayused;
     while (n > 1) do
          begin
          n := (n + 1) div 2;
          inc(workcount);
          if (st = arrayptr^[p+n].strptr^) then
               begin
               bfind := (p+n);
               exit;
               end
          else begin
               inc(workcount);
               if (st > arrayptr^[p+n].strptr^)  then
                    p := p + n;
               end;
          end;
     bfind := 0;
     end;



Function STRA_object.lfind(st : string) : STRA_NdxType;  {exact match}
var n : STRA_NdxType;
     begin
     n := 0;
     while (n < arrayused) do
          begin
          inc(n);
          if length(st) = length(arrayptr^[n].strptr^) then
               begin
               if st = arrayptr^[n].strptr^ then
                    begin
                    lfind := n;
                    exit;
                    end;
               end;
          end;
     lfind := 0;
     end;


Function STRA_object.find(st : string) : STRA_NdxType;
     begin
     if arraysorted then
          find := STRA_object.bfind(st)
     else find := STRA_object.lfind(st);
     end;

Function  STRA_object.findsection(secttag,sectname : string) : STRA_NdxType;
                              { scan for match/ 0 if not }
var n : STRA_NdxType;
    s : string;
     begin
     n := 0;
     while (n < arrayused) do
          begin
          inc(n);
          s := CheckSectionID(arrayptr^[n].strptr^,secttag);
          if s <> '' then
               begin
               if CompareUpL(s,sectname,length(sectname)) then
                    begin
                    findsection := n;
                    exit;
                    end;
               end;
          end;
     findsection := 0;
     end;



{PAGE}
{SECTION  *HOLD_object }
Procedure HOLD_object.init(n : HOLD_NdxType);
var l : longint;
    i : HOLD_NdxType;
     begin
     if n > STRA_BigArrayMax then n := STRA_BigArrayMax;
     MaxEntries := n;
     ArrHighVal := 0;
     comment := '';
     sepchar := ';' ;
     STRA_object.init(n);
     l := sizeof(HOLD_NumType) * n;
     if memavail > l then
          begin
          getmem(ArrNum,l);
          for i := 1 to arraymax do ArrNum^[i] := 0;
          end;
     end;


Procedure HOLD_object.done;
var l : longint;
     begin
     ArrHighVal := 0;
     l := sizeof(HOLD_NumType) * arraymax;
     IF (ArrNum <> NIL) and (l > 0) then
          begin
          FreeMem (ArrNum,l);
          ArrNum := NIL;
          end;
     STRA_object.done;
     end;



Function  HOLD_object.HighNum  : HOLD_NumType;
     begin
     HighNum := ArrHighVal;
     end;


Function  HOLD_object.findstr(st : string) : HOLD_NdxType;
var i,j  : HOLD_NdxType;
     begin
     j := 0;
     i := STRA_object.find(st);
     if i > 0 then j := i;
     findstr := j;
     end;


Function  HOLD_object.findnum(Num : HOLD_NumType) : HOLD_NdxType;
var i,j  : HOLD_NdxType;
    alldone : boolean;
     begin
     j := 0;
     alldone := false;
     if Num <= ArrHighVal then
          begin
          i := 0;
          while (i < ArrayUsed) and not alldone do
               begin
               inc(i);
               if ArrNum^[i] = Num then
                    begin
                    j := i;
                    alldone := true;
                    end;
               end;
          end;
     findnum := j;
     end;


Function  HOLD_object.count : HOLD_NdxType;
     begin
     count := ArrayUsed;
     end;



Function  HOLD_object.fetchNumN (n : HOLD_NdxType) : HOLD_NumType;
     begin
     if (n > 0) and (n <= ArrayUsed) then
          fetchNumN := ArrNum^[n]
     else fetchNumN := 0;
     end;


Function  HOLD_object.fetchStrN (n : HOLD_NdxType) : string;
     begin
     fetchStrN := STRA_object.fetchN(n);
     end;


Function HOLD_object.fetchN(n : HOLD_NdxType;var st :string; var Num :HOLD_NumType):boolean;
var ok : boolean;
     begin
     ok := true;
     if n > arrayused then ok := false;
     Num := fetchNumN(n);
     st := fetchStrN(n);
     fetchN := ok;
     end;


Function HOLD_object.append(st : string; Num : HOLD_NumType) : boolean;
var OK : boolean;
     begin
     OK := STRA_object.append(st);
     if OK then ArrNum^[ArrayUsed] := Num;
     if Num > ArrHighVal then ArrHighVal := Num;
     append := OK;
     end;


Function HOLD_object.storeN (n : HOLD_NdxType; st : string; Num : HOLD_NumType): Boolean;
var OK : boolean;
     begin
     OK := STRA_object.storeN(n,st);
     if OK then ArrNum^[n] := Num;
     if Num > ArrHighVal then ArrHighVal := Num;
     storeN := OK;
     end;


{$R-}

Procedure HOLD_object.swap(i,j : HOLD_NdxType);
var sptr  : StringPtr_type;
    Num   : HOLD_NumType;
     begin
     STRA_object.swap(i,j);
     Num   := ArrNum^[i];
     ArrNum^[i] := ArrNum^[j];
     ArrNum^[j] := Num;
     end;


procedure HOLD_object.sort;    {sorts based on string value }
var Gap,I,J,N  : integer;   { Algorithm needs negative 7/13/94 }
    s1,s2      : StringPtr_type;
     begin
     if arraysorted then exit;
     STRA_object.SortPreProcess;
     N   := STRA_object.count;
     Gap := N div 2;
     while (Gap > 0) do
         begin
         I := Gap;
         while (I < N) do
              begin
              J := I - Gap;
              s1 := arrayptr^[J+Gap+1].strptr;
              s2 := arrayptr^[J+1].strptr;
              while (J >= 0) and (s1^ < s2^) do
                   begin
                   HOLD_object.swap(J+1,J+Gap+1);
                   dec(J,Gap);
                   s1 := arrayptr^[J+Gap+1].strptr;
                   s2 := arrayptr^[J+1].strptr;
                   end;
              inc(I);
              end;
         Gap:=Gap div 2;
         end;
     STRA_object.SortPostProcess;
     arraysorted := true;
     end;

{$R+}




Procedure HOLD_object.dumpN(n : HOLD_NdxType);
var i  : HOLD_NdxType;
     begin
     if ArrayUsed < 1 then exit;
     if n > ArrayUsed then n := arrayused;
     writeln('dump   used: ',arrayused,'   max: ',ArrHighVal);
     for i := 1 to n do
          begin
          writeln(i:4,'  str [',arrayptr^[i].fetch,']  num [ ',
                  ArrNum^[i]:5,' ]');
          end;
     end;


Procedure HOLD_object.dump;
var i  : HOLD_NdxType;
     begin
     dumpN(9999);
     end;



Procedure HOLD_object.save(fname : string);
var i   : HOLD_NdxType;
    ok  : boolean;
    s   : string;
    tx  : TFILE_object;
     begin
     if arrayused < 1 then exit;
     tx.init(fname,true);
     if comment <> '' then
          begin
          s := comment;
          trim(s);
          ok := tx.append('!'+s);
          end;
     for i := 1 to arrayused do
          begin
          s := longintstr(ArrNum^[i],8);
          trim(s);
          ok := tx.append(trimstr(arrayptr^[i].fetch)+sepchar+s);
          end;
     tx.done;
     end;


Procedure HOLD_object.load(fname : string);
var s,st  : string;
    num   : HOLD_NumType;
    ok    : boolean;
    tx    : TFILE_object;
     begin
     comment := '';
     tx.init(fname,false);
     num  := 0;
     while tx.fetchnext(s) do
          begin
          if (num=0) and (s[1]='!') then
               begin
               delete(s,1,1);
               comment := s;
               end
          else begin
               st  := GetLeftStr(s,sepchar);
               num := strlong(GetString(s));
               ok  := HOLD_object.append(st,num);
               end;
          end;
     tx.done;
     end;



{PAGE}
{SECTION  *INFO_object }
Procedure INFO_object.init(max : STRA_NdxType);
var l : longint;
    i : STRA_NdxType;
     begin
     sepchar := ';';   { separator between key and data }
     UpShiftKey := true;
     infoheader.init;
     keystring.init(max);
     keyvalue.init(max);
     end;


procedure INFO_object.setsepchar(sep : char);
     begin
     sepchar := sep;   { separator between key and data }
     end;


procedure INFO_object.done;
var l : longint;
    i : STRA_NdxType;
    ok : boolean;
     begin
     l := sizeof(STR_object) * keystring.arraymax;
     IF (keystring.arrayptr <> NIL) and (l > 0) then
          begin
          for i := 1 to keystring.arraymax do
               begin
               keystring.arrayptr^[i].dispose;
               keyvalue.arrayptr^[i].dispose;
               end;
          end;
     IF (keystring.arrayptr <> NIL) and (l > 0) then
          begin
          FreeMem (keystring.arrayptr,l);
          keystring.arrayptr := NIL;
          FreeMem (keyvalue.arrayptr,l);
          keyvalue.arrayptr := NIL;
          end;
     keystring.arrayused := 0;
     keystring.arraysorted := false;
     keyvalue.arrayused := 0;
     keyvalue.arraysorted := false;
     end;


procedure INFO_object.clear;
var l : longint;
    i : STRA_NdxType;
    ok : boolean;
     begin
     keystring.clear;
     keyvalue.clear;
     end;



Function  INFO_object.Count : STRA_NdxType;
     begin
     Count := keystring.count;
     end;


Function  INFO_object.sorted : boolean;
     begin
     sorted := keystring.sorted;
     end;


Function  INFO_object.ArrayMaxSize : STRA_NdxType;
     begin
     ArrayMaxSize := keystring.ArrayMax;
     end;


Function  INFO_object.storeheader (s : string) : boolean;
     begin
     storeheader := infoheader.store(s);
     end;


Function  INFO_object.fetchheader : string;
     begin
     fetchheader := infoheader.fetch;
     end;


Function INFO_object.fetchkeyN(n : STRA_NdxType) : string;
var s : string;
     begin
     s := '';
     if n > 0 then s := keystring.fetchN(n);
     fetchkeyn := s;
     end;


Function INFO_object.fetchN(n : STRA_NdxType) : string;
var s : string;
     begin
     s := '';
     if n > 0 then s := keyvalue.fetchN(n);
     fetchn := s;
     end;


Function INFO_object.storeN(n : STRA_NdxType; ks,kv : string) : boolean;
var OK : boolean;
    s  : string;
     begin
     if UpShiftKey then s := UpCaseStr(ks)
     else s := ks;
     if n > 0 then
          begin
          OK := keystring.storeN(n,s);
          if OK then OK := keyvalue.storeN(n,kv);
          end
     else begin
          OK := keystring.append(s);
          if OK then OK := keyvalue.append(kv);
          end;
     storeN := OK;
     end;


Function INFO_object.append(ks,kv : string) : boolean;
var OK : boolean;
    s  : string;
     begin
     if UpShiftKey then s := UpCaseStr(ks)
     else s := ks;
     OK := keystring.append(s);
     if OK then OK := keyvalue.append(kv);
     append := OK;
     end;


Function INFO_object.store(ks,kv : string) : boolean;
var OK : boolean;
    n  : STRA_NdxType;
    s  : string;
     begin
     if UpShiftKey then s := UpCaseStr(ks)
     else s := ks;
     n := keystring.find(s);
     if n > 0 then
          begin
          OK := keystring.storeN(n,s);
          if OK then OK := keyvalue.storeN(n,kv);
          end
     else begin
          OK := keystring.append(s);
          if OK then OK := keyvalue.append(kv);
          end;
     store := OK;
     end;


Function INFO_object.decodeandstore(str : string) : boolean;
var ks,kv : string;
    i     : integer;
    ok    : boolean;
     begin
     ok := false;
     kv := str;
     i := pos(sepchar,kv);
     if i > 0 then
          begin
          ks := GetLeftStr(kv,sepchar);
          trim(ks); trim(kv);
          ok := INFO_object.store(ks,kv);
          end;
     decodeandstore := ok;
     end;


Function INFO_object.decodeandappend(str : string) : boolean;
var ks,kv : string;
    i     : integer;
    ok    : boolean;
     begin
     ok := false;
     kv := str;
     i := pos(sepchar,kv);
     if i > 0 then
          begin
          ks := GetLeftStr(kv,sepchar);
          trim(ks); trim(kv);
          ok := INFO_object.append(ks,kv);
          end;
     decodeandappend := ok;
     end;


Function INFO_object.fetch(ks : string) : string;
var n     : STRA_NdxType;
    s,str : string;
     begin
     if UpShiftKey then s := UpCaseStr(ks)
     else s := ks;
     str := '';
     n := keystring.find(s);
     if n > 0 then str := keyvalue.fetchN(n);
     fetch := str;
     end;



Function  INFO_object.find(ks : string) : STRA_NdxType;
{ returns index number of matching key}
var s : string;
     begin
     if UpShiftKey then s := UpCaseStr(ks)
     else s := ks;
     find := keystring.search(s,EQmode);
     end;


Procedure INFO_object.dump;
var i  : STRA_NdxType;
     begin
     writeln('Info object dump  ', keystring.count);
     if keystring.count < 1 then exit;
     for i := 1 to keystring.count do
          begin
          writeln(i:4,' [',keystring.fetchN(i),']  [',
                           keyvalue.fetchN(i),']');
          end;
     writeln('');
     end;


Procedure INFO_object.save(fname : string);
var i  : STRA_NdxType;
    OK : boolean;
    prefix : string[2];
    TEXTF : TFILE_object;
     begin
     if keystring.count < 1 then exit;
     TEXTF.init(fname,true);
     if keystring.sorted then prefix := '**'
     else prefix := '*';
     TEXTF.append(prefix+infoheader.fetch);
     for i := 1 to keystring.count do
         begin
         TEXTF.append(keystring.fetchN(i)+sepchar+keyvalue.fetchN(i));
         end;
     TEXTF.done;
     end;


Procedure INFO_object.load(fname : string);
var s     : string;
    TEXTF : TFILE_object;
    sortedflag : boolean;
     begin
     sortedflag := false;
     TEXTF.init(fname,false);
     while TEXTF.fetchnext(s) do
         begin
         if (INFO_object.count = 0) and (s[1] = '*') then
              begin
              delete(s,1,1);
              if s[1] = '*' then
                   begin
                   delete(s,1,1);
                   sortedflag := true;
                   end;
              infoheader.store(s);
              end
         else decodeandstore(s);
         end;
     TEXTF.done;
     if sortedflag then
          begin
          keystring.modified := false;
          keystring.arraysorted := true;
          end
     else sort;
     end;


Procedure INFO_object.loadremove(fname : string; tch : char);
var s,s1 : string[127];
    sortedflag : boolean;
    TEXTF : TFILE_object;
     begin
     sortedflag := false;
     TEXTF.init(fname,false);
     while TEXTF.fetchnext(s) do
         begin
         if (INFO_object.count = 0) and (s[1] = '*') then
              begin
              delete(s,1,1);
              if s[1] = '*' then
                   begin
                   delete(s,1,1);
                   sortedflag := true;
                   end;
              infoheader.store(s);
              end
         else begin
              s1 := GetLeftStr(s,tch);
              decodeandstore(s1);
              end;
         end;
     TEXTF.done;
     if sortedflag then
          begin
          keystring.modified := false;
          keystring.arraysorted := true;
          end
     else sort;
     end;


Procedure INFO_object.loadunique(fname : string; tch : char);
var s,s1 : string[127];
    TEXTF : TFILE_object;
    sortedflag : boolean;
     begin
     sortedflag := false;
     TEXTF.init(fname,false);
     while TEXTF.fetchnext(s) do
         begin
         if (INFO_object.count = 0) and (s[1] = '*') then
              begin
              delete(s,1,1);
              if s[1] = '*' then
                   begin
                   delete(s,1,1);
                   sortedflag := true;
                   end;
              infoheader.store(s);
              end
         else begin
              s1 := GetLeftStr(s,tch);
              decodeandappend(s1);
              end;
         end;
     TEXTF.done;
     if sortedflag then
          begin
          keystring.modified := false;
          keystring.arraysorted := true;
          end
     else sort;
     end;


{$R-}

Procedure INFO_object.swap(i,j : STRA_NdxType);
     begin
     keystring.swap(i,j);
     keyvalue.swap(i,j);
     end;


procedure INFO_object.sort;
var Gap,I,J,N : STRA_NdxType;
    s1,s2      : string;
     begin
     if sorted then exit;
     N   := INFO_object.count;
     Gap := N div 2;
     while (Gap > 0) do
         begin
         I := Gap;
         while (I < N) do
              begin
              J := I - Gap;
              s1 := keystring.fetchN(J+Gap+1);
              s2 := keystring.fetchN(J+1);
              while (J >= 0) and (s1 < s2) do
                   begin
                   INFO_object.swap(J+1,J+Gap+1);
                   dec(J,Gap);
                   s1 := keystring.fetchN(J+Gap+1);
                   s2 := keystring.fetchN(J+1);
                   end;
              inc(I);
              end;
         Gap:=Gap div 2;
         end;
     keystring.modified := false;
     keystring.arraysorted := true;
     end;
{$R+}



{PAGE}
{SECTION  *LOCATION_object }
Procedure LOCATION_object.init(max : STRA_NdxType);
     begin
     GetDir(0,olddir);
     newdir := '';
     INFO_object.init(max);
     sortflag := true;
     saveflag := true;
     end;


Procedure LOCATION_object.load(fn : string);
     begin
     INFO_object.SetSepChar('=');
     if fn <> '' then pLocFile := fn; { PbDATA, def "c:\location.loc"}
     INFO_object.load(pLocFile);
     if sortflag then INFO_object.sort;
     end;


Function  LOCATION_object.DirNstr(n : STRA_NdxType) : string;
     begin
     DirNstr := keyvalue.fetchN(n);
     end;


Function  LOCATION_object.DirTagstr(tag : string) : string;
var n : STRA_NdxType;
     begin
     n := keystring.find(tag);
     if n > 0 then
          DirTagstr := keyvalue.fetchN(n)
     else DirTagstr := '';
     end;


Procedure LOCATION_object.gotoDir(dir : string);
var fn : string;
     begin
     newdir := dir;
     writeln('GO TO [',newdir,']');
     if DirExistsMSG(addbackslash(newdir),'',' ') then ChangeDir(newdir);
     end;


Procedure LOCATION_object.gotoTag(tag : string);
var n : STRA_NdxType;
     begin
     n := keystring.find(tag);
     if n > 0 then
          begin
          newdir := keyvalue.fetchN(n);
          gotoDir(newdir);
          end
     else begin
          writeln('gotoTag - not found [',tag,']');
          end;
     end;


Procedure LOCATION_object.done;
     begin
     INFO_object.store('.',olddir);
     if saveflag then INFO_object.save(pLocFile);
     INFO_object.done;
     end;


{PAGE}
{SECTION  *TFILE_object }
procedure TFILE_object.init(fn : string; create : boolean);
var fdir   : string[40];
    fname  : string[8];
    fext   : string[4];
     begin
     opened   := false;
     filename := fn;
     linenum  := 0;
     PosCurr  := 0;
     err := 0;
     encrypt := false;
     encrypted := false;
     if create or FileExists(filename) then
             TFILE_object.open(filename,create);
     end;


procedure TFILE_object.InitAppend(fn : string);
var fdir   : string[40];
    fname  : string[8];
    fext   : string[4];
     begin
     opened   := false;
     filename := fn;
     linenum  := 0;
     PosCurr  := 0;
     err := 0;
     encrypt := false;
     encrypted := false;
     if not FileExists(fn) then
          begin
          TFILE_object.open(filename,true);
          end
     else begin
          assign(fil,fn);
          {$I-} System.Append(fil); {$I+}
          opened := not IOResultErrChk;
          end;
     end;


procedure TFILE_object.done;
     begin
     TFILE_object.close;
     end;


Function TFILE_object.IOResultErrChk : boolean;
     begin
     err := IORESULT;
     if err = 0 then
          IOResultErrChk := false
     else IOResultErrChk := true;
     if err <> 0 then
          writeln('tfile error [',filename,']  ',err);
     end;



Procedure TFILE_object.open(fn : string; create : boolean);
     begin
     linenum := 0;
     encrypted := false;
     if opened then TFILE_object.close;
     assign(fil,fn);
     if create then
          {$I-} ReWrite(fil) {$I+}
     else begin
          {$I-} Reset(fil); {$I+}
          end;
     opened := not IOResultErrChk;
     end;


Procedure TFILE_object.seek(l : longint);
var ok : boolean;
    ch  : char;
     begin
     if opened then ok := TextSeek(fil,l);
     PosCurr := l;
     linenum := -1; {no way top compute}
     end;


Function  TFILE_object.currentposition : longint;
     begin
     PosCurr := textpos(fil);
     currentposition := PosCurr;
     end;


Function TFILE_object.fetchnext(var s : string) : boolean;
var ok : boolean;
     begin
     ok := false;
     s := '';
     if opened and not EOF(fil) then
          begin
         {$I-} SYSTEM.readln(fil,s); {$I+}
          if not IOResultErrChk then
               begin
               inc(linenum);
               ok := true;
               if IsNCHANTed(s) then
                    begin
                    encrypted := true;
                    s := UnCompressStr(RVERTstr(s)); { decryption }
                    end;
               end;
          end;
     fetchnext := ok;
     end;


Function  TFILE_object.append(s : string) : boolean;
var ok : boolean;
    s1 : string;
     begin
     ok := false;
     if encrypt then s1 := NCHANTstr(s)
     else s1 := s;
     if opened then
          begin
         {$I-}
          SYSTEM.writeln(fil,s1);
         {$I+}
          if not IOResultErrChk then
               begin
               inc(linenum);
               ok := true;
               end;
          end;
     append := ok;
     end;



Procedure TFILE_object.OUT(s : string);
var ok : boolean;
    s1 : string;
     begin
     ok := false;
     if encrypt then s1 := NCHANTstr(s)
     else s1 := s;
     if opened then
          begin
         {$I-}
          SYSTEM.writeln(fil,s1);
         {$I+}
          if not IOResultErrChk then
               begin
               inc(linenum);
               ok := true;
               end;
          end;
     end;



procedure TFILE_object.close;
     begin
     if opened then
          begin
         {$I-} SYSTEM.Close(fil); {$I+}
          IOResultErrChk;
          opened := false;
          end;
     end;


procedure TFILE_object.clearfile;
var fn : string;
     begin
     fn := filename;
     TFILE_object.close;
     TFILE_object.open(fn,true);      { do a REWRITE }
     end;


procedure TFILE_object.refreshfile;
var fn : string;
     begin
     fn := filename;
     TFILE_object.close;
     TFILE_object.open(fn,false);      { do a RESET }
     end;


Function  TFILE_object.error : boolean;
     begin
     if err <> 0 then error := true
     else error := false;
     end;

end.
