program spirals;

(********************************************************************)
{ The algorithms for the spirals came from a program written for the
  HP 110 Portable by Dick Nungester, 01/02/85.

  Modified for MS-DOS EGA/CGA displays by J. F. Herlocker 05/31/88.

  Kaleidoscope added 08/02/88. Original algorithms by Tom Swan, from
  Turbo Technix May/June 1988, pg. 25 }
(********************************************************************)


{$R-}  {Range checking MUST be off for Kaleidoscope, otherwise Tom's quad
        mirror approach blows up.}

Uses
	Crt, Graph, Dos, Drivers;

const
	MaxSpirals = 34;	{ number of spirals in library }
	VersionNo = '2.5b';	{ latest version number of this program }

type
	InOutOrBoth = (Inside,Outside,Both);

	Spiral = record
		IOB		: InOutOrBoth;	{form spiral on inside of ring,
							outside, or both}
		Rr		: real;		{radius of ring}
		Rd		: real;		{radius of disk}
		Rp		: real;		{radius of pen location on disk}
		PtsPerRev	: real;		{# of points to plot for each disk
							revolution}
	  end; { Spiral record }

	Choice	=	(R,L,S,K,H,B1,B2);	{ used for parameter line choices }
								{ see HelpMsg for explanations }

var
	SpiralSet 	: array[1..MaxSpirals] of Spiral;

	GraphDriver	: integer;	{ The Graphics device driver }
	GraphMode		: integer;	{ The Graphics mode value }
	MaxX, MaxY	: word;		{ The maximum resolution of the screen }
	ErrorCode		: integer;	{ Reports any graphics errors }
	MaxColor		: word;		{ The maximum color value available }
	OldExitProc	: Pointer;	{ Saves exit procedure address }

	XCenter, YCenter 	: word;	{ Center of spirograph }
	MaxAllowedRad 		: word;	{ Max spiral radius }

	{ variables used to define spirals }
	InOutBoth	: string[1];
	Rr, Rd, Rp	: real;
	PointsPerRev	: real;

	ResultCode	: integer; 	{ result of string-to-value convert }
	MaxUserRad	: real;		{ maximum radius in user-units }
	Revs			: real;		{ number of total disk revolutions }

	ColorNum	:	integer;	{ Current foreground color }
	SpiralNum	:	integer;	{ current spiral number }

	MaxShow	:	integer;	{ most spirals that will be shown before clearing }
	Count	:	integer;	{ loop counter }

	IsEGA	:	boolean;	{ do we have an EGA? }

	ch		:	char;	{ used to clear buffer }

	Param1	:	Choice;	{ used for primary display options }
	Param2	:	Choice;	{ used for secondary options }

	BackColor	:	integer;	{ color of background }
	PalNum	:	integer;	{ current palette number }

(* * * * * * * * * * * * * * * * * * * * * * *)

{$F+}
procedure MyExitProc;
begin
	ExitProc := OldExitProc;	{ Restore exit procedure address }
	CloseGraph;			{ Shut down the graphics system }
end; { MyExitProc }
{$F-}

(* * * * * * * * * * * * * * * * * * * * * * *)

function ValText(NumberIn : real): string;	{ convert number to string }
var
	StrOut	: string[8];
begin
	if int(NumberIn) = NumberIn
		then str(NumberIn:3:0,StrOut)
		else str(NumberIn:5:2,StrOut);
	ValText := StrOut;
end;

(* * * * * * * * * * * * * * * * * * * * * * *)

procedure ParamParse;	{ see HelpMsg procedure for parameter explanations }
	begin
	if ParamCount = 0
		then Param1 := H	{ default is help message }
		else if (ParamStr(1) = 'R') or (ParamStr(1) = 'r') then Param1 := R
		else if (ParamStr(1) = 'S') or (ParamStr(1) = 's') then Param1 := S
		else if (ParamStr(1) = 'K') or (ParamStr(1) = 'k') then Param1 := K
		else if (ParamStr(1) = 'L') or (ParamStr(1) = 'l') then Param1 := L
		else Param1 := H;
	if (ParamCount = 2) and ( (ParamStr(2) = 'B1') or (ParamStr(2) = 'b1') )
			then Param2 := B1
		else if ( (ParamStr(2) = 'B2') or (ParamStr(2) = 'b2') )
			then Param2 := B2
		else Param2 := H;
	end; { ParamParse }


(* * * * * * * * * * * * * * * * * * * * * * *)

procedure RegisterDrivers;	{ register graphics drivers }

{ Note that this procedure assumes you have already turned the .BGI files
  into the unit DRIVERS.TPU, as explained in GRLINK.PAS on your TP 4.0 disk. }

	procedure Abort(Msg : string);
	begin
		Writeln(Msg, ': ', GraphErrorMsg(GraphResult));
		Delay(3000);
		Halt(1);
	end; { Abort procedure }

begin
	if RegisterBGIdriver(@CGADriverProc) < 0 then Abort('CGA');
	if RegisterBGIdriver(@EGAVGADriverProc) < 0 then Abort('EGA/VGA');
end; { RegisterDrivers procedure }

(* * * * * * * * * * * * * * * * * * * * * * *)

procedure GetSpirals(SpiralNumber : integer);

{ Initializes the library of spirals when the number passed to it
  is zero; otherwise returns the specifications of the spiral
  requested. }

begin {GetSpirals}

{initialize spiral set}

if SpiralNumber = 0 then  {passing 0 to the procedure initializes the spirals}
	begin
	with SpiralSet[1] do
		begin
			IOB		:= Both;
			Rr		:= 5;
			Rd		:= 1;
			Rp		:= 1.5;
			PtsPerRev := 20;
		end;
	with SpiralSet[2] do
		begin
			IOB		:= Outside;
			Rr		:= 20;
			Rd		:= 17;
			Rp		:= 37;
			PtsPerRev := 40;
		end;
	with SpiralSet[3] do
		begin
			IOB		:= Outside;
			Rr		:= 20;
			Rd		:= 1;
			Rp		:= 22;
			PtsPerRev := 20;
		end;
	with SpiralSet[4] do
		begin
			IOB		:= Inside;
			Rr		:= 50;
			Rd		:= 49;
			Rp		:= 1;
			PtsPerRev := 4;
		end;
	with SpiralSet[5] do
		begin
			IOB		:= Inside;
			Rr		:= 50;
			Rd		:= 1;
			Rp		:= 49;
			PtsPerRev := 6;
		end;
	with SpiralSet[6] do
		begin
			IOB		:= Outside;
			Rr		:= 100;
			Rd		:= 1;
			Rp		:= 101;
			PtsPerRev := 3.06;
		end;
	with SpiralSet[7] do
		begin
			IOB		:= Inside;
			Rr		:= 100;
			Rd		:= 1;
			Rp		:= 150;
			PtsPerRev := 3.96;
		end;
	with SpiralSet[8] do
		begin
			IOB		:= Outside;
			Rr		:= 100;
			Rd		:= 1;
			Rp		:= 101;
			PtsPerRev := 2.02;
		end;
	with SpiralSet[9] do
		begin
			IOB		:= Outside;
			Rr		:= 100;
			Rd		:= 1;
			Rp		:= 101;
			PtsPerRev := 2.05;
		end;
	with SpiralSet[10] do
		begin
			IOB		:= Inside;
			Rr		:= 100;
			Rd		:= 1;
			Rp		:= 99;
			PtsPerRev := 2.02;
		end;
	with SpiralSet[11] do
		begin
			IOB		:= Outside;
			Rr		:= 92;
			Rd		:= 1;
			Rp		:= 91;
			PtsPerRev := 3;
		end;
	with SpiralSet[12] do
		begin
			IOB		:= Inside;
			Rr		:= 49;
			Rd		:= 1;
			Rp		:= 50;
			PtsPerRev := 3;
		end;
	with SpiralSet[13] do
		begin
			IOB		:= Inside;
			Rr		:= 46;
			Rd		:= 462.319;
			Rp		:= 508.194;
			PtsPerRev := 17;
		end;
	with SpiralSet[14] do
		begin
			IOB		:= Inside;
			Rr		:= 27;
			Rd		:= 28;
			Rp		:= 1;
			PtsPerRev := 6;
		end;
	with SpiralSet[15] do
		begin
			IOB		:= Inside;
			Rr		:= 13;
			Rd		:= 1;
			Rp		:= 12;
			PtsPerRev := 14;
		end;
	with SpiralSet[16] do
		begin
			IOB		:= Inside;
			Rr		:= 19;
			Rd		:= 27.3;
			Rp		:= 46.3;
			PtsPerRev := 9;
		end;
	with SpiralSet[17] do
		begin
			IOB		:= Inside;
			Rr		:= 28;
			Rd		:= 293.45;
			Rp		:= 321.45;
			PtsPerRev := 4;
		end;
	with SpiralSet[18] do
		begin
			IOB		:= Outside;
			Rr		:= 22;
			Rd		:= 23;
			Rp		:= 1;
			PtsPerRev := 2;
		end;
	with SpiralSet[19] do
		begin
			IOB		:= Both;
			Rr		:= 2;
			Rd		:= 3;
			Rp		:= 1;
			PtsPerRev := 15;
		end;
	with SpiralSet[20] do
		begin
			IOB		:= Both;
			Rr		:= 18;
			Rd		:= 19;
			Rp		:= 37;
			PtsPerRev := 5;
		end;
	with SpiralSet[21] do
		begin
			IOB		:= Both;
			Rr		:= 12;
			Rd		:= 16.019;
			Rp		:= 4.019;
			PtsPerRev := 19;
		end;
	with SpiralSet[22] do
		begin
			IOB		:= Both;
			Rr		:= 69;
			Rd		:= 1;
			Rp		:= 70;
			PtsPerRev := 2;
		end;
	with SpiralSet[23] do
		begin
			IOB		:= Inside;
			Rr		:= 84;
			Rd		:= 1;
			Rp		:= 83;
			PtsPerRev := 16;
		end;
	with SpiralSet[24] do
		begin
			IOB		:= Both;
			Rr		:= 43;
			Rd		:= 1304.765;
			Rp		:= 1261.765;
			PtsPerRev := 8;
		end;
	with SpiralSet[25] do
		begin
			IOB		:= Inside;
			Rr		:= 86;
			Rd		:= 87;
			Rp		:= 1;
			PtsPerRev := 5;
		end;
	with SpiralSet[26] do
		begin
			IOB		:= Both;
			Rr		:= 40;
			Rd		:= 1071.67;
			Rp		:= 1111.67;
			PtsPerRev := 2;
		end;
	with SpiralSet[27] do
		begin
			IOB		:= Inside;
			Rr		:= 47;
			Rd		:= 40.9;
			Rp		:= 87.9;
			PtsPerRev := 3;
		end;
	with SpiralSet[28] do
		begin
			IOB		:= Both;
			Rr		:= 62;
			Rd		:= 63;
			Rp		:= 125;
			PtsPerRev := 8;
		end;
	with SpiralSet[29] do
		begin
			IOB		:= Inside;
			Rr		:= 74;
			Rd		:= 1;
			Rp		:= 73;
			PtsPerRev := 4;
		end;
	with SpiralSet[30] do
		begin
			IOB		:= Outside;
			Rr		:= 7;
			Rd		:= 8;
			Rp		:= 15;
			PtsPerRev := 2;
		end;
	with SpiralSet[31] do
		begin
			IOB		:= Both;
			Rr		:= 3;
			Rd		:= 1;
			Rp		:= 2;
			PtsPerRev := 14;
		end;
	with SpiralSet[32] do
		begin
			IOB		:= Inside;
			Rr		:= 4;
			Rd		:= 5;
			Rp		:= 1;
			PtsPerRev := 17;
		end;
	with SpiralSet[33] do
		begin
			IOB		:= Both;
			Rr		:= 3;
			Rd		:= 4;
			Rp		:= 7;
			PtsPerRev := 14;
		end;
	with SpiralSet[34] do
		begin
			IOB		:= Outside;
			Rr		:= 21;
			Rd		:= 22;
			Rp		:= 43;
			PtsPerRev := 5;
		end;

	end  {end of spiral set initialization}

	else  {get specific spiral characteristics}
		begin
			case SpiralSet[SpiralNumber].IOB of
				Inside :  InOutBoth := 'I';
				Outside:  InOutBoth := 'O';
				Both :	InOutBoth := 'B';
			end; {case}
			Rr := SpiralSet[SpiralNumber].Rr;
			Rd := SpiralSet[SpiralNumber].Rd;
			Rp := SpiralSet[SpiralNumber].Rp;
			PointsPerRev := SpiralSet[SpiralNumber].PtsPerRev;
		end;

end; {procedure GetSpirals}

(* * * * * * * * * * * * * * * * * * * * * * *)


procedure DispNotice;	{ display quick notice of author and escape procedure }

var
	LoopCount	:	longint;		{ used to delay display of msg, unless
							  user hits a key }
	EndLoop	:	longint;		{ when to stop }

begin
	TextBackground(blue);		{ put background color first for ClrScr }
	ClrScr;
	TextColor(cyan);
	writeln('Spirals version ', VersionNo, ', for PCs with CGA or EGA graphics boards.');
	writeln('Written by J. F. Herlocker.');
	writeln;
	writeln('Based on the HP 110 spirograph simulator by Dick Nungester');
	writeln('And the Kaleidoscope routine by Tom Swan.');
	writeln;
	writeln;
	TextColor(red);
	writeln('Press any key to exit at finish of current pattern.');
	LoopCount := 0;
	EndLoop := 20;					{ # of 1/4 secs to delay }
	repeat
		Delay(250);				{ wait 1/4 second }
		LoopCount := LoopCount +1;
		if KeyPressed then LoopCount := EndLoop;
	until LoopCount >= EndLoop;
	if KeyPressed then ch := ReadKey;		{ clear keyboard buffer }

end; { DispNotice procedure }

(* * * * * * * * * * * * * * * * * * * * * * *)

procedure HelpMsg;		{ display list of correct parameters, give user
					  chance to input via menu }
begin
	TextBackground(black);		{ put background color first for ClrScr }
	ClrScr;
	TextColor(red);
	writeln('Syntax: SPIRALS <R|K|L|S> [B1|B2]');
	writeln;
	TextColor(cyan);
	writeln('R = Random spiral patterns');
	writeln('K = "Kaleidocope" patterns');
	writeln('L = Spirals from a library of patterns are choosen at random');
	writeln('S = Library spiral patterns are stepped through one by one');
	writeln;
	writeln('B1 = "starry" background (except "K" option)');
	writeln('B2 = random background colors');
	writeln;
	writeln('Note that B1 and B2 are optional');
	writeln;
	writeln('Program shows this screen if no parameters are given.');
	writeln;
	writeln;
	write('Main option (R, K, L, S): ');
	ch := ReadKey;
	if (ch = 'R') or (ch = 'r') then Param1 := R
		else if (ch = 'K') or (ch = 'k') then Param1 := K
		else if (ch = 'L') or (ch = 'l') then Param1 := L
		else if (ch = 'S') or (ch = 's') then Param1 := S
		else Param1 := R;
	writeln(UpCase(ch));
	writeln;
	if Param1 = K then write('Secondary option (B2 or none): B')
		else write('Secondary option (B1, B2, or none): B');
	ch := ReadKey;
	if ch = '1' then Param2 := B1
		else if ch = '2' then Param2 := B2
		else Param2 := H;
	writeln(ch);
	DispNotice;	{ make sure the credits get shown }
end; { HelpMsg }

(* * * * * * * * * * * * * * * * * * * * * * *)

procedure Initialize;

{ Initialize graphics and report any errors that may occur }
begin
	ParamParse;				{ Check the command line parameters }
	if Param1 = H then HelpMsg	{ "H" means invalid parameter choice }
		else DispNotice;			{ display credits & instructions }
	RegisterDrivers;				{ register EGA and CGA drivers }

	{ when using Crt and graphics, turn off Crt's memory-mapped writes }
	DirectVideo := False;
	OldExitProc := ExitProc;			{ save previous exit proc }
	ExitProc := @MyExitProc;			{ insert our exit proc in chain }
	GraphDriver := Detect;				{ use autodetection }
	InitGraph(GraphDriver, GraphMode, '');	{ activate graphics }
	ErrorCode := GraphResult;			{ error? }
	if ErrorCode <> grOk then
		begin
			Writeln('Whoops! You must have either a CGA or EGA graphics');
			Writeln('card installed to run this program. Sorry about that.');
			Halt(1);		{ if no graphics card, why go on? }
		end;


	GetSpirals(0);		{ initialize spiral library }
	ColorNum := 1;		{ initialize global variables }
	SpiralNum := 1;
	PalNum := 1;
	Count := 1;
	IsEGA := true;				{ assume EGA }
	MaxShow := 10;				{ show 10 spirals in layers }
	if GraphDriver = CGA then
		begin
			SetGraphMode(CGAC1);  	{ 320 x 200 four color - Black,
								  LightCyan, LightMagenta, White}
			IsEGA := false;		{ not an EGA }
			MaxShow := 3;			{ layers don't look good on CGA }
		end;
	Randomize;			{ init random number generator }
	MaxColor := GetMaxColor;	{ Get the maximum allowable drawing color }
	MaxX := GetMaxX;		{ Get screen resolution values }
	MaxY := GetMaxY;
	XCenter := MaxX div 2;
	YCenter := MaxY div 2;
	MaxAllowedRad := YCenter-2;
end; { Initialize }

(* * * * * * * * * * * * * * * * * * * * * * *)

{ The next two procedures are almost verbatim from Dick Nungester's
  original code.  Very well done, as you'll see. }

procedure SpiroGraph(Rr,Rd,Rp,PtsPerRev,Revs :real; Outer :boolean);
(********************************************************************)
(*
(* Definition of terms:
(*
(* Rr = radius of ring.
(* Rd = radius of disk (disk rolls outside or inside ring).
(* Rp = radius on disk, of pen location.
(* PtsPerRev = Number of points to plot for each disk revolution.
(* Revs = total number of revolutions of disk to plot.
(* Outer = true (do outer curve), or false (do inner curve).
(*
(* X,Y = coordinate axes through center of ring.
(* Xr,Yr = coordinate axes, parallel to XY axes, but through
(*	   the center of the disc.
(* Eps = (epsilon), the angle of rotation of the disk for each plotted
(*	 point (relative to Xr,Yr axes).
(* c = coordinates of point C in the middle of the disc.
(* p = coordinates of point P where the pen is located.
(* 1,2 = coordinates of last point, this point, respectively.
(*
(*    A straightforward derivation of the equations of a spirograph
(* plot lead to:
(*
(* X = (Rr+Rd)*COS((Rd/Rr)*ALPHA) + Rp*COS(ALPHA)
(*        ^
(* Y = (Rr+Rd)*SIN((Rd/Rr)*ALPHA) + Rp*SIN(ALPHA)
(*        ^                       ^
(*    These equations define the OUTER curve, while the inner curve is
(* defined by equations with sign changes in the positions indicated
(* by ^.  ALPHA is the angle of rotation of the disk, in the Xr,Yr
(* reference frame.  Notice the first term is just the motion of
(* the center of the disk, and the second term superimposes the
(* rotation of the pen around the center of the disk.
(*    Implementing the above equations in a straightforward manner
(* is very slow to execute due to several trig functions per plotted
(* point.  The implementation that follows uses instead an iterative
(* technique, where each new point to plot is based solely on the
(* previous point and four trig CONSTANTS defined outside the plotting
(* loop.  This improved technique is about 2.5 times faster than
(* than the formula given above.
(********************************************************************)

var
	Xc1,  Yc1		: real;	{ center of disk, point C }
	Xrp1, Yrp1	: real;	{ pen, point P, relative  }
	Xp1,  Yp1		: real;	{ pen, point P, absolute  }

	Xc2,  Yc2	  	: real;	{ same as above, 'next set' }
	Xrp2, Yrp2	: real;
	Xp2,  Yp2		: real;

	Alpha		: real;	{ angle of rotation of disk }
	MaxAlpha		: real;	{ stop angle }
	Eps			: real;	{ change in Alpha each point }

	CosEpsD, SinEpsD	: real;	{ trig of angle epsilonDisk }
	CosEpsR, SinEpsR	: real;	{ trig of angle epsilonRing }

begin { SpiroGraph }

	{ Initialize }
	Alpha := 0;
	MaxAlpha := Revs*2*pi;
	Eps := 2*pi/PtsPerRev;
	CosEpsD := cos(Eps);
	SinEpsD := sin(Eps);
	CosEpsR := cos((Rd/Rr)*Eps);
	SinEpsR := sin((Rd/Rr)*Eps);

	if Outer then begin	{ Initialize OUTER curve }
		Xc1 := 0;
		Yc1 := Rr+Rd;
		Xrp1 := 0;
		Yrp1 := Rp;
	end
		else begin	  { Initialize INNER curve }
		Xc1 := 0;
		Yc1 := Rr-Rd;
		Xrp1 := 0;
		Yrp1 := -Rp;
	end;

	Xp1 := Xc1+Xrp1;	{ Absolute pen coordinates }
	Yp1 := Yc1+Yrp1;

	repeat
	if Outer then begin  { do OUTER curve }
		Xc2 := Xc1*CosEpsR-Yc1*SinEpsR;
		Yc2 := Yc1*CosEpsR+Xc1*SinEpsR;
		Xrp2 := Xrp1*CosEpsD-Yrp1*SinEpsD;
		Yrp2 := Yrp1*CosEpsD+Xrp1*SinEpsD;
	end else begin	 { do INNER curve }
		Xc2 := Xc1*CosEpsR-Yc1*SinEpsR;
		Yc2 := Yc1*CosEpsR+Xc1*SinEpsR;
		Xrp2 := Xrp1*CosEpsD+Yrp1*SinEpsD;
		Yrp2 := Yrp1*CosEpsD-Xrp1*SinEpsD;
	end;

	Xp2 := Xc2 + Xrp2;
	Yp2 := Yc2 + Yrp2;

	Line(Round(Xp1+XCenter), Round(Yp1+YCenter),
		Round(Xp2+XCenter), Round(Yp2+YCenter));

	Xc1 := Xc2;
	Yc1 := Yc2;
	Xrp1 := Xrp2;
	Yrp1 := Yrp2;
	Xp1 := Xp2;
	Yp1 := Yp2;

	Alpha := Alpha + Eps;

	until Alpha >= MaxAlpha; {end repeat}

end; { SpiroGraph }

(* * * * * * * * * * * * * * * * * * * * * * *)

procedure DoSpirals;

	begin

{ Define how many revolutions of the disk to make before the
  plot is stopped.  If Rr and Rd are both integers with
  with no common factors, Revs = Rr will always yield the
  best stopping point, since Revs*2*pi*Rd = N*2*pi*Rr is
  satisfied.  No check is made for "Rr and Rd integer with
  no common factors", but the user is told the plot stops
  after Rr revolutions of the disk, in the intro message. }

		Revs := Rr;

{ scale all radius values to the screen.  Assume all
  radii are positive values, but no magnitude assumptions. }

		if (InOutBoth = 'I')
		then
			MaxUserRad := abs(Rr - Rd) + Rp
		else
			MaxUserRad := Rr + Rd + Rp;
		if MaxUserRad = 0 then MaxUserRad := 1;
		Rr := Rr/MaxUserRad*MaxAllowedRad;
		Rd := Rd/MaxUserRad*MaxAllowedRad;
		Rp := Rp/MaxUserRad*MaxAllowedRad;


		if (InOutBoth = 'O') or (InOutBoth = 'B') then
			SpiroGraph(Rr, Rd, Rp, PointsPerRev, Revs, true);
		if (InOutBoth = 'I') or (InOutBoth = 'B') then
			SpiroGraph(Rr, Rd, Rp, PointsPerRev, Revs, false);
	end;  {DoSpirals }

(* * * * * * * * * * * * * * * * * * * * * * *)

procedure BackgroundStuff;

	procedure StarBackground;	{ put pixels randomly on the screen }

	var
		Count :	integer;
		begin
		{ rotate palettes for CGA }
		if not IsEGA then SetGraphMode( random (4) );
		if not (Param1 = K) then		{ Kaleido can't handle stars }
			for count := 1 to (MaxX*MaxY div 10) do
				PutPixel( random(MaxX), random(MaxY), random(MaxColor + 1) );
		end;	{ StarBackground }


	procedure RndBackground;		{ choose random background color for EGA,
							  random palette & background for CGA }

		begin
			BackColor := random( 16 ) + 1;
			if IsEGA then begin
				SetBkColor(BackColor);
				end
			else begin
				SetGraphMode( random (4) );	{ switch palette at random }
				SetBkColor(BackColor);
				end

		end; { RndBackground }

	begin
		case Param2 of
			B1	:	StarBackground;
			B2	:	RndBackground;
		end; {case}

	end; { BackgroundStuff }

(* * * * * * * * * * * * * * * * * * * * * * *)

procedure ShowSeq;  { show each spiral and color in sequence }

{ The loops are handled by simple counters rather than more
  sophisticated control structures to allow for the user to
  break at any point with a key press. }

	begin
		if SpiralNum > MaxSpirals then SpiralNum := SpiralNum - MaxSpirals;
		GetSpirals(SpiralNum);
		ClearViewPort;
		BackgroundStuff;		{ do background routines }
		SpiralNum := SpiralNum + 1;
		if ColorNum > MaxColor then ColorNum := ColorNum - MaxColor;
		ColorNum := ColorNum + 1;
		SetColor(ColorNum);

		if isEGA then
		begin
			if PalNum > 64 then PalNum := 1;	{ cycle through palette }
			PalNum := PalNum + 1;
			SetPalette(ColorNum,PalNum)
		end;

		DoSpirals;
		delay(250);	{ pause 1/4 sec -- just enough to avoid jerkiness }

	end; {procedure ShowSeq}

(* * * * * * * * * * * * * * * * * * * * * * *)

procedure ShowLibrary;  { show spirals from library in random order & color }

	begin
		{ get number between 1 and MaxSpirals }
          SpiralNum := random( MaxSpirals ) + 1;
		GetSpirals(SpiralNum);

		{ For the EGA, choose colors at random; for the CGA, 	}
		{ step through each of three colors. 				}
		if IsEGA then
			{ get random number between 1 and MaxColor }
			begin
				ColorNum := ColorNum + 1;
				SetColor(ColorNum);
				{ cycle through palette }
			end
			{ cycle through colors 1 - 3 }
			else
			begin
				if ColorNum > 3 then ColorNum := 1;
				SetColor(ColorNum);
				ColorNum := ColorNum + 1
			end;

		Count := Count + 1;			{ clear the screen occasionally }
		if Count = MaxShow + 1 then begin
			ClearViewPort;
			BackgroundStuff;		{ do background routines }
			Count := 1;
			end;

		DoSpirals;

	end;  {procedure ShowLibrary}

(* * * * * * * * * * * * * * * * * * * * * * *)

procedure ShowRandom;  { show spirals with random parameters }

	begin
		if random > 0.6 then InOutBoth := 'I'			{ in 60% }
			else if random > 0.7 then InOutBoth := 'O'	{ out 28% }
				else InOutBoth := 'B';				{ both 12% }
		Rr := random(100) + 2;  { real number between 2 and 101 }
		if random > 0.5 then Rd := Rr + 1
			else if random > 0.5 then Rd := 1
			else Rd := ( random * Rr) * Rr + 1;
		if random > 0.5 then Rp := Rr + Rd
			else Rp := abs(Rr - Rd);
		PointsPerRev := random(20) + 1;

		Count := Count + 1;			{ clear the screen occasionally }
		if Count = MaxShow + 1 then
			begin
				ClearViewPort;
				BackgroundStuff;		{ do background routines }
				Count := 1;
			end;

		if IsEGA then
			{ get number between 1 and MaxColor }
				begin
				ColorNum := random(MaxColor) + 1;
                    SetColor(ColorNum);
				end
			{ or step through 3 CGA colors }
			else begin
				if ColorNum > 3 then ColorNum := 1;
				SetColor(ColorNum);
				ColorNum := ColorNum + 1
				end;

		DoSpirals;

	end;  { procedure ShowRandom }

(* * * * * * * * * * * * * * * * * * * * * * *)

procedure Kaleidoscope;

{ Display four quadrant "kaleidoscope".
  This entire procedure was taken with very little modification
  from the program written by Tom Swan, appearing in TURBO TECHNIX,
  May/June 1988. }

VAR		XMax, YMax	: Integer;	{ Max X, Y values }
		X1, Y1, X2, Y2 : Integer;	{ Line endings }
		dX1,dY1,dX2,dY2 : Integer;	{ Change in X1,Y1,X2,Y2 }
		DisplayPeriod	: Word;		{ Time between clearing }
		LinePeriod	: Word;		{ Time each pattern lives }

	procedure Initialize;	{ Perform various intializations }

	begin
		DisplayPeriod := 0;		{ Force call to NewDisplayPeriod }
		XMax := GetMaxX DIV 2;	{ Set X and Y maximums to middle }
		YMax := GetMaxY DIV 2;	{   of display resolution }

	   {	Restrict viewport to 1/4 of entire display. With clipping
		off, this centers the origin (0,0) and makes mirror
		images in the four quadrants easy to draw. }

		SetViewPort( XMax, YMax, GetMaxX, GetMaxY, ClipOff );
	end; { Initialize }


	procedure NewDisplayPeriod;
	{ Clear screen and initialize DisplayPeriod,
	  controlling length of time between screen clears: }

	begin
		ClearDevice;		{ Clear entire display }
		BackgroundStuff;	{ do background routines - addition to
						  original program }
		SetViewPort( XMax, YMax, GetMaxX, GetMaxY, ClipOff );
		DisplayPeriod := 6 + random( 24 );  { 6..29 }
	end; { NewDisplayPeriod }


	procedure NewValues;
	{ Select coordinates, movements, LinePeriod,
	  and line color at random: }

	begin
		X1  := random( XMax + 1 );	{ X1 <- 0..XMax}
		Y1  := random( YMax + 1 );	{ Y1 <- 0..YMax}
		X2  := random( XMax + 1 );	{ X2 <- 0..XMax}
		Y2  := random( YMax + 1 );	{ Y2 <- 0..YMax}
		X1  := random( XMax + 1 );	{ X1 <- 0..XMax}
		dX1 := random( 16 ) - 8;		{ dX1 <- -8..+7 }
		dY1 := random( 16 ) - 8;		{ dY1 <- -8..+7 }
		dX2 := random( 16 ) - 8;		{ dX2 <- -8..+7 }
		dY2 := random( 16 ) - 8;		{ dY2 <- -8..+7 }
		LinePeriod := 5 + random(120);	{ LinePeriod <- 5..124 }
		ColorNum := ColorNum + 1;
		if ColorNum > MaxColor then ColorNum := 1;
		SetColor( ColorNum );
		if isEGA then
		begin  { sequence through palette to get more variety }
			if PalNum > 64 then PalNum := 1;
			PalNum := PalNum + 1;
			SetPalette(ColorNum, PalNum)
		end;
	end; { NewValues }


	procedure MoveCoordinates;
	{ Adjust line coordinates, making lines appear to move: }

	begin
		X1 := X1 + dX1;		{ Add appropriate "delta" }
		Y1 := Y1 + dY1;		{ value to line end       }
		X2 := X2 + dX2;		{ coordinates             }
		Y2 := Y2 + dY2;
	end; { MoveCoordinates }


	procedure DrawLines;  { Draw lines mirrored in four quadrants }

	begin
		Line( -X1, -Y1, -X2, -Y2 );	{ upper left quadrant }
		Line( -X1,  Y1, -X2,  Y2 );	{ lower left quadrant }
		Line(  X1, -Y1,  X2, -Y2 );	{ upper right quadrant }
		Line(  X1,  Y1,  X2,  Y2 );	{ lower right quadrant }
	end; { DrawLines }


begin { Kaleidoscope }

	Initialize;
	REPEAT
		IF DisplayPeriod <= 0 THEN NewDisplayPeriod;   { Clear screen }
		NewValues;
		WHILE ( LinePeriod > 0 ) AND ( NOT KeyPressed ) DO
		begin
			Delay( 5 );			{ Set speed limit }
			MoveCoordinates;		{ Animate display }
			DrawLines;			{ Draw mirror images }
			LinePeriod := LinePeriod - 1;
		end;
		DisplayPeriod := DisplayPeriod - 1
	UNTIL KeyPressed;

	{ Note that the key buffer is NOT cleared. Let the same key
	  exit from the main program as well. }

end;  { Kaleidoscope }



(*********************************************************************)
(*					M A I N   P R O G R A M				    *)
(*********************************************************************)
begin { main }

	Initialize;		{ set up system for graphics, display help msg, etc. }

	{ put up backgrounds, except for ShowSeq, which handles its own }
	if Param1 <> S then	BackgroundStuff;

	Repeat

		case Param1 of
			R	:	ShowRandom;
			S	:    ShowSeq;
			K	:	Kaleidoscope;
			L	:	ShowLibrary;
		end; {case}

	until KeyPressed;

	ch := ReadKey;		{ empty keyboard buffer }
	CloseGraph;
end. { main }
