/*****	last modified Mon Aug 07 23:30:19 1989
*
*	biomorph.c -- generates fractal biomorphs
*
*	Version 1.01.  Written in Turbo C 2.0.  Assumes system contains an EGA
*	with 256K and a color monitor, and that EGAVGA.BGI has been converted
*	and added to GRAPHICS.LIB.  Compilation options: -k- -G -O.  It'll run
*	faster if you have a math coprocessor, but it isn't required.
*
*	This program is based on work by Clifford J. Pickover, as
*	described in A. K. Dewdney's "Computer Recreations" column
*	in the July 1989 issue of Scientific American.  Actually, it's
*	based on Dewdney's psuedo-code, and stays pretty close to it.
*
*	Images are generated by reiterating the function f(Z) = Z^n + C, where Z
*	and C are both complex numbers.  C is a constant.  Initial values of Z are
*	in the range -1.5 to 1.5 (by default) on the complex plane.  Other
*	equations can be used, and will eventually be added.
*
*	Input values:	"Viewport size" is the size of one side of the display area.
*
*					"Window size" is the area on the complex plane used for
*					the plot.  The center is always 0; thus, the default window
*					size of 3 describes an area going from -1.5 to 1.5.
*
*					"n" is the integer power of Z.  3 is the default, but 2
*					and 5 (at least) work nicely too.
*
*					"C" is a complex constant used in calculating the plot.
*					Different values will result in different shapes.
*
*	Scaling:		The window size is automatically scaled to the viewport
*					size.  In other words, if you change the viewport size, but
*					leave the window size the same, the plot will grow or
*					shrink to match the viewport.  The change the size of the
*					plotted image, use a bigger window size to reduce the
*					"magnification" or a smaller size to increase the
*					"magnification".
*
*	This program and source code are released into the public domain.
*	I welcome any comments, changes, or improvements!
*
*		8/5/89
*
*		Dave Seidel
*		9 Glenwood Blvd.
*		Hudson, NY  12534
*		(518) 828-8456
*
*		CIS: 71511,2217
*
*-----------------------------------------------------------------------------
*
*	Changes in version 1.01:
*
*		-	Ctrl-C now gives you the opportumity to start over.  The program
*			remembers the values you used before you interrupted, so you can
*			change or keep what you want to.  This is convenient is you start
*			start a plot and want to restart with color, or with a different
*			size or something.
*		-	You can now plot in color, if you want.  The color of a point is
*			based on the absolute value of the Z that generated that point.
*		-	You can choose your own exponent for Z, providing a much greater
*			range of possible plots.
*		-	The plotting is now a little faster.  However, color plot are
*			slower than black-and-white plots, and the higher the exponent,
*			the slower it gets.
*		-	Slightly nicer input screens.
*
******************************************************************************/

#include <conio.h>
#include <ctype.h>
#include <dos.h>
#include <graphics.h>
#include <math.h>
#include <setjmp.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/*	stubs for stuff we don't need */
void _setargv (void) {}
void _setenvp (void) {}

/*	for Ctrl-C; this is the same as kbhit(), but without function-call
	overhead */
#define CHKKBD()			_AH = 0x0b; \
							geninterrupt(0x21)

/*	write a title centered over the viewport */
#define	TITLE(s,c)			setcolor(c); \
							outtextxy((v.right/2)-(textwidth((s))/2), \
									  wy-(2*textheight((s))), (s))

/*	write a caption centered under the viewport */
#define	CAPTION(s,c)		setcolor(c); \
							outtextxy((v.right/2)-(textwidth((s))/2), \
									  (wy+kmax)+textheight((s)), (s))

#define INPUTF(desc,var)	cprintf("Enter %s, or  for last value (%f): ", desc, var); gets(s); \
							if (strlen(s)) var = atof(s)

#define INPUTI(desc,var)	cprintf("Enter %s, or  for last value (%d): ", desc, var); gets(s); \
							if (strlen(s)) var = atoi(s)

#define SQR(a)				((a)*(a))

/*	quicker than comparing cabs(z) to 10 */
#define CTEST(n)			(((n).x*(n).x)+((n).y*(n).y) > 100 ? 1 : 0)

/*	add complex number n to complex number m, put result in n */
#define CADD(n,m)			(n).x += (m).x; \
							(n).y += (m).y

typedef enum {false, true}	bool;

char s[80],
	*titl = "BIOMORPH 1.01",
	*titlbox = "\r\nĿ"
			   "\r\n %s "
			   "\r\n\r\n\r\n",
	*eqn = "f(Z) = Z  Z + C",
	*cap1 = "Plotting, press Ctrl-C to quit...",
	*cap2 = "Done, press any key to clear.";

/* used by setjmp/longjmp to save/restore program state */
static jmp_buf jmp_entry;

/* default input values */
static int view_size = 100,
		   window_size = 3;
static struct complex c = { 0.5, 0.0 };
static int e = 3;
static bool color = false;


/* local prototypes */
int				quit	(void);
struct complex	cpow	(struct complex q, int n);

/*
**	this is where it all happens...
*/
void main (void)
{
	int g_driver, g_mode, wx, wy, j, xasp, yasp, kmax, ch;
	register int k, n;
	struct complex z0, z;
	float aspect_ratio, voff, vmul, kmul;
	struct viewporttype v;
	bool done;

	/* set up Ctrl-C to our handler */
	setcbrk(1);
	ctrlbrk(quit);

	/* set up reentry point */
	setjmp(jmp_entry);

	/* get input */
	textattr(23);
	clrscr();
	textattr(112);
	cprintf(titlbox, titl);
	textattr(23);
	while (1)
	{
		INPUTI("viewport size", view_size);
		if (view_size > 400)
		{
			putch('\a');
			cprintf("Maximum value is 400!\r\n");
			view_size = 100;
		}
		else
			break;
	}
	INPUTI("window size in complex plane", window_size);
	textattr(30);
	cprintf("\r\n    Basic function: %s\r\n\r\n", eqn);
	textattr(23);
	INPUTI("n (integer power of Z)", e);
	INPUTF("real part of C", c.x);
	INPUTF("imaginary part of C", c.y);
	cputs("\r\n");
	cprintf("(C)olor or (B)lack-and-white (default: no color): ");
	done = false;
	while (!done)
	{
		ch = getch();
		switch (toupper(ch))
		{
			case 'C':
				color = true;
				done = true;
				break;
			case 'B':
			case 13:
				color = false;
				done = true;
				break;
			default:
				putch('\a');
				break;
		}
	}

	/* initialize graphics */
	if (registerbgidriver(EGAVGA_driver) < 0)
		exit(1);
	g_driver = EGA;
	g_mode = EGAHI;
	initgraph(&g_driver, &g_mode, "");
	if (graphresult() < 0)
	{
		cputs("\r\nSorry, you must have an EGA to run this program!\r\n");
		exit(1);
	}
	setlinestyle(SOLID_LINE, NULL, THICK_WIDTH);
	getviewsettings(&v);
	getaspectratio(&xasp, &yasp);
	aspect_ratio = (float)xasp/(float)yasp;

	/* set up parameters */
	kmax = view_size * aspect_ratio;			/* y-axis, scaled for aspect */
	wx = ((v.right-view_size)/2)-1;				/* top-left of viewport      */
	wy = ((v.bottom-kmax)/2)-1;
	voff = -((float)window_size/2);				/* complex plane window size */
	vmul = (float)window_size/(float)view_size;	/* x increment in window     */
	kmul = (float)window_size/kmax;				/* y increment in window     */

	TITLE(titl, RED);
	CAPTION(cap1, RED);
	rectangle(wx-2, wy-2, wx+view_size+2, wy+kmax+2);
	setviewport(wx, wy, wx+view_size, wy+kmax, 1);
	for (j = 1; j <= view_size; j++)
	{
		for (k = 1; k <= kmax; k++)
		{
			CHKKBD();
			z0.x = voff+(vmul*j);	/* set up an initial value for Z */
			z0.y = voff+(kmul*k);
			z = z0;
			for (n = 1; n <= 10; n++)
			{
				/* compute Zn+1 = Zn^E + C */
				z = cpow(z, e);
				CADD(z, c);
				if (fabs(z.x) > 10 || fabs(z.y) > 10 || CTEST(z))
					break;
			}
			if (fabs(z.x) < 10 || fabs(z.y) < 10)
				continue;
			else
			{
				if (color)
					putpixel(j-1, k-1, ((int)cabs(z))%16);
				else
					putpixel(j-1, k-1, WHITE);
			}
		}
	}
	setviewport(v.left, v.top, v.right, v.bottom, 1);
	CAPTION(cap1, BLACK);
	CAPTION(cap2, RED);
	getch();
	quit();
}	/* main */


/*
**	invoked when aborting or when plot finishes,
**	allows user to quit or restart using last input values
*/
int quit (void)
{
	int ch;
	bool done = false;

	closegraph();
	textattr(23);
	clrscr();
	textattr(112);
	cprintf(titlbox, titl);
	textattr(23);
	cputs("(R)estart\r\n"
		  "(Q)uit\r\n\r\n"
		  "Enter choice: ");
	while (!done)
	{
		ch = getch();
		switch (toupper(ch))
		{
			case 'R':
				longjmp(jmp_entry, 0);
				done = true;
				break;
			case 'Q':
				done = true;
				break;
			default:
				putch('\a');
				break;
		}
	}
	textattr(LIGHTGRAY);
	clrscr();
	textattr(112);
	cprintf(titlbox, titl);
	textattr(LIGHTGRAY);
	cputs("Written 8/89 by David A. Seidel (71511,2217)\r\n"
		  "Based on work by Clifford J. Pickover,\r\n"
		  "as described in A. K. Dewdney's \"Computer Recreations\" column\r\n"
		  "in the July 1989 issue of Scientific American.\r\n");
	return(0);
}	/* quit */


/*
**	raises complex number q to integer power n
*/
struct complex cpow (struct complex q, int n)
{
	struct complex tmp;
	double xx, yy;

	switch (n)
	{
		case 0:
			tmp.x = tmp.y = 0.0;
			break;
		case 1:
			tmp = q;
			break;
		case 2:
			tmp.x = SQR(q.x) - SQR(q.y);
			tmp.y = 2 * q.x * q.y;
			break;
		case 3:
			xx = SQR(q.x);
			yy = SQR(q.y);
			tmp.x = q.x * (xx - (3*yy));
			tmp.y = q.y * ((3*xx) - yy);
			break;
		default:
			xx = q.x;
			yy = q.y;
			while (n > 1)
			{
				tmp.x = (xx * q.x) - (yy * q.y);
				tmp.y = (xx * q.y) + (yy * q.x);
				n--;
				if (n > 1)
				{
					xx = tmp.x;
					yy = tmp.y;
				}
			}
			break;
	}
	return(tmp);
}	/* cpow */
