/* a print utility, more UNIX compatible than others
 *
 * pr [ option ] ... [ file ] ...
 *
 */

/* Compiles with Microsoft C, version 4.0, using the small memory model.
   -- in order to support wildcarded file names, be sure to include
      SSETARGV.OBJ at link-time */

#define LINT_ARGS			/* Enable type-checking */

#include <stdio.h>
#include <malloc.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <ctype.h>
#include <fcntl.h>
#include <io.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <time.h>
#include <assert.h>

/* Useful Defines */

#define TRUE	1
#define FALSE	0
#define FF	'\014'
#define CR	'\013'
#define MAXFILES	15

/* global definitions */

struct wholepage {			/* keep track of locations on page */
    char *line_begin;
    char *line_now;
} *ppage, xbigline;

struct wholepage *bigline = &xbigline;

    
/* Start of parameters from Command line */

extern unsigned char	isinit;			/* # of characters in  printer
						   initialization string*/

extern unsigned char	isdeinit;		/* # of characters in  printer
						   de-initialization string*/
extern char		ofile[];		/* Output file name */
extern char		initstr[];		/* printer initialization string */
extern char		deinitstr[];		/* printer de-initialization string */
extern unsigned	char	if_lineno;		/* TRUE if numbering lines */
extern unsigned	char	if_cutoff;		/* shall we wrap long lines */
extern unsigned	char	lines;			/* lines per page */
extern unsigned	char	width;			/* page width */
extern unsigned	char	tab;			/* spaces per tab */
extern unsigned	char	columns;		/* # columns to print */
extern unsigned char	isheader;		/* TRUE if user-supplied header */
extern unsigned char	lmargin;		/* # spaces for left margin */
extern unsigned int	startpage;		/* starting page number */
extern unsigned char	multiple;		/* true if printing multiple
						   files at once */
extern char		separator;		/* character to use instead of white space */
extern char		header[300];		/* header text */
extern char		if_mask7;		/* Mask out high-order byte */
extern char		usestdin;		/* Use standard input */

/* End of Patchable Parameters */

FILE		*ofileptr = stdprn;		/* pointer to output file */
long		*timeptr;			/* pointer to a time value */
unsigned	printheader = TRUE;		/* true if printing headers */
unsigned	useff = FALSE;			/* true if using form-feeds */
unsigned	page;				/* current page number */
int		pwidth;				/* Width minus margin */
int		if_pause = FALSE;		/* TRUE if -u (pause) parameter */
int		started = FALSE;		/* TRUE if printing begun */
int		endpage;			/* # of last text line on page */
unsigned	files = 0;			/* index into fp list */
FILE		*fp[MAXFILES];			/* array of files for multiple printing */
struct		stat fileinfo;			/* File status block from fstat call */
int		lineno;				/* Current line number */
char		*bigline_ptr = NULL;		/* pointer to split-line location */
char		*bigline_start = NULL;		/* pointer to big line */
int		single;				/* TRUE if single column */

/* Forward function declarations */

void usage();
void pr(FILE *[]);
void allocpage();
void init_page();
void cleanup();
int prpage(FILE *[]);
int buildpage(FILE *[]);
int buildline(struct wholepage *, int, FILE *, int, char, int);
int inhex(char *, char *);


/* ------------------------------------------------------------------------ */

/* main -- process arguments and cycle through list of files */

main(argc,argv)
int argc;
char *argv[];
{
    int i;
    char *s;
    int headnext, initnext, deinitnext, ofilenext;
    long current_time;
    char curr_time_str[27], file_time_str[27];
    char tempstr[300];

    time(&current_time);				/* get the time */
    strcpy (curr_time_str, ctime(&current_time));	/* convert to ASCII */
    curr_time_str[24] = 0;				/* null out cr */

    header[0] = 0;
    while (--argc > 0 && (*++argv)[0] == '-') {
	headnext = initnext = deinitnext = ofilenext = FALSE;
	for (s = argv[0]+1; *s; s++) {
	  if (ofilenext) {
		strcpy (ofile, s);
		ofilenext = FALSE;
		*(s+1) = 0;
	  }
	  else
	    switch (tolower(*s)) {
		case 'a':			/* Initialization string */
			initnext = TRUE;
			if (*(s+1)) usage();
			break;

		case 'b':
	    		if (!isdigit(*++s))
	    		    usage();
			columns = *s & 0x0f;
			break;

		case 'p':			/* starting page # */
			startpage = atoi(++s);
			if (startpage == 0)
			    usage();
			i = strspn(s, "0123456789");
			s = s + i - 1;
			break;

		case 'h':			/* user-supplied header */
			isheader = headnext = TRUE;
			if (*(s+1)) usage();
			break;

		case 'w':			/* line width */
			width = atoi(++s);
			if (width == 0)
			    usage();
			i = strspn(s, "0123456789");
			s = s + i - 1;
			break;

		case 'l':			/* lines per page */
			lines = atoi(++s);
			if (lines == 0)
			    usage();
			i = strspn(s, "0123456789");
			s = s + i - 1;
			break;
			
		case 'e':			/* Exact print - leave off
		    				   page-headers, trailers */
			printheader = FALSE;
			useff = FALSE;			/* automatically */
			break;

		case 't':			/* tab width */
			tab = atoi(++s);
			i = strspn(s, "0123456789");
			s = s + i - 1;
			break;

		case 'u':
			if_pause = TRUE;
			break;

		case 'i':			/* left margin */
			lmargin = atoi(++s);
			if (lmargin == 0)
			    usage();
			i = strspn(s, "0123456789");
			s = s + i - 1;
			break;

		case 'f':			/* use form-feeds */
			useff = TRUE;
			break;

		case 's':			/* column-separator character */
			separator = *++s;
			if (separator == 0)
			    usage();
			break;

		case 'm':			/* multiple files/page */
			multiple = ! multiple;
			break;

		case 'n':			/* line-numbers */
			if_lineno = ! if_lineno;
			break;

		case 'o':
			ofilenext = TRUE;
			break;

		case 'c':			/* Don't cut-off lines */
			if_cutoff = ! if_cutoff;
			break;

		case 'x':			/* Mask out high-order bit */
			if_mask7 = ! if_mask7;
			break;

		case 'y':			/* Use standard input */
			usestdin = ! usestdin;
			break;

		case 'z':			/* de-Initialization string */
			deinitnext = TRUE;
			if (*(s+1)) usage();
			break;

		default:
			usage();
		}
	}
	if (headnext) {
	    strcpy(header, *++argv);
	    argc--;
	}
	if (ofilenext) {
	    strcpy(ofile, *++argv);
	    argc--;
	}
	if (initnext) {
	    ++argv;
	    isinit = i = inhex(initstr, *argv);
	    if (i < 0) usage();
	    argc--;
	}
	if (deinitnext) {
	    isdeinit = i = inhex(deinitstr, *++argv);
	    if (i < 0) usage();
	    argc--;
	}
    }

/* Finished processing flags */

    if ((columns == 1) && (!multiple))
        single = TRUE;
    else
	single = FALSE;
    if (isheader && lmargin) {
	for (i = 0; i<lmargin; i++)
	    tempstr[i] = ' ';
	tempstr[i] = 0;
	strcat(tempstr, header);
	strcpy(header, tempstr);
    }
    if (isheader) {
	while ((width - strlen(header)) > 10)
	    strcat(header, " ");
	header[width-10] = 0;
    }
    if ((argc == 0) && (!usestdin))
	usage();
    pwidth = width - lmargin;			/* get printable width */
    if (ofile[0]) {
	if (strcmpi(ofile, "stdout") == 0)
	    ofileptr = stdout;
	else
	    if ((ofileptr = fopen(ofile, "w")) == NULL) {
		fprintf(stderr, "Couldn't open output file %s\n", ofile);
		exit(1);
	    }
    }
    else
	assert (setmode(fileno(stdprn), O_TEXT) != -1);

    s = initstr;
    while (isinit--)
	putc(*s++,ofileptr);

    allocpage();

    while (argc--) {
	if (isheader == 0) {			/* if no user-supplied header */
	    for (i=0; i<lmargin; i++)
		header[i] = ' ';
	    strcpy(&header[i], *argv);		/* start header with file name */
	}
    	fp[files] = fopen(*argv,"r");
	if ((fp[files] == NULL) || stat(*argv, &fileinfo)) {
	    fprintf(stderr, "pr: can't open %s\n",*argv);
	    exit(1);
	}
    	if (!multiple) {
	    if (isheader == 0) {
		strcpy (file_time_str, ctime(&fileinfo.st_atime));
		file_time_str[24] = 0;
		strcat(header, "    ");
		if ((width - strlen(header) - 10) > 21)
		    strcat(header, &file_time_str[4]);
		if ((width - strlen(header))  > 47) {
		    strcat(header, "     Printed ");
		    strcat(header, &curr_time_str[4]);
		}
		while ((width - strlen(header)) > 10)
		    strcat(header, " ");
	    }
	    pr(fp);
	}
	else
	    ++files;
	++argv;
    }
    if (usestdin)
	fp[files] = stdin;
    if (multiple || usestdin) {
	if (!isheader) header[0] = 0;
	pr(fp);
    }
    cleanup();
    s = deinitstr;
    if (useff)
	putc('\f', ofileptr);
    else
	if (printheader)
	    for (i = 0; i < 3; i++)
		putc('\n', ofileptr);
    while (isdeinit--)
	putc(*s++,ofileptr);
}

/* ------------------------------------------------------------------------ */

void pr(fp)
FILE *fp[];
{
	page = 1;
	lineno = 0;
	if (printheader) {
		if (useff) endpage = lines - 5;
		else endpage = lines - 3;
	}
	else {
		useff = FALSE;
		endpage = lines;
	}

	while (prpage(fp) != EOF) {
		++page;
	}
}

/* ------------------------------------------------------------------------ */

void allocpage()
{
	struct wholepage *pt;


	if (single) {
		if ((bigline_start = malloc(1024)) == NULL)
			perror("out of memory");
		xbigline.line_begin = bigline_start;
		xbigline.line_now   = bigline_start;
	}
	else {
	    ppage = (struct wholepage *) malloc(lines*sizeof(struct wholepage));
	    if (ppage == NULL)
		perror("out of memory");

	    for (pt = ppage;pt < (ppage + lines);++pt) {
		if ((pt->line_begin=malloc(width+2)) == NULL)
			perror("out of memory");
		pt->line_now = pt->line_begin;
		pt->line_now [0] = 0;
	    }
	}
}

/* ------------------------------------------------------------------------ */

void init_page()
{
    struct wholepage *pt;

    if (!single)
	for (pt = ppage;pt < (ppage + lines);++pt) {
		pt->line_now = pt->line_begin;
		pt->line_now [0] = 0;
	}
}

/* ------------------------------------------------------------------------ */

void cleanup()
{
	struct wholepage *pt;

    if (single)
	free (bigline_start);
    else {
	for (pt = ppage;pt < (ppage + lines);++pt)
		if (pt->line_begin != 0)
			free(pt->line_begin);
	free((char *)ppage);
	ppage = NULL;
    }
}

/* ------------------------------------------------------------------------ */

prpage(fp)
FILE *fp[];
{
	int i, j;
	char *p;
	int iseof;
	struct wholepage *pt;

	if ((page >= startpage) && if_pause) {
		cputs("\nReady the next page in the printer - hit any key");
		getch();
	}
	iseof = buildpage(fp);
	if (page < startpage)
	    return iseof;

	j = 0;
	if ((!started) && (!useff) && printheader)
	    j = 2;
	started = 1;
	pt = ppage + j;
	if (!single)
	    for (i=j;i<endpage;++i) {
		p = (pt++) ->line_begin;
		while (*p) putc(*p++,ofileptr);
		putc('\n',ofileptr);
	    }
	if (useff) putc('\f',ofileptr);	
	else if (printheader)
	    for (i=0;i<3;++i)
		putc('\n',ofileptr);
	return iseof;
}

/* ------------------------------------------------------------------------ */

buildpage(fp)
FILE *fp[];
{
    int i, j, k;			/* temporaries */
    int vpos;				/* current line number */
    int colwid, colwida;		/* column widths */
    int start;				/* starting position on line */
    int retval;				/* value returned from buildline --
					   may be 0, CR, FF, or EOF */
    int startline;			/* 1st text line on page */
    int if_print_this;			/* TRUE if page >= starting page # */
    int iseof;				/* value to return from here --
					   may be 0 or EOF */
    char *p;
    char q;

    init_page();
    vpos = 0;
    iseof = 0;
    if_print_this = (page >= startpage);
    if (printheader && (!useff) ) {
    	if (single && if_print_this && started &&(!if_pause))
	    fprintf(ofileptr, "\n\n");
	vpos += 2;
    }
    if (printheader) {
    	vpos += 3;
    	if (if_print_this)
	    if (single)
		fprintf(ofileptr,"%s  Page%4d\n\n\n",header,page);
	    else
		sprintf((ppage+vpos)->line_begin,"%s  Page%4d",header,page);
    }

/* Variables */

    if (columns > 1)
    	colwid = (pwidth/columns) -1;
    if (multiple) {
    	if (if_lineno) {
	    colwid  = ((pwidth - 7) / files) - 1;
	    colwida = colwid + 7;
	}
	else
	    colwida = colwid = (pwidth/files) -1;
    }
    start = 0;

/* Single column case */

    if (single) {
	for (;vpos<endpage;++vpos) {
	    if (if_cutoff) {
		xbigline.line_now = xbigline.line_begin;
		lineno++;
		retval = buildline(bigline,pwidth,fp[0],lmargin,0,if_lineno);
		p = xbigline.line_begin;
		if (i = ((retval == CR) && !if_lineno))
		    vpos--;
		if (if_print_this) {
		    while (*p) putc(*p++, ofileptr);
		    if (i)
			putc('\r',ofileptr);
		    else
			putc('\n',ofileptr);
		}
	    }
	    else {
		j = (if_lineno ? 7 : 0);
	    	if (bigline_ptr == NULL) {
		    xbigline.line_now = xbigline.line_begin;
		    lineno++;
		    retval = buildline(bigline, 1024, fp[0],0,0,if_lineno);
		    if (retval == EOF) {
			fclose(fp[0]);
			iseof = EOF;
			break;
		    }
		    bigline_ptr = bigline_start;
		    j = 0;
		}
		k = lmargin + j;
		if (if_print_this)
		    while (k--)
			putc(' ', ofileptr);
		if (strlen(bigline_ptr) <= (pwidth - j)) {
		    p = bigline_ptr;
		    if (i = ((retval == CR) && !if_lineno))
			vpos--;
		    if (if_print_this) {
			while (*p) putc(*p++, ofileptr);
			if (i)
			    putc('\r',ofileptr);
			else
			    putc('\n',ofileptr);
		    }
		    bigline_ptr = NULL;
		}
		else {
		    k = pwidth - j;
		    if (if_print_this) {
			while (k--)
			    putc(*bigline_ptr++, ofileptr);
			putc('\n', ofileptr);
		    }
		    else bigline_ptr += k;
		}
	    }
	    if (retval == EOF) {
		fclose(fp[0]);
		iseof = EOF;
		break;
	    }
	    if (retval == FF)
		break;
	}
    if (if_print_this)
	for (++vpos;vpos<endpage;++vpos)
	    putc('\n', ofileptr);
    return iseof;
    }

/* Multiple column case */

    if (!multiple && columns > 1) {
	startline = vpos;
	j = lmargin;
	q = 0;
	for (i=0;i<columns;++i) {
	    for (vpos = startline;vpos<endpage;++vpos) {
	    	if (i)
		    j = start - strlen((ppage+vpos)->line_begin);
		assert (j >= 0);
		lineno++;
		if (buildline(ppage+vpos,colwid,fp[0], j, q, if_lineno)
		    == EOF) {
			fclose(fp[0]);
			return EOF;
		}
	    }
	    if (i == 0)
		start = lmargin;
	    start += colwid;
	    q = separator;
	} /* end for each column */
	return 0;
    } /* end column mode */


/* Multiple file case */

    if (multiple) {
	startline = vpos;
	j = lmargin;
	q = 0;
	k = if_lineno;
	for (i=0;i<files;++i) {
	    for (vpos = startline;vpos<endpage;++vpos) {
		if (i)
		    j =  start - strlen((ppage+vpos)->line_begin);
		else
		    lineno++;
		assert (j >= 0);
		if (
		    fp[i] != NULL && 
		    buildline(ppage+vpos,colwida,fp[i], j, q, k)
		    == EOF
		    ) {
			fclose(fp[i]);
			fp[i] = NULL;
		    }
		} /* end for each line */
		if (i == 0)
		    start = lmargin;
		start += colwida;
		q = separator;
		k = 0;
		colwida = colwid;
	    } /* end for each column */
	    for (i=0;i<files;++i) if (fp[i] != NULL) return 0;
	    return EOF;
    } /* end multiple */

    return 0;
}

/* ------------------------------------------------------------------------ */

buildline(pg, w, fp, llmargin, qsep, if_lineno)
struct wholepage *pg;
int w;
FILE *fp;
int llmargin;
char qsep;
int if_lineno;

{
    int hpos;
    int c;

    hpos = 0;
    if (if_lineno || (qsep && (qsep != ' '))) {
	while (llmargin) {
	    *(pg -> line_now)++ = ' ';
	    --llmargin;
	}
	if (qsep) {
	    *(pg -> line_now)++ = qsep;
	    qsep = 0;
	} 
    }
    if (if_lineno &&  (w > 9)) {
	sprintf(pg->line_now, "%*d ", 6, lineno);
	w -= 7;
	pg -> line_now += 7;
    }
    while ((c = getc(fp)) != EOF) {
	if ((c!= '\n')&& (c!='\r')&&(c!='\f')) {
	    while (llmargin) {
		*(pg -> line_now)++ = ' ';
		--llmargin;
	    }
	    if (qsep) {
		*(pg -> line_now)++ = qsep;
		qsep = 0;
	    } 
	}
	switch (c) {
	    case '\n':
		*(pg -> line_now) = 0;
		return 0;
	    case '\r':
		*(pg -> line_now) = 0;
		return CR;
	    case '\f':
		if ((c = getc(fp)) != '\n')
		    ungetc(c, fp);
		if (!multiple) {
		    *(pg -> line_now) = 0;
		    return FF;
		}
		if (c == '\n')
		    return 0;
		break;
	    case '\t':
		if (tab) {
		    if (hpos < w) {
			*(pg -> line_now)++ = ' ';
			++hpos;
		    }
		    while ((hpos % tab) && (hpos < w)) {
			*(pg -> line_now)++ = ' ';
			++hpos;
		    }
		break;
		}
	    default:
		if (if_mask7) {
		    c = c & 0x7f;
		    if (c < ' ') c = 0;
		}
		if (c && hpos < w)
		    *(pg -> line_now)++ = c;
		++hpos;
	}
    }
     *(pg ->line_now) = 0;
    if (c == EOF) return EOF;
    return 0;
}

/* ------------------------------------------------------------------------ */

void usage()

{
	puts("Usage: pr -options file (file ...)");
	puts("       options are:");
	puts("              a     next argument is printer initialization string");
	puts("              b#    print text in # columns(1-9)");
	puts("              c     don't Chop long lines (wrap them)");
	puts("              e     Exact print (no headers,trailers)");
	puts("              f     use FormFeeds between pages");
	puts("              h     Next argument is a page header");
	puts("              i#    Indent each line by # spaces");
	puts("              l#    specify Lines per page (default 66)");
	puts("              m     multiple files per page (one per column)");
	puts("              n     Number the lines of the file");
	puts("              o     followed by output file name (or 'stdout')");
	puts("              p#    start printing at designated Page");
	puts("              s^    Use charcter '^' as column separator");
	puts("              t#    Tabs expand to # characters");
	puts("              u     Pause between sheets");
	puts("              w#    Width of page is # characters (default 80)");
	puts("              x     Mask out high-order bit of each character");
	puts("              y     use standard input");
	puts("              z     next argument is printer de-initialization string");
	exit(0);
}

/* Convert a string from hexadecimal to bytes */
/* Returns # bytes in the string, -1 if illegal input */

inhex(q, p)

char *q, *p;

{
    int i;

    i = 0;
    for (;*p;p++) {
	if ((*p != ' ') && (*p != ',')) {
	    if ((!isascii(*p)) || (!isascii(*(p+1))) ||
		(!isxdigit(*p)) || (!isxdigit(*(p+1))))
			return -1;
	    i++;
	    *q = ((*p > '9') ? (toupper(*p) - 'A' + 10) : (*p - '0')) * 16;
	    *q++ += (*(++p) > '9') ? (toupper(*p) - 'A' + 10) : (*p - '0');
	}
    }
    *q = 0;
    return i;
}
