/**********************************
 * FFILTERS.H
 *
 * Basic Font Filtering Scheme
 *
 * James Bumgardner 1987
 ***********************************/

#define SIZE_LIMIT	152
#define ESC		27

struct {
	int fil26;
	char nul1,font_type;
	int nul2,base_line,width,height;
	char orient,fixed;
	int symset,pitch,points,nul3;
	char nul4,style,weight,type_face;
} font_header;

struct {
	char fil4,nul1,fil14,cil1;
	char orient,nul2;
	int left_offset,top_offset,width,height,delta_x;
} char_header;

char ibuf[2000],*bitmap = &ibuf[sizeof(char_header)];

char pmap[2][SIZE_LIMIT][SIZE_LIMIT];

FILE *ifile,*ofile;

main(argc,argv)
char *argv[];
{
	int total_bytes,horiz_bytes,true_width,arg;

	if (argc < 3)
		erxit("Please Specify an Input and Output File");

	if (argc >= 4)	arg = atoi(argv[3]);
	else		arg = 0;	
	
	if ((ifile = fopen(argv[1],"r")) == NULL)
		erxit("Can't open Input File\n");
	if ((ofile = fopen(argv[2],"w")) == NULL)
		erxit("Can't open Output File\n");

	if ((total_bytes = get_font_header()) == EOF)
		erxit("Invalid Font File");
	process_font_header(arg);
	write_font_header(total_bytes);
		
	printf("Processing  ");
	while ((total_bytes = get_char_header()) != EOF)
	{
		fix_header();

		horiz_bytes = (total_bytes-sizeof(char_header))/char_header.height;

		translate_bitmap(char_header.height,horiz_bytes);

		true_width = horiz_bytes << 3;
		process_char(char_header.height,true_width,arg);

		horiz_bytes = (char_header.width-1 >> 3)+1;
		translate_pmap(char_header.height,horiz_bytes);

		total_bytes = sizeof(char_header)+ horiz_bytes * char_header.height;
		unfix_header();
		fprintf(ofile,"\033(s%dW",total_bytes);
		fwrite(ibuf,total_bytes,1,ofile);
	}
}

translate_bitmap(h,w)
{
	int y,x;
	char *p,*op;
	unsigned char byt,bit;
	p = bitmap;
	for (y = 0; y < h; ++y) {
		op = pmap[0][y];
		for (x = 0; x < w; ++x) {
			byt = *p++;
			for (bit = 0x80; bit; bit >>= 1)
			{
				if (byt & bit)		*op++ = 1;
				else			*op++ = 0;
			}
		}
	}
}

translate_pmap(h,w)
{
	int y,x,rx;
	char *p;
	unsigned char byt,bit;
	p = bitmap;
	for (y = 0; y < h; ++y) {
		rx = 0;
		for (x = 0; x < w; ++x) {
			byt = 0;
			for (bit = 0x80; bit; bit >>= 1)
			{
				if (pmap[1][y][rx])	byt |= bit;
				++rx;
			}
			*p++ = byt;
		}

	}
}

get_font_header()
{
	unsigned int tmp;
	int c;
	char *p;
	
	while ((c = getc(ifile)) != EOF && c != ESC)
		;
	if (c == EOF)
		erxit("Invalid Font File");
	tmp = getw(ifile);

	if (tmp == ')s'	)
	{
		tmp = get_number();
		fread(ibuf,tmp,1,ifile);
		memcpy(&font_header,ibuf,sizeof(font_header));
		font_header.base_line = iswap(font_header.base_line);
		font_header.width = iswap(font_header.width);
		font_header.height = iswap(font_header.height);
		if (font_header.width > SIZE_LIMIT ||
		    font_header.height > SIZE_LIMIT)
		{
			fprintf(stderr,"Characters are too big\n");
			return(EOF);
		}
		return(tmp);
	}
	return(EOF);	
}

write_font_header(t)
{
	fprintf(ofile,"\033)s%dW",t);
	font_header.base_line = iswap(font_header.base_line);
	font_header.width = iswap(font_header.width);
	font_header.height = iswap(font_header.height);
	memcpy(ibuf,&font_header,sizeof(font_header));
	fwrite(ibuf,t,1,ofile);
}

get_char_header()
{
	int tmp;
	int c,cno;
	while ((c = getc(ifile)) != EOF && c != ESC)
		;
	if (c == EOF)		return(EOF);
	tmp = getw(ifile);

	if (tmp != '*c')	return(EOF);
	cno = get_number();

	printf("\b%c",cno);
	fprintf(ofile,"\033*c%dE",cno);

	c = getc(ifile);		/* skip escape */
	tmp = getw(ifile);
	if (tmp != '(s')	return(EOF);

	tmp = get_number();

	fread(ibuf,tmp,1,ifile);
	return(tmp);
}

fix_header()
{
	memcpy(&char_header,ibuf,sizeof(char_header));
	char_header.width = iswap(char_header.width);
	char_header.height = iswap(char_header.height);
	char_header.delta_x = iswap(char_header.delta_x);
	char_header.left_offset = iswap(char_header.left_offset);
	char_header.top_offset = iswap(char_header.top_offset);
}

unfix_header()
{
	char_header.width = iswap(char_header.width);
	char_header.height = iswap(char_header.height);
	char_header.delta_x = iswap(char_header.delta_x);
	char_header.left_offset = iswap(char_header.left_offset);
	char_header.top_offset = iswap(char_header.top_offset);
	memcpy(ibuf,&char_header,sizeof(char_header));
}

iswap(n)
unsigned int n;
{
	return(n >> 8 | n << 8);
}

get_number()
{
	char *p;
	int c;
	p = ibuf;
	while ((c = getc(ifile)) != EOF && isdigit(c))
		*p++ = c;
	*p = 0;
	return(atoi(ibuf));
}

erxit(msg)
char *msg;
{
	fprintf(stderr,msg);
	exit();
}

