/**
 **
 **	LJ.C  -- A printing utility for the Hp LaserJet
 **
 **	This program prints a series of files on the LaserJet
 **	printer.  The files are printed in a "landscape" font at
 **	16.6, 27.2 or 30 character to the inch.  To take advantage of this 
 **	density, two "pages" of information from the file are 
 **	printed on each piece of paper (left and right halves).  The
 **	small fonts now supported are:
 **			Line Printer  27.2 pitch  5 point    0U
 **			Gothic Light  30   pitch  4.4 point  0U
 ** or equivalent.  These fonts can be found in the third party font
 **	cartridge:
 **					Super Cartridge 1 (tm)
 **		  from		IQ Engineering
 **					P.O. Box 60955
 **					Sunnyvale, CA 94086
 ** Options now include:
 **		1. Printing 132 column listing (with small fonts)
 **		2. Allowing lines too long to print (i.e. greater than 80
 **	         or 132 characters) to wrap to the next line.
 **		3. Variable tabs to allow for printing text that has
 **          other than 8 column spacing.
 **		4. IBM manual binder format with hole punch markers
 **	         and individual page numbers on each half. Because
 **			 of the extra room required for hole punches the number
 **			 of printable characters on a line are reduced as follows:
 **					   PITCH		# CHAR ON LINE
 **					   Normal 			78 
 **					 S0 (27 cpi)		130
 **					 S1 (30 cpi)		132
 **		5.	Printing of an optional heading on each page.
 **
 **
 **	Usage is:	LJ [-w][-s][-m][-tx][-h"text"] file1 [options] file2 [options] file3 ...
 **
 **	Where file# is a valid MS-DOS filename, included on the
 **	command line.  The file name may also contain wild card characters
 ** (e.g. * or ?) and be intermixed with various options.
 **
 ** This program is now compatible with MS C V5.0.  Although
 ** it should be, no attempt has been made to verify that it is now backwards
 ** compatible with Lattice C.   B.J.L.
 **
 **	Options:
 **		-w	long lines wrap to next line
 **		-sx	use of small fonts (<=5 points) for 132 column lists where:
 **				x=0 or blank  =>  27.2 cpi font
 **				x=1           =>  30 cpi font
 **		-m	manual format for 5 1/2 x 8 1/2 binders
 **		-tx tab size of x where x is greater than 0
 **		-h"text" Cause 'text' to be printed as a heading on each page
 **
 **	Modification History:
 **
 **	Joe Barnhart	original version					May 5, 1985
 **	Ray Ducan		date and time stamping				May 22, 1985
 **	Joe Barnhart	revised date stamping				June 6, 1985
 **	Ray Moon		modified for CI86					December 13, 1985
 **					& revised EOF test
 **
 **	Bruce Lontka (CIS 73647,1702   SOURCE AAJ455) 		August 20, 1988
 **					Added wide print, small fonts,      
 **					variable tab stops, IBM manual 
 **					format, center line, wild card file
 **					names, page headings, null argument 
 **					litany and MS C V5.0 compatibility.
 **
 **/

#define LATTICE	1 	/* Lattice C */
#define CI86	2	/* CI86 */ 
#define MSC		3 	/* Microsoft C V5.0 */
#define COMPILER 	MSC

#if COMPILER == LATTICE
#include <h\stdio.h>
#else 
#include <stdio.h>
#endif

#if COMPILER == MSC
#include <time.h>			/* MSC supports time and date functions */
#endif

#define PAGE	'\f'	/* for compilers without '\f' */

/*
**	Options definitions
*/
int	wide =	FALSE,		/* wide printing */
	small =	FALSE,		/* small font option */
	manual =FALSE,		/* manual format option */
	very_small=FALSE,	/* very small font in use */
	tabs = 8;			/* tab expansion, default to 8 */

int		wide_error = FALSE;	/* line too wide error */
float 	char_per_in = 16.6;	/* default characters per inch (Line printer) */
float	gutter = 0.25;		/* default left gutter */
int		symbol_set = 8;		/* default symbol set is Roman-8 */
int		lines_per_in = 8;	/* default at 8 lines an inch */
int		lines_per_pg = 56;	/* default at 56 lines per page */
int		line_width = 80,	/* default number of characters on a line */
		side,				/* side of page */
		left_mar,			/* left margin */
		right_mar;			/* right margin */

char 	datestr[10], 		/* date string */
		timestr[10];		/* time string */
char	title[100]= "";		/* title */
#define LEFT 0				/* left side of page */
#define RIGHT 1				/* right side of page */

#if COMPILER == CI86
typedef struct {
		unsigned short ax,bx,cx,dx,si,di,ds,es;
	} REGSET;
#endif

#if COMPILER == LATTICE
typedef struct {
 		int ax, bx, cx, dx, si, di;
	} REGSET;
#endif

main(argc, argv)
int argc;
char *argv[];
{

	int filenum = 1;			/* command line argument index */
		
	FILE *fp, *prn, *fopen();
	
	if (argc < 2) {
		/*
		**	If no arguments given then respond with the 'how to' litany
		*/
		printf("LaserJet Side by Side Printing Utility Ver. 2.00\n");
		printf("Usage:\n");
		printf("    lj [-r][-w][-s][-m][-tx][-h\"text\"] file1 [[options]file2 [...]]\n");
		printf("       [] = optional argument\n");
		printf("	   -h\"text\" Optional page heading\n");
		printf("       -m manual format for 5 1/2 x 8 1/2 binders\n");
		printf("       -r reset options to default values (use for multiple files)\n");
		printf("       -sx use of small fonts where\n");
		printf("            x=0 or blank means 27 cpi font\n");
		printf("            x=1 means 30 cpi font\n");
		printf("       -tx tab size of x where x>0 \n");
		printf("       -w long lines wrap to next line\n");
		printf("       file#  name of text file to be printed (may contain wild cards)\n");
		exit(1);
	}	
	
							
#if COMPILER == MSC
	if( (prn = fopen("PRN","wt") ) == NULL) {
#else
	if( (prn = fopen("PRN:", "w") ) == NULL) {
#endif
		printf("Error opening printer as file.\n");
	} else {
		/*
		**	Begin the scan of the command line
		*/		
		_strtime(timestr);
		_strdate(datestr);
		for(; filenum < argc; filenum++) {

			if (*argv[filenum] == '-') {
				/*
				**	A potential option has been encountered
				*/
				switch (tolower(argv[filenum][1])) {
						
					case 'w' :	/* wide word wrap */
						wide = TRUE;
						printf("Printing with long line word wrap\n");
						break;
							
					case 's' : /* small font */
						small = TRUE;
						symbol_set = 0;			/* Set for Ascii */
						lines_per_pg = 84;		/* more lines */
						if (argv[filenum][2] == '1') {
							/*
							**	Very small 30 pitch requested
							*/
							very_small = TRUE;
							char_per_in = 30;		/* small line printer type face */
							printf ("Using 30 cpi pitch font\n");
						} else {
							char_per_in = 27.2;
							printf("Using 27 cpi pitch font\n");
						}	
						break;
							
					case 'h' : /* Heading */
						strcpy (title,&argv[filenum][2]);
						printf ("Heading: %s\n",title);
						break;
						
					case 'm' : /* manual format */
						manual = TRUE;
						gutter = 0.44;
						printf("Printing in 5 1/2 x 8 1/2 manual format\n");
						break;
							
					case 'r' : /* reset options to defaults */
						wide =	FALSE;		/* wide printing */
						small =	FALSE;		/* small font option */
						manual =FALSE;		/* manual format option */
						very_small=FALSE;	/* very small font in use */
						tabs = 8;			/* tab expansion, default to 8 */
                
						wide_error = FALSE;	/* line too wide error */
						char_per_in = 16.6;	/* default characters per inch (Line printer) */
						gutter = 0.25;		/* default left gutter */
						symbol_set = 8;		/* default symbol set is Roman-8 */
						lines_per_in = 8;	/* default at 8 lines an inch */
						lines_per_pg = 56;	/* default at 56 lines per page */
						line_width = 80;	/* default number of characters on a line */
						*title = 0;			/* no title */
						printf("All options reset to default values\n");
						break;

					case 't' : /* tab stops */
						tabs = atoi(&argv[filenum][2]);
						if (tabs < 1) {
							printf("Invalid tab stop of %d\n",tabs);
							exit(1);
						}
						printf("Tab stops set to %d spaces\n",tabs);
						break;	
            
					default :			
						/*
						**	No match was found!
						*/
						printf("Invalid option '%s' encountered",argv[filenum]);
						exit(1);
						break;
				}	
				continue;		/* get next argument */
			}	
            
			/*
			**	Set up for maximum number of characters on a line.  This is required
			**	because of the dead band around the perimeter of the HP-LJ page.
			*/
			if (manual) {
				if (!small) {
					line_width = 78;
				} else {
					if (very_small) {
						line_width = 132;
					} else {
						line_width = 130;
					}
				}
			} else {
				if (!small) {
					line_width = 80;
				} else {
					line_width = 132;
				}
			}			

			/* initialize the LaserJet for landscape printing*/
			fprintf(prn, "\033E\033&l1O\033(%dU\033(s%1.1fH\033&l%dd6E",symbol_set,char_per_in,lines_per_pg/7);
			/*        reset^  landscp^  symb^       cpi^       lpi^ ^6 lines top margin */



			fp = fopen(argv[filenum], "r");
			if (fp == NULL)
				printf("file %s doesn't exist.\n", argv[filenum]);
			else {
				printf("Now printing %s\n", argv[filenum]);
				printfile(fp, prn, argv[filenum]);
				fclose(fp);
				if (wide_error) {
					printf("  Some lines in %s were truncated\n", argv[filenum]);
					wide_error = FALSE;
				}	
			}
		}
		fprintf(prn, "\015\033E");	/* reset LaserJet */
	}
}

printfile(fp,prn,filename)
	FILE *fp,*prn;
	char *filename;
{
	int pagenum = 1;
	
	while( !feof(fp)) {
		left_mar = char_per_in * gutter;	/* 1/4 in.left margin */
		if (manual) {
			/*
			**	For IBM manual format, insert 3 ring binder punches
			*/
			fprintf(prn, "\033&l0E");	 /* set at 0,0 */
			    /*           topm^ */
		
			box (prn,450,60);	/* 1.5 in.down 3/8 in.from margin */
			box (prn,1275,60); 	/* 4.25 in.down */
			box (prn,2100,60); 	/* 6 in.down */
		}	
		fprintf(prn, "\033&l6E\033&a0r%dm%dL\015",left_mar+line_width,left_mar);	 /* set left half */
		    /*          topm ^ row 0^ rtm^ ^lfm (rtm = rt margin, lfm = left margin) */   
		side = LEFT;
		printpage(fp,prn,line_width);			/* print page */
		if (manual) {
			stamp(prn, filename, pagenum++);	/* title */
		}
		if( !feof(fp)) {			/* if more .. */
			/* 
			**	Draw a separator 7.5 in. line in the middle of the page, 
			**	one half inch from the top.
			*/ 
			fprintf(prn, "\033&l0E");	 /* set at 0,0 */
			    /*           topm^ */
		
			if (manual) {
				/*
				**	For IBM manual format, insert 3 ring binder punches
				*/
				box (prn,450,1690);		/* 1.5 in.down 3/8 in.from center margin */
				box (prn,1275,1690); 	/* 4.25 in.down */
				box (prn,2100,1690); 	/* 6 in.down */

				fprintf(prn,"\033*p150y1590X\033*c2a2250b0P");
					/*  x,y position^    rectangle^      ^solid fill */

			} else {
				fprintf(prn,"\033*p150y1590X\033*c2a2100b0P");
					/*  x,y position^    rectangle^      ^solid fill */

			}			
			left_mar = char_per_in * (5.5 + gutter) - (small?2:0);		/* Mid page + 1/4 in.*/
			fprintf(prn, "\033&l6E\033&a0r%dm%dL\015",left_mar+line_width,left_mar);	 /* set right half */
 		    /*             topm^   row 0^ rtm^ ^lfm  (rtm = rt margin, lfm = left margin) */   
			side = RIGHT;
			printpage(fp,prn,line_width);		/* print another */
			if (manual) {
				stamp(prn, filename, pagenum++);	/* title */
			}
		}
		if (!manual) {
			stamp(prn, filename, pagenum++);	/* title */
		}	
		fputc(PAGE, prn);			/* kick paper */
	}
}

/*
**	The 'printpage' proceedure will contruct a 'page'.  In this case, one
**	side of the double facing final output page.
*/
printpage(fp,prn,max_col)
	FILE *fp, *prn;
	int	 max_col;		/* maximum column for printing */
{
	int c,line,col;
	
	line = col = 0;
	while(line < lines_per_pg)
		switch(c = fgetc(fp)) {
			case '\n':				/* newline found */
				col = 0;			/* zero colunm */
				line++;				/* adv line cnt */
				fputc('\n',prn);
				break;
			case '\t':				/* TAB found */
				do
					fputc('\040',prn);
				while ((++col % tabs) != 0);
				break;
			case PAGE:				/* page break or */
			case EOF:				/* EOF found */
				line = lines_per_pg;		/* force terminate */
				break;
			default:				/* no special case */
				if (col >= max_col) {
					if (wide) {
						/*
						**	word wrap wide lines
						*/
						fprintf(prn,"\n    ");
						col = 4;
						line++;
					} else {
						wide_error = TRUE;
						break;
					}		
				}	
				col++;
				fputc(c,prn);		/* print character */
				break;
		}
}

/*
**	The 'stamp' proceedure places the file name and time and date stamp
**	on the current page.
*/
stamp(prn, filename,pagenum)
	FILE *prn;
	char *filename;
	int pagenum;
{
	int lm;
	
	if (small) {
		fprintf(prn, "\033(8U\033(s16.6H");
		/*             symb^      cpi^  */
	}	

	if (manual) {
		if (side == LEFT) {
			lm = gutter * 16.6;
		} else {
			lm = 16.6 * (5.5 + gutter);		/* Mid page + 1/4 in.*/
		}	
		fprintf(prn, "\033&a0r%dm%dL",lm+(small?132:80),lm);	 /* set right half */
 	    /*             row 0^ rtm^ ^lfm  */   
		fprintf(prn, "\015\033&a%dR",lines_per_pg + (small?3:2) ); 	/* move to title row */
		/*	        row position^ */
		fprintf(prn, "File: %-46.46s", filename);
	} else {
		lm = gutter * 16.6;
		fprintf(prn, "\033&a%dl%dM",lm,lm+165);			/* widen margins */
		/*               lfm^  ^rtm */
		fprintf(prn, "\015\033&a%dR",lines_per_pg + (small?3:2) ); 	/* move to row 58 */
		/*	        row position^ */

		fprintf(prn, "File: %-113s", filename);
	}	
	fprintf(prn, "Page %-3d %8.8s %8.8s", pagenum, datestr, timestr);
	if (strlen(title) > 0) {
		/*
		**	Optional Heading 
		*/
		fprintf(prn, "\015\033&l0E\033&a3R");	 /* set at 3 lines from top */
		    /*              topm^   row^ */
		fprintf(prn, "%s", title);
	}			    
	if (small) {
		fprintf(prn, "\033(%dU\033(s%1.1fH",symbol_set,char_per_in);
		/*              symb^      cpi^  */
	}	
}

/*
**	This proceedure creates a 1/4 in. square, 10% shaded box with a cross
**	in the center.  It is used as a positioner for a hole punch.
*/
box (prn,y,x)
FILE	*prn;	/* printer file */
int 	x,		/* x position (in PCL dots) of box center */
	 	y;		/* y position (in PCL dots) of box center */
{
	fprintf(prn,"\033*p%dx%dY\033*c75a75b10g2P",x-37,y-37);	/* shaded box 1/4 in. square */
	/*             x,y pos^ rectangle^  ^10% fill */
	fprintf(prn,"\033*p%dx%dY\033*c3a75b0P",x-1,y-37);		/* horiz crossbar */
	/*             x,y pos^ rectangle^  ^solid fill */
	fprintf(prn,"\033*p%dx%dY\033*c74a3b0P",x-37,y-1);		/* vert crossbar */
	/*             x,y pos^ rectangle^  ^solid fill */
}

#if COMPILER != MSC
/*
**	Support for compiliers that do not contain the time and date function
**	in their library.
*/ 
_strdate(datestr)
	char *datestr;
{
	REGSET regs;
	int month, day, year;
	
	regs.ax = 0x2a00;
#if COMPILER == CI86
	sysint21(&regs,&regs);
#else
	int86(0x21,&regs,&regs);
#endif
	month = (regs.dx >> 8) & 255;
	day = regs.dx & 255;
	year = regs.cx - 1900;
	sprintf(datestr, "%02d/%02d/%02d", month, day, year);
}

_strtime(timestr)
	char *timestr;
{
	REGSET regs;
	int hours,mins;
	
	regs.ax = 0x2c00;
#if COMPILER == CI86
	sysint21(&regs,&regs);
#else
	int86(0x21,&regs,&regs);
#endif
	hours = (regs.cx >> 8) & 255;
	mins = regs.cx & 255;
	sprintf(timestr, "%02d:%02d", hours, mins);
}
	
#endif
	
					