/*
**
**	LJ2UP -- A printing utility for the HP LaserJet
**
**	This program prints a series of files on the LaserJet printer.	The
**	files are printed in a "landscape" font at 17 characters to the inch.
**	To take advantage of this density, two "pages" of information from
**	the file are printed on each piece of paper (left and right halves).
**
**	Usage is:	LJ2UP  [-psvtdo] file1 file2 file3 ...
**
**			p	Page length
**			s	Set Forth "screenfile" mode
**			v	Turn off vertical line on page
**			t	Set tabs
**			d	Supress date and file stamp
**			o	Re-direct output to device or file
**			file	Any MS-DOS file spec, including wildcards
**				and directories
**
**	Joe Barnhart	original version for Lattice C		May 5, 1985
**	Ray Duncan	date and time stamping			May 22, 1985
**	Joe Barnhart	revised date stamping			June 6, 1985
**	Ray Duncan	modified for Microsoft C		Oct. 18, 1985
**	Joe Barnhart	added wildcards, split long lines,
**			command line switches, screenfiles	July 24, 1986
**	Joe Barnhart	added pathname support, prt redirect	Oct 26, 1986
**	Joe Barnhart	added vertical line to page		Apr 3, 1987
**  wfk added OS/2 support 10/31/89
*/

#include <stdio.h>
#include <stdlib.h>
#include "getargs.h"
#define INCL_DOS
#include <os2.h>

#define FALSE	0
#define TRUE	1
#define ENV	"LJ"		/* name of environment string */
#define ENVLEN	20		/* max characters in environment str */
#define DEFPRT	"PRN"		/* default printer device */
#define DEFLINE 58		/* default lines per page */
#define DEFCOL	80		/* default number of columns */
#define MAXLINE	68		/* height of page in lines */
#define PAGE	'\014'          /* form feed */
#define TAB	8		/* width of one tab stop */
#define LEFT	0		/* printing on left side of page */


/*
**	The following defines provide control over the Hewlett-Packard 
**	LaserJet and LaserJet PLUS printers.
*/

#define CLEAR		fprintf(prn, "\033E")
#define INITLJ		fprintf(prn, "\033&l1o8D\033(s17H\033&l64F\033&l0L")
#define TOPMAR(x)	fprintf(prn, "\033&l%dE", x)
#define MARGIN(l,r)	fprintf(prn, "\033&a0r%dm%dL\015",r,l )
#define ROW(r)		fprintf(prn, "\033&a%dR",r )
#define RHALF		MARGIN( 92,172 )
#define LHALF		MARGIN( 5,85 )
#define MIDDLE		MARGIN( 88,90 )

typedef struct {
	    int ax, bx, cx, dx, si, di, cflag;
	} REGSET;

char	*fsearch(), *fnext(), *fpath();

char	*ljenv[2];		/* array of pointers to strings */
int	line;			/* global line number */
int	pagenum;		/* global page number (physical) */
int	nostamp = FALSE;	/* date and time stamping on/off */
int	forthscr = FALSE;	/* Forth screen mode on/off */
int	nograf = FALSE;		/* graphics (boxes) on/off */
int	ibm = FALSE;		/* IBM print mode, 66 lines/page */
int	side = LEFT;		/* which side of page now printing */
int	lastline = DEFLINE;	/* last printed line */
int	maxline = DEFLINE;	/* number of lines per page */
int	maxcol = DEFCOL;	/* length of one line */
int	tab = TAB;		/* value of tab stops */
char	*prtdev = DEFPRT;	/* print device name */
char	*printname;		/* name of file being printed */
FILE	*prn;			/* printer device stream */

ARG	argtab[] = {
	{ 'd', BOOLEAN,	&nostamp,	"turn off date and time stamp"	},
	{ 's', BOOLEAN, &forthscr,	"enable Forth screenfile mode"	},
	{ 'v', BOOLEAN, &nograf,	"disable vertical line"		},
	{ 't', INTEGER,	&tab,		"spacing of tab stops"		},
	{ 'p', INTEGER, &maxline,	"print length in lines"		},
	{ 'i', BOOLEAN, &ibm,		"IBM mode, 66 lines, no FF's"	},
	{ 'o', STRING,  (int *)&prtdev,	"output device or file"		}
};
#define TABSIZE	( sizeof(argtab) / sizeof(ARG) )


main(argc, argv)
int   argc;
char  *argv[];
{
    int   filenum, top;
    int   dummy;
    char  *fspec, *path, *filename, *p;
    char  fullname[64];
    FILE  *fp, *fopen();


    printf( "LaserJet landscape lister, version 2.0\n" );
    printf( "Copyright (c) 1987 by Joe Barnhart\n\n" );

    /* First check for default preferences with LJ=-xxx enviromnent str */

    if( ( p = getenv( ENV ) ) != NULL )
    {
	ljenv[0] = "";
	ljenv[1] = p;
	dummy = 2;
	getargs( dummy, ljenv, argtab, TABSIZE );
    }

    /* Next, scan the command line for switches that override defaults */

    argc = getargs( argc, argv, argtab, TABSIZE );
    if( !ibm )
    {
	if( forthscr )
	    top = 5;
	else
	    top = (MAXLINE - maxline) / 2;
	if( nostamp )
	    lastline = maxline;
	else
	    lastline = maxline-2;
    }
    else
    {
	top = 0;
	lastline = maxline = 65;
	nostamp = TRUE;
    }

    /* Open printer and begin */

    prn = fopen( prtdev, "wb" );
    if( prn == NULL )
    {
	printf( "Error opening print device: %s\n", prtdev );
	exit( 1 );
    }

    CLEAR;
    INITLJ;
    TOPMAR( top );
    for( filenum = 1; filenum < argc; filenum++ ) {
	fspec = argv[filenum];
	path = fpath( fspec );
	filename = fsearch( fspec );
	if( filename == NULL )
	    printf("File %s not found.\n", fspec );
	else 
	    do {
		strcpy( fullname, path );
		strcat( fullname, filename );
		printname = filename;
		if (forthscr)
		    printforth( fullname );
		else
		    printfile( fullname );
		filename = fnext();
	    } while( filename != NULL );
    }
    CLEAR;
    fclose( prn );
}


printfile( filename )
char *filename;
{
    FILE *fp, *fopen();
    register char  c;
    register col;

    fp = fopen( filename ,"r" );
    if( fp == NULL ) {
	    printf( "Error opening file: %s.\n", filename );
	    return;
    }
    else
	    printf( "Now printing file: %s\n", filename );
    line = col = 0;
    pagenum = 1;
    LHALF;
    side = LEFT;
    while((c = fgetc(fp)) != EOF)
	    switch(c) {
	      case '\012':                        /* newline found */
		      newline();
		      col = 0;
		      break;
	      case '\015':			/* CR found */
		      fputc('\015',prn);
		      break;
	      case  '\011':                       /* TAB found */
		      do
		        fputc('\040',prn);
		      while ( (++col % tab) != 0 );
		      break;
	      case PAGE:				/* Page break */
		      newpage();
		      col = 0;
		      break;
	      default:				/* no special case */
		      fputc(c,prn);			/* print character */
		      col++;
		      if ( col > maxcol ) {		/* break long lines */
		        newline();
		        col = 0;
		      }
		    break;
	    }
    stamp();
    kickpage();
    fclose( fp );
}


printforth( filename )
char *filename;
{
    int fd;
    int screen;
    int endfile;

    fd = open( filename ,0 );
    if( fd == -1 ) {
	printf( "Error opening file: %s.\n", filename );
	return;
    }
    else
	printf( "Now printing file: %s\n", filename );
    screen = line = 0;
    pagenum = 1;
    LHALF;
    side = LEFT;
    endfile = FALSE;
    while( !endfile ) {
	endfile = printscr( fd, screen );
	if( (screen+1 % 3) == 0 )		/* three screens per page */
	    newpage();
	screen++;
    }
    stamp();
    kickpage();
    close( fd );
}


printscr( fd, screen )
int fd;
int screen;
{
    char text[65];
    int endfile;
    int ln;

    text[64] = '\0';
    endfile = read( fd, text, 64 );
    if( endfile <= 0 )
	return( TRUE );				/* and quit if end of file */
    fprintf( prn, "       Screen %d", screen );
    newline();
    println( text, 0 );
    for( ln=1; ln<16; ln++ ) {			/* print remaining 15 lines */
	endfile = read( fd, text, 64 );
	if( endfile > 0 ) 
	    println( text, ln );
    }
    newline();					/* two blank lines to */
    newline();					/* separate screens */
    return( FALSE );
}


println( text, ln )
char *text;
int ln;
{
    int skipcol;
    blanks( text );
    fprintf( prn, "   %2d: ", ln );
    if( *text != '\0' )
	fprintf( prn, "%s", text );
    newline();
}


blanks( textptr )
char *textptr;
{
    register char *endptr;
    register i;

    endptr = textptr + 64;
    while( (*endptr=='\b') && (endptr>=textptr) )
	endptr--;				/* scan off trailing blanks */
    *( ++endptr ) = '\0';			/* and force end of line */
}


newline()
{
    line++;
    if ( line > lastline )
	newpage();
    else {
	fputc('\015',prn);			/* send CR - LF sequence */
	fputc('\012',prn);
    }
}


newpage()
{
    stamp();					/* title stamp */
    line = 0;
    if ( side == LEFT )
	RHALF;					/* LaserJet to right half */
    else {
	kickpage();
	LHALF;					/* LaserJet to left half */
    }
    pagenum++;
    side = !side;
}


vline()
{
    int i;
    MIDDLE;
    for( i=0; i<=2*maxline; i++ )
    {
	fputc( '|', prn );
	fputc( '\015', prn );			/* overlap vert bars with */
	fputc( '\033', prn );			/* half line feed */
	fputc( '=', prn );
    }
}


kickpage()
{
    if( !nograf )
	vline();				/* draw line down middle */
    fputc(PAGE, prn);				/* kick out paper */
}


stamp()
{
    char datestr[10], timestr[10];
    if( nostamp )
	return;					/* no date stamping */
    timestamp( timestr );
    datestamp( datestr );
    if( !nograf )				/* no vertical line */
    {
	fputc( '\015', prn );
	ROW( maxline );
	fprintf( prn, "File: %-48s", printname );
	fprintf( prn, "%s   %s   ", datestr, timestr );
	fprintf( prn, "Page %2d", pagenum );
    }
    else if( !(pagenum & 0x01) )		/* stamp only on even pages */
    {
	MARGIN( 5,172 );			/* widen margins */
	ROW( maxline );				/* move to bottom row */
	fprintf( prn, "File: %-133s", printname );
	fprintf( prn, "Page: %-3d", (pagenum+1) / 2 );
	fprintf( prn, "   %s   %s", datestr, timestr);
    }
}


datestamp( datestr )
char  *datestr;
{
  DATETIME date;

  DosGetDateTime(&date);
  sprintf(datestr,"%02d/%02d/%02d",date.month, date.day, (date.year % 100));
}

timestamp( timestr )
char	*timestr;
{
  DATETIME date;

  DosGetDateTime(&date);
  sprintf(timestr,"%02d:%02d",date.hours, date.minutes);
}
