/* p.c - Output C/ASM source code file as Postscript */

/* Last modified: 07-Jan-1991 */

#include <stdio.h>
#include <process.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <stdlib.h>

void top_of_doc(void);		/* Called at top of document */
void bot_of_doc(void);		/* Called at foot of document */
void top_of_page(void); 	/* Called at top of each page */
void bot_of_page(void); 	/* Called at foot of each page */
char *readline(char *where, int howmuch, FILE *infile);
void sign_on(void);
void process_options(int argc, char **argv);
void help(void);

FILE *infile, *outfile; 	/* input and output files */

struct {			/* Page size, in points */
    long x;
    long y;
} pagesize = {595, 841};

static char mname[][4]= {"Jan","Feb","Mar","Apr","May","Jun",
			"Jul","Aug","Sep","Oct","Nov","Dec"};

struct {			/* Page margins, in points */
    long top;
    long bottom;
    long left;
    long right;
} margins = {50, 50, 72, 36};

#ifndef OS2

#define FACESIZE 64		/* Max length of font name */

struct _FONTMETRICS {
    char szFamilyname[FACESIZE];    /* Subset of OS/2 font definition */
    long lEmHeight;
    long lExternalLeading;
    long lAveCharWidth;
} font = {"Courier",8,0,6};

#endif

int tabset = 8; 		/* Where to set tabs on the line */

struct tm *tm_time;		/* Microsoft C time stuff */
time_t now;

long x,y;
int linecount, pagecount = 1;	/* Line and page counters */

char filename[255];		/* HPFS filenames can be 254 chars long */
char printer[255] = {"LPT1"};	/* Default to LPT1 */

main(int argc, char **argv)
{
    char linebuf[255];

    if(argc < 2) {
	fprintf(stderr,"Usage: p [-options] filename\n");
	fprintf(stderr,"p -? for help");
	exit(1);
    }

    /* Deal with command-line options */
    process_options(argc,argv);

    /* Open the input file */
    if((infile = fopen(filename,"r")) == NULL) {
	fprintf(stderr,"Unable to open input file %s",filename);
	exit(1);
    }

    /* Open the printer */
    if((outfile = fopen(printer,"w")) == NULL) {
	fprintf(stderr,"Unable to open printer %s",printer);
	exit(1);
    }

    /* Try to read the first line, exit if we fail */
    if(readline(linebuf, sizeof(linebuf), infile) == NULL)
	exit(1);
    /* Read the clock */
    time(&now);
    tm_time = localtime(&now);
    /* Perform top-of-document processing, e.g. output Postscript header */
    top_of_doc();

    while(!feof(infile)) {
	/* Position within top margin of page */
	y = pagesize.y - margins.top - 2 * (font.lEmHeight + font.lExternalLeading);
	/* Perform top-of-page processing, e.g. print page header */
	top_of_page();

	/* Now work down the page, line by line */
	while(y > margins.bottom + 2 * (font.lEmHeight + font.lExternalLeading ) && !feof(infile)) {
	    /* Move down a line */
	    y -= font.lEmHeight + font.lExternalLeading;
	    /* Move to left margin */
	    x = margins.left;
	    fprintf(outfile,"%ld %ld moveto",x,y);
	    /* Output line of text, in brackets, followed by 'show' */
	    fprintf(outfile,"(%s) show\n",linebuf);
	    /* Count that line, and then read the next */
	    linecount++;
            readline(linebuf,sizeof(linebuf),infile);
	}
	/* Perform bottom-of-page processing */
	bot_of_page();
    }
    bot_of_doc();
    return 0;
}

void top_of_doc()
{
    /* Output the Postscript header comments */
    fprintf(outfile,"%%!PS-Adobe-2.0\n");
    fprintf(outfile,"%%%%Title: PSLIST Postscript File Lister\n");
    fprintf(outfile,"%%%%Creator: Les Bell and Associates Pty Ltd\n\n");

    /* Pick and size the appropriate font */
    fprintf(outfile,"/%s findfont %ld scalefont setfont\n",font.szFamilyname,font.lEmHeight);

    /* Set the page and line counters */
    pagecount = linecount = 1;
}

void bot_of_doc()
{
    fprintf(outfile,"\004");
}

void top_of_page()
{
    /* Go to the top left of the page */
    fprintf(outfile,"%ld %ld moveto ",margins.left, pagesize.y - margins.top);

    /* Print header: filename first */
    fprintf(outfile,"(File: %s) show\n",filename);

    /* Position half the date/time width to the left of the center */
    fprintf(outfile,"%ld %ld moveto ",
	pagesize.x / 2 - 18 * font.lAveCharWidth / 2,
	pagesize.y - margins.top);
    /* and then output the date */
    fprintf(outfile,"(%02d %3s %02d %02d:%02d:%02d) show\n",
	tm_time->tm_mday,
	mname[tm_time->tm_mon],
	tm_time->tm_year,
	tm_time->tm_hour,
	tm_time->tm_min,
	tm_time->tm_sec);

    /* and finally move to top right and output the page number */
    fprintf(outfile,"%ld %ld moveto ",
	pagesize.x - margins.right - (7 * font.lAveCharWidth),
	pagesize.y - margins.top);
    fprintf(outfile,"(Page %d) show\n",pagecount);
}

void bot_of_page()
{
    /* Issue a Postscript showpage command */
    fprintf(outfile,"showpage\n");
    pagecount++;
    /* and display a dot on the screen to show we haven't hung */
    printf(".");
}

/*******************************************************************\
*	    Read a line, expanding tabs to spaces		    *
*	    and escaping embedded parentheses			    *
\*******************************************************************/

char *readline(char *where, int howmuch, FILE *infile)
{
    int c, i = 0;
    char *temp;

    temp = where;

    /* Read a char at a time until newline or end-of-file or overflowing buffer */
    while((c = getc(infile)) != EOF && c != '\n' && i < howmuch) {
	/* If it's a tab, insert spaces until a tab stop */
	if(c == '\t') {
	    where[i++] = ' ';
	    while (i%tabset)
		where[i++] = ' ';
	}
	/* If it's a backslash, insert two backslashes */
        else if(c == '\\') {
            where[i++] = '\\';
            where[i++] = '\\';
	}
	/* If it's a bracket, make it backslash-bracket */
	else if(c == '(' || c == ')') {
	    where[i++] = '\\';
	    where[i++] = (char)c;
	}
	/* None of the above, just plonk it in the buffer */
        else
            where[i++] = (char)c;
    }
    /* We've reached the end of the line, append a null character */
    where[i] = '\0';
    /* If we read nothing and are at end-of-file, return NULL */
    if (c == EOF && i == 0)
	return NULL;
    /* otherwise, return the address of the buffer - that's temp */
    else
	return temp;
}

void sign_on()
{
    printf("--------------------------------------------------------------------\n");
    printf("Postscript Print Formatter. (C) 1990 Les Bell and Associates Pty Ltd\n");
    printf("--------------------------------------------------------------------\n");
}

/*******************************************************************\
*	    Process command-line options			    *
\*******************************************************************/

void process_options(argc,argv)
int argc;
char **argv;
{
    int i,j,k;

    /* For each word on the command line in turn, after the first */
    for(i = 1; i < argc; i++) {
	/* If the first character is dash or slash, it's an option */
	if (argv[i][0] == '-' || argv[i][0] == '/') {
	    /* Look at the next character */
	    j = 1;
	    while (argv[i][j]) {
		switch(toupper(argv[i][j])) {
		    case 'F':
			/* If it's an F, set the font */
			for(j++, k = 0; font.szFamilyname[k] = argv[i][j] ;j++,k++);
			j--;
			break;
		    case 'S':
			/* If it's an S, set the font size */
			font.lEmHeight = atol(argv[i]+j+1);
			while(argv[i][j]) j++; j--;
			break;
		    case 'T':
			/* If it's a T, change the tab settings */
			tabset = atoi(argv[i]+j+1);
			while(argv[i][j]) j++; j--;
			break;
		    case 'O':
			/* If it's an O, change the output printer device */
			for(j++, k = 0; printer[k] = argv[i][j] ;j++,k++);
			j--;
			break;
		    case '?':
			/* For ?, provide help */
			help();
		}
		j++;
	    }
	}
	else
	    /* This is the filename */
	    strcpy(filename,argv[i]);
    }
}

void help(void)
{
    sign_on();
    printf("Usage: p [-options] filespec\n");
    printf("Options:\n\t-fName\tSet font\n\t-snn\tSet font size\n");
    printf("\t-tnn\tSet tab spacing\n\t-oDev\tSet output device/file\n");

    exit(0);
}
