/******
*
*   module:      lorenz.c
*   descripton:  program that draws the Lorenz Attractor, a chaotic function
*                as seen on the television show NOVA "THE STRANGE NEW
*                SCIENCE OF CHAOS"
*
*                this program is adapated from a similar program by
*                Marty Belles 9-2-89
*
*  programmer:   Bryan A. Woodruff (Woodruff Software Systems) 6/2/90
*
******/

#include <windows.h>
#include <math.h>
#include "lorenz.h"

long FAR PASCAL WndProc(HWND, unsigned, WORD, LONG);
void DrawLorenz(HWND);

int     iPrevX, iPrevY, iPrevDef;
short   xClient, yClient;
short   nRed, nGreen, nBlue;
double  dX, dY, dZ, dThetaX, dThetaY, dThetaZ, dCosX, dCosY, dCosZ,
        dSinX, dSinY, dSinZ;

WORD    wColor = IDM_RANDOM, wPlane = IDM_YZPLANE;


int PASCAL WinMain(hInstance, hPrevInstance, lpszCmdLine, nCmdShow)
HANDLE  hInstance, hPrevInstance;
LPSTR   lpszCmdLine;
int     nCmdShow;
{
   HWND         hWnd;
   MSG          msg;
   WNDCLASS     wndclass;
   static char  szAppName[] = "lorenz";
   static char  szVersion[] = "Lorenz Attractor v1.0";

   if (hPrevInstance)
      return (FALSE);

   wndclass.style         = CS_HREDRAW | CS_VREDRAW;
   wndclass.lpfnWndProc   = WndProc;
   wndclass.cbClsExtra    = 0;
   wndclass.cbWndExtra    = 0;
   wndclass.hInstance     = hInstance;
   wndclass.hIcon         = NULL;
   wndclass.hCursor       = LoadCursor (NULL, IDC_ARROW);
   wndclass.hbrBackground = GetStockObject (WHITE_BRUSH);
   wndclass.lpszMenuName  = szAppName;
   wndclass.lpszClassName = szAppName;

   if (!RegisterClass(&wndclass))
      return FALSE;

   hWnd = CreateWindow(szAppName, szVersion, WS_OVERLAPPEDWINDOW,
                        CW_USEDEFAULT, 0, CW_USEDEFAULT, 0, NULL, NULL,
                        hInstance, NULL);

   ShowWindow(hWnd, nCmdShow);
   UpdateWindow(hWnd);

   while (TRUE) {
      if (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE)) {
         if (msg.message == WM_QUIT)
            break;
         TranslateMessage(&msg);
         DispatchMessage(&msg);
       }
       else
          DrawLorenz(hWnd);
   }
   return msg.wParam;

} /* end of WinMain */

/******
*
*     function: DrawLorenz
*  description: calculates and draws Lorenz Attractor
*
*               *** adapted from FRACTAL PROGRAMMING IN C, pp. 87 - 90 ***
*
******/

#define ONETHIRD (1.0 / 3.0)

void DrawLorenz(hWnd)
HWND hWnd;
{
   HPEN    hPen;
   HDC     hDC;

   int     iX, iY;
   double  dt = 0.01, dt2 = dt / 2.0;
   double  d0_x, d0_y, d0_z, d1_x, d1_y, d1_z, d2_x, d2_y, d2_z,
           d3_x, d3_y, d3_z, xt, yt, zt;

   int     rand();


   d0_x = 10.0 * (dY - dX) * dt2;
   d0_y = (-dX * dZ + 28.0 * dX - dY) * dt2;
   d0_z = (dX * dY - 8.0 * dZ / 3.0) * dt2;
   xt = dX + d0_x;
   yt = dY + d0_y;
   zt = dZ + d0_z;
   d1_x = 10.0 * (yt - xt) * dt2;
   d1_y = (-xt * zt + 28.0 * xt - yt) * dt2;
   d1_z = (xt * yt - 8.0 * zt / 3.0) * dt2;
   xt = dX + d1_x;
   yt = dY + d1_y;
   zt = dZ + d1_z;
   d2_x = 10.0 * (yt - xt) * dt;
   d2_y = (-xt * zt + 28.0 * xt - yt) * dt;
   d2_z = (xt * yt - 8.0 * zt / 3.0) * dt;
   xt = dX + d2_x;
   yt = dY + d2_y;
   zt = dZ + d2_z;
   d3_x = 10.0 * (yt - xt) * dt2;
   d3_y = (-xt * zt + 28.0 * xt - yt) * dt2;
   d3_z = (xt * yt - 8.0 * zt / 3.0) * dt2;
   xt = dX + d3_x;
   yt = dY + d3_y;
   zt = dZ + d3_z;

   dX += (d0_x + d1_x + d1_x + d2_x + d3_x) * ONETHIRD;
   dY += (d0_y + d1_y + d1_y + d2_y + d3_y) * ONETHIRD;
   dZ += (d0_z + d1_z + d1_z + d2_z + d3_z) * ONETHIRD;

   switch (wColor) {
      case IDM_BLACKONWHITE :
         nRed   = 0;
         nGreen = 0;
         nBlue  = 0;
         break;
      case IDM_WHITEONBLACK :
         nRed   = 255;
         nGreen = 255;
         nBlue  = 255;
         break;
      case IDM_RANDOM:
         if (((iY < 0) && (iPrevY >= 0)) ||
             ((iY > 0) && (iPrevY <= 0))) {
            nRed   = (short) (rand () % 255);
            nGreen = (short) (rand () % 255);
            nBlue  = (short) (rand () % 255);
         }
   }

   hDC = GetDC(hWnd);
   hPen = CreatePen(PS_SOLID, 1, RGB(nRed, nGreen, nBlue));
   SelectObject(hDC, hPen);
   SetMapMode(hDC, MM_ISOTROPIC);
   SetWindowExt(hDC, 500, 500);
   SetViewportExt(hDC, xClient / 2, -yClient / 2);
   SetViewportOrg(hDC, xClient / 2, yClient / 2);
   MoveTo(hDC, iPrevX, iPrevY);
   switch (wPlane) {
      case IDM_XYPLANE:
         iX = (int)(10.0 * dX - 250.0);
         iY = (int)(10.0 * dY);
         break;
      case IDM_YZPLANE:
         iX = (int)(10.0 * dZ - 250.0);
         iY = (int)(10.0 * dY);
         break;
      case IDM_XYZPLANE:
         iX = (int)((dX * dSinX + dY * dSinY + dZ * dSinZ) * 10.0 - 250.0);
         iY = (int)((dX * dCosX + dY * dCosY + dZ * dCosZ) * 10.0);
   }
   if (!iPrevDef)
      MoveTo(hDC, iX, iY);
   LineTo(hDC, iX, iY);
   iPrevX = iX;
   iPrevY = iY;
   iPrevDef = TRUE;
   ReleaseDC(hWnd, hDC);
   DeleteObject(hPen);

} /* end of DrawLorenz */

BOOL FAR PASCAL AboutDlgProc(hDlg, iMessage, wParam, lParam)
HWND      hDlg;
unsigned  iMessage;
WORD      wParam;
LONG      lParam;
{
   switch (iMessage) {
      case WM_INITDIALOG:
         break;

      case WM_COMMAND:
         switch (wParam) {
            case IDOK:
               EndDialog (hDlg, 0);
               break;

            default:
               return FALSE;
         }
         break;

      default:
         return FALSE;

   }
   return TRUE;

} /* end of AboutDlgProc */

long FAR PASCAL WndProc(hWnd, iMessage, wParam, lParam)
HWND      hWnd;
unsigned  iMessage;
WORD      wParam;
LONG      lParam;
{
   static FARPROC  lpfnAboutDlgProc;
   static HWND     hInstance;

   HMENU  hMenu;
   HDC    hDC;

   switch (iMessage) {
      case WM_CREATE:
         hInstance = ((LPCREATESTRUCT) lParam) -> hInstance;
         lpfnAboutDlgProc = MakeProcInstance(AboutDlgProc, hInstance);
         break;

      case WM_SIZE:
         dX = 0.0;
         dY = 1.0;
         dZ = 0.0;
         iPrevDef = FALSE;

         xClient = LOWORD(lParam);
         yClient = HIWORD(lParam);
         SetClassWord (hWnd, GCW_HBRBACKGROUND,
                       GetStockObject ((wColor == IDM_BLACKONWHITE) ?
                                        WHITE_BRUSH : BLACK_BRUSH));
         InvalidateRect (hWnd, NULL, TRUE);
         break;

      case WM_COMMAND:
         switch (wParam) {
            case IDM_WHITEONBLACK:
            case IDM_BLACKONWHITE:
            case IDM_RANDOM:
               hMenu = GetMenu(hWnd);
               CheckMenuItem(hMenu, wColor, MF_UNCHECKED);
               wColor  = wParam;
               CheckMenuItem(hMenu, wColor, MF_CHECKED);
               SetClassWord (hWnd, GCW_HBRBACKGROUND,
                             GetStockObject ((wParam == IDM_BLACKONWHITE) ?
                                              WHITE_BRUSH : BLACK_BRUSH));

               InvalidateRect (hWnd, NULL, TRUE);
               break;

            case IDM_XYPLANE:
            case IDM_YZPLANE:
            case IDM_XYZPLANE:
               hMenu = GetMenu(hWnd);
               CheckMenuItem(hMenu, wPlane, MF_UNCHECKED);
               wPlane = wParam;
               CheckMenuItem(hMenu, wPlane, MF_CHECKED);
               if (wPlane == IDM_XYZPLANE) {
                  dThetaX = 0.7853981633974;
                  dThetaY = 0.0;
                  dThetaZ = 1.570796326795;
                  dCosX = cos(dThetaX);
                  dCosY = cos(dThetaY);
                  dCosZ = cos(dThetaZ);
                  dSinX = sin(dThetaX);
                  dSinY = sin(dThetaY);
                  dSinZ = sin(dThetaZ);
               }
               iPrevDef = FALSE;
               InvalidateRect (hWnd, NULL, TRUE);
               break;

            case IDM_ABOUT:
               DialogBox (hInstance, "LorenzAbout", hWnd,
                          lpfnAboutDlgProc);
               break;

      }
      break;

      case WM_DESTROY:
         PostQuitMessage(0);
         break;

      default:
         return DefWindowProc (hWnd, iMessage, wParam, lParam);
   }

   return 0L;

} /* end of WndProc */
