/***
 * Estrazione, Controllo e Ordinamento dei Dati Kepleriani NASA 2-linee
 * 12/2/94 - Giovanni Mazzola - IT9XXS - mc3626@mclink.it
 *******/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#define NUMBER(array) sizeof(array)/sizeof(array[0])
#define BUFFER 85
#define MAX_EL 255

/* NASA 2-Lines data structure */
struct DATA {
   char satname[20];
	char first_row[80];
   char second_row[80];
} keps[MAX_EL];

/* Files involved */
FILE *fp_in, *fp_out, *fp_ref;

/* Global variables */
int i = 0;

/* Text-file-position (NASA row 1) */
fpos_t posrow1;

/* Prototypes declarations */
int checksum(char row[]);
int substr(char *s_string, int s_offset, int s_length);
int extract(void);
int strcmp();

int main(int argc, char *argv[])
{
    int found;              /* Flag (was the requested satellite found or not ?)*/
	 int success = 0;        /* How many requested satellites have been found */
    int valid = 0;          /* How many valid satellites have been found */
    char wsat[80];          /* Max satellites to find */

	if(argc < 3)
	{
		puts("");
		puts("* * NASA 2-Lines filtering/checking utility * *\n");
		puts("Syntax: SORTNENG <file_to_scan> <keps_file> [sat_names_file] [> log]\n");
		puts("IT9XXS Giovanni Mazzola - 12/2/94 - Internet: mc3626@mclink.it");
		puts("\07");
		exit(1);
	}

    /* Check against equal file-names */
	if((strcmpi(argv[1], argv[2]) == 0) || (strcmpi(argv[2], argv[3]) == 0))
	{
		printf("Error: file names must be different !\07");
		exit(1);
	}

	/* Open source file */
	if ((fp_in = fopen(argv[1], "r")) == NULL)
	{
		fprintf(stderr, "\nError: unable to open input file %s\07\n", strupr(argv[1]));
		exit(1);
	}

	/* Open reference file */
	if(argc == 4)
	{
		if ((fp_ref = fopen(argv[3], "r")) == NULL)
		{
			fprintf(stderr, "\nError: unable to open reference file %s\07\n", strupr(argv[3]));
			exit(1);
		}
	}

	/* Extract all valid NASA keps from source file */
	printf("Scanning %s ...", argv[1]);
	if((valid = extract()) == 0 )
	{
		printf("Sorry... no keps found in %s !", argv[1]);
		exit(1);
	}

	/* Sort the array holding the element set */
	qsort(keps, valid, sizeof(keps[MAX_EL]), strcmp);

	/* Open output file */
	if ((fp_out = fopen(argv[2], "w")) == NULL)
	{
		fprintf(stderr, "\nError: unable to open output file %s\07\n", strupr(argv[2]));
		exit(1);
	}

	/* If no reference file was passed from the command line,     */
	/* write-out the *entire* element set extracted and sorted... */
	if(argc < 4)
	{
		for(i=0; i < valid; i++)
		{
			fprintf(fp_out, "%s", keps[i].satname);
			fprintf(fp_out, "%s", keps[i].first_row);
			fprintf(fp_out, "%s", keps[i].second_row);
		}
		printf("\n%s has been succesfully created (%d elements).\n", strupr(argv[2]), valid);
	}
	else /* extract the requested satellites only */
	{
		printf("\nExtracting requested satellites...\n");

		while(fgets(wsat, BUFFER, fp_ref) != NULL)
		{
			if(isalnum(* wsat))
			{
				for(i = 0; i < valid; i++)
				{
					if(strcmpi(wsat, keps[i].satname) == 0)
					{
						fprintf(fp_out, "%s", keps[i].satname);
						fprintf(fp_out, "%s", keps[i].first_row);
						fprintf(fp_out, "%s", keps[i].second_row);
						printf("Found %s", wsat);
						success++;
						break;
					}
				}
				if(i == valid) printf("Not found: %s\07", wsat);
			}
		}
		printf("\n%s has been succesfully created (%d elements).\n", strupr(argv[2]), success);
	}
	fcloseall();
	return(0);
}


/***
 * Input-file scanning
 *******/

int extract(void)
{
	while(!feof(fp_in))
	{
		fgets(keps[i].satname, BUFFER, fp_in);
		fgetpos(fp_in, &posrow1);
		fgets(keps[i].first_row, BUFFER, fp_in);
		fgets(keps[i].second_row, BUFFER, fp_in);

		/* Object number must be the same on the two lines */
		if(strnicmp(keps[i].first_row + 2, keps[i].second_row + 2, 5) == 0 &&
			((strchr(keps[i].first_row, 'U') - keps[i].first_row) == 7))
		{
			/* Verify the checksum for both rows */
			if((checksum(keps[i].first_row) != keps[i].first_row[68] - 48) ||
				(checksum(keps[i].second_row) != keps[i].second_row[68] - 48))
			{
				printf("\nChecksum error, line read was:\n%s\n", keps[i].satname);
			}
			else /* if checksum is valid */
			{
				printf("%s", keps[i].satname);
				i++;
			}
		}
		  else /* advance one row */
		{
			if(!feof(fp_in)) fsetpos(fp_in, &posrow1);
		}
	}
	return(i);
}

/***
 * Checksum calculator
 *******/
int checksum(char row[])
{
	int k, csum;
	csum = 0;

	 /* Sum all numbers togheter     */
	 /* plus 'minus sign' (counts 1) */
	 /* The others count 0 (ignored) */
	for(k = 0; k < 68; k++)
	{
		if(isdigit(row[k])) csum = csum + row[k] - 48;
		if(row[k] == '-') csum++;
	}
	return(csum %= 10);
}
