//-------------------------------------------------------------------------
//
//                  Fractal Landscape Generator v1.2
//                   Copyright (c) 1992 Steve Anger
//                This program is freely distributable
//
//   FRGEN is a utility to generate fractal landscapes and shapes using
// successive triangle sub-division. The fractal data can be output as PoV
// or Vivid scene description files or as raw triangle data.
//
//                                        Internet: steve.anger@rose.uucp
//                                            (alt) lsuc!rose.uucp!steve.anger
//                         You Can Call me Ray BBS: (708)358-5611
//-------------------------------------------------------------------------

#include <fstream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include "vector.hpp"

#ifdef __TURBOC__
#include <graphics.h>
#include <conio.h>
#include <dos.h>
extern unsigned _stklen = 8096;  // Larger stack size for recursion
#endif

typedef enum bool {false = 0, true = 1};
typedef enum OFormat {POV, VIVID, RAW};

const char ver[] = "v1.2";
const int  VMAX = (3*256) + 1;

// Function prototypes
void usage(void);
void abortmsg (char *msg, int exit_code);
void add_ext (char *fname, char *ext, bool force);
void process_args (int argc, char *argv[]);
void file_args(void);
void process_option (char *s);
void write_options(void);
Vector noise_vector (Vector& v);
bool read_triangle (Vector& a, Vector& b, Vector& c,
                       int& fix_ab, int& fix_bc, int& fix_ca);
void tri_fractal (Vector& a, Vector& b, Vector& c,
                  int fix_ab, int fix_bc, int fix_ca, int level);
void set_view (void);
void calc_bounds (Vector *vlist, int count, Vector& center, float& radius);
void plot_tri (const Vector& a, const Vector& b, const Vector& c);
int init_display (void);
void close_display (void);
void swap (float &a, float &b);
void check_abort (void);


// Global variables
fstream f, g;
Vector  viewpoint;      // Where you're looking from
Vector  lookat;         // Where you're looking at
Vector  nx, ny, nz;     // Basis vectors for viewpoint
Vector  nscale;         // Noise scaling factors
Vector  nbias;          // Noise biasing factors
int     depth;          // Depth of recursion
int     seed;           // Seed for random number generator
bool    display;        // Set to true if display preview enabled
bool    swap_yz;        // Swap y and z coords?
OFormat format;         // Output file format
char    infile[64];     // Input file name
char    outfile[64];    // Output file name
long    tri_count;      // Count of triangles generated
Vector  gmin_v;
Vector  gmax_v;
Vector  kmin, kmax;
Vector  vlist[VMAX];    // A list of the verticies of the last 256 triangles
int     vindex;         // Index into vlist[]
int     input_line;     // Current input line being parsed
long    mark2;
long    mark4;


int main (int argc, char* argv[])
{
    Vector a, b, c;

    cerr << "\n                 Fractal Landscape Generator " << ver << "\n";
    cerr << "                  Copyright (c) 1992 Steve Anger\n\n";

    randomize();
    seed = random(10000);
    infile[0] = '\0';
    outfile[0] = '\0';
    viewpoint = Vector (10.0, 10.0, -10.0);
    lookat = Vector (0.0, 0.0, 0.0);
    nscale = Vector (0.10, 0.10, 0.10);
    nbias = Vector (0.0, 0.0, 0.0);
    depth = 3;
    display = false;
    swap_yz = false;
    format = POV;
    tri_count = 0;
    input_line = 0;
    vindex = 0;

    for (int i = 1; i < argc; i++) {
        if (argv[i][0] != '-' && argv[i][0] != '/') {
            if (infile[0] == '\0') {
                strcpy (infile, argv[i]);
                add_ext (infile, "fr", false);
            }
            else if (outfile[0] == '\0') {
                strcpy (outfile, argv[i]);

            }
            else
                abortmsg ("Too many file names specified.", 1);
        }
    }

    if (infile[0] == '\0'  ||  outfile[0] == '\0') {
        usage();
        exit (1);
    }

    f.open (infile, ios::in);
    if (!f) abortmsg ("Error opening input file.", 1);

    file_args();  // Get options from input file

    process_args (argc, argv);  // Get the command line options

    switch (format) {
        case POV:   add_ext (outfile, "dat", false);
                    break;
        case VIVID: add_ext (outfile, "v", false);
                    break;
        case RAW:   add_ext (outfile, "raw", false);
                    break;
    }

    g.open (outfile, ios::out);
    if (!g) abortmsg ("Error opening output file.", 1);

    if (depth < 1)
        abortmsg ("Recurse depth (-r) must be >= 1", 1);

    if (display) {
        set_view();

        if (init_display() < 0)
            abortmsg ("Unable to initialize graphics display.", 1);
    }

    write_options();

    if (format == POV) {
        g << "declare DefaultTexture = texture\n";
        g << "    color red 1.0 green 1.0 blue 1.0\n";
        g << "    ambient 0.1\n";
        g << "    diffuse 0.6\n";
        g << "end_texture\n\n";

        g << "composite\n";
    }

    int fix_ab, fix_bc, fix_ca;
    gmin_v = Vector (+1e8, +1e8, +1e8);
    gmax_v = Vector (-1e8, -1e8, -1e8);

    while (read_triangle (a, b, c, fix_ab, fix_bc, fix_ca)) {
        tri_fractal (a, b, c, fix_ab, fix_bc, fix_ca, depth);
    }

    // Add a bounding shape to the entire fractal
    if (format == POV) {
        g << "    bounded_by\n";
        g << "\tintersection\n";
        g << "\t    plane <+1 0 0> " << +gmax_v.x << " end_plane\n";
        g << "\t    plane <-1 0 0> " << -gmin_v.x << " end_plane\n";
        g << "\t    plane <0 +1 0> " << +gmax_v.y << " end_plane\n";
        g << "\t    plane <0 -1 0> " << -gmin_v.y << " end_plane\n";
        g << "\t    plane <0 0 +1> " << +gmax_v.z << " end_plane\n";
        g << "\t    plane <0 0 -1> " << -gmin_v.z << " end_plane\n";
        g << "\tend_intersection\n";
        g << "    end_bound\n";
        g << "end_composite\n\n";
    }

    f.close();
    g.close();

    cerr << "\n" << tri_count << " triangles generated.\n";

    if (display) {
        cerr << "Press return.\n";
        getchar();
        close_display();
    }

    return 0;
}


void usage()
{
    cerr << "Usage: frgen infile[.fr] outfile[.dat] [[-,/]options]\n";
    cerr << "Options: -sxnnn   Scale noise in x direction by nnn (0.0 - 1.0)\n";
    cerr << "         -synnn     '     '    ' y     '      '  '     '     ' \n";
    cerr << "         -sznnn     '     '    ' z     '      '  '     '     ' \n";
    cerr << "         -bxnnn   Bias noise in  x direction by nnn (-1.0 - 1.0)\n";
    cerr << "         -bynnn     '    '    '  y     '      '  '     '     '  \n";
    cerr << "         -bznnn     '    '    '  z     '      '  '     '     '  \n";
    cerr << "         -vxnnn   Set viewpoint x coord to nnn\n";
    cerr << "         -vynnn    '      '     y   '    '  ' \n";
    cerr << "         -vznnn    '      '     z   '    '  ' \n";
    cerr << "         -lxnnn   Set lookat x coord to nnn\n";
    cerr << "         -lynnn    '     '   y   '    '  ' \n";
    cerr << "         -lznnn    '     '   z   '    '  ' \n";
    cerr << "         -ennn    Use nnn as seed for random number generator\n";
    cerr << "         -rnnn    Maximum recursion depth of nnn\n";
    cerr << "         -d       Display generated fractal on screen\n";
    cerr << "         -o[n]    Set output format (-op = PoV, -ov = Vivid, -or = raw)\n";
    cerr << "         -x       Exchange Y and Z coords on output.\n";
}


void abortmsg (char *msg, int exit_code)
{
    if (display)
        close_display();

    cerr << msg << "\n";
    exit (exit_code);
}


void add_ext (char *fname, char *ext, bool force)
{
    int i;

    for (i = 0; i < strlen(fname); i++)
        if (fname[i] == '.') break;

    if (fname[i] == '\0' || force) {
        fname[i] = '.';
        strcpy (&fname[i+1], ext);
    }
}


void swap (float &a, float &b)
{
    float temp = a;
    a = b;
    b = temp;
}


void process_args (int argc, char* argv[])
{
    for (int i = 1; i < argc; i++) {
        if (argv[i][0] == '-' || argv[i][0] == '/')
            process_option (&argv[i][1]);
    }
}


void file_args()
{
    char line[100] = "";
    char last_ch = ' ';

    do {
        f.getline (line, 100);
        input_line++;
    } while (!f.eof() && line[0] == ';');

    if (f.eof())
        return;

    for (int i = 0; i < strlen (line); i++) {
        if ((line[i] == '-' || line[i] == '/') && isspace(last_ch))
            process_option (&line[i+1]);

        last_ch = line[i];
    }
}


void process_option (char *s)
{
    switch (s[0]) {
        case 'v':
            switch (s[1]) {
                case 'x': sscanf (&s[2], "%f", &viewpoint.x);
                          break;
                case 'y': sscanf (&s[2], "%f", &viewpoint.y);
                          break;
                case 'z': sscanf (&s[2], "%f", &viewpoint.z);
                          break;
                default : cerr << "Invalid option -v" << s[1] << "\n";
            }
            break;

        case 'l':
            switch (s[1]) {
                case 'x': sscanf (&s[2], "%f", &lookat.x);
                          break;
                case 'y': sscanf (&s[2], "%f", &lookat.y);
                          break;
                case 'z': sscanf (&s[2], "%f", &lookat.z);
                          break;
                default : cerr << "Invalid option -l" << s[1] << "\n";
            }
            break;

        case 's':
            switch (s[1]) {
                case 'x': sscanf (&s[2], "%f", &nscale.x);
                          break;
                case 'y': sscanf (&s[2], "%f", &nscale.y);
                          break;
                case 'z': sscanf (&s[2], "%f", &nscale.z);
                          break;
                default : cerr << "Invalid option -s" << s[1] << "\n";
            }
            break;

        case 'b':
            switch (s[1]) {
                case 'x': sscanf (&s[2], "%f", &nbias.x);
                          break;
                case 'y': sscanf (&s[2], "%f", &nbias.y);
                          break;
                case 'z': sscanf (&s[2], "%f", &nbias.z);
                          break;
                default : cerr << "Invalid option -b" << s[1] << ", ignored\n";
            }
            break;

        case 'e': sscanf (&s[1], "%d", &seed);
                  break;

        case 'r': sscanf (&s[1], "%d", &depth);
                  break;

        case 'd': display = true;
                  break;

        case 'x': swap_yz = true;
                  break;

        case 'o' :
            switch (s[1]) {
                case 'p': format = POV;    break;
                case 'v': format = VIVID;  break;
                case 'r': format = RAW;    break;
                default : cerr <<"Invalid option -o" << s[1] << ", ignored\n";
            }
            break;


        default : cerr << "Invalid option -" << s[1] << ", ignored\n";
    }
}


void write_options()
{
    if (format == POV || format == VIVID) {
        g << "/*\n";
        g << "     Generated with FRGEN " << ver << " from file " << infile << "\n";
        g << "     Options in effect:";
        g << " -sx" <<  nscale.x << " -sy" << nscale.y << " -sz" << nscale.z;
        g << " -bx" << nbias.x << " -by" << nbias.y << " -bz" << nbias.z;
        g << " -r" << depth << " -e" << seed << "\n";
        g << "*/\n\n";
    }
}


bool read_triangle (Vector& a, Vector& b, Vector& c,
                    int& fix_ab, int& fix_bc, int& fix_ca)
{
    char line[100] = "";
    char msg[40];
    int n;

    do {
        f.getline (line, 100);
        input_line++;
    } while (!f.eof()  &&  line[0] == ';');

    if (f.eof())
        return false;

    a = Vector (0.0, 0.0, 0.0);
    b = Vector (0.0, 0.0, 0.0);
    c = Vector (0.0, 0.0, 0.0);

    fix_ab = 0;
    fix_bc = 0;
    fix_ca = 0;

    n = sscanf (line, "%f %f %f %f %f %f %f %f %f %d %d %d",
                &a.x, &a.y, &a.z, &b.x, &b.y, &b.z, &c.x, &c.y, &c.z,
                &fix_ab, &fix_bc, &fix_ca);

    if (n == EOF)
        return false;

    if (!(n == 9 || n == 12)) {
        sprintf (msg, "Error in input file, line %d", input_line);
        abortmsg (msg, 1);
    }

    return true;
}


Vector noise_vector (Vector& v)
{
    // Generate a random vector that is a function of the
    // vector v's position
    Vector noise;

    // seed the rand # generator with a mish-mash of the x, y, and z coords
    srand (seed ^ (long)(-23465*v.x) ^ (long)(17365*v.y) ^ (long)(5364*v.z));

    noise.x = nscale.x * (2.0*rand()/RAND_MAX - 1.0 + nbias.x);
    noise.y = nscale.y * (2.0*rand()/RAND_MAX - 1.0 + nbias.y);
    noise.z = nscale.z * (2.0*rand()/RAND_MAX - 1.0 + nbias.z);

    return noise;
}


void tri_fractal (Vector& a, Vector& b, Vector& c,
                  int fix_ab, int fix_bc, int fix_ca, int level)
{
    Vector ab, bc, ca, oa, ob, oc, norm;
    float ab_len, bc_len, ca_len;

    if (level == 0) {
        oa = a;
        ob = b;
        oc = c;

        check_abort();

        if (swap_yz) {
            swap (oa.y, oa.z);
            swap (ob.y, ob.z);
            swap (oc.y, oc.z);
        }

        switch (format) {
            case POV:
                g << "\t    triangle ";
                g << "<" << oa << "> <" << ob << "> <" << oc << ">";
                g << " end_triangle\n";
                break;

            case VIVID:
                norm = (b - a) * (c - a);
                if (mag(norm) > 0.0)
                    norm = norm / mag(norm);

                g << "patch = {\n";
                g << "\tvertex = " << oa << "; normal = " << norm << ";\n";
                g << "\tvertex = " << ob << "; normal = " << norm << ";\n";
                g << "\tvertex = " << oc << "; normal = " << norm << ";\n";
                g << "}\n\n";
                break;

            case RAW:
                g << oa << " " << ob << " " << oc << "\n";
                break;
        }

        // Keep a list of the verticies of the last 256 triangles
        // generated (used to calculate bounding spheres).
        vlist[vindex] = a;   vindex = (vindex + 1) % VMAX;
        vlist[vindex] = b;   vindex = (vindex + 1) % VMAX;
        vlist[vindex] = c;   vindex = (vindex + 1) % VMAX;

        // Keep track of the min/max limits of the fractal
        gmin_v = min (gmin_v, a);
        gmin_v = min (gmin_v, b);
        gmin_v = min (gmin_v, c);
        gmax_v = max (gmax_v, a);
        gmax_v = max (gmax_v, b);
        gmax_v = max (gmax_v, c);

        tri_count++;

        if (display)
            plot_tri (a, b, c);
    }
    else {
        Vector noise_ab, noise_bc, noise_ca, norm;

        // Find the midpoints of the three line segments
        ab = (a + b)/2.0;
        bc = (b + c)/2.0;
        ca = (c + a)/2.0;

        ab_len = mag(a - b);
        bc_len = mag(b - c);
        ca_len = mag(c - a);

        // Compute the normal to the triangle
        norm = (b - a) * (c - a);
        norm = norm/mag(norm);

        // Create some noise proportional to the edge lengths
        noise_ab = ab_len * noise_vector (ab);
        noise_bc = bc_len * noise_vector (bc);
        noise_ca = ca_len * noise_vector (ca);

        // Don't let any 'fixed' points move out of the plane
        // of the triangle (remove the noise's normal component)
        if (fix_ab) noise_ab = noise_ab * norm;
        if (fix_bc) noise_bc = noise_bc * norm;
        if (fix_ca) noise_ca = noise_ca * norm;

        // Perturb the three vectors
        ab = ab + noise_ab;
        bc = bc + noise_bc;
        ca = ca + noise_ca;

        if (format == POV) {
            if (level == 4) {
                mark4 = tri_count;
                g << "  composite\n";
            }

            if (level == 2 || (level < 2 && level == depth)) {
                mark2 = tri_count;
                g << "    object\n";
                g << "\tunion\n";
            }
        }

        tri_fractal (a,  ab, ca, fix_ab, 0, fix_ca, level-1);
        tri_fractal (b,  bc, ab, fix_bc, 0, fix_ab, level-1);
        tri_fractal (c,  ca, bc, fix_ca, 0, fix_bc, level-1);
        tri_fractal (ab, bc, ca, 0,  0, 0,  level-1);

        Vector center;
        float radius;

        if (format == POV) {
            // Add a bounding sphere for every set of 16 triangles
            if (level == 2 || (level < 2 && level == depth)) {
                calc_bounds (vlist, int(3*(tri_count - mark2)), center, radius);

                g << "\tend_union\n\n";
                g << "\tbounded_by\n";
                g << "\t    sphere <" << center << "> " << radius << " end_sphere\n";
                g << "\tend_bound\n\n";
                g << "\ttexture DefaultTexture end_texture\n";
                g << "    end_object\n\n";
            }

            // Add another bounding sphere for every set of 256 triangles
            if (level == 4) {
                calc_bounds (vlist, int(3*(tri_count - mark4)), center, radius);

                g << "    bounded_by\n";
                g << "      sphere <" << center << "> " << radius << " end_sphere\n";
                g << "    end_bound\n";
                g << "  end_composite\n\n";
            }
        }
    }
}


void set_view()
{
    // Calculate a set of base vectors for the new viewpoint
    nz = lookat - viewpoint;
    nz = nz/mag(nz);

    ny = Vector (0.0, 1.0, 0.0) - nz*(nz.y/(nz % nz));
    ny = ny/mag(ny);

    nx = nz * ny;
}


void calc_bounds (Vector *vlist, int count, Vector& center, float& radius)
{
    // Find a sphere enclosing all of the points in vlist[]
    Vector  min_v, max_v;
    float r;
    int i, start;

    start = (vindex + VMAX - count) % VMAX;

    min_v = Vector (+1e8, +1e8, +1e8);
    max_v = Vector (-1e8, -1e8, -1e8);

    for (i = start; i != vindex ; i = (i + 1) % VMAX) {
        min_v = min (min_v, vlist[i]);
        max_v = max (max_v, vlist[i]);
    }

    center.x = (min_v.x + max_v.x)/2.0;
    center.y = (min_v.y + max_v.y)/2.0;
    center.z = (min_v.z + max_v.z)/2.0;

    radius = 0.0;
    for (i = start; i != vindex; i = (i + 1) % VMAX) {
        r = mag (vlist[i] - center);
        if (r > radius)
             radius = r;
    }
}


void plot_tri (const Vector& a, const Vector& b, const Vector& c)
{
#ifdef __TURBOC__
    Vector pa, pb, pc, ta, tb, tc;
    int ax, ay, bx, by, cx, cy;

    ta = a - viewpoint;
    tb = b - viewpoint;
    tc = c - viewpoint;

    pa = Vector (ta % nx, ta % ny, ta % nz);
    pb = Vector (tb % nx, tb % ny, tb % nz);
    pc = Vector (tc % nx, tc % ny, tc % nz);

    ax = int((0.75*pa.x/pa.z + 0.5)*getmaxx());
    bx = int((0.75*pb.x/pb.z + 0.5)*getmaxx());
    cx = int((0.75*pc.x/pc.z + 0.5)*getmaxx());
    ay = int((0.5 - pa.y/pa.z)*getmaxy());
    by = int((0.5 - pb.y/pb.z)*getmaxy());
    cy = int((0.5 - pc.y/pc.z)*getmaxy());

    setcolor (LIGHTBLUE);
    moveto (ax, ay);
    lineto (bx, by);
    lineto (cx, cy);
    lineto (ax, ay);
#endif
}


int init_display()
{
#ifdef __TURBOC__
    int gdriver = DETECT, gmode, errorcode;

    initgraph (&gdriver, &gmode, "");
    errorcode = graphresult();
    if (errorcode != grOk) {
        cerr << "ERROR:" << grapherrormsg (errorcode) << "\n\n";
        return -1;
    }
#endif

    return 0;
}


void close_display()
{
#ifdef __TURBOC__
    restorecrtmode();
#endif
}


void check_abort()
{
#ifdef __TURBOC__
    static int cnt = 0;

    ++cnt;

    if (cnt >= 10) {
        cnt = 0;

        if (kbhit())
            abortmsg ("Aborted!", 1);
    }
#endif
}


