#include <dos.h>
#include <stdio.h>
#include <string.h>

#pragma pack(1)

/****************************************
*   Register definitions                *
****************************************/
union   REGS    Regs;
struct  SREGS   SRegs;

#define AX      Regs.x.ax
#define BX      Regs.x.bx
#define CX      Regs.x.cx
#define DX      Regs.x.dx
#define SI      Regs.x.si
#define DI      Regs.x.di

#define AH      Regs.h.ah
#define AL      Regs.h.al
#define BH      Regs.h.bh
#define BL      Regs.h.bl
#define CH      Regs.h.ch
#define CL      Regs.h.cl
#define DH      Regs.h.dh
#define DL      Regs.h.dl

#define CY      Regs.cflag

#define CS      SRegs.cs
#define DS      SRegs.ds
#define ES      SRegs.es
#define SS      SRegs.ss

/****************************************
*   VESA function definitions           *
****************************************/
#define GetInfo             0
#define GetModeInfo         1
#define SetMode             2
#define GetMode             3
#define SaveRestoreState    4
#define GetSetRAMWindow     5

/****************************************
*   VESA subfunction definitions        *
****************************************/
#define GetStateBufferSize  0
#define SaveState           1
#define RestoreState        2
#define SetWindow           0
#define GetWindow           1

/****************************************
*   VESA information structure          *
****************************************/
struct VESAInfo
{
    unsigned char           Signature[4];
    unsigned int            VESAVersion;
    unsigned char far *     OEMString;
    unsigned char           Capabilities[4];
    unsigned int far *      ModeList;
    unsigned char           Reserved[238];
};
typedef struct VESAInfo *       VESAInfoPtr;

/****************************************
*   VESA mode information structure     *
****************************************/
struct VESAModeInfo
{
    unsigned int            ModeAttributes;
    unsigned char           WindowAAttributes;
    unsigned char           WindowBAttributes;
    unsigned int            WindowGranularity;
    unsigned int            WindowSize;
    unsigned int            WindowASegment;
    unsigned int            WindowBSegment;
    void (far *             SelectWindow)(void);
    unsigned int            BytesPerScanLine;

    unsigned int            WidthInPixels;
    unsigned int            HeightInPixels;
    unsigned char           CharacterWidthInPixels;
    unsigned char           CharacterHeightInPixels;
    unsigned char           Planes;
    unsigned char           BitsPerPixel;
    unsigned char           Banks;
    unsigned char           MemoryModel;
    unsigned char           BankSize;
    unsigned char           Reserved[227];
};
typedef struct VESAModeInfo *   VESAModeInfoPtr;

/****************************************
*   Mode attributes                     *
****************************************/
#define ModeSupported       0x0001
#define OptionalInfo        0x0002
#define BIOSOutputSupport   0x0004
#define ColorMode           0x0008
#define GraphicsMode        0x0010

/****************************************
*   Window attributes                   *
****************************************/
#define WindowSupported     0x0001
#define WindowIsReadable    0x0002
#define WindowIsWritable    0x0004

/****************************************
*   VESA modes                          *
****************************************/
#define Mode640by400by256   0x0100
#define Mode640by480by256   0x0101
#define Mode800by600by16    0x0102
#define Mode800by600by256   0x0103
#define Mode1024by768by16   0x0104
#define Mode1024by768by256  0x0105
#define Mode1280by1024by16  0x0106
#define Mode1280by1024by256 0x0107

/****************************************
*   Mode test information structure     *
****************************************/
struct ModeTest
{
    unsigned int            Mode;
    unsigned int            Width;
    unsigned int            Height;
    unsigned int            Depth;
    unsigned int            KB;
};

/****************************************
*   Send VESA interrupt                 *
****************************************/

void intVESA(int Function)
{
    AL = Function;
    AH = 0x4f;
    int86x(0x10, &Regs, &Regs, &SRegs);
}

/****************************************
*   Select RAM window                   *
****************************************/

int SelectRAMWindow(int Window, int Page)
{
    BH = SetWindow;
    BL = Window;
    DX = Page;
    intVESA(GetSetRAMWindow);
    return !((int)((unsigned int)AH));
}

/****************************************
*   Get VESA information                *
****************************************/
void GetVESAInformation(VESAInfoPtr Info)
{
    segread(&SRegs);
    ES = DS;
    DI = (unsigned int)Info;
    intVESA(GetInfo);                           // Get VESA information
    if  (   (AL != 0x4f) ||
            (AH != 0) ||
            (strncmp(Info -> Signature, "VESA", 4) != 0)
        )
    {
        puts("Video interface does not support VESA calls.");
        exit(15);
    }
};

/****************************************
*   Test RAM                            *
****************************************/
unsigned int TestRAM(unsigned int Mode, VESAModeInfoPtr ModeInfo)
{
    unsigned int        AUsed           = 0;
    unsigned int        BUsed           = 0;
    unsigned int        WriteWindow;
    unsigned int        ReadWindow;
    unsigned int        Interval;
    unsigned int        Granularity;
    unsigned int        RAMSize;
    unsigned int        WindowSet;
    unsigned char far * WritePtr;
    unsigned char far * ReadPtr;
    unsigned int        Done            = 0;

    if ((ModeInfo -> WindowAAttributes & WindowSupported) != 0)
    {
        if ((ModeInfo -> WindowAAttributes & WindowIsReadable) != 0)
        {
            AUsed = 1;
            ReadWindow = 0;
            FP_SEG(ReadPtr) = ModeInfo -> WindowASegment;
        }
        if ((ModeInfo -> WindowAAttributes & WindowIsWritable) != 0)
        {
            AUsed = 1;
            WriteWindow = 0;
            FP_SEG(WritePtr) = ModeInfo -> WindowASegment;
        }
    }
    if ((ModeInfo -> WindowBAttributes & WindowSupported) != 0)
    {
        if ((ModeInfo -> WindowBAttributes & WindowIsReadable) != 0)
        {
            BUsed = 1;
            ReadWindow = 1;
            FP_SEG(ReadPtr) = ModeInfo -> WindowBSegment;
        }
        if ((ModeInfo -> WindowBAttributes & WindowIsWritable) != 0)
        {
            BUsed = 1;
            WriteWindow = 1;
            FP_SEG(WritePtr) = ModeInfo -> WindowBSegment;
        }
    }
    FP_OFF(ReadPtr) = 0;
    FP_OFF(WritePtr) = 0;
    BX = Mode;
    intVESA(SetMode);
    if (AH != 0)
    {
        return 0;
    }
    Granularity = ModeInfo -> WindowGranularity;
    Interval = ModeInfo -> WindowSize / Granularity;
    RAMSize = 0;
    do
    {
        if (AUsed)
        {
            WindowSet = SelectRAMWindow(0, RAMSize);
            if (!WindowSet) Done = 1;
        }
        if (BUsed)
        {
            WindowSet = SelectRAMWindow(1, RAMSize);
            if (!WindowSet) Done |= 1;
        }
        if (*ReadPtr == 0xaa) Done = 1;
        else
        {
            *WritePtr = 0xaa;
            if (*ReadPtr != 0xaa) Done = 1;
            else RAMSize += Interval;
        }
    } while (!Done);
    RAMSize *= Granularity;
    return RAMSize;
}

/****************************************
*   Test VESA Mode                      *
****************************************/
void TestMode(struct ModeTest * ModeTestInfo)
{
    unsigned int        Mode = ModeTestInfo -> Mode;
    struct VESAModeInfo ModeInfo;

    segread(&SRegs);
    CX = Mode;
    ES = DS;
    DI = (unsigned int)&ModeInfo;
    intVESA(GetModeInfo);
    if (AH != 0)
    {
        ModeTestInfo -> Width = 0;
        ModeTestInfo -> Height = 0;
        ModeTestInfo -> Depth = 0;
        ModeTestInfo -> KB = 0;
    }
    else if ((ModeInfo.ModeAttributes & OptionalInfo) == 0)
    {
        switch ((Mode & 7) >> 1)
        {
            case    0:
                ModeTestInfo -> Width = 640;
                ModeTestInfo -> Height = ((Mode & 1) == 0) ? 400 : 480;
                ModeTestInfo -> Depth = 256;
                break;
            case    1:
                ModeTestInfo -> Width = 800;
                ModeTestInfo -> Height = 600;
                ModeTestInfo -> Depth = ((Mode & 1) == 0) ? 16 : 256;
                break;
            case    2:
                ModeTestInfo -> Width = 1024;
                ModeTestInfo -> Height = 768;
                ModeTestInfo -> Depth = ((Mode & 1) == 0) ? 16 : 256;
                break;
            case    3:
                ModeTestInfo -> Width = 1280;
                ModeTestInfo -> Height = 1024;
                ModeTestInfo -> Depth = ((Mode & 1) == 0) ? 16 : 256;
                break;
        }
        ModeTestInfo -> KB = TestRAM(Mode, &ModeInfo);
    }
    else
    {
        ModeTestInfo -> Width = ModeInfo.WidthInPixels;
        ModeTestInfo -> Height = ModeInfo.HeightInPixels;
        ModeTestInfo -> Depth = 1 << ModeInfo.BitsPerPixel;
        ModeTestInfo -> KB = TestRAM(Mode, &ModeInfo);
    }
}

/****************************************
*   Main                                *
****************************************/

void main(void)
{
    char                Buf[16];
    int                 i;
    int                 ModeCnt         = 0;
    int                 ErrorFlag       = 0;
    unsigned int        CurrentMode;
    unsigned int far *  ModePtr;
    struct VESAInfo     Info;
    struct ModeTest     ModeTestInfo[16];

    GetVESAInformation(&Info);
    ModePtr = Info.ModeList;
    while (*ModePtr != -1)
    {
        ModeTestInfo[ModeCnt++].Mode = *ModePtr++;
    }
    if (ModeCnt == 0)
    {
        puts("No VESA modes supported.");
        exit(15);
    }
    intVESA(GetMode);
    if (AH != 0)
    {
        puts("Cannot get current video mode.");
        exit(15);
    }
    CurrentMode = BX;
    for (i = 0; i < ModeCnt; i++)
    {
        TestMode(&ModeTestInfo[i]);
    }
    BX = CurrentMode;
    intVESA(SetMode);
    puts("Video interface supports VESA calls.");
    printf("OEM string = %Fs\n\n", Info.OEMString);
    puts("Mode (hex)      Resolution (WxHxD)      Video RAM (KB)");
    for (i = 0; i < ModeCnt; i++)
    {
        printf("   %04x", ModeTestInfo[i].Mode);
        if (ModeTestInfo[i].Width == 0)
        {
            puts("*");
            ErrorFlag = 1;
        }
        else
        {
            sprintf
                (   Buf,
                    "%dx%dx%d",
                    ModeTestInfo[i].Width,
                    ModeTestInfo[i].Height,
                    ModeTestInfo[i].Depth
                );
            if (ModeTestInfo[i].KB != 0)
            {
                printf
                    (   "           %-13s              %d\n",
                        Buf,
                        ModeTestInfo[i].KB
                    );
            }
            else
            {
                printf
                    (   "           %-13s              *\n",
                        Buf
                    );
                ErrorFlag = 1;
            }
        }
    }
    if (ErrorFlag)
    {
        puts("\n* indicates unavailable information.");
    }
}
