*******************************************************************************
*                                                                             *
*                             Virtual List system                             *
*                                                                             *
*                            Demonstration program                            *
*                                                                             *
*                            1991 Jayson R. Minard                            *
*                                                                             *
**
*                                                                             *
* This program demonstrates the capabilities and flexibility of the VList     *
* library.  The following are shown:                                          *
*                                                                             *
*          Pulldown menus                                                     *
*          Prompts                                                            *
*          Popup menus                                                        *
*          Picklists                                                          *
*                                                                             *
**
*                                                                             *
* All major features are preceded by comments containing the text '!LOOK'.    *
* This will allow you to search and find the features of the VList library.   *
*                                                                             *
*******************************************************************************

#INCLUDE WARN.HDR
#INCLUDE DATE.HDR
#INCLUDE STRING.HDR
#INCLUDE SYSTEM.HDR
#INCLUDE DATABASE.HDR
#INCLUDE IO.HDR
#INCLUDE COLORS.HDR
#INCLUDE KEYS.HDR

#INCLUDE vlist.hdr
#INCLUDE vmouse.hdr

VARDEF EXTERN
  BYTE  __color_std,;
        __color_enhcd
ENDDEF

*******************************************************************************
*                              global variables                               *
*******************************************************************************

VARDEF
  LOGICAL is_mono
  LOGICAL set_delete_on
  LOGICAL is_mouse
  LOGICAL main_loop

  LOGICAL at_end
  LOGICAL at_front

  *- menu list
  LONG    menu_list
  LONG    prompt_list

  *- colors
  BYTE  d_main_std,;
        d_main_enhcd,;
        d_main_border,;
        d_main_back,;
        d_main_data,;
        d_menu_std,;
        d_menu_enhcd,;
        d_menu_hi_std,;
        d_menu_hi_enhcd,;
        d_menu_border,;
        d_menu_skip

ENDDEF

*******************************************************************************
*                            database definitions                             *
*******************************************************************************

DBFDEF people
  CHAR( 20 )  first_name
  CHAR( 20 )  last_name
  CHAR( 40 )  salutation
  CHAR( 40 )  address1
  CHAR( 40 )  address2
  CHAR( 30 )  city
  CHAR( 2 )   state
  CHAR( 10 )  zip
  CHAR( 12 )  phone1
  CHAR( 12 )  phone2

  DATE        f_contact      && first contact date
  DATE        l_contact      && last contact date
  DATE        n_contact      && next contact date
ENDDEF

INDEXDEF
  CHAR( 40 )  i_people     UPPER( people->last_name +;
                                  people->first_name )
ENDDEF


*******************************************************************************
*                                Draw_Border()                                *
*******************************************************************************

PROCEDURE Draw_Border
  PARAMETERS VALUE LOGICAL active,;
             VALUE LOGICAL clear_middle

  VARDEF
    CHAR( 8 ) border_chars
    CHAR( 1 ) fill_middle
  ENDDEF

  IF active
    border_chars = &double_box
  ELSE
    border_chars = &single_box
  ENDIF

  IF clear_middle
    fill_middle = " "
  ELSE
    fill_middle = ""
  ENDIF

  FILL( 3, 5, 17, 73, border_chars, fill_middle, d_main_border, d_main_std, 6 )

ENDPRO

*******************************************************************************
*                                DRAW_TITLES()                                *
*******************************************************************************

PROCEDURE Draw_Titles
  VARDEF
    BYTE  old_std, old_enhcd
  ENDDEF

  old_std   = __color_std
  old_enhcd = __color_enhcd

  __color_std = d_main_std
  @ 4,  7 SAY "Last name: "
  @ 4, 40 SAY "First name: "
  @ 6,  7 SAY "Salutation: "
  @ 8,  7 SAY "Address: "
  @11,  7 SAY "City: "
  @11, 45 SAY "State: "
  @11, 56 SAY "Zip: "
  @13,  7 SAY "Phone 1: "
  @13, 30 SAY "Phone 2: "
  @15,  7 SAY "First contact: "
  @15, 32 SAY "Last: "
  @15, 48 SAY "Next: "

  __color_enhcd = old_enhcd
  __color_std   = old_std

ENDPRO


*******************************************************************************
*                                 DRAW_REC()                                  *
*******************************************************************************

PROCEDURE Draw_Rec
  VARDEF
    BYTE  old_std, old_enhcd
  ENDDEF

  old_std   = __color_std
  old_enhcd = __color_enhcd

  __color_std   = d_main_data
  @ 4, 18 SAY people->last_name
  @ 4, 52 SAY people->first_name
  @ 6, 19 SAY people->salutation
  @ 8, 16 SAY people->address1
  @ 9, 16 SAY people->address2
  @11, 13 SAY people->city
  @11, 52 SAY people->state PICTURE "@!"
  @11, 61 SAY people->zip PICTURE "XXXXX-XXXX"
  @13, 16 SAY people->phone1 PICTURE "999-999-9999"
  @13, 39 SAY people->phone2 PICTURE "999-999-9999"
  @15, 22 SAY people->f_contact
  @15, 38 SAY people->l_contact
  @15, 54 SAY people->n_contact

  __color_enhcd = old_enhcd
  __color_std   = old_std

ENDPRO


*******************************************************************************
*                                  GET_REC()                                  *
*******************************************************************************

PROCEDURE Get_Rec
  VARDEF
    BYTE  old_std, old_enhcd
  ENDDEF

  old_std   = __color_std
  old_enhcd = __color_enhcd

  __color_std   = d_main_data
  __color_enhcd = d_main_enhcd
  @ 4, 18 GET people->last_name
  @ 4, 52 GET people->first_name
  @ 6, 19 GET people->salutation
  @ 8, 16 GET people->address1
  @ 9, 16 GET people->address2
  @11, 13 GET people->city
  @11, 52 GET people->state PICTURE "@!"
  @11, 61 GET people->zip PICTURE "XXXXX-XXXX"
  @13, 16 GET people->phone1 PICTURE "999-999-9999"
  @13, 39 GET people->phone2 PICTURE "999-999-9999"
  @15, 22 GET people->f_contact
  @15, 38 GET people->l_contact
  @15, 54 GET people->n_contact

  CURSOR_ON()
  READ
  CURSOR_OFF()

  __color_enhcd = old_enhcd
  __color_std   = old_std

ENDPRO

*******************************************************************************
*                                    Add()                                    *
*******************************************************************************

PROCEDURE Add
  PARAMETERS       LOGICAL    ret_val,;
             CONST CHAR( 40 ) add_str

  * !LOOK:  The following code segment shows the addition of elements into
  *         a list.  The code determines if the element is to be skipped
  *         or not (horizontal lines in a menu are skipped).

  IF ret_val = .F.
    RETURN
  ENDIF

  IF AT( "?", add_str ) > 0 .AND. AT( "...", add_str ) > 0
    IF .NOT. Vlist_Append( menu_list, &jl_skip, &jl_default, add_str )
      ret_val = .F.
    ENDIF

  ELSE
    IF .NOT. Vlist_Add( menu_list, add_str )
      ret_val = .F.
    ENDIF

  ENDIF

ENDPRO


*******************************************************************************
*                                 SET_MENUS()                                 *
*******************************************************************************

PROCEDURE Setup_Menus
  VARDEF
    LOGICAL ret_val
  ENDDEF

  * !LOOK:  The following code segment shows the creation of a pulldown
  *         and popup menu list.

  menu_list = VLIST_INIT( 50 )
  IF menu_list = 0
    __color_std = &black_white
    CLEAR
    ? "!ERROR:  cannot initialize pulldown menu system due to insufficient memory"
    ?
    QUIT 1
  ENDIF

  ADD( ret_val, "1.1 @1  |  ^People DBF  " )
  ADD( ret_val, "1.1.0   |" )
  ADD( ret_val, "1.1.1   | ^Append " )
  ADD( ret_val, "1.1.2   | ^Modify " )
  ADD( ret_val, "1.1.?   |..." )
  ADD( ret_val, "1.1.3   | ^Browse " )
  ADD( ret_val, "1.2 @15 |  ^Utilities  " )
  ADD( ret_val, "1.2.0   |" )
  ADD( ret_val, "1.2.1   | ^Reindex " )
  ADD( ret_val, "1.2.2   | ^Pack " )
  ADD( ret_val, "1.2.?   |..." )
  ADD( ret_val, "1.2.3   | ^Quit " )

  ADD( ret_val, "2.0     | Quit? " )
  ADD( ret_val, "2.1     |   ^Yes  " )
  ADD( ret_val, "2.2     |   ^No   " )

  IF ret_val = .F.
    __color_std = &black_white
    CLEAR
    ? "!ERROR:  setting pulldown options due to insufficient memory"
    ?
    QUIT 1
  ENDIF

ENDPRO

*******************************************************************************
*                               SETUP_PROMPTS()                               *
*******************************************************************************

PROCEDURE SETUP_PROMPTS

  prompt_list = VLIST_INIT( 40 )

  IF prompt_list = 0
    __color_std = &black_white
    CLEAR
    ? "!ERROR:  cannot initialize prompt system due to insufficient memory"
    ?
    QUIT 1
  ENDIF

  * !LOOK:  This code sets up a list to simulate the @PROMPT command

  VList_Add( prompt_list, "1.1 @20,3  |  ^Next  " )
  VList_Add( prompt_list, "1.2 @20,11 |  ^Previous  " )
  VList_Add( prompt_list, "1.3 @20,23 |  ^Top  " )
  VList_Add( prompt_list, "1.4 @20,30 |  ^Bottom  " )
  VList_Add( prompt_list, "1.5 @20,40 |  [ ] ^Deleted  " )
  VList_Add( prompt_list, "1.6 @20,55 |  [ ] ^SET DELETE ON  " )

ENDPRO


*******************************************************************************
*                               ALTER_PROMPTS()                               *
*******************************************************************************

PROCEDURE Alter_Prompts

  * !LOOK:  we can now alter elements in an existing list.

  IF .NOT. A_DELETED( people )
    Vlist_Edit( prompt_list, 5, "1.5 @20, 40 |  [ ] ^Deleted  " )
  ELSE
    Vlist_Edit( prompt_list, 5, "1.5 @20, 40 |  [X] ^Deleted  " )
  ENDIF

  IF at_end
    Vlist_Set_Status( prompt_list, 1, &jl_skip )
    Vlist_Set_Status( prompt_list, 4, &jl_skip )
  ELSE
    Vlist_Set_Status( prompt_list, 1, &jl_normal )
    Vlist_Set_Status( prompt_list, 4, &jl_normal )
  ENDIF

  IF at_front
    Vlist_Set_Status( prompt_list, 2, &jl_skip )
    Vlist_Set_Status( prompt_list, 3, &jl_skip )
  ELSE
    Vlist_Set_Status( prompt_list, 2, &jl_normal )
    Vlist_Set_Status( prompt_list, 3, &jl_normal )
  ENDIF

ENDPRO

*******************************************************************************
*                                 DBF_LINE()                                  *
*******************************************************************************

FUNCTION CHAR Dbf_Line
  VARDEF
    CHAR one_line
    CHAR temp_str
  ENDDEF

  one_line = STR( A_RECNO( people ), 5, 0 )
  temp_str = TRIM( people->last_name ) + ", " + TRIM( people->first_name )
  temp_str = LEFT( temp_str + SPACE( 40 - LEN( temp_str ) ), 40 )
  one_line = one_line + temp_str + ""
  one_line = one_line + people->phone1 + ""
  one_line = one_line + people->phone2

  RETURN one_line
ENDPRO

*******************************************************************************
*                                Handle_Select                                *
*******************************************************************************
* This routine is called when an element is selected from a pulldown menu     *
* system.                                                                     *
*******************************************************************************

FUNCTION UINT Handle_Select
  PARAMETERS VALUE LONG handle,;
       VALUE UINT system,;
       VALUE UINT over,;
       VALUE UINT down

  VARDEF
    BYTE old_std, old_enhcd
    UINT decision, save_var
    ULONG old_rec
    LONG  browse_list
    UINT  browse_left, browse_right, browse_top, browse_current
    UINT  temp_cnt, temp_first
  ENDDEF

  DO CASE
    CASE over = 1
      SAVE_AREA( 2, 1, 8, 12 )
      Draw_Border( .T., .T. )

      DO CASE
        CASE over = 1 .AND. down = 1   && add
          old_rec = A_RECNO( people )

          Draw_Titles()
          !people APPEND BLANK
          Get_Rec()
          IF LASTKEY() = 27
            !people DELETE
            !people GOTO old_rec
          ENDIF

          Draw_Rec()
        CASE over = 1 .AND. down = 2   && modify
          Draw_Titles()
          Get_Rec()
          Draw_Rec()
        CASE over = 1 .AND. down = 3   && browse
          old_rec = A_RECNO( people )

          * !LOOK:  the VLIST system can be used for database browse simulation:

          browse_list = VLIST_INIT( 250 )
          IF browse_list <> 0

            !people GOTO TOP

            temp_first = 1
            DO WHILE .NOT. A_EOF( people )
              IF .NOT. Vlist_Add( browse_list, Dbf_Line() )
                EXIT
              ENDIF

              IF A_RECNO( people ) = old_rec
                temp_first = Vlist_Number( browse_list )
              ENDIF

              !people SKIP
            ENDDO

            browse_current = temp_first
            IF browse_current > 12
              browse_top = browse_current - 6

              DO WHILE ( Vlist_Max( browse_list ) - browse_top + 1 ) < 12
                browse_top = browse_top - 1
              ENDDO

            ELSE
              browse_top = 1
            ENDIF

            browse_left    = 6
            browse_right   = 100

            old_std   = __color_std
            old_enhcd = __color_enhcd
            __color_std   = d_main_std
            __color_enhcd = d_main_enhcd

            * !LOOK:  the following is the actual display of the database
            *         browse simulation.  This is the normal PICK-LIST
            *         replacement.

            Vlist_Pick( browse_list, 4, 6, 15, 71,;
                        browse_top, browse_current,;
                        0, 0, browse_left, browse_right, 1, 0,;
                        Vlist_Default_Key_Handler,;
                        .F., .T., .F., is_mouse, .F. )

            __color_std   = old_std
            __color_enhcd = old_enhcd

            IF LASTKEY() = 27
              !people GOTO old_rec
            ELSE
              old_rec = VAL( LEFT( Vlist_Cstr( browse_list ), 5 ) )
              !people GOTO old_rec
            ENDIF

            Vlist_Clear( browse_list )
          ENDIF

          Draw_Border( .T., .T. )
          Draw_Titles()
          Draw_Rec()
      ENDCASE

      Draw_Border( .F., .F. )
      RESTORE_AREA()

    CASE over = 2 .AND. down = 1   && reindex
      SAVE_SCREEN()
      FILL( 10, 32, 13, 49, &double_box, " ", d_menu_border, d_menu_std, 6 )
      @11, 34 SAY "ReINDEXing..."
      !people REINDEX
      RESTORE_AREA()
    CASE over = 2 .AND. down = 2   && pack
      SAVE_SCREEN()
      FILL( 10, 33, 13, 47, &double_box, " ", d_menu_border, d_menu_std, 6 )
      @11, 35 SAY "PACKing..."
      !people PACK
      !people REINDEX
      RESTORE_AREA()
    CASE over = 2 .AND. down = 3
      decision = 2

      * !LOOK:  call to show the pulldown menu:

      VList_PopUp( menu_list, 2, decision, Vlist_Default_Menu_Handler,;
                   10, 34, 24,;
                   &double_box,;
                   d_menu_border, d_menu_std,;
                   .T., .T., save_var, .T., .F., is_mouse, .T. )

      IF decision = 1 .AND. LASTKEY() <> 27
        main_loop = .F.
        RETURN 1
      ENDIF

    OTHERWISE
  ENDCASE

  RETURN 0
ENDPRO



*******************************************************************************
*                                FORCE_MAIN()                                 *
*******************************************************************************

PROCEDURE Force_Main
  PARAMETERS CONST CHAR( 128 ) dos_cmd_line

  VARDEF
    CHAR( 128 )  our_cmd_line
    BYTE         old_std, old_y, old_x
    LOGICAL      re_index
    UINT         selected
    INT          lkey
    CHAR( 8 )    old_time
    UINT         pull_over, pull_down
  ENDDEF

  *- save machine state

  old_std = __color_std
  old_y   = ROW()
  old_x   = COL()
  SAVE_SCREEN()

  SET STATUS OFF
  SET SCOREBOARD OFF
  SET EXCLUSIVE ON

  our_cmd_line = TRIM( UPPER( dos_cmd_line ) )

  *- check for /M parameter to signify forced mono mode.

  IF AT( "/M", our_cmd_line ) > 0
    is_mono = .T.
  ELSE
    is_mono = .NOT. ISCOLOR()
  ENDIF

  *- check for database existance

  SET ALIAS i_people TO "people.fdx"

  re_index = .F.
  IF .NOT. EXIST( "people.dbf" )
    BUILD "people.dbf" FROM ALIAS people
    re_index = .T.
  ELSE
    IF .NOT. EXIST( "people.fdx" )
      re_index = .T.
    ENDIF

  ENDIF

  OPEN "people.dbf" ALIAS people EXCLUSIVE

  IF re_index
    !people INDEX i_people
  ENDIF

  !people SET INDEX TO i_people

  *- set color variables

  IF is_mono
    d_main_std       = &black_white
    d_main_enhcd     = &white_black
    d_main_border    = &black_light_grey
    d_main_data      = &black_light_grey
    d_main_back      = &black_black
    d_menu_std       = &black_light_grey
    d_menu_enhcd     = &white_black
    d_menu_hi_std    = &black_white
    d_menu_hi_enhcd  = &white_black
    d_menu_border    = &black_light_grey
    d_menu_skip      = &black_light_grey
  ELSE
    d_main_std       = &black_white
    d_main_enhcd     = &blue_white
    d_main_border    = &black_light_grey
    d_main_data      = &black_light_grey
    d_main_back      = &black_light_cyan
    d_menu_std       = &blue_white
    d_menu_enhcd     = &black_white
    d_menu_hi_std    = &blue_light_cyan
    d_menu_hi_enhcd  = &black_light_cyan
    d_menu_border    = &blue_light_grey
    d_menu_skip      = &blue_light_grey
  ENDIF

  SET DELETE OFF
  set_delete_on = .F.

  *- draw initial screen

  __color_std      = d_main_std
  __color_enhcd    = d_main_enhcd
  __color_skip     = d_menu_skip
  __color_bar      = d_main_border
  __color_tab      = d_main_border
  __color_hi_std   = d_menu_hi_std
  __color_hi_enhcd = d_menu_hi_enhcd
  __scroll_offset  = 1
  __embed_char     = "^"

  CLEAR

  FILL( 2, 0, 22, 79, "", "", d_main_back, d_main_back, 0 )
  __color_std = d_main_border
  @ 1, 0 TO  1, 79
  @23, 0 TO 23, 79
  __color_std = d_main_std
  @24, 0 SAY CENTER( "VLIST Demonstration (c) 1991 Jayson R. Minard", 80 )

  Draw_Border( .F., .T. )

  Draw_Titles()
  main_loop = .T.

  Setup_Menus()
  Setup_Prompts()

  IF Vmouse_Init() = 0
    is_mouse = .F.
  ELSE
    is_mouse = .T.
    Vmouse_Reset()
  ENDIF

  selected = 1
  at_end   = A_EOF( people )
  at_front = .T.
  pull_over = 1
  pull_down = 1

  FILL( 19, 1, 22, 78, &double_box, " ", d_menu_border, d_menu_std, 6 )
  CURSOR_OFF()
  Alter_Prompts()

  DO WHILE main_loop
    Draw_Rec()

    __color_std   = d_menu_std
    __color_enhcd = d_menu_enhcd
    selected = Vlist_Prompt( prompt_list, 1, selected,;
                             Vlist_Default_Prompt_Handler,;
                             is_mouse, .T. )

    lkey = LASTKEY()
    DO CASE
      CASE lkey = 27
        FILL( 19, 1, 22, 78, &single_box, "", d_menu_border, d_menu_std, 6 )
        __color_std = d_main_border
        @ 1, 0 TO  1, 79 DOUBLE
        __color_std = d_menu_std
        @ 0, 0 CLEAR TO 0, 79

        * !LOOK:  show the pulldown menu

        VList_Pulldown( menu_list, 1, pull_over, pull_down,;
                        Handle_Select, Vlist_Default_Menu_Handler,;
                        0, 2, 22,;
                        0, &single_box, d_menu_border, d_menu_std,;
                        .T., .F., is_mouse, .F., .T. )

        FILL( 19, 1, 22, 78, &double_box, "", d_menu_border, d_menu_std, 6 )
        __color_std = d_main_border
        @ 1, 0 TO  1, 79
        @ 0, 0 CLEAR TO 0, 79
        __color_std = d_menu_std

      CASE selected = 1 && next
        !people SKIP 2
        IF A_EOF( people )
          at_end = .T.
          selected = 2
        ELSE
          at_end = .F.
        ENDIF

        !people SKIP - 1

        at_front = A_BOF( people )

        Alter_Prompts()

      CASE selected = 2        && previous
        !people SKIP -2
        IF A_BOF( people )
          at_front = .T.
          selected = 1
        ELSE
          at_front = .F.
          !people SKIP
        ENDIF

        at_end = A_EOF( people )

        Alter_Prompts()

      CASE selected = 3 && top
        !people GOTO TOP
        at_front = .T.
        at_end = A_EOF( people )
        selected = 1

        Alter_Prompts()

      CASE selected = 4 && bottom
        !people GOTO BOTTOM
        at_end = .T.
        at_front = A_BOF( people )
        selected = 2

        Alter_Prompts()

      CASE selected = 5 && deleted()
        IF A_DELETED( people )
          !people RECALL
        ELSE
          !people DELETE
        ENDIF

        Alter_Prompts()

      CASE selected = 6 && set delete
      * !LOOK:  alter an existing prompt to act like a push-button

        IF set_delete_on
          SET DELETE OFF
          Vlist_Edit( prompt_list, 6, "1.6 @20, 55 |  [ ] ^SET DELETE ON  " )
        ELSE
          SET DELETE ON
          Vlist_Edit( prompt_list, 6, "1.6 @20, 55 |  [X] ^SET DELETE ON  " )
        ENDIF

        set_delete_on = .NOT. set_delete_on
    ENDCASE

  ENDDO

  *- reset machine state
  CURSOR_ON()

  IF is_mouse
    Vmouse_Reset()
    Vmouse_Cursor( .F. )
  ENDIF

  CLOSE ALL
  __color_std = old_std
  RESTORE_AREA()
  @ old_y, old_x

ENDPRO