/*
 * Boot installator.
 * Author: Serge Vakulenko, <vak@kiae.su>
 */

#include <stdio.h>
#include <fcntl.h>
#include <dos.h>

#define BOOTFILE	"boot.bin"
#define SAVEFILE	"bootsave.bin"

#define PartAddr        0x1be	/* Offset to partition table */
#define ValidationAddr  0x1fe	/* Offset to validation bytes */
#define MAGIC		0xaa55	/* Validation tag */

typedef struct PartitionEntry {
	unsigned char	bootIndicator;
	unsigned char	beginHead;
	unsigned char	beginSector;
	unsigned char	beginCyl;
	unsigned char	systemId;
	unsigned char	endHead;
	unsigned char	endSector;
	unsigned char	endCyl;
	unsigned short	relSectorLow;
	unsigned short	relSectorHigh;
	unsigned short	numSectorsLow;
	unsigned short	numSectorsHigh;
} PartitionEntry;

char bootRecord [512];
char bootProg [512];

main (argc, argv)
char **argv;
{
	int rc, i, fd;
	char reply [80], *bootfile;

	printf ("Boot installer, version 1.0, Copyright (C) Serge Vakulenko\n\n");

	if (argc > 2) {
		printf ("Usage: bootinst [bootfile]\n");
		return (-1);
	}
	bootfile = argc>1 ? argv[1] : BOOTFILE;

	rc = bootio (0, 0, 0, 1, bootRecord);
	if (rc) {
		fprintf (stderr, "Error %d reading boot record\n", rc);
		return (-1);
	}
	if (*(short *) &bootRecord [ValidationAddr] != (short) MAGIC) {
		fprintf (stderr, "Bad master boot record!\n");
		return (-1);
	}

	printtable ();

	fd = open (bootfile, O_RDONLY|O_BINARY);
	if (fd < 0) {
		fprintf (stderr, "Cannot read file %s\n", bootfile);
		return (-1);
	}
	if (read (fd, bootProg, 512) != 512) {
		fprintf (stderr, "Error reading %s\n", bootfile);
		return (-1);
	}
	close (fd);
	if (*(short *) &bootProg [ValidationAddr] != (short) MAGIC) {
		fprintf (stderr, "Bad boot image\n");
		return (-1);
	}

	printf ("\nAre you sure you want to install new boot? (yes/no) ");
	fgets (reply, sizeof (reply), stdin);
	if (stricmp (reply, "y\n") && stricmp (reply, "yes\n"))
		return (0);

	close (creat (SAVEFILE, 0664));
	fd = open (SAVEFILE, O_WRONLY|O_BINARY);
	if (fd < 0) {
		fprintf (stderr, "Cannot write to file %s\n", SAVEFILE);
		return (-1);
	}
	if (write (fd, bootRecord, 512) != 512) {
		fprintf (stderr, "Error writing to %s\n", SAVEFILE);
		return (-1);
	}
	close (fd);

	memcpy (bootRecord, bootProg, PartAddr);

	rc = bootio (1, 0, 0, 1, bootRecord);
	if (rc) {
		fprintf (stderr, "Error %d updating boot record\n", rc);
		return (-1);
	}
	printf ("New boot record successfully installed\n");
	return (0);
}

bootio (wr, head, cyl, sect, buf)
char *buf;
{
	return (biosdisk (wr ? 3 : 2, 0x80, head, cyl, sect, 1, buf));
}

static char head1 [] =	"Ŀ\n";
static char head2 [] =	" PartitionCan Boot   Beginning      Ending     RelativeNumber of\n";
static char head3 [] =	"N  Type   BootPartHead Cyl  SectHead Cyl  Sect Sectors  Sectors \n";
static char head4 [] =	"Ĵ\n";
static char mid1 [] =	"%d%-9.9s%-4.4s%-4.4s%3u %3u %3u %3u %3u %3u %8lu %8lu \n";
static char mid2 [] =	"Ĵ\n";
static char foot [] =	"\n";

printtable ()
{
	PartitionEntry *part = (PartitionEntry *) (bootRecord + PartAddr);
	int i, cb;
	long relSectors;
	long numSectors;
	char *typeString, block [512];

	printf (head1);
	printf (head2);
	printf (head3);
	printf (head4);

	for (i=0; i<4; ++i) {
		switch (part->systemId) {
		default:	typeString = "  ?????";   cb = 1; break;
		case 0:		typeString = "  empty";   cb = 0; break;
		case 1:		typeString = "  dos-12";  cb = 1; break;
		case 2:		typeString = "  xenix";   cb = 1; break;
		case 3:		typeString = "xenix usr"; cb = 0; break;
		case 4:		typeString = "  dos-16";  cb = 1; break;
		case 5:		typeString = "  extend";  cb = 0; break;
		case 6:		typeString = "  bigdos";  cb = 1; break;
		case 7:		typeString = "  hpfs";    cb = 1; break;
		case 0x75:	typeString = "  pcix";    cb = 1; break;
		case 0xdb:	typeString = "  cp/m";    cb = 1; break;
		case 0xff:	typeString = "  bbt";     cb = 0; break;
		case 0x08:	typeString = "  aix";     cb = 0; break;
		case 0x09:	typeString = " coherent"; cb = 1; break;
		case 0x0A:	typeString = "  os/2";    cb = 1; break;
		case 0x10:	typeString = "  opus";    cb = 1; break;
		case 0x40:	typeString = "venix 286"; cb = 1; break;
		case 0x50:
		case 0x51:	typeString = "  dm";      cb = 1; break;
		case 0x52:	typeString = "microport"; cb = 1; break;
		case 0x56:	typeString = "  gb";      cb = 1; break;
		case 0x61:
		case 0xE1:
		case 0xE3:
		case 0xE4:
		case 0xF1:
		case 0xF4:	typeString = "  speed";   cb = 1; break;
		case 0x63:	typeString = "  unix";    cb = 1; break;
		case 0x64:	typeString = "novell286"; cb = 1; break;
		case 0x65:	typeString = "novell386"; cb = 1; break;
		case 0x80:	typeString = "old minix"; cb = 1; break;
		case 0x81:	typeString = "  minix";   cb = 1; break;
		case 0x82:	typeString = "  linux";   cb = 1; break;
		case 0x93:	typeString = "  amoeba";  cb = 1; break;
		case 0x94:	typeString = "amoebaBBT"; cb = 0; break;
		case 0xA5:	typeString = "  386bsd";  cb = 1; break;
		case 0xB7:	typeString = "  bsdi";    cb = 1; break;
		case 0xB8:	typeString = "bsdi swap"; cb = 0; break;
		case 0xF2:	typeString = " dos sec";  cb = 1; break;
		}
		relSectors = *(long*)(&part->relSectorLow);
		numSectors = *(long*)(&part->numSectorsLow);

		if (cb && (bootio (0, part->beginHead, part->beginCyl,
		    part->beginSector, block) != 0 ||
		    *(short *) &block [ValidationAddr] != (short) MAGIC))
			cb = 0;

		printf (mid1, i+1, typeString, cb ? " yes" : " no",
			(part->bootIndicator & 0x80) ? " yes" : " no",
			part->beginHead, part->beginCyl, part->beginSector,
			part->endHead, part->endCyl, part->endSector,
			relSectors, numSectors);
		if (i < 3)
			printf (mid2);
		++part;
	}
	printf (foot);
}
