/*--SOURCE FILE (PROGRAM) DSP.C-----------------------REV 91.06.04-*/

/*-----------------------------------------------------------------*
 *  
 *  Copyright 1991-1992, Thomas G. Harold, All Rights Reserved.
 *
 *  Usage of this source code is free to individuals using it for
 *  personal/private use.  Educational institutions are also granted
 *  free use for educational purposes.  Commercial use or inclusion
 *  of this code requires a site licence.
 *
 *
 *                              DISCLAIMER
 *      This program is distributed "as is".  I disclaim all 
 *      warranties, expressed or implied, including, without 
 *      limitation, the warranties of merchantability and of 
 *      fitness for any purpose.  I will assume no liability for 
 *      damages, direct or consequential, which may result from 
 *      the use of this program. 
 *
 *  Modifications to this source code are allowed so long as all
 *  copyright notices and disclaimers remain intact.  Modified copies
 *  of this source code may not be distributed in any form.
 *
 *  All correspondence concerning this source code should be 
 *  directed to the following address:
 *
 *      Thomas G. Harold
 *      2206 Tracey's Rd.
 *      Sparks, MD. 21152
 *
 *  Or by using Compuserve MAIL:
 *
 *      Thomas G. Harold - 71750,3724
 *
 *-----------------------------------------------------------------*/

/*  File:       dsp.c
    Author:     Thomas G. Harold
    LastEdit:   January 2, 1992
    Created:    September 19, 1991
    Version:    1.1
    Purpose:    Disk speed program.

    History:    1.0     Sep/Oct 1991
                    Original version.

                1.1     Oct/Dec 1991
                    Added % complete for each pass & creation of test file

    Notes:

        dsp.c was written entirely in ANSI C and should be completely portable
        to other environments.  

        Read Sequential:
            - Sequential read, start to finish, of entire test file.

        Read Random:
            - Random reads from test file.

        Write Sequential:
            - Sequential write, start to finish, of entire test file.

        Write Random:
            - Random writes to test file.

        Read/Write Sequential:
            - Sequential write, then read, start to finish, of entire test file.

        Read/Write Random:
            - Random reads from or writes to test file.

    Usage:


    DSP type mode min_pass min_time max_time blocks block_size [test_id rpt_file]

            #   argument    type        possible values
            1   type        char        R | W | RW
            2   mode        char        S | R
            3   min_pass    int         1..32768
            4   min_time    long        30..604800
            5   max_time    long        30..604800
            6   blocks      int         1..32768
            7   block_size  int         1..32768
            8   test_id     char        8 characters
            9   rpt_file    char        report file to append to

*/

/*==Includes=======================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/*==Defines========================================================*/

#define dspSEQUENTIAL           1
#define dspRANDOM               2

#define dspREAD                 1
#define dspWRITE                2
#define dspREAD_WRITE           3

#define dspMIN_MINPASSES        1
#define dspMAX_MINPASSES        32768
#define dspMIN_TIME             30
#define dspMAX_TIME             604800
#define dspMIN_BLOCKS           1
#define dspMAX_BLOCKS           32768
#define dspMIN_SIZE             1
#define dspMAX_SIZE             32768

/*==Typedefs=======================================================*/
/*==Struct & Union Defs============================================*/
/*==Static (Global) Variables======================================*/

/* The following must remain intact & unchanged */
static char COPYRIGHT[] =   "Copyright 1991, Thomas G. Harold, All Rights Reserved.";
static char OBJECTNAME[] =  "DSP - 1.1";
static char MODULEDATE[] =  "10/27/1991";   /* Last edit date */
/* This ends the legally required static variables */

static char PROGRAMNAME[] =     "DSP";
static char VERSIONNUMBER[] =   "1.1";
static char PROGRAMCOMMENT[] =  "Disk Speed Program";

static int rpt_flag = 0;      /* set report flag to FALSE */
static char TESTFILE[] = "dsp.$$$";
static char RESULTSFILE[128] = "";
static char TESTBATCHID[128] = "";
static FILE *frpt = NULL;
static FILE *fp = NULL;

static long min_time = 60;
static long max_time = 150;

static size_t block_size = 1024;
static size_t block_count = 256;
static int rpt_freq = 16;           /* report % complete each 16 blocks */
static int passes = 16;

static int test_mode = dspRANDOM;
static int test_type = dspREAD_WRITE;
static char test_mstr[2] = "R";
static char test_tstr[3] = "RW";

static unsigned char *test_data;    /* pointer to malloc()'ed buffer */


/*==Extern Variable Declarations===================================*/
/*==Private Function Prototypes====================================*/
/*-----------------------------------------------------------------*
 * fname                        type
 * Input:                       ...
 * Output:                      ...
 * Externs Used:
 * Externs Changed:
 * Statics Used:
 * Statics Changed:
 * Purpose:
 * Return Values:
 *-----------------------------------------------------------------*/
int header (void);
int qheader (void);
int create_data (void);
int create_file (void);
int modify_data (unsigned char *ptr);
int report (int iterations, time_t elapsed, long ttl_bytes, long io_count);
int delete_data (void);
int w_s_test (void);
int w_r_test (void);
int r_s_test (void);
int r_r_test (void);
int rw_s_test (void);
int rw_r_test (void);


/*==Main===========================================================*/

int main
(
    int argc,
    char *argv[]
)
    {
        time_t timer;
        long foo;
        char *sptr;

        fprintf (stderr,"%s %s - %s\n",PROGRAMNAME,VERSIONNUMBER,PROGRAMCOMMENT);
        fprintf (stderr,"    %s\n",COPYRIGHT);

        /* Check command line parameters: */

        if ((argc < 8) || ((argc > 8) && (argc < 10)))
        {
            fprintf (stderr,"Usage:\n");
            fprintf (stderr,"%s type mode passes min_time max_time blocks block_size [test_ID test_file]\n",PROGRAMNAME);
            fprintf (stderr,"      type = R | W | RW (Read, Write, Read/Write)\n");
            fprintf (stderr,"      mode = S | R (Sequential/Random)\n");
            fprintf (stderr,"    passes = 1..32768\n");
            fprintf (stderr,"  min_time = 30..604800 seconds\n");
            fprintf (stderr,"  max_time = 30..604800 seconds\n");
            fprintf (stderr,"    blocks = 1..32767\n");
            fprintf (stderr,"block_size = 1..32767 bytes\n");
            fprintf (stderr,"   test_ID = 1..8 character test batch ID\n");
            fprintf (stderr," test_file = test results file name\n");
            fprintf (stderr,"\n");
        }

        if (strcmp (argv[1],"R") == 0) test_type = dspREAD;
        else if (strcmp (argv[1],"W") == 0) test_type = dspWRITE;
        else if (strcmp (argv[1],"RW") == 0) test_type = dspREAD_WRITE;
        else
        {
            fprintf (stderr,"Test type must be either R, W, or RW\n");
            exit (1);
        }
        strcpy (test_tstr,argv[1]);

        if (strcmp (argv[2],"S") == 0) test_mode = dspSEQUENTIAL;
        else if (strcmp (argv[2],"R") == 0) test_mode = dspRANDOM;
        else
        {
            fprintf (stderr,"Test mode must either be S or R\n");
            exit (1);
        }
        strcpy (test_mstr,argv[2]);

        sptr = argv[3];
        foo = strtol (sptr,&sptr,0);
        if ((foo < dspMIN_MINPASSES) || (foo >= dspMAX_MINPASSES))
        {
            fprintf (stderr,"Passes must be between %li and %li.\n",dspMIN_MINPASSES,dspMAX_MINPASSES);
            exit (1);
        }
        passes = (int) foo;

        sptr = argv[4];
        foo = strtol (sptr,&sptr,0);
        if ((foo < dspMIN_TIME) || (foo >= dspMAX_TIME))
        {
            fprintf (stderr,"Minimum time must be between %li and %li.\n",dspMIN_TIME,dspMAX_TIME);
            exit (1);
        }
        min_time = foo;

        sptr = argv[5];
        foo = strtol (sptr,&sptr,0);
        if ((foo < dspMIN_TIME) || (foo >= dspMAX_TIME))
        {
            fprintf (stderr,"Maximum time must be between %li and %li.\n",dspMIN_TIME,dspMAX_TIME);
            exit (1);
        }
        max_time = foo;

        sptr = argv[6];
        foo = strtol (sptr,&sptr,0);
        if ((foo < dspMIN_BLOCKS) || (foo >= dspMAX_BLOCKS))
        {
            fprintf (stderr,"Number of blocks must be between %li and %li.\n",dspMIN_BLOCKS,dspMAX_BLOCKS);
            exit (1);
        }
        block_count = (size_t) foo;
        rpt_freq = (block_count / 50);
        if (rpt_freq < 16) rpt_freq = 16;

        sptr = argv[7];
        foo = strtol (sptr,&sptr,0);
        if ((foo < dspMIN_SIZE) || (foo >= dspMAX_SIZE))
        {
            fprintf (stderr,"Block size must be between %li and %li.\n",dspMIN_SIZE,dspMAX_SIZE);
            exit (1);
        }
        block_size = (size_t) foo;

        if (argc > 8)
        {
            strcpy (TESTBATCHID,argv[8]);
            TESTBATCHID[8] = '\0';
            strcpy (RESULTSFILE,argv[9]);
            frpt = fopen (RESULTSFILE,"a");
            if (frpt == NULL)
            {
                fprintf (stderr,"Error opening report file:  %s for append.\n",RESULTSFILE);
                exit (1);
            }
            fclose (frpt);
            rpt_flag = 1;       /* set report file flag to TRUE */
        }

        /* ------------------------ */

        time (&timer);          /* get current time */
        srand ((int) timer);    /* seed random number generator */

        header();               /* print header */

        if (create_data() != 0) exit (1);   /* create test data block */

        switch (test_type)
        {
            case dspREAD:
                switch (test_mode)
                {
                    case dspSEQUENTIAL:
                        if (r_s_test() != 0) exit (1);
                        break;
                    case dspRANDOM:
                        if (r_r_test() != 0) exit (1);
                        break;
                }
                break;
            case dspWRITE:
                switch (test_mode)
                {
                    case dspSEQUENTIAL:
                        if (w_s_test() != 0) exit (1);
                        break;
                    case dspRANDOM:
                        if (w_r_test() != 0) exit (1);
                        break;
                }
                break;
            case dspREAD_WRITE:
                switch (test_mode)
                {
                    case dspSEQUENTIAL:
                        if (rw_s_test() != 0) exit (1);
                        break;
                    case dspRANDOM:
                        if (rw_r_test() != 0) exit (1);
                        break;
                }
                break;
        }

        if (delete_data() != 0) exit (1);   /* delete test data block */

        exit (0);
    }


/*==Private Function Code==========================================*/
/*-----------------------------------------------------------------*
 * header()
 *      Prints standard header to screen.  Output is not redirectable
 *      to a file.
 *-----------------------------------------------------------------*/
int header
(
    void
)
    {
        long ttl_size;
        ttl_size = ((long) block_size * (long) block_count);

        fprintf (stderr,"\n");

        fprintf (stderr,"    Test Type:  ");
        switch (test_type)
        {
            case dspREAD:
                fprintf (stderr,"READ ");
                break;
            case dspWRITE:
                fprintf (stderr,"WRITE ");
                break;
            case dspREAD_WRITE:
                fprintf (stderr,"READ/WRITE ");
                break;
            default:
                fprintf (stderr,"UNKNOWN ");
        }
        switch (test_mode)
        {
            case dspSEQUENTIAL:
                fprintf (stderr,"SEQUENTIAL ");
                break;
            case dspRANDOM:
                fprintf (stderr,"RANDOM ");
                break;
            default:
                fprintf (stderr,"UNKNOWN ");
        }
        fprintf (stderr,"\n");
        fprintf (stderr,"\n");

        fprintf (stderr,"    Min. Time:  %10li sec       Block Count:  %10u\n",min_time,block_count);
        fprintf (stderr,"    Max. Time:  %10li sec        Block Size:  %10u bytes\n",max_time,block_size);
        fprintf (stderr,"  Min. Passes:  %10i            Total Size:  %10lu bytes\n",passes,ttl_size);
        fprintf (stderr,"\n");

        return (0);
    }


/*-----------------------------------------------------------------*
 * qheader() 
 *      Prints header for test results.
 *-----------------------------------------------------------------*/
int qheader
(
    void
)
    { 
        fprintf (stderr,"PASS -TIME- -AVG TIME- --KB/SEC-- --IO/SEC-- ---SEC/IO---\n");
        return (0);
    }



/*-----------------------------------------------------------------*
 * create_data()
 *      Creates data block of size needed for the test run.
 *-----------------------------------------------------------------*/
int create_data
(
    void
)
    {
        int i;

        test_data = NULL;
        test_data = malloc (block_size);
        if (test_data == NULL) 
        {
            fprintf (stderr,"\aUnable to allocate room for test data in memory\n");
            return (1);
        }
        for (i = 0; i < block_size; i++)
        {
            test_data[i] = (rand() % 256);
        }

        return (0);
    }



/*-----------------------------------------------------------------*
 * create_file()
 *      Creates the data file used by the tests.
 *-----------------------------------------------------------------*/
int create_file
(
    void
)
    {
        int i;

        fprintf (stderr,"Creating test file...");

        if (fp != NULL) 
        {
            fprintf (stderr,"\acreate_file() called while fp is already open\n");
            return (1);
        }

        if ((fp = fopen (TESTFILE,"wb")) == NULL) 
        {
            fprintf (stderr,"\aUnabled to create %s\n",TESTFILE);
            return (1);
        }

        for (i = 0; i < block_count; i++)
        {
            if (fwrite (test_data, block_size, 1, fp) != 1) 
            {
                fprintf (stderr,"\aError while creating %s\n",TESTFILE);
                fclose (fp);
                remove (TESTFILE);
                return (1);
            }
            if ((i % rpt_freq) == 0)
                fprintf (stderr,"%3li%%\b\b\b\b",(long) (((long) i * 100L) / (long) block_count));
            modify_data (test_data);
        }

        fclose (fp);

        fprintf (stderr,"finished.\n");

        return (0);
    }


/*-----------------------------------------------------------------*
 * modify_data()
 *      Randomly modifies bytes within the current block being written
 *      so that it is different from the block that was read.  Some
 *      smart caches are able to determine whether or not a block of
 *      data is different from what was read from the disk.
 *-----------------------------------------------------------------*/
int modify_data
(
    unsigned char *ptr
)
    {
        int i,j;
        unsigned char *sptr;

        sptr = ptr;
        j = (32 / (1 + (block_size / 32))) + 1;     /* modify 32 bytes */
        for (i = 0; i < block_size; i += j);        /* in the test data block */
        {
            *sptr = (rand() % 256);
            sptr += j;
        }

        return (0);
    }


/*-----------------------------------------------------------------*
 * report()
 *      Reports the results of one iteration to the screen.  (Displays
 *      a running total of the results.)
 *-----------------------------------------------------------------*/
int report
(
    int iterations,
    time_t elapsed,
    long ttl_bytes,
    long io_count
)
    {
        static float avg_time = 0.0;    /* Average time per iteration (pass) */
        static float kb_sec = 0.0;      /* Average Kb/sec overall */
        static float io_sec = 0.0;      /* Avg. I/O's per sec overall */
        static float sec_io = 0.0;      /* Avg. sec per I/O overall */

        if ((elapsed > 0) && (iterations > 0))
        {
            avg_time = (float) elapsed / (float) iterations;
            kb_sec = ((float) (ttl_bytes / 1024) / (float) elapsed);
            io_sec = (float) io_count / (float) elapsed;
            sec_io = (float) elapsed / (float) io_count;
        }

        fprintf (stderr,"%4i %6li %10.2f %10.2f %10.2f %12.8f\n",
            iterations, elapsed, avg_time, kb_sec, io_sec, sec_io);

        return (0);
    }


/*-----------------------------------------------------------------*
 * delete_data()
 *      Frees memory used by the data buffer.
 *-----------------------------------------------------------------*/
int delete_data
(
    void
)
    {
        free (test_data);
        return (0);
    }


/*-----------------------------------------------------------------*
 * w_s_test()
 *      Write Sequential test.
 *-----------------------------------------------------------------*/
int w_s_test
(
    void
)
    {
        int iters = 0;      /* # of iterations completed */
        int blks = 0;       /* # of blocks to read/write */
        long seek_pos = 0;  /* next seek position */
        long bytes_io = 0;  /* total bytes read/written (io_cnt * block_size) */
        long io_cnt = 0;    /* # of i/o calls */
        time_t start_time, finish_time, elapsed;    /* time variables */

        if (create_file () != 0) return (1);        /* create file */
        fp = fopen (TESTFILE,"rb+");                /* open file for testing */
        if (fp == NULL)                             /* check for error */
        {
            fprintf (stderr,"\aError opening test file %s for use as write only\n",TESTFILE);
            return (1);
        }
        setbuf (fp,NULL);       /* file is now unbuffered */
        rewind (fp);            /* rewind file to beginning */

        fprintf (stderr,"\n");
        fprintf (stderr,"Write Sequential:\n");
        fprintf (stderr,"    - Sequential writes to the text file.\n");
        fprintf (stderr,"\n");

        qheader();              /* print results header */

        iters = 0;              /* reset iterations count */
        time (&start_time);     /* check start time */

        /* main loop */         /* do while within time constraints */
        do                      /* and the minimum # of passes are complete */
        {
            for (blks = 0; blks < block_count; blks++)
            {
                seek_pos = block_size * blks;
                fseek (fp, seek_pos, SEEK_SET);

                if (fwrite (test_data, block_size, 1, fp) != 1) 
                {
                    fprintf (stderr,"\aError while writing to test file %s during test\n", TESTFILE);
                    remove (TESTFILE);
                    return (1);
                }

                if ((blks % rpt_freq) == 0)
                    fprintf (stderr,"%3li%%\b\b\b\b",(long) (((long) blks * 100L) / (long) block_count));
                modify_data (test_data);    /* randomly modify data block */

                io_cnt++;
            }
            bytes_io = block_size * io_cnt;     /* calculate bytes read/written */

            iters++;                    /* increment iterations count */
            time (&finish_time);                        /* get finish time */
            elapsed = (finish_time - start_time);       /* elapsed time (sec) */
            report (iters, elapsed, bytes_io, io_cnt);  /* report results */
        }
        while (!((elapsed > max_time) || ((iters >= passes) && (elapsed >= min_time))));

        /* WRITE REPORT FILE */
        if (rpt_flag)
        {
            frpt = fopen (RESULTSFILE,"a");
            if (frpt == NULL)
            {
                fprintf (stderr,"Error opening report file:  %s for append.\n",RESULTSFILE);
                exit (1);
            }
            else
            {
                float kb_sec, io_sec;

                kb_sec = ((float) (bytes_io / 1024) / (float) elapsed);
                io_sec = (float) io_cnt / (float) elapsed;

                fprintf (frpt,"%8s %2s %1s %5i %5i %4i %6li %10li %10li %10.2f %10.2f\n",
                    TESTBATCHID, test_tstr, test_mstr, block_count, block_size, iters, elapsed, bytes_io, io_cnt, kb_sec, io_sec);
                fclose (frpt);
            }
        }

        fclose (fp);                    /* close test file */
        remove (TESTFILE);              /* erase test file */

        return (0);
    }


/*-----------------------------------------------------------------*
 * w_r_test()
 *      Write Random test.
 *-----------------------------------------------------------------*/
int w_r_test
(
    void
)
    {
        int iters = 0;      /* # of iterations completed */
        int blks = 0;       /* # of blocks to read/write */
        long seek_pos = 0;  /* next seek position */
        long bytes_io = 0;  /* total bytes read/written (io_cnt * block_size) */
        long io_cnt = 0;    /* # of i/o calls */
        time_t start_time, finish_time, elapsed;    /* time variables */

        if (create_file () != 0) return (1);        /* create file */
        fp = fopen (TESTFILE,"rb+");                /* open file for testing */
        if (fp == NULL)                             /* check for error */
        {
            fprintf (stderr,"\aError opening test file %s for use as write only\n",TESTFILE);
            return (1);
        }
        setbuf (fp,NULL);       /* file is now unbuffered */
        rewind (fp);            /* rewind file to beginning */

        fprintf (stderr,"\n");
        fprintf (stderr,"Write Random:\n");
        fprintf (stderr,"    - Randomly chosen writes to the test file.\n");
        fprintf (stderr,"\n");

        qheader();              /* print results header */

        iters = 0;              /* reset iterations count */
        time (&start_time);     /* check start time */

        /* main loop */         /* do while within time constraints */
        do                      /* and the minimum # of passes are complete */
        {
            for (blks = 0; blks < block_count; blks++)
            {
                seek_pos = (int) drand (block_count) - 1;   /* calc seek pos */
                seek_pos *= block_size;
                fseek (fp, seek_pos, SEEK_SET);

                if (fwrite (test_data, block_size, 1, fp) != 1) 
                {
                    fprintf (stderr,"\aError while writing to test file %s during test\n", TESTFILE);
                    remove (TESTFILE);
                    return (1);
                }

                if ((blks % rpt_freq) == 0)
                    fprintf (stderr,"%3li%%\b\b\b\b",(long) (((long) blks * 100L) / (long) block_count));
                modify_data (test_data);    /* randomly modify data block */

                io_cnt++;
            }
            bytes_io = block_size * io_cnt;     /* calculate bytes read/written */

            iters++;                    /* increment iterations count */
            time (&finish_time);                        /* get finish time */
            elapsed = (finish_time - start_time);       /* elapsed time (sec) */
            report (iters, elapsed, bytes_io, io_cnt);  /* report results */
        }
        while (!((elapsed > max_time) || ((iters >= passes) && (elapsed >= min_time))));

        /* WRITE REPORT FILE */
        if (rpt_flag)
        {
            frpt = fopen (RESULTSFILE,"a");
            if (frpt == NULL)
            {
                fprintf (stderr,"Error opening report file:  %s for append.\n",RESULTSFILE);
                exit (1);
            }
            else
            {
                float kb_sec, io_sec;

                kb_sec = ((float) (bytes_io / 1024) / (float) elapsed);
                io_sec = (float) io_cnt / (float) elapsed;

                fprintf (frpt,"%8s %2s %1s %5i %5i %4i %6li %10li %10li %10.2f %10.2f\n",
                    TESTBATCHID, test_tstr, test_mstr, block_count, block_size, iters, elapsed, bytes_io, io_cnt, kb_sec, io_sec);
                fclose (frpt);
            }
        }

        fclose (fp);                    /* close test file */
        remove (TESTFILE);              /* erase test file */

        return (0);
    }


/*-----------------------------------------------------------------*
 * r_s_test()
 *      Read Sequential test.
 *-----------------------------------------------------------------*/
int r_s_test
(
    void
)
    {
        int iters = 0;      /* # of iterations completed */
        int blks = 0;       /* # of blocks to read/write */
        long seek_pos = 0;  /* next seek position */
        long bytes_io = 0;  /* total bytes read/written (io_cnt * block_size) */
        long io_cnt = 0;    /* # of i/o calls */
        time_t start_time, finish_time, elapsed;    /* time variables */

        if (create_file () != 0) return (1);        /* create file */
        fp = fopen (TESTFILE,"rb+");                /* open file for testing */
        if (fp == NULL)                             /* check for error */
        {
            fprintf (stderr,"\aError opening test file %s for use as read/write\n",TESTFILE);
            return (1);
        }
        setbuf (fp,NULL);       /* file is now unbuffered */
        rewind (fp);            /* rewind file to beginning */

        fprintf (stderr,"\n");
        fprintf (stderr,"Read Sequential:\n");
        fprintf (stderr,"    - Sequential read from the test file.\n");
        fprintf (stderr,"\n");

        qheader();              /* print results header */

        iters = 0;              /* reset iterations count */
        time (&start_time);     /* check start time */

        /* main loop */         /* do while within time constraints */
        do                      /* and the minimum # of passes are complete */
        {
            for (blks = 0; blks < block_count; blks++)
            {
                seek_pos = block_size * blks;
                fseek (fp, seek_pos, SEEK_SET);

                if (fread(test_data, block_size, 1, fp) != 1) 
                {
                    fprintf (stderr,"\aError while reading from test file %s during test\n", TESTFILE);
                    remove (TESTFILE);
                    return (1);
                }

                if ((blks % rpt_freq) == 0)
                    fprintf (stderr,"%3li%%\b\b\b\b",(long) (((long) blks * 100L) / (long) block_count));
                modify_data (test_data);    /* randomly modify data block */

                io_cnt++;
            }
            bytes_io = block_size * io_cnt;     /* calculate bytes read/written */

            iters++;                    /* increment iterations count */
            time (&finish_time);                        /* get finish time */
            elapsed = (finish_time - start_time);       /* elapsed time (sec) */
            report (iters, elapsed, bytes_io, io_cnt);  /* report results */
        }
        while (!((elapsed > max_time) || ((iters >= passes) && (elapsed >= min_time))));

        /* WRITE REPORT FILE */
        if (rpt_flag)
        {
            frpt = fopen (RESULTSFILE,"a");
            if (frpt == NULL)
            {
                fprintf (stderr,"Error opening report file:  %s for append.\n",RESULTSFILE);
                exit (1);
            }
            else
            {
                float kb_sec, io_sec;

                kb_sec = ((float) (bytes_io / 1024) / (float) elapsed);
                io_sec = (float) io_cnt / (float) elapsed;

                fprintf (frpt,"%8s %2s %1s %5i %5i %4i %6li %10li %10li %10.2f %10.2f\n",
                    TESTBATCHID, test_tstr, test_mstr, block_count, block_size, iters, elapsed, bytes_io, io_cnt, kb_sec, io_sec);
                fclose (frpt);
            }
        }

        fclose (fp);                    /* close test file */
        remove (TESTFILE);              /* erase test file */

        return (0);
    }


/*-----------------------------------------------------------------*
 * r_r_test()
 *      Read Random test.
 *-----------------------------------------------------------------*/
int r_r_test
(
    void
)
    {
        int iters = 0;      /* # of iterations completed */
        int blks = 0;       /* # of blocks to read/write */
        long seek_pos = 0;  /* next seek position */
        long bytes_io = 0;  /* total bytes read/written (io_cnt * block_size) */
        long io_cnt = 0;    /* # of i/o calls */
        time_t start_time, finish_time, elapsed;    /* time variables */

        if (create_file () != 0) return (1);        /* create file */
        fp = fopen (TESTFILE,"rb+");                /* open file for testing */
        if (fp == NULL)                             /* check for error */
        {
            fprintf (stderr,"\aError opening test file %s for use as read only\n",TESTFILE);
            return (1);
        }
        setbuf (fp,NULL);       /* file is now unbuffered */
        rewind (fp);            /* rewind file to beginning */

        fprintf (stderr,"\n");
        fprintf (stderr,"Read Random:\n");
        fprintf (stderr,"    - Randomly chosen reads from the test file.\n");
        fprintf (stderr,"\n");

        qheader();              /* print results header */

        iters = 0;              /* reset iterations count */
        time (&start_time);     /* check start time */

        /* main loop */         /* do while within time constraints */
        do                      /* and the minimum # of passes are complete */
        {
            for (blks = 0; blks < block_count; blks++)
            {
                seek_pos = (int) drand (block_count) - 1;   /* calc seek pos */
                seek_pos *= block_size;
                fseek (fp, seek_pos, SEEK_SET);

                if (fread(test_data, block_size, 1, fp) != 1) 
                {
                    fprintf (stderr,"\aError while reading from test file %s during test\n", TESTFILE);
                    remove (TESTFILE);
                    return (1);
                }

                if ((blks % rpt_freq) == 0)
                    fprintf (stderr,"%3li%%\b\b\b\b",(long) (((long) blks * 100L) / (long) block_count));
                modify_data (test_data);    /* randomly modify data block */

                io_cnt++;
            }
            bytes_io = block_size * io_cnt;     /* calculate bytes read/written */

            iters++;                    /* increment iterations count */
            time (&finish_time);                        /* get finish time */
            elapsed = (finish_time - start_time);       /* elapsed time (sec) */
            report (iters, elapsed, bytes_io, io_cnt);  /* report results */
        }
        while (!((elapsed > max_time) || ((iters >= passes) && (elapsed >= min_time))));

        /* WRITE REPORT FILE */
        if (rpt_flag)
        {
            frpt = fopen (RESULTSFILE,"a");
            if (frpt == NULL)
            {
                fprintf (stderr,"Error opening report file:  %s for append.\n",RESULTSFILE);
                exit (1);
            }
            else
            {
                float kb_sec, io_sec;

                kb_sec = ((float) (bytes_io / 1024) / (float) elapsed);
                io_sec = (float) io_cnt / (float) elapsed;

                fprintf (frpt,"%8s %2s %1s %5i %5i %4i %6li %10li %10li %10.2f %10.2f\n",
                    TESTBATCHID, test_tstr, test_mstr, block_count, block_size, iters, elapsed, bytes_io, io_cnt, kb_sec, io_sec);
                fclose (frpt);
            }
        }

        fclose (fp);                    /* close test file */
        remove (TESTFILE);              /* erase test file */

        return (0);
    }


/*-----------------------------------------------------------------*
 * rw_s_test()
 *      Read/Write Sequential test.
 *-----------------------------------------------------------------*/
int rw_s_test
(
    void
)
    {
        int iters = 0;      /* # of iterations completed */
        int blks = 0;       /* # of blocks to read/write */
        long seek_pos = 0;  /* next seek position */
        long bytes_io = 0;  /* total bytes read/written (io_cnt * block_size) */
        long io_cnt = 0;    /* # of i/o calls */
        time_t start_time, finish_time, elapsed;    /* time variables */

        if (create_file () != 0) return (1);        /* create file */
        fp = fopen (TESTFILE,"rb+");                /* open file for testing */
        if (fp == NULL)                             /* check for error */
        {
            fprintf (stderr,"\aError opening test file %s for use as read/write\n",TESTFILE);
            return (1);
        }
        setbuf (fp,NULL);       /* file is now unbuffered */
        rewind (fp);            /* rewind file to beginning */

        fprintf (stderr,"\n");
        fprintf (stderr,"Read/Write Sequential:\n");
        fprintf (stderr,"    - Read from file then write back to test file.\n");
        fprintf (stderr,"\n");

        qheader();              /* print results header */

        iters = 0;              /* reset iterations count */
        time (&start_time);     /* check start time */

        /* main loop */         /* do while within time constraints */
        do                      /* and the minimum # of passes are complete */
        {
            for (blks = 0; blks < block_count; blks++)
            {
                seek_pos = block_size * blks;
                fseek (fp, seek_pos, SEEK_SET);
                if (fread(test_data, block_size, 1, fp) != 1) 
                {
                    fprintf (stderr,"\aError while reading from test file %s during test\n", TESTFILE);
                    remove (TESTFILE);
                    return (1);
                }

                modify_data (test_data);    /* randomly modify data block */
                io_cnt++;

                seek_pos = block_size * blks;
                fseek (fp, seek_pos, SEEK_SET);
                if (fwrite (test_data, block_size, 1, fp) != 1) 
                {
                    fprintf (stderr,"\aError while writing to test file %s during test\n", TESTFILE);
                    remove (TESTFILE);
                    return (1);
                }

                if ((blks % rpt_freq) == 0)
                    fprintf (stderr,"%3li%%\b\b\b\b",(long) (((long) blks * 100L) / (long) block_count));
                modify_data (test_data);    /* randomly modify data block */
                io_cnt++;
            }
            bytes_io = block_size * io_cnt;     /* calculate bytes read/written */

            iters++;                    /* increment iterations count */
            time (&finish_time);                        /* get finish time */
            elapsed = (finish_time - start_time);       /* elapsed time (sec) */
            report (iters, elapsed, bytes_io, io_cnt);  /* report results */
        }
        while (!((elapsed > max_time) || ((iters >= passes) && (elapsed >= min_time))));

        /* WRITE REPORT FILE */
        if (rpt_flag)
        {
            frpt = fopen (RESULTSFILE,"a");
            if (frpt == NULL)
            {
                fprintf (stderr,"Error opening report file:  %s for append.\n",RESULTSFILE);
                exit (1);
            }
            else
            {
                float kb_sec, io_sec;

                kb_sec = ((float) (bytes_io / 1024) / (float) elapsed);
                io_sec = (float) io_cnt / (float) elapsed;

                fprintf (frpt,"%8s %2s %1s %5i %5i %4i %6li %10li %10li %10.2f %10.2f\n",
                    TESTBATCHID, test_tstr, test_mstr, block_count, block_size, iters, elapsed, bytes_io, io_cnt, kb_sec, io_sec);
                fclose (frpt);
            }
        }

        fclose (fp);                    /* close test file */
        remove (TESTFILE);              /* erase test file */

        return (0);
    }


/*-----------------------------------------------------------------*
 * rw_r_test()
 *      Read/Write random test.
 *-----------------------------------------------------------------*/
int rw_r_test
(
    void
)
    {
        int iters = 0;      /* # of iterations completed */
        int blks = 0;       /* # of blocks to read/write */
        long seek_pos = 0;  /* next seek position */
        long bytes_io = 0;  /* total bytes read/written (io_cnt * block_size) */
        long io_cnt = 0;    /* # of i/o calls */
        time_t start_time, finish_time, elapsed;    /* time variables */

        if (create_file () != 0) return (1);        /* create file */
        fp = fopen (TESTFILE,"rb+");                /* open file for testing */
        if (fp == NULL)                             /* check for error */
        {
            fprintf (stderr,"\aError opening test file %s for use as read/write\n",TESTFILE);
            return (1);
        }
        setbuf (fp,NULL);       /* file is now unbuffered */
        rewind (fp);            /* rewind file to beginning */

        fprintf (stderr,"\n");
        fprintf (stderr,"Read/Write Random:\n");
        fprintf (stderr,"    - Randomly chosen reads from or writes to the test file.\n");
        fprintf (stderr,"\n");

        qheader();              /* print results header */

        iters = 0;              /* reset iterations count */
        time (&start_time);     /* check start time */

        /* main loop */         /* do while within time constraints */
        do                      /* and the minimum # of passes are complete */
        {
            for (blks = 0; blks < block_count; blks++)
            {
                seek_pos = (int) drand (block_count) - 1;   /* calc seek pos */
                seek_pos *= block_size;
                fseek (fp, seek_pos, SEEK_SET);

                if ((rand() % 2) == 1)  /* randomly choose read/write */
                {
                    if (fwrite (test_data, block_size, 1, fp) != 1) 
                    {
                        fprintf (stderr,"\aError while writing to test file %s during test\n", TESTFILE);
                        remove (TESTFILE);
                        return (1);
                    }
                }
                else 
                {
                    if (fread(test_data, block_size, 1, fp) != 1) 
                    {
                        fprintf (stderr,"\aError while reading from test file %s during test\n", TESTFILE);
                        remove (TESTFILE);
                        return (1);
                    }
                }

                if ((blks % rpt_freq) == 0)
                    fprintf (stderr,"%3li%%\b\b\b\b",(long) (((long) blks * 100L) / (long) block_count));
                modify_data (test_data);    /* randomly modify data block */

                io_cnt++;
            }
            bytes_io = block_size * io_cnt;     /* calculate bytes read/written */

            iters++;                    /* increment iterations count */
            time (&finish_time);                        /* get finish time */
            elapsed = (finish_time - start_time);       /* elapsed time (sec) */
            report (iters, elapsed, bytes_io, io_cnt);  /* report results */
        }
        while (!((elapsed > max_time) || ((iters >= passes) && (elapsed >= min_time))));

        /* WRITE REPORT FILE */
        if (rpt_flag)
        {
            frpt = fopen (RESULTSFILE,"a");
            if (frpt == NULL)
            {
                fprintf (stderr,"Error opening report file:  %s for append.\n",RESULTSFILE);
                exit (1);
            }
            else
            {
                float kb_sec, io_sec;

                kb_sec = ((float) (bytes_io / 1024) / (float) elapsed);
                io_sec = (float) io_cnt / (float) elapsed;

                fprintf (frpt,"%8s %2s %1s %5i %5i %4i %6li %10li %10li %10.2f %10.2f\n",
                    TESTBATCHID, test_tstr, test_mstr, block_count, block_size, iters, elapsed, bytes_io, io_cnt, kb_sec, io_sec);
                fclose (frpt);
            }
        }

        fclose (fp);                    /* close test file */
        remove (TESTFILE);              /* erase test file */

        return (0);
    }

/*--End of File:  DSP.C --*/

