
(*
 * SAFMT  - format system activity report file from
 *          sorted system activity log file captured by SA.
 *
 * Copyright (C) 1987 Samuel H. Smith,
 * all commercial rights reserved.
 *
 * Written by: s.h.smith, 31-jan-87
 *
 *)

const
   version = 'SAFMT v1.3 (3/6/92 SHS)';

   copyright: string[80] = 'Copyright (C) 1992 Samuel H. Smith; All Rights Reserved';

type
   anystring = string[80];

   summary = record
      time:       real;          {total elapsed time}
      runs:       integer;       {count of runs}
      reads:      real;          {count of reads}
      writes:     real;          {count of writes}
      others:     real;          {count of other system calls}
   end;


var
   con:        text;

   line:       anystring;
   name:       anystring;

   cmd:        summary;
   other:      summary;

   iopt:       boolean;       {-i (exclude i/o counts) option}
   vopt:       boolean;       {-v (exclude headings) option}
   jopt:       boolean;       {-j (combine commands run only once) option}
   sopt:       boolean;       {-s (summary sysact format) option}

   inbuf:      array[1..10000] of char;
   outbuf:     array[1..10000] of char;


(*
 * log entry format:
 *    0        1         2         3         4         5         6
 *    123456789012345678901234567890123456789012345678901234567890
 *    SYSACT.COM   01-31-87 22:11:44 22:11:44.78 RRRR WWWW OOOO RRRRR
 *    |            |        |        |           |    |    |    |
 *    |            |        |        |           |    |    |    +--runs (dec)
 *    |            |        |        |           |    |    |   (optional)
 *    |            |        |        |           |    |    |
 *    |            |        |        |           |    |    +--misc calls (hex)
 *    |            |        |        |           |    |
 *    |            |        |        |           |    +-------writes (hex)
 *    |            |        |        |           |
 *    |            |        |        |           +------------reads  (hex)
 *    |            |        |        |
 *    |            |        |        +------------------------runtime
 *    |            |        |
 *    |            |        +---------------------------------terminate time
 *    |            |
 *    |            +------------------------------------------terminate date
 *    |
 *    +-------------------------------------------------------filename
 *)


{extract hex digits from a specified position in the current input line}
function hex(from: integer): real;
const
   digit: array['0'..'F'] of integer =
       (0,1,2,3,4,5,6,7,8,9,0,0,0,0,0,0,0,10,11,12,13,14,15);
   function nibble(fm: integer): real;
   begin
      nibble := int( digit[upcase(line[fm])] shl 4 +
                     digit[upcase(line[fm+1])] );
   end;
begin
   hex := nibble(from) * 256.0 + nibble(from+2);
end;


{extract the elapsed runtime from the current input line}
function jtime: real;
   function get(from: integer): real;
   begin
      get := int( (ord(line[from]  )-ord('0')) * 10 +
                  (ord(line[from+1])-ord('0')) );
   end;
begin
   jtime := get(32) * 3600.0 +
            get(35) * 60.0 +
            get(38) +
            get(41) / 100.0;
end;


{update the current cmd totals from the input line}
procedure update_totals;
var
   mult:  real;
   code:  integer;
begin
   mult := 0;
   if length(line) > 58 then
      val(copy(line,59,5),mult,code)
   else
      mult := 1;

   with cmd do
   begin
      time := time + jtime * mult;
      runs := runs + trunc(mult);

      if not iopt then
      begin
         reads := reads + hex(44) * mult;
         writes := writes + hex(49) * mult;
         others := others + hex(54) * mult;
      end;
   end;
end;


{zero the current cmd totals}
procedure initial_totals;
begin
   with cmd do
   begin
      name := copy(line,1,12);
      time := 0;
      reads := 0;
      writes := 0;
      others := 0;
      runs := 0;
   end;
end;


{display report titles based on -v and -i options}
procedure write_titles;
begin
   if not vopt then
   begin

      if not iopt then
      begin
         writeln('  Total    Average  -- Sys Call Averages -- ');
         writeln('  Time      Time     Reads   Writes   Misc  Runs     Command');
         writeln('---------- --------- ------- ------- ------- ------ --------------');
      end
      else

      begin
         writeln('  Total    Average  ');
         writeln('  Time      Time    Runs     Command');
         writeln('---------- --------- ------ --------------');
      end;
   end;
end;


{output a line of totals based on -j and -i options; updates other totals
 instead of producing output when runs=1}
procedure write_totals;

   function timeformat(n: real): anystring;

      function number(n: real): anystring;
      var
         buf: anystring;
      begin
         while n > 10000.0 do
            n := n - 10000;
         str(trunc(n) mod 100:0,buf);
         if length(buf) < 2 then
            buf := '0' + buf;
         number := buf;
      end;

   begin
      timeformat := number(n / 3600.0) + ':' +
                    number(n / 60.0)   + ':' +
                    number(n) + '.'    +
                    number(n * 100);
   end;

   function hexformat(n: real): anystring;

      function digit(n: real): anystring;
      var
         i: integer;
      begin
         while n > 16384.0 do
            n := n - 16384.0;
         i := trunc(n) and 15;
         if i > 9 then
            i := i + 7;
         digit := chr(i + ord('0'));
      end;

   begin
      hexformat := digit(n / 4096.0) + digit(n / 256.0) +
                   digit(n / 16.0)  +  digit(n);
   end;

begin
   with cmd do
   begin
      if jopt and (runs=1) then
      begin
         other.time := other.time + time;
         other.reads := other.reads + reads;
         other.writes := other.writes + writes;
         other.others := other.others + others;
         other.runs := other.runs + 1;
      end
      else

      if runs = 0 then
      else

      if sopt then
         writeln(name,' ',
                 copy(line,14,18),
                 timeformat(time/int(runs)),' ',
                 hexformat(reads/int(runs)),' ',
                 hexformat(writes/int(runs)),' ',
                 hexformat(others/int(runs)),' ',
                 runs)
      else
      if iopt then
         writeln(time:10:1,' ',
                 time/int(runs):9:2,' ',
                 runs:6,'   ',
                 name)
      else
         writeln(time:10:1,' ',
                 time/int(runs):9:2,' ',
                 reads/int(runs):7:0,' ',
                 writes/int(runs):7:0,' ',
                 others/int(runs):7:0,' ',
                 runs:6,'   ',
                 name);
   end;
end;


{process command line options}
procedure get_options;
var
   i:   integer;
   opt: anystring;

begin
   iopt := false;
   jopt := false;
   vopt := false;
   sopt := false;

   for i := 1 to paramcount do
   begin
      opt := paramstr(i);

      if opt[1] in ['-','/'] then
         delete(opt,1,1);

      case upcase(opt[1]) of

         'I':  iopt := true;

         'J':  jopt := true;

         'V':  vopt := true;

         'S':  begin
                  sopt := true;
                  iopt := false;
                  jopt := false;
                  vopt := true;
               end;

         else
            writeln(con);
            writeln(con,'invald option: ',opt);
            writeln(con);
            writeln(con,'usage:    SAFMT <INFILE >OUTFILE [options]');
            writeln(con);
            writeln(con,'options:  -i    exclude i/o counts');
            writeln(con,'          -v    exclude headings');
            writeln(con,'          -j    combine commands run only once');
            writeln(con,'          -s    produce SYSACT summary format');
            writeln(con);
            writeln(con,'INFILE must be a sorted SYSACT.LOG activity file');
            halt;
      end;
   end;

end;


{main program}
begin
   assign(con,'con');
   rewrite(con);

   settextbuf(output,outbuf);
   settextbuf(input,inbuf);

   writeln(con,version);
   get_options;

   write_titles;

   readln(line);
   initial_totals;
   other := cmd;
   update_totals;

   {process each record in the system activity log}
   while not eof do
   begin
      readln(line);

      {accumulate repeats on the current record}
      if name = copy(line,1,12) then
         update_totals
      else
      begin
         {different record, output accumulators and start new current}
         write_totals;
         initial_totals;
         update_totals;
      end;
   end;

   {output the last record}
   write_totals;

   if jopt then
   begin
      jopt := false;
      cmd := other;
      name := '**others';
      write_totals;
   end;

end.

