
; ResMode.asm
; Enhanced modes for VGA-cards
; Assembler: Turbo Assembler 3.0

; Copyright (c) 1992 by Olaf Hess
; (Fr alle des Deutschen mchtigen: Olaf He)
; Hof Kappelberg 1
; D - 3509 Altmorschen
; Germany

; CompuServe ID: 100 031, 3536

; This program installs itself resident and sets the screen resolution in
; text mode to non-standard modes like 80x30, 94x34, etc.
; Call it without parameters for a complete listing of the possible values.

; Attention:
; ResMode uses the VGA-monitor about 10% beyond it's original specifications.
; Use it with care! If your monitor makes a funny sound after invoking one
; of ResMode's enhanced modes turn the monitor off immediately.

; I hereby declare that whoever who uses this program does so on his or her
; own risk. I am in no way liable for any consequences of the use of ResMode.

; This program is free software and may be used or changed at one's own
; convenience as long as the original copyright notice is kept.
; Please mark individual changes in the source code.
; If you do any changes or if you have suggestion, comments, bug reports etc:
; Please send me a message!

; PrintScreen equ 1 ; Uncomment when pressing PrintScreen should cause a
                    ; modus refresh
; ScrollLock equ 1 ; Uncomment when pressing ScrollLock should cause a
                   ; modus refresh

.8086

; Int 21h, function 09h : Display string
disp    macro string
        mov dx, offset string
        mov ah, 09h
        int 21h
endm

; Int 21h, function 25h : Set interrupt vector
set_vector macro intr, oldint, newint
        cli
        mov word ptr oldint, bx
        mov word ptr oldint + 2, es
        mov dx, offset newint
        mov al, intr
        mov ah, 25h
        int 21h
        sti
endm

; Int 21h, function 25h : Set interrupt vektor back
set_vector_back macro interrupt, orgvector
        cli
        mov dx, word ptr es:orgvector
        mov ax, word ptr es:orgvector + 2
        mov ds, ax
        mov al, interrupt
        mov ah, 25h
        int 21h
        sti
endm

; Int 21h, function 31h : Keep program resident
keep_process macro return_code, last_byte
        mov al, return_code
        mov dx, offset last_byte
        mov cl, 4
        shr dx, cl
        inc dx
        mov ah, 31h
        int 21h
endm

; Int 21h, function 35h : Get interrupt vector
get_vector macro interrupt
        mov al, interrupt
        mov ah, 35h
        int 21h
endm

; Int 21h, function 4Ch : Halt program
end_process macro return_code
        mov al, return_code
        mov ah, 4Ch
        int 21h
endm

; Int 21h, function 49h : Release memory
release_memory macro
        mov ah, 49h
        int 21h
endm

; Int 10h, function 0 : Initialize screen modus
clrscr  macro
        mov ax, 0003
        int 10h
endm

OutByte macro PortNo, Value ; Write a byte to a port
        mov al, Value
        mov dx, PortNo
        out dx, al
endm

OutWord macro PortNo, Value ; Write a word to a port
        mov ax, Value
        mov dx, PortNo
        out dx, ax
endm

OutAX   macro PortNo ; Write the word in AX to a port
        mov dx, PortNo
        out dx, ax
endm

OutAL   macro PortNo ; Write the byte in AL to a port
        mov dx, PortNo
        out dx, al
endm

InWord  macro PortNo ; Read a word from a port
        mov dx, PortNo
        in ax, dx
endm

code segment
        assume cs: code, ds: code, es: nothing, ss: nothing

        org 80h
Parameter db ?

        org 100h

start:
        jmp begin

oldint10 dd ? ; Old int 10h vector
oldint2F dd ? ; Old int 2Fh vector

IFDEF PrintScreen
oldint05 dd ?
ENDIF

IFDEF ScrollLock
oldint09 dd ?
ENDIF

MouseInst db ? ; Mouse installed
action   db ? ; Action flag
Rows     db ? ; Number of Rows on screen
Columns  db ? ; Number of columns on screen
CursorX  db ? ; Cursor Position
CursorY  db ? ; Cursor Position

VGA_Off proc far ; Turn the VGA card off
        cli
        OutByte 03C4h, 0
        OutByte 03C5h, 1
        OutByte 03C4h, 23
        InWord 03D5h
        and ax, 127
        OutAX 03D5h
        OutByte 03D4h, 17
        InWord 03D5h
        and ax, 127
        OutAX 03D5h
        ret
VGA_Off  endp

VGA_On  proc far ; Turn the VGA card on
        OutByte 03D4h, 17
        InWord 03D5h
        or ax, 128
        OutAX 03D5h
        OutByte 03D4h, 23
        InWord 03D5h
        or ax, 128
        OutAX 03D5h
        OutByte 03C4h, 0
        OutByte 03C5h, 3
        sti
        ret
VGA_On  endp

VGA_94_Columns proc far ; Set 94 columns
        InWord 03CCh
        and ax, 0F3h
        or ax, 4
        OutAX 03C2h
        OutByte 03C4h, 1
        InWord 03C5h
        or ax, 1
        OutAX 03C5h
        OutByte 03D4h, 0
        OutByte 03D5h, 108
        OutByte 03D4h, 1
        OutByte 03D5h, 93
        OutByte 03D4h, 2
        OutByte 03D5h, 94
        OutByte 03D4h, 3
        OutByte 03D5h, 143
        OutByte 03D4h, 4
        OutByte 03D5h, 98
        OutByte 03D4h, 5
        OutByte 03D5h, 142
        OutByte 03D4h, 19
        OutByte 03D5h, 47
        InWord 03DAh
        OutByte 03C0h, 19
        OutByte 03C0h, 0
        OutByte 03C0h, 32
        OutByte 03C0h, 32
        ret
VGA_94_Columns endp

Set_480_Lines proc far ; Set 480 scan-lines
        InWord 03CCh
        or ax, 192
        OutAX 03C2h
        OutByte 03D4h, 6
        OutByte 03D5h, 11
        OutByte 03D4h, 7
        OutByte 03D5h, 62
        OutByte 03D4h, 9
        OutByte 03D5h, 79
        OutByte 03D4h, 16
        OutByte 03D5h, 234
        OutByte 03D4h, 17
        OutByte 03D5h, 140
        OutByte 03D4h, 18
        OutByte 03D5h, 223
        OutByte 03D4h, 21
        OutByte 03D5h, 231
        OutByte 03D4h, 22
        OutByte 03D5h, 4
        ret
Set_480_Lines endp

CharHeight proc far ; Set character height
        ; CharHeight in CX
        xor ax, ax
        mov es, ax
        cli
        mov es:[485h], cl
        sti
        OutByte 03D4h, 9
        InWord 03D5h
        and ax, 0E0h
        add ax, cx
        dec ax
        OutAX 03D5h
        OutByte 03D4h, 10
        mov al, cl

        cmp al, 12
        jbe kl_gl_1
        dec al
    kl_gl_1:
        sub al, 2
        OutAL 03D5h
        OutByte 03D4h, 11

        mov al, cl
        cmp al, 12
        jbe kl_gl_2
        dec al
    kl_gl_2:
        dec al
        OutAL 03D5h
        ret
CharHeight endp

SetVGAModus proc far ; Set a new VGA textmodus
        ; Get cursor position
        mov ah, 03
        mov bh, 00
        int 10h
        mov cs:CursorX, dl
        mov cs:CursorY, dh

        ; Start of the routines to set the new modus
        mov ax, 1201h
        mov bl, 30h
        cmp cs:Rows, 43
        je Lines_43
        inc al
    Lines_43:
        int 10h

        mov ax, 1101h ; al = 1
        xor bl, bl
        cmp cs:Rows, 34
        je Lines_34
        inc al ; al = 2
        cmp cs:Rows, 30
        ja Lines_34
        add al, 2 ; al = 4
    Lines_34:
        int 10h
        call VGA_Off
        cmp cs:Columns, 94
        jne Columns_80
        call VGA_94_Columns
    Columns_80:
        cmp cs:Rows, 30
        je Call480Lines
        cmp cs:Rows, 34
        je Call480Lines
        cmp cs:Rows, 60
        jne No480Lines

    Call480Lines:
        call Set_480_Lines

    No480Lines:
        cmp cs:Rows, 30
        ja Test34
        mov cx, 16
        call CharHeight
        jmp CallVGAOn
    Test34:
        cmp cs:Rows, 34
        ja Bigger34
        mov cx, 14
        call CharHeight
        jmp CallVGAOn

    Bigger34:
        mov cx, 8
        call CharHeight

    CallVGAOn:
        call VGA_On

        xor ax, ax
        mov es, ax
        mov al, cs:Columns
        cli
        mov es:[44Ah], al
        mov al, cs:Rows
        dec al
        mov es:[484h], al
        sti

        ; Set cursor position
        mov ah, 02
        mov bh, 00
        mov dh, cs:CursorY
        mov dl, cs:CursorX
        int 10h

        ret
SetVGAModus endp

int10handler proc far ; The new int 10h handler

        jmp short int10

int10id db "RM"                 ; Program id string
    int10:
        sti         ; Enable interrupts n
        push ds     ; Save ds
        push cs     ; ds = cs
        pop ds

        cmp cs:action, 1 ; Called by itself?
        je CallOld       ; Yes => Call old interrupt

        cmp ax, 0003h    ; Set modus 03h?
        je SetLines      ; Yes: Set our own modus
        cmp ax, 0083h    ; Set modus 83h?
        je SetLines      ; Yes: Set our own modus

    CallOld:
        pushf
        call cs:[oldint10] ; Call original int 10h routine
        jmp exit           ; Jump to end

    SetLines:       ; Set the new modus
        push ax
        push bx
        push cx
        push dx
        push es
        mov cs:action, 1  ; Tell the program that it's calling itself

        int 10h           ; AX is still 0003h or 0083h, set this modus first
        call SetVGAModus  ; New set the enhanced modus

        mov cs:action, 0 ; Tell the program that is isn't calling itself anymore
        pop es
        pop dx
        pop cx
        pop bx
        pop ax

    exit:
        pop ds  ; Pop registers
        ret 2   ; Bye, release stack
int10handler endp

int2Fhandler proc far

    jmp short int2F

id2Fid  db "RM"

    int2F:
        sti
        push ds

        push cs
        pop ds

        cmp ah, ''
        jne Call_Old_2F
        cmp al, '1'
        je IsInst

    Call_Old_2F:
        pop ds
        jmp cs:[oldint2F]

    IsInst:
        mov ax, 00FFh
        pop ds
        ret 2
int2Fhandler endp


IFDEF PrintScreen

int05handler proc far ; The new int 05h handler

        jmp short int05

id      db "RM"                 ; Program id string

int05:
        sti                     ; Enable interrupts
        push ds

        push cs                 ; ds = cs
        pop ds

        push ax
        push bx
        push cx
        push dx
        push es

        cmp cs:MouseInst, 1
        jne NoMouse_1

        ; Hide mouse cursor
        mov ax, 02h
        int 33h
    NoMouse_1:

        call SetVGAModus

        cmp cs:MouseInst, 1
        jne NoMouse_2

        ; Show mouse cursor
        mov ax, 01h
        int 33h
    NoMouse_2:

        pop es
        pop dx
        pop cx
        pop bx
        pop ax

        pop ds
        ret 2
endp

ENDIF ; PrintScreen

IFDEF ScrollLock

int09handler proc far   ; The new int 09h handler
        jmp short int09

id09    db "RM"

    int09:
        sti         ; Enable interrupts
        push ax
        push ds

        xor ax, ax  ; ax = 0
        mov ds, ax  ; ds = 0
        mov ah, ds:[0418h] ; Read keyboard statusbyte 2
        and ah, 00010000b ; Mask bits
        cmp ah, 00010000b ; ScrollLock pressed?
        ;       76543210
        jne quit09        ; No --> Quit

        cli
        mov ah, 11101111b
        and ds:[0418h], ah ; Set Scroll Lock bit to 0
        mov ah, 11101111b ; Set Scroll Lock active bit to 0
        and ds:[0417h], ah
        sti
        jmp ScrLockPressed

    quit09:
        pop ds      ; Call the old interrupt routine
        pop ax
        jmp cs:[oldint09]

    ScrLockPressed: ; Set the new VGA modus

        push bx
        push cx
        push dx
        push es

        cmp cs:MouseInst, 1
        jne NoMouse_3

        ; Hide mouse cursor
        mov ax, 02h
        int 33h
    NoMouse_3:

        call SetVGAModus

        cmp cs:MouseInst, 1
        jne NoMouse_4

        ; Show mouse cursor
        mov ax, 01h
        int 33h

    NoMouse_4:

        pop es
        pop dx
        pop cx
        pop bx
        sti
        jmp quit09 ; Call the old interrupt routine

int09handler endp

ENDIF ; ScrollLock

; The following code can be overwritten after initialization

transient   equ this byte

intro       db 10, 13, "ResMode V4.1", 10, 13, 10, 13, "$"

syntax      db "ResMode enables enhanced text modes.", 10, 13
            db "Syntax: ResMode [Columns] [Rows]", 10, 13
            db "[Columns] in 80, 94", 10, 13
            db "[Rows] in 25, 30, 34, 43, 50, 60", 10, 13, 10, 13
            db "Program must only be run on systems with a "
            db "VGA-compatible graphics card.", 10, 13
            db "$"

okmessage   db "ResMode resident installed, "
            db "de-installation by calling it again.", 10, 13

IFDEF PrintScreen
            db "Force modus refresh by pressing [PrintScreen].", 10, 13
ENDIF

IFDEF ScrollLock
            db "Force modus refresh by pressing [ScrollLock].", 10, 13
ENDIF
            db "$"

uninstmsg   db "Program removed from memory.", 10, 13, "$"

no_vga      db "No VGA graphics adaptor active!"
            db 10, 13, "$"

NotLastTsr  db "Cannot uninstall. Another TSR has been loaded on "
            db "top of Resmode.", 10, 13, "$"

begin:
        ; Check whether program is already installed
        mov ah, ''
        mov al, '1'
        int 2Fh
        cmp ax, 00FFh
        je uninst ; Is installed --> Try to uninstall

        jmp install ; Jump to installation routine

    uninst:     ; Try to uninstall the program
        get_vector 2Fh                  ; Get interrupt vector
        cmp word ptr es:[bx+2], "MR"    ; Does it point to ResMode?
        jne Cannot_Uninstall            ; No --> A TSR has been loaded later

        get_vector 10h
        cmp word ptr es:[bx+2], "MR"
        jne Cannot_Uninstall

IFDEF PrintScreen
        get_vector 05h
        cmp word ptr es:[bx+2], "MR"
        jne Cannot_Uninstall
ENDIF

IFDEF ScrollLock
        get_vector 09h
        cmp word ptr es:[bx+2], "MR"
        jne Cannot_Uninstall
ENDIF

        ; Restore original interrupt vectors
        set_vector_back 2Fh, oldint2F
        set_vector_back 10h, oldint10

IFDEF PrintScreen
        set_vector_back 05h, oldint05
ENDIF

IFDEF ScrollLock
        set_vector_back 09h, oldint09
ENDIF

        release_memory ; Release the memory occupied by ResMode
        push cs
        pop ds

        clrscr  ; Set 80x25 screen mode

        ; Display messages
        disp intro
        disp uninstmsg
        end_process 0

    Cannot_Uninstall:
        ; Another TSR that uses one of the interrupt vectors in use by ResMode
        ; has been loaded after ResMode --> Uninstall not possible.
        disp intro
        disp NotLastTsr
        end_process 1

    install:    ; Start of installation routine

        ; Test whether a VGA graphics card is active
        mov ax, 1A00h
        int 10h
        cmp al, 1Ah
        jne No_VGA_Active
        cmp bl, 7
        je VGA_Active
        cmp bl, 8
        je VGA_Active

    No_VGA_Active:
        disp intro
        disp syntax
        disp no_vga
        end_process 1

    VGA_Active:
        lea bx, Parameter   ; Load address of parameters
        mov ah, [bx]        ; Length of parameter string
        cmp ah, 0           ; No parameter?
        jne Param1

        ; No parameter --> Display message
        disp intro
        disp syntax
        end_process 1

    Param1:
        inc bx              ; Point to first character
        mov dl, 1           ; Parameter no. 1

    Param2:
        xor cx, cx          ; Initialize counter

    NextChar:
        inc bx              ; Increment to next character
        mov al, [bx]        ; Load to al
        cmp al, ' '         ; Is it a space?
        je NextChar         ; Ignore it
        cmp al, 13          ; Is it a return?
        je SearchEnd        ; End of search
        cmp al, 0           ; Compare with 0
        je SearchEnd        ; End of search
        cmp al, '0'         ; Character < '0'?
        jl NextChar         ; Yes --> ignore it
        cmp al, '9'         ; Character > '9'
        ja NextChar         ; Yes --> ignore it
        sub al, '0'         ; Convert char to byte
        inc cx              ; Increment counter
        cmp cx, 1           ; Is it the first number?
        ja SecondChar       ; No --> Jump to second char
        mov dh, 10
        mul dh              ; Multiply  al with 10
        mov dh, al          ; Result is in al
        jmp NextChar        ; Continue search

    SecondChar:
        add dh, al          ; Second character, increment dh
        cmp dl, 2           ; Is it the second parameter?
        je SearchEnd        ; Yes --> Search is over
        inc dl              ; No --> Increment parameter count
        cmp cx, 2           ; Does the parameter have the correct length?
        jne DispSyntax      ; No --> Message
        mov cs:Columns, dh  ; Save value
        jmp Param2          ; Continue search

    SearchEnd:              ; End of search
        cmp cx, 2           ; Does the parameter have the correct length?
        jne DispSyntax      ; No --> Message

        mov cs:Rows, dh     ; Save value

        ; Compare whether values are valid
        cmp cs:Columns, 80
        je CmpRows
        cmp cs:Columns, 94
        jne DispSyntax

    CmpRows:
        cmp cs:Rows, 25
        je SetVectors
        cmp cs:Rows, 30
        je SetVectors
        cmp cs:Rows, 34
        je SetVectors
        cmp cs:Rows, 43
        je SetVectors
        cmp cs:Rows, 50
        je SetVectors
        cmp cs:Rows, 60
        je SetVectors

    DispSyntax: ; Display message
        disp intro
        disp syntax
        end_process 1


    ; Set the interrupt vectors to the routines in ResMode
    SetVectors:

        ; Check for mouse
        mov cs:MouseInst, 0
        mov ax, 0000h
        int 33h
        cmp AX, 0000h
        je NoMouseInst
        mov cs:MouseInst, 1

    NoMouseInst:

        ; Release the program's environment
        mov ax, cs:02Ch
        mov es, ax
        release_memory

        ; Set the interrupts for the corresponding routines
        get_vector 10h
        set_vector 10h, oldint10, int10handler

        get_vector 2Fh
        set_vector 2Fh, oldint2F, int2Fhandler

IFDEF PrintScreen
        get_vector 05h
        set_vector 05h, oldint05, int05handler
ENDIF

IFDEF ScrollLock
        get_vector 09h
        set_vector 09h, oldint09, int09handler
ENDIF
        mov cs:action, 0    ; Initialize flag
        clrscr              ; Set the new screen modus

        disp intro          ; Display messages
        disp okmessage

        keep_process 0, transient   ; Terminate the program resident

code    ends
end     start

; End of file RESMODE.ASM
