(* Print Ascii or Ansi text from StdIn as Postscript file to StdOut *)
MODULE PSPrint;

IMPORT Str, FIO, Lib;
FROM Storage IMPORT ALLOCATE, DEALLOCATE;
FROM EnvirOps IMPORT FindFile;
FROM MyStr    IMPORT PosCaps, Trim;

TYPE
  CS (* CharSet *)        = ( Cour, Symb, Graph, 
                              ProcInvNot, ProcOnehalf, ProcOnequarter,
                              ProcIndex, ProcThreequarters);
  CC (* CharConversion *) = RECORD
                              ch: CHAR; (* char to convert to *)
                              cs: CS;   (* charset to convert to *)
                            END;
  CM (* CharMap *)        = ARRAY CHAR OF CC;

  ParamSetMap             = SET OF CS;

CONST
  HasParams  = ParamSetMap {Cour, Symb, Graph, ProcIndex};
  CRLF       = 15C+12C;
  DefMaxlines= 62;
  DefMaxcols = 80;
  DefLMargin = 36;
  DefTMargin = 30;
  DefSpacing = 11;
  DefFontsize= 10;
  HeaderFile = 'psheader.ps';
  SwitchChars= Str.CHARSET{'/','-'};
  NeedEscape= Str.CHARSET{'(',')','\'};
  EscapeChar= '\';
  (* two character mappings are supported: IBM_437 and
     Windows_ANSI 1 *)
  IBM_437 = CM(
    CC( ' ', Cour ),  (* 0C *)
    CC( ' ', Cour ),  (* 1C, Smiley *)
    CC( ' ', Cour ),  (* 2C, inverted Smiley *)
    CC( 251C,Symb ),  (* 3C, heart *)
    CC( 250C,Symb ),  (* 4C, diamond *)
    CC( 247C,Symb ),  (* 5C, club *)
    CC( 252C,Symb ),  (* 6C, spade *)
    CC( 267C,Symb ),  (* 7C, bullet as filled circle *)
    CC( ' ', Cour ),  (* 10C block with hollow circle *)
    CC( ' ', Cour ),  (* 11C circle *)
    CC( ' ', Cour ),  (* 12C block with circle *)
    CC( ' ', Cour ),  (* 13C male symbol *)
    CC( ' ', Cour ),  (* 14C female symbol *)
    CC( ' ', Cour ),  (* 15C single note *)
    CC( ' ', Cour ),  (* 16C double note *)
    CC( ' ', Cour ),  (* 17C spoked wheel *)
    CC( 20C, Graph),  (* 20C triangle right *)
    CC( 21C, Graph),  (* 21C triangle left *)
    CC( ' ', Cour ),  (* 22C vertical doubleheaded arrow *)
    CC( ' ', Cour ),  (* 23C double exclamation mark *)
    CC( 266C,Cour ),  (* 24C, paragraph mark *)
    CC( 236C,Cour ),  (* 25C, section, german paragraph *)
    CC( 334C,Graph),  (* 26C  bottom block *)
    CC( ' ', Cour ),  (* 27C vertical doubleheaded arrow with underbar *)
    CC( 255C,Symb ),  (* 30C, up arrow *)
    CC( 257C,Symb ),  (* 31C, down arrow *)
    CC( 256C,Symb ),  (* 32C, right arrow *)
    CC( 254C,Symb ),  (* 33C, left arrow *)
    CC( 300C,Graph),  (* 34C  lower left corner *)
    CC( 253C,Symb ),  (* 35C, doubleheaded arrow horizontal *)
    CC( 36C, Graph),  (* 36C triangle up *)
    CC( 37C, Graph),  (* 37C triangle down *)
    CC( ' ', Cour ),  (* Space *)
    CC( '!', Cour ),  (* ! *)
    CC( '"', Cour ),  (* " *)
    CC( '#', Cour ),  (* # *)
    CC( '$', Cour ),  (* $ *)
    CC( '%', Cour ),  (* % *)
    CC( '&', Cour ),  (* & *)
    CC( 47C, Cour ),  (* ' *)
    CC( '(', Cour ),  (* ( *)
    CC( ')', Cour ),  (* ) *)
    CC( '*', Cour ),  (* * *)
    CC( '+', Cour ),  (* + *)
    CC( ',', Cour ),  (* , *)
    CC( '-', Cour ),  (* - *)
    CC( '.', Cour ),  (* . *)
    CC( '/', Cour ),  (* / *)
    CC( '0', Cour ),  (* 0 *)
    CC( '1', Cour ),  (* 1 *)
    CC( '2', Cour ),  (* 2 *)
    CC( '3', Cour ),  (* 3 *)
    CC( '4', Cour ),  (* 4 *)
    CC( '5', Cour ),  (* 5 *)
    CC( '6', Cour ),  (* 6 *)
    CC( '7', Cour ),  (* 7 *)
    CC( '8', Cour ),  (* 8 *)
    CC( '9', Cour ),  (* 9 *)
    CC( ':', Cour ),  (* : *)
    CC( ';', Cour ),  (* ; *)
    CC( '<', Cour ),  (* < *)
    CC( '=', Cour ),  (* = *)
    CC( '>', Cour ),  (* > *)
    CC( '?', Cour ),  (* ? *)
    CC( '@', Cour ),  (* @ *)
    CC( 'A', Cour ),  (* A *)
    CC( 'B', Cour ),  (* B *)
    CC( 'C', Cour ),  (* C *)
    CC( 'D', Cour ),  (* D *)
    CC( 'E', Cour ),  (* E *)
    CC( 'F', Cour ),  (* F *)
    CC( 'G', Cour ),  (* G *)
    CC( 'H', Cour ),  (* H *)
    CC( 'I', Cour ),  (* I *)
    CC( 'J', Cour ),  (* J *)
    CC( 'K', Cour ),  (* K *)
    CC( 'L', Cour ),  (* L *)
    CC( 'M', Cour ),  (* M *)
    CC( 'N', Cour ),  (* N *)
    CC( 'O', Cour ),  (* O *)
    CC( 'P', Cour ),  (* P *)
    CC( 'Q', Cour ),  (* Q *)
    CC( 'R', Cour ),  (* R *)
    CC( 'S', Cour ),  (* S *)
    CC( 'T', Cour ),  (* T *)
    CC( 'U', Cour ),  (* U *)
    CC( 'V', Cour ),  (* V *)
    CC( 'W', Cour ),  (* W *)
    CC( 'X', Cour ),  (* X *)
    CC( 'Y', Cour ),  (* Y *)
    CC( 'Z', Cour ),  (* Z *)
    CC( '[', Cour ),  (* [ *)
    CC( '\', Cour ),  (* \ *)
    CC( ']', Cour ),  (* ] *)
    CC( '^', Cour ),  (* ^ *)
    CC( '_', Cour ),  (* _ *)
    CC( '`', Cour ),  (* ` *)
    CC( 'a', Cour ),  (* a *)
    CC( 'b', Cour ),  (* b *)
    CC( 'c', Cour ),  (* c *)
    CC( 'd', Cour ),  (* d *)
    CC( 'e', Cour ),  (* e *)
    CC( 'f', Cour ),  (* f *)
    CC( 'g', Cour ),  (* g *)
    CC( 'h', Cour ),  (* h *)
    CC( 'i', Cour ),  (* i *)
    CC( 'j', Cour ),  (* j *)
    CC( 'k', Cour ),  (* k *)
    CC( 'l', Cour ),  (* l *)
    CC( 'm', Cour ),  (* m *)
    CC( 'n', Cour ),  (* n *)
    CC( 'o', Cour ),  (* o *)
    CC( 'p', Cour ),  (* p *)
    CC( 'q', Cour ),  (* q *)
    CC( 'r', Cour ),  (* r *)
    CC( 's', Cour ),  (* s *)
    CC( 't', Cour ),  (* t *)
    CC( 'u', Cour ),  (* u *)
    CC( 'v', Cour ),  (* v *)
    CC( 'w', Cour ),  (* w *)
    CC( 'x', Cour ),  (* x *)
    CC( 'y', Cour ),  (* y *)
    CC( 'z', Cour ),  (* z *)
    CC( '{', Cour ),  (* { *)
    CC( '|', Cour ),  (* | *)
    CC( '}', Cour ),  (* } *)
    CC( '~', Cour ),  (* ~ *)
    CC( ' ', Cour ),  (*  *)
    CC( '', Cour ),  (* , this and the following national language chars *)
    CC( '', Cour ),  (*   are reencoded into the Courier charset in the *)
    CC( '', Cour ),  (*   postscript header file we use *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( ' ', ProcInvNot ),  (*  *)
    CC( 330C,Symb ),  (*  *)
    CC( ' ', ProcOnehalf ),  (*  *)
    CC( ' ', ProcOnequarter ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Cour ),  (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( '', Graph ), (*  *)
    CC( 'a', Symb ),  (*  *)
    CC( 373C,Cour ),  (*  *)
    CC( 'G', Symb ),  (*  *)
    CC( 'p', Symb ),  (*  *)
    CC( 'S', Symb ),  (*  *)
    CC( 's', Symb ),  (*  *)
    CC( 'm', Symb ),  (*  *)
    CC( 't', Symb ),  (*  *)
    CC( 'F', Symb ),  (*  *)
    CC( 'Q', Symb ),  (*  *)
    CC( 'W', Symb ),  (*  *)
    CC( 'd', Symb ),  (*  *)
    CC( 245C,Symb ),  (*  *)
    CC( 306C,Symb ),  (*  *)
    CC( 316C,Symb ),  (*  *)
    CC( 307C,Symb ),  (*  *)
    CC( 272C,Symb ),  (*  *)
    CC( 261C,Symb ),  (*  *)
    CC( 263C,Symb ),  (*  *)
    CC( 243C,Symb ),  (*  *)
    CC( 363C,Symb ),  (*  *)
    CC( 365C,Symb ),  (*  *)
    CC( 270C,Symb ),  (*  *)
    CC( 273C,Symb ),  (*  *)
    CC( 260C,Symb ),  (*  *)
    CC( 267C,Symb ),  (*  *)
    CC( 327C,Symb ),  (*  *)
    CC( 326C,Symb ),  (*  *)
    CC( 'n', ProcIndex ),  (*  *)
    CC( '2', ProcIndex ),  (*  *)
    CC( ' ', Cour ),  (*  *)
    CC( '', Cour )   (*  *)
    );

  Windows_ANSI = CM(
    CC( ' ', Cour ),  (* 0C *)
    CC( ' ', Cour ),  (* 1C *)
    CC( ' ', Cour ),  (* 2C *)
    CC( ' ', Cour ),  (* 3C *)
    CC( ' ', Cour ),  (* 4C *)
    CC( ' ', Cour ),  (* 5C *)
    CC( ' ', Cour ),  (* 6C *)
    CC( ' ', Cour ),  (* 7C *)
    CC( ' ', Cour ),  (* 10C *)
    CC( ' ', Cour ),  (* 11C *)
    CC( ' ', Cour ),  (* 12C *)
    CC( ' ', Cour ),  (* 13C *)
    CC( ' ', Cour ),  (* 14C *)
    CC( ' ', Cour ),  (* 15C *)
    CC( ' ', Cour ),  (* 16C *)
    CC( ' ', Cour ),  (* 17C *)
    CC( ' ', Cour ),  (* 20C *)
    CC( ' ', Cour ),  (* 21C *)
    CC( ' ', Cour ),  (* 22C *)
    CC( ' ', Cour ),  (* 23C *)
    CC( ' ', Cour ),  (* 24C *)
    CC( ' ', Cour ),  (* 25C *)
    CC( ' ', Cour ),  (* 26C *)
    CC( ' ', Cour ),  (* 27C *)
    CC( ' ', Cour ),  (* 30C *)
    CC( ' ', Cour ),  (* 31C *)
    CC( ' ', Cour ),  (* 32C *)
    CC( ' ', Cour ),  (* 33C *)
    CC( ' ', Cour ),  (* 34C *)
    CC( ' ', Cour ),  (* 35C *)
    CC( ' ', Cour ),  (* 36C *)
    CC( ' ', Cour ),  (* 37C *)
    CC( ' ', Cour ),  (* Space *)
    CC( '!', Cour ),  (* ! *)
    CC( '"', Cour ),  (* " *)
    CC( '#', Cour ),  (* # *)
    CC( '$', Cour ),  (* $ *)
    CC( '%', Cour ),  (* % *)
    CC( '&', Cour ),  (* & *)
    CC( 47C, Cour ),  (* ' *)
    CC( '(', Cour ),  (* ( *)
    CC( ')', Cour ),  (* ) *)
    CC( '*', Cour ),  (* * *)
    CC( '+', Cour ),  (* + *)
    CC( ',', Cour ),  (* , *)
    CC( '-', Cour ),  (* - *)
    CC( '.', Cour ),  (* . *)
    CC( '/', Cour ),  (* / *)
    CC( '0', Cour ),  (* 0 *)
    CC( '1', Cour ),  (* 1 *)
    CC( '2', Cour ),  (* 2 *)
    CC( '3', Cour ),  (* 3 *)
    CC( '4', Cour ),  (* 4 *)
    CC( '5', Cour ),  (* 5 *)
    CC( '6', Cour ),  (* 6 *)
    CC( '7', Cour ),  (* 7 *)
    CC( '8', Cour ),  (* 8 *)
    CC( '9', Cour ),  (* 9 *)
    CC( ':', Cour ),  (* : *)
    CC( ';', Cour ),  (* ; *)
    CC( '<', Cour ),  (* < *)
    CC( '=', Cour ),  (* = *)
    CC( '>', Cour ),  (* > *)
    CC( '?', Cour ),  (* ? *)
    CC( '@', Cour ),  (* @ *)
    CC( 'A', Cour ),  (* A *)
    CC( 'B', Cour ),  (* B *)
    CC( 'C', Cour ),  (* C *)
    CC( 'D', Cour ),  (* D *)
    CC( 'E', Cour ),  (* E *)
    CC( 'F', Cour ),  (* F *)
    CC( 'G', Cour ),  (* G *)
    CC( 'H', Cour ),  (* H *)
    CC( 'I', Cour ),  (* I *)
    CC( 'J', Cour ),  (* J *)
    CC( 'K', Cour ),  (* K *)
    CC( 'L', Cour ),  (* L *)
    CC( 'M', Cour ),  (* M *)
    CC( 'N', Cour ),  (* N *)
    CC( 'O', Cour ),  (* O *)
    CC( 'P', Cour ),  (* P *)
    CC( 'Q', Cour ),  (* Q *)
    CC( 'R', Cour ),  (* R *)
    CC( 'S', Cour ),  (* S *)
    CC( 'T', Cour ),  (* T *)
    CC( 'U', Cour ),  (* U *)
    CC( 'V', Cour ),  (* V *)
    CC( 'W', Cour ),  (* W *)
    CC( 'X', Cour ),  (* X *)
    CC( 'Y', Cour ),  (* Y *)
    CC( 'Z', Cour ),  (* Z *)
    CC( '[', Cour ),  (* [ *)
    CC( '\', Cour ),  (* \ *)
    CC( ']', Cour ),  (* ] *)
    CC( '^', Cour ),  (* ^ *)
    CC( '_', Cour ),  (* _ *)
    CC( '`', Cour ),  (* ` *)
    CC( 'a', Cour ),  (* a *)
    CC( 'b', Cour ),  (* b *)
    CC( 'c', Cour ),  (* c *)
    CC( 'd', Cour ),  (* d *)
    CC( 'e', Cour ),  (* e *)
    CC( 'f', Cour ),  (* f *)
    CC( 'g', Cour ),  (* g *)
    CC( 'h', Cour ),  (* h *)
    CC( 'i', Cour ),  (* i *)
    CC( 'j', Cour ),  (* j *)
    CC( 'k', Cour ),  (* k *)
    CC( 'l', Cour ),  (* l *)
    CC( 'm', Cour ),  (* m *)
    CC( 'n', Cour ),  (* n *)
    CC( 'o', Cour ),  (* o *)
    CC( 'p', Cour ),  (* p *)
    CC( 'q', Cour ),  (* q *)
    CC( 'r', Cour ),  (* r *)
    CC( 's', Cour ),  (* s *)
    CC( 't', Cour ),  (* t *)
    CC( 'u', Cour ),  (* u *)
    CC( 'v', Cour ),  (* v *)
    CC( 'w', Cour ),  (* w *)
    CC( 'x', Cour ),  (* x *)
    CC( 'y', Cour ),  (* y *)
    CC( 'z', Cour ),  (* z *)
    CC( '{', Cour ),  (* { *)
    CC( '|', Cour ),  (* | *)
    CC( '}', Cour ),  (* } *)
    CC( '~', Cour ),  (* ~ *)
    CC( ' ', Cour ),  (*  *)
    (* the following special chars are reencoded into the Courier charset 
       in the postscript header file we use *)
    CC( ' ', Cour ),  (* 128   *)
    CC( ' ', Cour ),  (* 129   *)
    CC( 270C,Cour ),  (* 130 quotesinglbase  *)
    CC( 237C,Cour ),  (* 131 florin  *)
    CC( 271C,Cour ),  (* 132 quotedblbase *)
    CC( 274C,Cour ),  (* 133 ellipsis *)
    CC( 262C,Cour ),  (* 134 dagger *)
    CC( 263C,Cour ),  (* 135 daggerdbl *)
    CC( 303C,Cour ),  (* 136 circumflex ^ *)
    CC( 275C,Cour ),  (* 137 perthousand *)
    CC( 325C,Cour ),  (* 138 Scaron *)
    CC( 323C,Cour ),  (* 139 guilsinglleft < *)
    CC( 352C,Cour ),  (* 140 OE ligatur *)
    CC( ' ', Cour ),  (* 141 *)
    CC( ' ', Cour ),  (* 142 *)
    CC( ' ', Cour ),  (* 143 *)
    CC( ' ', Cour ),  (* 144 *)
    CC( 047C,Cour ),  (* 145 quoteright *)
    CC( '`', Cour ),  (* 146 quoteleft *)
    CC( 272C,Cour ),  (* 147 quotedblright *)
    CC( 322C,Cour ),  (* 148 quotedblleft *)
    CC( 267C,Cour ),  (* 149 bullet *)
    CC( 261C,Cour ),  (* 150 endash *)
    CC( 320C,Cour ),  (* 151 emdash *)
    CC( 304C,Cour ),  (* 152 ~ tilde *)
    CC( 324C,Symb ),  (* 153 Trademark *)
    CC( 326C,Cour ),  (* 154 scaron *)
    CC( 324C,Cour ),  (* 155 guilsinglright > *)
    CC( 372C,Cour ),  (* 156 oe ligatur *)
    CC( ' ', Cour ),  (* 157 *)
    CC( ' ', Cour ),  (* 158 *)
    CC( 327C,Cour ),  (* 159 Ydieresis *)
    CC( ' ', Cour ),  (* 160  *)
    CC( '', Cour ),  (* 161 exclamdown *)
    CC( '', Cour ),  (* 162 cent *)
    CC( '', Cour ),  (* 163 sterling *)
    CC( 330C,Cour ),  (* 164 currency *)
    CC( '', Cour ),  (* 165 yen *)
    CC( 246C,Graph),  (* 166 BrokenBar *)
    CC( 236C,Cour ),  (* 167 section *)
    CC( 310C,Cour ),  (* 168 dieresis *)
    CC( 323C,Symb ),  (* 169 copyright *)
    CC( '', Cour ),  (* 170 ordfeminine *)
    CC( '', Cour ),  (* 171 guillemotleft *)
    CC( 330C,Symb ),  (* 172 logicalnot *)
    CC( '-', Cour ),  (* 173 dash (?) *)
    CC( 322C,Symb ),  (* 174 registered *)
    CC( 305C,Cour ),  (* 175 macron *)
    CC( 312C,Cour ),  (* 176 ring *)
    CC( 261C,Symb ),  (* 177 plusminus *)
    CC( '2', ProcIndex ),  (* 178 index 2 *)
    CC( '3', ProcIndex ),  (* 179 index 3 *)
    CC( 302C,Cour ),  (* 180 acute acent *)
    CC( 'm', Symb ),  (* 181 my *)
    CC( 266C,Cour ),  (* 182 paragraph  *)
    CC( 327C,Symb ),  (* 183 dotmath *)
    CC( 313C,Cour ),  (* 184 cedilla acent *)
    CC( '1', ProcIndex ),  (* 185 index 1 *)
    CC( 260C,Symb ),  (* 186 degree *)
    CC( '', Cour ),  (* 187 guillemotright *)
    CC( ' ', ProcOnequarter ),  (* 188 onequarter  *)
    CC( ' ', ProcOnehalf ),  (* 189 onehalf  *)
    CC( ' ', ProcThreequarters ),  (* 190 threequarters *)
    CC( '', Cour ),  (* 191 questiondown *)
    CC( 331C,Cour ),  (* 192 Agrave *)
    CC( 332C,Cour ),  (* 193 Aacute *)
    CC( 333C,Cour ),  (* 194 Acircumflex *)
    CC( 334C,Cour ),  (* 195 Atilde *)
    CC( '', Cour ),  (* 196 Adieresis *)
    CC( '', Cour ),  (* 197 Aring *)
    CC( '', Cour ),  (* 198 AE ligatur *)
    CC( '', Cour ),  (* 199 Ccedilla *)
    CC( 335C,Cour ),  (* 200 Egrave *)
    CC( '', Cour ),  (* 201 Eacute *)
    CC( 336C,Cour ),  (* 202 Ecircumflex *)
    CC( 337C,Cour ),  (* 203 Edieresis *)
    CC( 340C,Cour ),  (* 204 Igrave *)
    CC( 341C,Cour ),  (* 205 Iacute *)
    CC( 342C,Cour ),  (* 206 Icircumflex *)
    CC( 343C,Cour ),  (* 207 Idieresis *)
    CC( ' ', Cour ),  (* 208 old english Eth *)
    CC( '', Cour ),  (* 209 Ntilde *)
    CC( 344C,Cour ),  (* 210 Ograve *)
    CC( 345C,Cour ),  (* 211 Oacute *)
    CC( 346C,Cour ),  (* 212 Ocircumflex *)
    CC( 347C,Cour ),  (* 213 Otilde *)
    CC( '', Cour ),  (* 214 Odieresis *)
    CC( 264C,Symb ),  (* 215 multiply *)
    CC( 351C,Cour ),  (* 216 Oslash *)
    CC( 353C,Cour ),  (* 217 Ugrave *)
    CC( 354C,Cour ),  (* 218 Uacute *)
    CC( 355C,Cour ),  (* 219 Ucircumflex *)
    CC( '', Cour ),  (* 220 Udieresis *)
    CC( ' ', Cour ),  (* 221 Yacute *)
    CC( ' ', Cour ),  (* 222 old english Thorn *)
    CC( 373C,Cour ),  (* 223 germandbls  *)
    CC( '', Cour ),  (* 224 agrave *)
    CC( '', Cour ),  (* 225 aacute *)
    CC( '', Cour ),  (* 226 acircumflex *)
    CC( 356C,Cour ),  (* 227 atilde *)
    CC( '', Cour ),  (* 228 adieresis *)
    CC( '', Cour ),  (* 229 aring *)
    CC( '', Cour ),  (* 230 ae ligatur *)
    CC( '', Cour ),  (* 231 ccedilla *)
    CC( '', Cour ),  (* 232 egrave *)
    CC( '', Cour ),  (* 233 eacute *)
    CC( '', Cour ),  (* 234 ecircumflex *)
    CC( '', Cour ),  (* 235 edieresis *)
    CC( '', Cour ),  (* 236 igrave *)
    CC( '', Cour ),  (* 237 iacute *)
    CC( '', Cour ),  (* 238 icircumflex *)
    CC( '', Cour ),  (* 239 idieresis *)
    CC( ' ', Cour ),  (* 240 old english eth *)
    CC( '', Cour ),  (* 241 ntilde *)
    CC( '', Cour ),  (* 242 ograve *)
    CC( '', Cour ),  (* 243 oacute *)
    CC( '', Cour ),  (* 244 ocircumflex *)
    CC( 357C,Cour ),  (* 245 otilde *)
    CC( '', Cour ),  (* 246 odieresis *)
    CC( 270C,Symb ),  (* 247 divide  *)
    CC( 371C,Cour ),  (* 248 oslash *)
    CC( '', Cour ),  (* 249 ugrave *)
    CC( '', Cour ),  (* 250 uacute *)
    CC( '', Cour ),  (* 251 ucircumflex *)
    CC( '', Cour ),  (* 252 udieresis *)
    CC( ' ', Cour ),  (* 253 yacute *)
    CC( ' ', Cour ),  (* 254 old englsih thorn *)
    CC( '', Cour )   (* 255 ydieresis *)
    );
TYPE
  PFILEENTRY= POINTER TO FILEENTRY;
  FILEENTRY = RECORD
                fname: ARRAY [0..72] OF CHAR;
                next : PFILEENTRY;
              END;
VAR
  infile, outfile: FIO.File;
  sendHeader     : BOOLEAN; 
  Filestack      : PFILEENTRY;
  Charmap        : CM;
  maxlines, maxcol : CARDINAL; 
  lmargin, tmargin, spacing, fontsize: CARDINAL; 

PROCEDURE Push( s: ARRAY OF CHAR );
  VAR
    p: PFILEENTRY;
  BEGIN (* AddToStack *)
    NEW(p);
    Str.Copy(p^.fname, s);
    p^.next := Filestack;
    Filestack := p;
  END Push;

PROCEDURE Pop( VAR s: ARRAY OF CHAR ): BOOLEAN;
  VAR p: PFILEENTRY;
  BEGIN (* Pop *)
    IF Filestack # NIL THEN
      p := Filestack;
      Str.Copy( s, p^.fname );
      Filestack := p^.next;
      DISPOSE(p);
      RETURN TRUE;
    ELSE
      RETURN FALSE;
    END;
  END Pop;

PROCEDURE Error( errparm, errdesc: ARRAY OF CHAR );
  CONST
    f::= FIO.ErrorOutput;
  BEGIN (* Error *)
    FIO.WrChar( f, 7C );
    FIO.WrStr( f, errparm );
    FIO.WrStr( f, ': ');
    FIO.WrStr( f, errdesc );
    FIO.WrLn( f );
  END Error;

PROCEDURE GiveInfo();
  CONST
    Info =
(*%T GERMAN *)
'PSPRINT: Wandelt ASCII- oder ANSI-Textfiles in Postscriptfiles um.'+CRLF+
12C+
'Syntax: psprint [{/|-}{?|h|n|p|a|t|l|f|s|z|c}] [filename [filename]...]'+CRLF+
12C+
'Gro- oder Kleinschreibung ist beliebig. Als Prfix fr Schalter kann'+CRLF+
'wahlweise / oder - verwendet werden. Die Reihenfolge der Angaben ist'+CRLF+
'beliebig. Zwischen Schalter und zugehrigen Argumenten drfen keine Leer-'+CRLF+
'zeichen stehen, = und : sind erlaubt aber nicht notwendig.'+CRLF+
12C+
'Schalter:'+CRLF+
'  /?  zeigt diese Hilfsinformationen an'+CRLF+
'  /h  wie /?'+CRLF+
'  /n  verhindert die Ausgabe des Postscript-Headerfiles'+CRLF+
'  /p  lenkt die Ausgabe direkt auf den Drucker um.'+CRLF+
'  /a  nimmt Windows-ANSI anstelle von IBM-ASCII als Zeichensatz an.'+CRLF+
12C+
'  In den folgenden Zeilen steht ## fr eine ganze Zahl.'+CRLF+
'  /t=## Setzt den oberen Rand in Points (1/72 inch). Default= 30'+CRLF+
'  /l=## Setzt den linken Rand in Points (1/72 inch). Default= 36'+CRLF+
'  /f=## Setzt die Schriftgre in Points (1/72 inch). Default= 10'+CRLF+
'  /s=## Setzt den Zeilenabstand in Points (1/72 inch). Default= 11'+CRLF+
'  /z=## Setzt Zahl der gedruckten Zeilen pro Seite. Default= 62'+CRLF+
'  /c=## Setzt Zahl der gedruckten Zeichen pro Zeile. Default= 80'+CRLF+
12C+
'PSPRINT ist ein Filterprogramm, es liest den zu konvertierenden Text '+CRLF+
'normalerweise von der Standardeingabe und gibt das Postscriptfile auf der '+CRLF+
'Standardausgabe aus. Ein- und Ausgabe knnen so von der DOS-Kommandozeile aus'+CRLF+
'umgelenkt werden (z. B. dir | psprint /p druckt das Directorylisting auf'+CRLF+
'einem Postscript-Drucker aus). Man kann aber auch ein oder mehrere Datei-'+CRLF+
'namen fr umzuwandelnde Textfiles angeben.'+CRLF+
'Vorsicht beim ndern der Defaults fr Rand etc.! Es erfolgt keine Prfung der'+CRLF+
'gesetzten Werte, z.B. ob die geforderte Zahl von Zeichen auch in eine Zeile'+CRLF+
'pat. Wenn Sie Schriftgre und Zeilenabstand ndern mssen Sie die Zahl der'+CRLF+
'Zeichen pro Zeile und Zeilen pro Seite selbst anpassen!'+CRLF;
(*%E *)
(*%F GERMAN *)
'PSPRINT: Converts ASCII- or ANSI-textfiles to Postscript files.'+CRLF+
12C+
'Syntax: psprint [{/|-}{?|h|n|p|a|t|l|f|s|z|c}] [filename [filename]...]'+CRLF+
12C+
'Parameters are not case-sensitive. Both / or - can be used for switches.'+CRLF+
'Parameters and switches can be entered in any order.'+CRLF+
'No spaces are allowed between switches and their arguments, = and : are'+CRLF+
'allowed but not required.'+CRLF+
12C+
'Switches:'+CRLF+
'  /?  displays this help info'+CRLF+
'  /h  like /?'+CRLF+
'  /n  supresses output of the Postscript headerfiles'+CRLF+
'  /p  sends output to PRN instead of CON'+CRLF+
'  /a  assumes Windows-ANSI coding instead of IBM-ASCII (codepage 437)'+CRLF+
12C+
'  In the following lines ## represent a unsigned integer number.'+CRLF+
'  /t=## set top margin to ## points (1/72 inch). Default= 30'+CRLF+
'  /l=## set left margin to ## points (1/72 inch). Default= 36'+CRLF+
'  /f=## set character size to ## points (1/72 inch). Default= 10'+CRLF+
'  /s=## set linespacing to ## points (1/72 inch). Default= 11'+CRLF+
'  /z=## set number of printed lines per page. Default= 62'+CRLF+
'  /c=## set number of printed chars per line. Default= 80'+CRLF+
12C+
'PSPRINT is a DOS filter, it reads text from standard input or a file '+CRLF+
'and writes the postscript output to standard output (CON) or PRN.'+CRLF+
'Input and output can be redirected from the DOS command line.'+CRLF+
'E. g. "dir | psprint > prn" will print a directory listing on the postscipt'+CRLF+
'printer on device PRN). It is also possible to pass one or more filenames'+CRLF+
'to PRPRINT. The header file will only be send once in this case.'+CRLF+
'Take care if you change the defaults for margins etc.! The current version'+CRLF+
'of PSPRINT will not check if the values make sense.'+CRLF+
'If you change the character size and linespacing, you also have to change'+CRLF+
'the number of chars per line and lines per page to prevent loss of characters'+CRLF+
'during printing. Sorry for any inconvenience caused by this.';
(*%E *)
  BEGIN (* GiveInfo *)
    FIO.WrStr( FIO.StandardOutput, Info );
  END GiveInfo;

PROCEDURE SetValue( VAR value: CARDINAL; str: ARRAY OF CHAR );
  CONST
    Numbers = Str.CHARSET{'0'..'9'};
  VAR
    nstr: ARRAY [0..10] OF CHAR; 
    n, len   : CARDINAL; 
    ok  : BOOLEAN; 
  BEGIN (* SetValue *)
    (* move number to nstr *)
    n := 2;  (* skip switch *)
    len := Str.Length( str );
    nstr[0] := 0C;
    LOOP
      IF str[n] IN Numbers THEN EXIT END;
      INC( n );
      IF n >= len THEN EXIT END;
    END;
    IF n < len THEN
      LOOP
        IF str[n] IN Numbers THEN 
          Str.Append( nstr, str[n] );
        ELSE
          EXIT
        END;
        INC( n );
        IF n >= len THEN EXIT END;
      END;
      n := CARDINAL( Str.StrToCard( nstr, 10, ok ));
      IF ok THEN
        value := n
      ELSE
        Error( str, 'Keine ganze Zahl zur Basis 10!')
      END;
    ELSE
      Error( str, 'Zahlenangabe fehlt!');
    END;
  END SetValue;

PROCEDURE EvalParams():BOOLEAN;
  VAR
    n,i : CARDINAL;
    p : ARRAY [0..72] OF CHAR;
    fname_found: BOOLEAN;
  BEGIN (* EvalParams *)
    n := Lib.ParamCount();
    IF n > 0 THEN
      fname_found := FALSE;
      FOR i:= 1 TO n DO
        Lib.ParamStr( p, i );
        IF p[0] IN SwitchChars THEN
          Str.Lows( p );
          CASE p[1] OF 
            '?','h': GiveInfo; RETURN FALSE;
          | 'a'    : Charmap := Windows_ANSI;
          | 'c'    : SetValue( maxcol, p );
          | 'f'    : SetValue( fontsize, p );
          | 'l'    : SetValue( lmargin, p );
          | 'n'    : sendHeader := FALSE;
          | 'p'    : outfile := FIO.PrinterDevice;
          | 's'    : SetValue( spacing, p);
          | 't'    : SetValue( tmargin, p );
          | 'z'    : SetValue( maxlines, p );
          ELSE
            Error( p, 'unbekannter Schalter!');
            RETURN FALSE;
          END;
        ELSE
          (* should be a filename *)
          IF FIO.Exists( p ) THEN
            Push(p)
          ELSE
            Error( p, 'Datei nicht gefunden!');
            RETURN FALSE;
          END;
        END;
      END; (* FOR *)
    END;
    RETURN TRUE;
  END EvalParams;

PROCEDURE Defaults();
  BEGIN (* Defaults *)
    infile := FIO.StandardInput;
    outfile:= FIO.StandardOutput;
    sendHeader := TRUE;
    Filestack  := NIL;
    Charmap    := IBM_437;
    maxlines   := DefMaxlines;
    maxcol     := DefMaxcols;
    lmargin    := DefLMargin;
    tmargin    := DefTMargin;
    spacing    := DefSpacing;
    fontsize   := DefFontsize;
  END Defaults;

(*# save,  call( o_a_copy=>off, o_a_size=>on )*)
PROCEDURE SetPSDefaults();
  PROCEDURE PSDef( varname: ARRAY OF CHAR; value: CARDINAL );
    BEGIN (* PSDef *)
      FIO.WrStr( outfile, varname );
      FIO.WrChar( outfile, ' ');
      FIO.WrCard( outfile, value, 0 );
      FIO.WrStr(outfile, ' def'+CRLF);
    END PSDef;
  BEGIN (* SetPSDefaults *)
    IF lmargin # DefLMargin THEN
      PSDef( '/Lmargin', lmargin );
    END;
    IF tmargin # DefTMargin THEN
      PSDef( '/Tmargin', tmargin );
    END;
    IF spacing # DefSpacing THEN
      PSDef( '/Lspacing', spacing );
    END;
    IF fontsize # DefFontsize THEN
      PSDef( '/CharSize', fontsize );
    END;
  END SetPSDefaults;

PROCEDURE InitPS( fname: ARRAY OF CHAR );
  BEGIN (* InitPS *)
    FIO.WrStr( outfile, '%%FileName: ');
    FIO.WrStr( outfile, fname );
    FIO.WrLn( outfile );
    FIO.WrStr( outfile, '/mainVM save def '+CRLF+'Defaults'+CRLF );
    SetPSDefaults;
    FIO.WrStr( outfile, 'Normal GOTOP'+CRLF );
  END InitPS;

PROCEDURE ExitPS( lineno: CARDINAL );
  BEGIN (* ExitPS *)
    IF lineno > 1 THEN
      FIO.WrStr( outfile, ' Page ');
      FIO.WrLn( outfile );
    END;
    FIO.WrStr( outfile, 'mainVM restore'+CRLF+'%%EndFile'+CRLF );
  END ExitPS;

PROCEDURE Page( no: CARDINAL; eject: BOOLEAN );
  BEGIN (* Page *)
    IF eject THEN
      FIO.WrStr( outfile, ' Page ');
    END;
    FIO.WrStr( outfile, CRLF+'%%Page: ');
    FIO.WrCard( outfile, no, 0 );
    FIO.WrLn( outfile );
  END Page;

PROCEDURE ChangeState( state: CS );
  BEGIN (* ChangeState *)
    CASE state OF
      Cour : FIO.WrStr( outfile, ') S');
    | Symb : FIO.WrStr( outfile, ') aSymbol');
    | Graph: FIO.WrStr( outfile, ') aIBMChar');
    | ProcInvNot : FIO.WrStr( outfile, ' invertednot');
    | ProcOnehalf: FIO.WrStr( outfile, ' onehalf');
    | ProcOnequarter: FIO.WrStr( outfile, ' onequarter');
    | ProcIndex: FIO.WrStr( outfile, ') indexshow');
    | ProcThreequarters: FIO.WrStr( outfile, ' threequarters');
    END;
    FIO.WrLn( outfile );
  END ChangeState;

PROCEDURE CheckOverflow( VAR lineno, pageno: CARDINAL );
  BEGIN (* CheckOverflow *)
    IF lineno > maxlines THEN
      INC( pageno );
      Page( pageno, TRUE );
      lineno := 1;
    END;
  END CheckOverflow;

PROCEDURE WrCharOktal( f: FIO.File; ch: CHAR );
  VAR
    oStr : ARRAY [0..4] OF CHAR; 
    ok   : BOOLEAN; 
  BEGIN (* WrCharOktal *)
    Str.CardToStr( LONGCARD(ORD(ch)), oStr, 8, ok );
    WHILE Str.Length( oStr ) < 3 DO
      Str.Insert( oStr, '0', 0 );
    END;
    Str.Insert( oStr, EscapeChar, 0 );
    FIO.WrStr( f, oStr );
  END WrCharOktal;

PROCEDURE BeginString( state: CS );
  BEGIN (* BeginString *)
    IF state IN HasParams THEN
      FIO.WrChar( outfile, '(');
    END;
  END BeginString;

PROCEDURE PrintLine( inline: ARRAY OF CHAR; lineno, pageno: CARDINAL );
  VAR 
    ch : CHAR;
    state, newstate: CS;
    n, len, col  : CARDINAL; 
  BEGIN (* PrintLine *)
    state := Cour;
    BeginString(state);
    col := 0;
    Trim( inline, FALSE, TRUE );
    len := Str.Length( inline );
    IF len > 0 THEN
      FOR n:= 0 TO len-1 DO
        ch := Charmap[ inline[n] ].ch;
        newstate := Charmap[ inline[n] ].cs;
        INC( col );
        IF col > maxcol THEN
          ChangeState( state );
          FIO.WrStr( outfile, ' NL'+CRLF);
          col := 1;
          INC( lineno );
          CheckOverflow( lineno, pageno );
          BeginString(state);
        END;
        IF newstate # state THEN
          ChangeState( state );
          state := newstate;
          BeginString(state);
        END;
        IF ch IN NeedEscape THEN
          FIO.WrChar( outfile, EscapeChar );
          FIO.WrChar( outfile, ch );
        ELSIF ch = 10C (* Backspace *) THEN
          ChangeState( state );
          FIO.WrStr( outfile, ' BS ');
          IF col > 1 THEN DEC( col ); END;
          BeginString(state);
        ELSIF ch = 11C (* Tab *) THEN
          FIO.WrStr( outfile, '\t' );
        ELSIF ch = 14C (* Formfeed *) THEN
          ChangeState( state );
          FIO.WrStr( outfile, ' NL'+CRLF);
          col := 1;
          lineno := 1;
          Page( pageno, TRUE );
          BeginString(state);
        ELSIF state IN HasParams THEN
          IF (ch > CHR(127)) OR (ch < ' ') THEN
            WrCharOktal( outfile, ch )
          ELSE
            FIO.WrChar( outfile, ch );
          END
        END;
      END;
    END;
    ChangeState( state );
    FIO.WrStr( outfile, ' NL'+CRLF);
  END PrintLine;

PROCEDURE Print( f: FIO.File; fname: ARRAY OF CHAR );
  VAR
    iline: ARRAY [0..256] OF CHAR; 
    pageno, lineno: CARDINAL; 
  BEGIN (* Print *)
    FIO.EOF := FALSE;
    InitPS( fname );
    pageno := 1;
    lineno := 0;
    Page( pageno, FALSE );
    WHILE NOT FIO.EOF DO
      FIO.RdStr( f, iline );
      INC( lineno );
      CheckOverflow( lineno, pageno );
      PrintLine( iline, lineno, pageno );
    END;
    ExitPS( lineno );
  END Print;

(*# restore *)

PROCEDURE PrintHeader();
  CONST
    (* second line of ps header file should look like this: *)
    Title = '%%Title: LitRef Header V. 1.0';
  VAR
    f    : FIO.File;
    pbuf : POINTER TO ARRAY [1..2000H+FIO.BufferOverhead] OF BYTE;
    ok   : BOOLEAN; 
    line : ARRAY [0..128] OF CHAR; 
    lno  : CARDINAL; 
  BEGIN (* PrintHeader *)
    IF sendHeader THEN
      IF FindFile( HeaderFile, line, ok ) THEN
        f := FIO.OpenRead( line );
        FIO.EOF := FALSE;
        pbuf := NIL;
        NEW( pbuf );
        IF pbuf # NIL THEN
          FIO.AssignBuffer( f, pbuf^ );
        END;
        lno := 0;
        ok := TRUE;
        WHILE NOT FIO.EOF AND ok DO
          FIO.RdStr( f, line );
          FIO.WrStr( outfile, line );
          INC( lno );
          IF (lno = 2) THEN
            (* The second line is always the Title and should thus 
               contain the strings %%Title and Lit. Check for this.
               to make shure we have the correct header file here. *)
            IF (PosCaps( line, '%%Title' ) = MAX(CARDINAL)) OR
               (PosCaps( line, 'Lit') = MAX(CARDINAL))
            THEN
              Error( line, 'Vermutlich falsches Headerfile!');
              ok := FALSE;
            END;
          END;
          IF Str.Length( line ) < SIZE(line) THEN
            FIO.WrLn( outfile );
          END;
        END; (* WHILE *)
        FIO.Close( f );
        IF pbuf # NIL THEN
          DISPOSE( pbuf )
        END;
      ELSE
        Error( HeaderFile, 'Postscript-Headerfile nicht gefunden!');
        HALT;
      END; (* IF FindFile *)
      sendHeader := FALSE;
    END; (* IF sendHeader *)
  END PrintHeader;

PROCEDURE PrintFiles();
  VAR
    fname: ARRAY [0..72] OF CHAR; 
    f    : FIO.File;
    pbuf : POINTER TO ARRAY [1..2000H+FIO.BufferOverhead] OF BYTE;
  BEGIN (* PrintFiles *)
    NEW( pbuf );
    WHILE Pop( fname ) DO
      f := FIO.OpenRead( fname );
      IF pbuf # NIL THEN
        FIO.AssignBuffer( f, pbuf^ );
      END;
      PrintHeader;  (* this will be executed only once! *)
      Print( f, fname );
      FIO.Close( f );
    END;
    IF pbuf # NIL THEN
      DISPOSE( pbuf );
    END;
  END PrintFiles;

BEGIN
  Defaults;
  IF EvalParams() THEN
    (* check StdIn first *)
    IF FIO.Size( infile ) > 0 THEN
      PrintHeader;
      Print( infile, 'Standardeingabe' );
    ELSIF Filestack = NIL THEN
      (* nothing to process *)
      GiveInfo;
    END;
    IF Filestack # NIL THEN
      PrintFiles;
    END;
  END;
END PSPrint.
