; Structural Steel Shape Drawing Program
; Original program by Jeffrey David Wells
; Pellham Phillips Hagerman Architects and Engineers
; Springfield, MO
; (417) 865-1672

; This program will draw all of the structural steel
; shapes listed in the American Institute of Steel 
; Construction's (AISC) Steel Construction Manual, either
; the ASD Ninth edition or the LRFD first edition.
; Dimensional data for the various steel shapes are stored
; in ".dim" files.  The program needs to be able to find 
; these files--either by locating them in the path via the 
; findfile function, or in the special path specification
; "c:/acad/steel".

; To operate the program, load it with the load function 
; and type "steel".  Then supply the name of the steel shape.
; Data entry is case-insensitive, but you cannot use leading
; zeros for decimal numbers.  Type "HELP" in response to
; the shape name prompt in order to see a list of all the
; available shape types and data entry format samples.  
; The program does not alter the drawing environment.  It
; simply draws the shape on the current layer and UCS.
; if you want it to modify setvars or set a layer, you will
; need to add the necessary code.
; It may not be pretty, but it works!  If you make 
; improvements or find mistakes, please upload them for the 
; author. 

;;; PROGRAM REVISIONS by
;;; Chris Bryant 10/5/92 [CIS 72570,1012]
;;; Nashville, TN

;A. Separated code into modular form for readability and
;   ease of modification.

;B. Fixed error handling.

;C. Final shape is a polyline created from shape lines/arcs.
;   This was done for ease of editing (fewer picks). If
;   after the shape is drawn it needs to be moved, rotated, etc.
;   a selection set of <P>revious and <L>ast immediately after
;   running STEEL are all that is needed to select the shape.

;D. Added option to hatch shape. Pattern used is ACAD's "steel".
;   Hatch scale is set by using the current "DIMSCALE". If a different
;   pattern or pattern scale is to be used edit function (hshp).

;E. Added code to place entities on A.I.A. CAD Guideline layers.
;   To use different layers, search and replace for:
;   steel shape ---> layer "A-SECT-MCUT"
;   steel hatch ---> layer "A-SECT-PATT"

;F. Screen flips back to graphics automatically upon entering
;   a shape designation after the help screen.

;G. Added function to show loading progress.

; One other change I have made to my own personal program file
; that is not included here was to shorten variable names to
; four characters or less to save on node space. I'm still
; using real mode AutoCAD (R10) and need to save memory where
; I can... I hope these changes help. 

;----------------------------------------------------------------
; Load-time entertainment
;----------------------------------------------------------------
(setq cnt 0)
 
(prompt "\nLoading STEEL.LSP") 
(defun dot () 
  (setq cnt (1+ cnt))
  (princ ".")
) 
;--------------------------------------
; Error handler
;--------------------------------------
(dot) 
(defun newerr (NE) 
  (if (/= NE "Function cancelled") 
    (princ (strcat "\nError: " NE))
  ) 
  (command ".LAYER" "S" clyr "") 
  (setvar "CMDECHO" cmde)
  (setvar "BLIPMODE" blip)
  (setq *error* olderr)
  (princ)
) 
;---------------------------------------
; Parse the file name from shape name
;---------------------------------------
(dot) 
(defun get_shape_file (shape_name / i char) 
  (setq i 0)
  (while (and (/= char "0") 
           (/= char "1") 
           (/= char "2") 
           (/= char "3") 
           (/= char "4") 
           (/= char "5") 
           (/= char "6") 
           (/= char "7") 
           (/= char "8") 
           (/= char "9") 
           (/= char ".") 
           (/= char "X") 
           (<= i (strlen shape_name))
         ) 
    (setq i (1+ i) 
          char (substr shape_name i 1))
  ) 
  (substr shape_name 1 (- i 1))
) 
;-----------------------------------------
; Read lines from file till shape is found
;-----------------------------------------
(dot) 
(defun read_line (file_name shape_name / exit_code) 
  (setq a (open file_name "r") 
        len (strlen shape_name))
  (while (/= exit_code 1) 
    (setq str (read-line a))
    (if
      (or
        (or
          (= str nil)
          (= str "")
        )
        (and
          (= (substr str 1 len) shape_name) 
          (= (substr str (1+ len) 1) " ")
        )                ;end or
      )                  ;end or   
      (setq exit_code 1)
    )                    ;end if
  )                      ;end while
  (close a) 
  (setq str str)
) 
;--------------------------------------
; Help screen
;---------------------------------------
(dot) 
(defun get_help () 
  (textscr) 
  (princ "\nThe following are valid shape designations:") 
  (princ "\n\n\tW44X285\t\tW Shapes") 
  (princ "\n\tM14X18\t\tM Shapes") 
  (princ "\n\tS24X121\t\tS Shapes") 
  (princ "\n\tHP14X117\tHP Shapes") 
  (princ "\n\tC15X50\t\tAmerican Standard Channels") 
  (princ "\n\tMC18X58\t\tMiscellaneous Channels") 
  (princ "\n\tL8X8X1.125\tAngles") 
  (princ "\n\tWT18X179.5\tStructural Tees from W Shapes") 
  (princ "\n\tMT7X9\t\tStructural Tees from M Shapes") 
  (princ "\n\tST12X60.5\tStructural Tees from S Shapes") 
  (princ "\n\tP.5\t\tStandard Weight Pipe") 
  (princ "\n\tPX.5\t\tExtra Strong Pipe") 
  (princ "\n\tPXX2\t\tDouble-Extra Strong Pipe") 
  (princ "\n\tTS16X16X.625\tStructural Tubing") 
  (princ "\n\nFractions must be entered as decimal numbers.") 
  (princ "\nIf the fraction is less than 1, do not include a ") 
  (princ "\nleading zero.\n")
) 
;end defun
;-----------------------------------------
; Make selection set of shape entities
;-----------------------------------------
(dot) 
(defun smake () 
  (setq sset (ssadd))
  (while (setq ent_name (entnext ent_name))
    (ssadd ent_name sset)
  )
) 
;-----------------------------------------
; Join entities into single polyline shape
;-----------------------------------------
(dot) 
(defun pjoin () 
  (command ".PEDIT" (entlast) "y" "J" sset "" "X") 
  (setq sset (entlast))
) 
;-----------------------------------------
; Hatch steel shape
;-----------------------------------------
(dot) 
(defun hshp (/ yn) 
  (initget 1 "Yes No") 
  (setq yn (getkword "\nHatch the shape (Y/N)? "))
  (if (= yn "Yes") 
    (progn
      (command ".LAYER" "M" "A-SECT-PATT" "") 
      (command ".HATCH" "steel" (/ (getvar "dimscale") 2) 0 sset "")
    )
  )
) 
;---------------------------------------
; Draw W,M or HP-shape
;---------------------------------------
(dot) 
(defun w_shape () 
  (setq d (atof (substr str 16 14)) 
        tw (atof (substr str 30 14)) 
        bf (atof (substr str 44 14)) 
        tf (atof (substr str 58 14)) 
        k (atof (substr str 72 14)) 
        r (- k tf) 
        pt1 insert_pt 
        pt2 (polar insert_pt rot bf) 
        pt3 (polar pt2 (+ rot (/ pi 2)) tf) 
        pt4 (polar pt3 (+ rot pi) (/ (- bf (+ (* 2 r) tw)) 2)) 
        pt5 (polar (polar pt4 (+ rot pi) r) (+ rot (/ pi 2)) r) 
        pt6 (polar pt5 (+ rot (/ pi 2)) (- d (* 2 k))) 
        pt7 (polar (polar pt6 (+ rot (/ pi 2)) r) rot r) 
        pt8 (polar pt7 rot (/ (- bf (+ (* 2 r) tw)) 2)) 
        pt9 (polar pt8 (+ rot (/ pi 2)) tf) 
        pt10 (polar pt9 (+ rot pi) bf) 
        pt11 (polar pt10 (- rot (/ pi 2)) tf) 
        pt12 (polar pt11 rot (/ (- bf (+ (* 2 r) tw)) 2)) 
        pt13 (polar (polar pt12 rot r) (- rot (/ pi 2)) r) 
        pt14 (polar pt13 (- rot (/ pi 2)) (- d (* 2 k))) 
        pt15 (polar (polar pt14 (- rot (/ pi 2)) r) (+ rot pi) r) 
        pt16 (polar pt15 (+ rot pi) (/ (- bf (+ (* 2 r) tw)) 2))) ;end setq
  (command "line" pt1 pt2 pt3 pt4 "" "arc" pt4 "e" pt5 "d" pt16 "line" pt5 pt6 "" 
         "arc" pt7 "e" pt6 "d" pt11 "line" pt7 pt8 pt9 pt10 pt11 pt12 "" "arc" 
           pt12 "e" pt13 "d" pt8 "line" pt13 pt14 "" "arc" pt15 "e" pt14 "d" pt3 
         "line" pt15 pt16 pt1 ""
  )                                  ;end command
  (smake) 
  (pjoin) 
  (hshp)
) 
;end defun 
;---------------------------------------
; Draw S-shape
;---------------------------------------
(dot) 
(defun s_shape () 
  (setq d (atof (substr str 16 14)) 
        tw (atof (substr str 30 14)) 
        bf (atof (substr str 44 14)) 
        tf (atof (substr str 58 14)) 
        k (atof (substr str 72 14)) 
        temp_dist1 (* (/ (- bf tw) 4.0) (/ 1 6.0)) 
        temp_dist2 (- k tf temp_dist1) 
        pt1 insert_pt)               ;end setq
  (if (<= (- tf temp_dist1) temp_dist2) 
    (setq pt2 pt1)
    (setq pt2 (polar pt1 (+ rot (/ pi 2.0)) (- tf (+ temp_dist1 temp_dist2))))
  ) 
  (setq temp_pt1 (polar pt1 (+ rot (/ pi 2.0)) (- tf temp_dist1)) 
        pt3 (polar temp_pt1 (+ rot 0.16514868) temp_dist2) 
        temp_pt2 (polar (polar temp_pt1 rot (/ (- bf tw) 2.0)) (+ rot (/ pi 2.0)) 
                        (* (/ (- bf tw) 2.0) (/ 1 6.0))
                 ) 
        pt4 (polar temp_pt2 (+ rot pi 0.16514868) temp_dist2) 
        pt5 (polar temp_pt2 (+ rot (/ pi 2.0)) temp_dist2) 
        pt6 (polar pt1 (+ rot (/ pi 2.0)) (/ d 2.0)) 
        pt7 (polar pt6 rot bf) 
        pt8 (polar pt1 rot (/ bf 2.0)) 
        pt9 (polar pt8 (+ rot (/ pi 2.0)) bf) 
        pt10 (polar pt5 (+ rot (/ pi 2.0)) (- d (* k 2.0))) 
        pt11 (polar pt1 rot bf))     ;end setq
  (command "line" pt1 pt2 "") 
  (setq ent_list (ssget "L"))
  (command "mirror" "L" "" pt8 pt9 "N") 
  (setq ent_list (ssadd (entlast) ent_list))
  (command "arc" pt3 "e" pt2 "d" temp_pt1) 
  (setq ent_list (ssadd (entlast) ent_list))
  (command "mirror" "L" "" pt8 pt9 "N") 
  (setq ent_list (ssadd (entlast) ent_list))
  (command "line" pt3 pt4 "") 
  (setq ent_list (ssadd (entlast) ent_list))
  (command "mirror" "L" "" pt8 pt9 "N") 
  (setq ent_list (ssadd (entlast) ent_list))
  (command "arc" pt5 "e" pt4 "d" temp_pt2) 
  (setq ent_list (ssadd (entlast) ent_list))
  (command "mirror" "L" "" pt8 pt9 "N") 
  (setq ent_list (ssadd (entlast) ent_list))
  (command "mirror" ent_list "" pt6 pt7 "N" "line" pt5 pt10 "" "mirror" "L" "" 
           pt8 pt9 "N" "line" pt1 pt11 "" "mirror" "L" "" pt6 pt7 "N"
  )                                  ;end command
  (smake) 
  (pjoin) 
  (hshp)
) 
;end defun
;---------------------------------------
; Draw C or MC-shape
;---------------------------------------
(dot) 
(defun c_shape () 
  (setq d (atof (substr str 16 14)) 
        tw (atof (substr str 30 14)) 
        bf (atof (substr str 44 14)) 
        tf (atof (substr str 58 14)) 
        k (atof (substr str 72 14)) 
        temp_dist1 (* (/ (- bf tw) 2.0) (/ 1 6.0)) 
        temp_dist2 (- k tf temp_dist1) 
        pt1 insert_pt)
  (if (<= (- tf temp_dist1) temp_dist2) 
    (setq pt2 pt1)
    (setq pt2 (polar pt1 (+ rot (/ pi 2.0)) (- tf (+ temp_dist1 temp_dist2))))
  ) 
  (setq temp_pt1 (polar pt1 (+ rot (/ pi 2.0)) (- tf temp_dist1)) 
        pt3 (polar temp_pt1 (+ rot 0.16514868) temp_dist2) 
        temp_pt2 (polar (polar temp_pt1 rot (- bf tw)) (+ rot (/ pi 2.0)) 
                        (* (- bf tw) (/ 1 6.0))
                 ) 
        pt4 (polar temp_pt2 (+ rot pi 0.16514868) temp_dist2) 
        pt5 (polar temp_pt2 (+ rot (/ pi 2.0)) temp_dist2) 
        pt6 (polar pt1 (+ rot (/ pi 2.0)) (/ d 2.0)) 
        pt7 (polar pt6 rot bf) 
        pt8 (polar pt5 (+ rot (/ pi 2.0)) (- d (* k 2.0))) 
        pt9 (polar pt1 rot bf) 
        pt10 (polar pt9 (+ rot (/ pi 2.0)) d)) ;end setq
  (command "line" pt1 pt2 "") 
  (setq ent_list (ssget "L"))
  (command "arc" pt3 "e" pt2 "d" temp_pt1) 
  (setq ent_list (ssadd (entlast) ent_list))
  (command "line" pt3 pt4 "") 
  (setq ent_list (ssadd (entlast) ent_list))
  (command "arc" pt5 "e" pt4 "d" temp_pt2) 
  (setq ent_list (ssadd (entlast) ent_list))
  (command "line" pt1 pt9 "") 
  (setq ent_list (ssadd (entlast) ent_list))
  (command "mirror" ent_list "" pt6 pt7 "N" "line" pt5 pt8 "" "line" pt9 pt10 "") ;end comman-
                                     ;d
  (smake) 
  (pjoin) 
  (hshp)
) 
;end defun
;---------------------------------------
; Draw L-shape
;---------------------------------------
(dot) 
(defun l_shape () 
  (setq l_vert (atof (substr str 30 14)) 
        l_horiz (atof (substr str 44 14)) 
        l_thick (atof (substr str 58 14)) 
        k (atof (substr str 72 14)) 
        pt1 insert_pt 
        pt2 (polar pt1 rot l_horiz) 
        pt3 (polar pt2 (+ rot (/ pi 2.0)) l_thick) 
        pt4 (polar pt3 (+ rot pi) (- l_horiz k)) 
        temp_pt1 (polar pt3 (+ rot pi) (- l_horiz l_thick)) 
        pt5 (polar temp_pt1 (+ rot (/ pi 2.0)) (- k l_thick)) 
        pt6 (polar pt5 (+ rot (/ pi 2.0)) (- l_vert k)) 
        pt7 (polar pt6 (+ rot pi) l_thick)) ;end setq
  (command "line" pt1 pt2 pt3 pt4 "" "arc" pt4 "e" pt5 "d" temp_pt1 "line" pt5 
           pt6 pt7 pt1 ""
  )                                  ;end command
  (smake) 
  (pjoin) 
  (hshp)
) 
;end defun
;---------------------------------------
; Draw WT or MT-shape
;---------------------------------------
(dot) 
(defun wt_shape () 
  (setq d (atof (substr str 16 14)) 
        tw (atof (substr str 30 14)) 
        bf (atof (substr str 44 14)) 
        tf (atof (substr str 58 14)) 
        k (atof (substr str 72 14)) 
        r (- k tf) 
        pt1 insert_pt 
        pt2 (polar insert_pt rot bf) 
        pt3 (polar pt2 (+ rot (/ pi 2)) tf) 
        pt4 (polar pt3 (+ rot pi) (/ (- bf (+ (* 2 r) tw)) 2)) 
        pt5 (polar (polar pt4 (+ rot pi) r) (+ rot (/ pi 2)) r) 
        pt6 (polar pt5 (+ rot (/ pi 2)) (- d k)) 
        pt7 (polar pt6 (+ rot pi) tw) 
        pt8 (polar pt7 (- rot (/ pi 2.0)) (- d k)) 
        pt9 (polar (polar pt8 (- rot (/ pi 2)) r) (+ rot pi) r) 
        pt10 (polar pt9 (+ rot pi) (/ (- bf (+ (* 2 r) tw)) 2))) ;end setq
  (command "line" pt1 pt2 pt3 pt4 "" "arc" pt4 "e" pt5 "d" pt10 "line" pt5 pt6 
           pt7 pt8 "" "arc" pt9 "e" pt8 "d" pt3 "line" pt9 pt10 pt1 ""
  )                                  ;end command
  (smake) 
  (pjoin) 
  (hshp)
) 
;end defun
;---------------------------------------
; Draw ST-shape
;---------------------------------------
(dot) 
(defun st_shape () 
  (setq d (atof (substr str 16 14)) 
        tw (atof (substr str 30 14)) 
        bf (atof (substr str 44 14)) 
        tf (atof (substr str 58 14)) 
        k (atof (substr str 72 14)) 
        temp_dist1 (* (/ (- bf tw) 4.0) (/ 1 6.0)) 
        temp_dist2 (- k tf temp_dist1) 
        pt1 insert_pt)
  (if (<= (- tf temp_dist1) temp_dist2) 
    (setq pt2 pt1)
    (setq pt2 (polar pt1 (+ rot (/ pi 2.0)) (- tf (+ temp_dist1 temp_dist2))))
  ) 
  (setq temp_pt1 (polar pt1 (+ rot (/ pi 2.0)) (- tf temp_dist1)) 
        pt3 (polar temp_pt1 (+ rot 0.16514868) temp_dist2) 
        temp_pt2 (polar (polar temp_pt1 rot (/ (- bf tw) 2.0)) (+ rot (/ pi 2.0)) 
                        (* (/ (- bf tw) 2.0) (/ 1 6.0))
                 ) 
        pt4 (polar temp_pt2 (+ rot pi 0.16514868) temp_dist2) 
        pt5 (polar temp_pt2 (+ rot (/ pi 2.0)) temp_dist2) 
        pt6 (polar pt1 rot (/ bf 2.0)) 
        pt7 (polar pt6 (+ rot (/ pi 2.0)) d) 
        pt8 (polar pt5 (+ rot (/ pi 2.0)) (- d k)) 
        pt9 (polar pt8 rot tw) 
        pt10 (polar pt1 rot bf))
  (command "line" pt1 pt2 "" "mirror" "L" "" pt6 pt7 "N" "arc" pt3 "e" pt2 "d" 
           temp_pt1 "mirror" "L" "" pt6 pt7 "N" "line" pt3 pt4 "" "mirror" "L" "" 
           pt6 pt7 "N" "arc" pt5 "e" pt4 "d" temp_pt2 "mirror" "L" "" pt6 pt7 "N" 
         "line" pt5 pt8 "" "mirror" "L" "" pt6 pt7 "N" "line" pt1 pt10 "" "line" 
           pt8 pt9 ""
  ) 
  (smake) 
  (pjoin) 
  (hshp)
) 
;end defun
;---------------------------------------
; Draw P,PX or PXX-shape
;---------------------------------------
(dot) 
(defun pipe () 
  (setq temp_dist1 (atof (substr str 16 14)) 
        temp_dist2 (atof (substr str 30 14)))
  (command "circle" insert_pt "d" temp_dist1 "circle" insert_pt "d" temp_dist2) 
  (smake) 
  (hshp)
) 
;end defun
;---------------------------------------
; Draw TS-shape
;---------------------------------------
(dot) 
(defun tube () 
  (setq l_vert (atof (substr str 30 14)) 
        l_horiz (atof (substr str 44 14)) 
        l_thick (atof (substr str 58 14)) 
        temp_pt1 (polar insert_pt rot l_horiz) 
        temp_pt2 (polar temp_pt1 (+ rot (/ pi 2.0)) l_vert) 
        temp_pt3 (polar insert_pt (+ rot (/ pi 2.0)) l_vert) 
        pt1 (polar insert_pt rot (* l_thick 2.0)) 
        pt2 (polar pt1 rot (- l_horiz (* l_thick 4.0))) 
        pt3 (polar (polar pt2 rot (* l_thick 2.0)) (+ rot (/ pi 2.0)) (* l_thick 
                                                                         2.0
                                                                      )
            ) 
        pt4 (polar pt3 (+ rot (/ pi 2.0)) (- l_vert (* l_thick 4.0))) 
        pt5 (polar pt2 (+ rot (/ pi 2.0)) l_vert) 
        pt6 (polar pt1 (+ rot (/ pi 2.0)) l_vert) 
        pt7 (polar pt4 (+ rot pi) l_horiz) 
        pt8 (polar pt3 (+ rot pi) l_horiz) 
        pt9 (polar pt1 (+ rot (/ pi 2.0)) l_thick) 
        pt10 (polar pt2 (+ rot (/ pi 2.0)) l_thick) 
        pt11 (polar pt3 (+ rot pi) l_thick) 
        pt12 (polar pt4 (+ rot pi) l_thick) 
        pt13 (polar pt5 (- rot (/ pi 2.0)) l_thick) 
        pt14 (polar pt6 (- rot (/ pi 2.0)) l_thick) 
        pt15 (polar pt7 rot l_thick) 
        pt16 (polar pt8 rot l_thick)) ;end setq
  (command "line" pt1 pt2 "" "arc" pt2 "e" pt3 "d" temp_pt1 "line" pt3 pt4 "" 
         "arc" pt4 "e" pt5 "d" temp_pt2 "line" pt5 pt6 "" "arc" pt6 "e" pt7 "d" 
           temp_pt3 "line" pt7 pt8 "" "arc" pt8 "e" pt1 "d" insert_pt
  )                                  ;end command
  (smake) 
  (pjoin) 
  (setq ent_name (entlast) 
        e2 ent_name)
  (setq temp_pt1 (inters pt9 pt10 pt11 pt12 nil) 
        temp_pt2 (inters pt11 pt12 pt13 pt14 nil) 
        temp_pt3 (inters pt13 pt14 pt15 pt16 nil) 
        temp_pt4 (inters pt15 pt16 pt9 pt10 nil))
  (command "line" pt9 pt10 "" "arc" pt10 "e" pt11 "d" temp_pt1 "line" pt11 pt12 
         "" "arc" pt12 "e" pt13 "d" temp_pt2 "line" pt13 pt14 "" "arc" pt14 "e" 
           pt15 "d" temp_pt3 "line" pt15 pt16 "" "arc" pt16 "e" pt9 "d" temp_pt4
  )                                  ;end command
  (smake) 
  (pjoin) 
  (setq sset (ssadd) 
        sset (ssadd (entlast) sset) 
        sset (ssadd e2 sset))
  (hshp)
) 
;end defun
;---------------------------------------
; STEEL command
;---------------------------------------
(dot) 
(defun C:STEEL (/ cmde blip clyr shape_name str file_name file_spec insert_pt
                  rot shape_type d tw bf tf k pt1 pt2 pt3 pt4 pt5 pt6 pt7 pt8
                  pt9 pt10 pt11 pt12 pt13 pt14 pt15 pt16 temp_dist1 temp_dist2
                  temp_pt1 temp_pt2 temp_pt3 temp_pt4 l_vert l_horiz l_thick
                  sset ent_name e2)

  ;--------------------
  ; set environment
  ;--------------------
  (setq cmde (getvar "CMDECHO") 
        blip (getvar "BLIPMODE") 
        clyr (getvar "CLAYER") 
        olderr *error* 
        *error* newerr)
  (setvar "CMDECHO" 0)
  (setvar "BLIPMODE" 0)
  ;----------------------
  ; get steel shape name
  ;----------------------
  (while (or (= shape_name nil) (= shape_name "") (= shape_name "HELP")) 
    (initget 1) 
    (setq shape_name (strcase (getstring "\nEnter <HELP> or shape designation: ")))
    (if (= shape_name "HELP") (get_help))
  )
  (graphscr)
  (setq shape_type (get_shape_file shape_name))
  (if
    (and
      (= (setq file_name (findfile (strcat shape_type ".dim")))
          nil
      )
      (= (setq file_spec (open (strcat "C:/ACAD/STEEL/" shape_type ".dim") "r"))
           nil
      )
    )
    (progn
      (prompt "\n\nERROR:  Could not open data file...")
      (prin1)
    )
    (progn
      (if (/= file_spec nil)
        (progn
          (close file_spec)
          (setq file_name (strcat "C:/ACAD/STEEL/" shape_type ".dim"))
        )
      )
      (setq str (read_line file_name shape_name))
      (if (or (= str nil) (= str ""))
        (prompt "\n\nERROR:  Shape not in data file...")
        (progn
          (while (= insert_pt nil)
            (setq insert_pt (getpoint "\nInsertion point: "))
          )
          (if (= (setq rot (getangle insert_pt "\nRotation angle <0>: "))
                 nil
              )
            (setq rot 0.0)
          )
          ;---------------------------------------
          ; Draw shapes
          ;---------------------------------------
          (setq ent_name (entlast))
          (command ".LAYER" "M" "A-SECT-MCUT" "")
          (cond
            ((or (= shape_type "W") (= shape_type "M") (= shape_type "HP"))
             (w_shape)
            )
            ((= shape_type "S") (s_shape))
            ((or (= shape_type "C") (= shape_type "MC")) (c_shape))
            ((= shape_type "L") (l_shape))
            ((or (= shape_type "WT") (= shape_type "MT")) (wt_shape))
            ((= shape_type "ST") (st_shape))
            ((or (= shape_type "P") (= shape_type "PX") (= shape_type "PXX"))
             (pipe)
            )
            ((= shape_type "TS") (tube))
          )                          ;end cond
        )                            ;end progn
      )                              ;end if
    )                                ;end progn
  )                                  ;end if
  (command ".LAYER" "S" clyr "")
  (setvar "CMDECHO" cmde)
  (setvar "BLIPMODE" blip)
  (setq *error* olderr)
  (prin1)
) 
(princ " Done.") 
(setq cnt nil dot nil)
(princ)
(C:STEEL)
