
         {*********************************************************}
         {*                  BTVCALC.PAS 1.00                     *}
         {*        Copyright (c) TurboPower Software 1993.        *}
         {*                 All rights reserved.                  *}
         {*********************************************************}

program BTVCalc;  {-Calculate posible section lengths for VREC}

 {
   Using the VRec B-Tree Filer unit to support variable sized records
   requires you to choose a section size that VRec will use when accessing
   your data. Several B-Tree users have requested an automated method for
   determining this size. This program won't completely automate this
   process, but it will releive you from performing the calculations. You
   will still need to decide which of the proposed section sizes best fits
   your applications needs (space savings, speed, etc.).

   Execute BTVCalc followed by the minimum size of your record and then the
   maximum size of your record. Optionally, you can force BTVCalc to display
   non-standard section sizes by providing a third parameter, (-2). By
   default, BTVCalc will display section sizes that are powers of two.

   If the display scrolls or you want a permanent record of the BTVCalc's
   output, you can use DOS redirection to redirect the output of BTVCalc.

   Example: BTVCALC 115 364 > TEMP.TXT    (send to a file)
            BTVCALC 115 364 > PRN         (send to a printer)
            BTVCALC 115 354 | MORE        (pipe to more program)

   If you execute BTVCalc with no parameters, a short help screen is displayed.


   Rules: (From the B-Tree Filer manual)

    1. The section length should be kept as small as is consistant with the
       remaining rules
    2. The section length should typically be a multiple of 2
    3. The largest record shouldn't be much larger than about 8 sections
    4. A section shouldn't be much larger than the smallest version of the
       record
    5. The section length should account for the fact that VREC uses 6 bytes
       of the first section and 7 bytes of each subsequent section
 }

uses
  Crt;

var
  i                : Integer;
  L, Sec, Pwr      : LongInt;
  UsedSections     : LongInt;
  UnusedBytes,
  SectionSize,
  LastSecSize      : LongInt;
  MinRecSize,
  MaxRecSize       : LongInt;
  UsePowerOfTwo    : Boolean;
  UnusedAvg        : Real;

procedure Help;
begin
  WriteLn;
  WriteLn(' BTVCALC - Calculate section sizes for B-Tree Filer''s');
  WriteLn('           variable length record unit, VREC.');
  WriteLn;
  WriteLn('   Syntax: BTVCALC MinRecSize MaxRecSize [-2]');
  WriteLn;
  WriteLn('    Where: MinRecSize is the fixed part of the record.');
  WriteLn('           MaxRecSize is the largest record size.');
  WriteLn('           -2 means NOT to use power of 2 section sizes.');
  WriteLn;
  WriteLn(' DOS redirection can be used to send the output to a file.');
  Halt;
end;

function Power(X,Y: Real): Real;
{-Return X raised to the power of Y}
begin
  Power := Exp(Y*Ln(X));
end;

function OverHead(SectionsUsed: LongInt): LongInt;
{-Calculate the VRec Filer overhead for variable length record managemnt}
begin
  OverHead := 6 + (SectionsUsed-1) * 7;
end;

function BytesUnused(SectionSize,SectionsUsed,RecordSize: LongInt): LongInt;
{-Calculate the number of bytes that are not used}
begin
  BytesUnused := (SectionsUsed*SectionSize)-RecordSize-OverHead(SectionsUsed);
end;


begin  {BTVCalc}
  {get min and max record sizes}
  if ParamCount < 2 then Help;
  Val(ParamStr(1),MinRecSize,i);
  if (i <> 0) or (MinRecSize < 1) then Help;
  Val(ParamStr(2),MaxRecSize,i);
  if (i <> 0) or (MaxRecSize < MinRecSize) then Help;

  if ParamCount > 2 then
    UsePowerOfTwo := ParamStr(3) <> '-2'
  else
    UsePowerOfTwo := True;

  {Used to bypass dupicate section sizes}
  LastSecSize := 0;

  {Open standard output - so output can be redirected}
  Assign(Output,'');
  ReWrite(Output);

  WriteLn;
  WriteLn(' BTVCALC - Possible section sizes for B-Tree Filer''s');
  WriteLn('           variable length record unit, VREC.');
  WriteLn;
  WriteLn(' Minimum record size: ',MinRecSize);
  WriteLn(' Maximum record size: ',MaxRecSize);
  WriteLn;

  {calculate section sizes for 2 to 16 sections}
  for Sec := 2 to 16 do begin
    SectionSize := (MaxRecSize + OverHead(Sec)) div Sec;

    {work with even sized sections}
    if Odd(SectionSize) then Inc (SectionSize);

    {Find closest power of 2 if not turned off at the command line}
    if UsePowerOfTwo then begin
      Pwr := 2;
      while Round(Power(2,Pwr)) < SectionSize do Inc(Pwr);
      SectionSize := Round(Power(2,Pwr));
    end;

    {Skip sections sizes already displayed and those > 65536}
    if (SectionSize <> LastSecSize) and (SectionSize <= 65536) then begin
      LastSecSize := SectionSize;  {update our flag}

      WriteLn(' Sections of ',SectionSize,' Bytes each.');

      {Calculate sections used and unused bytes for MinRecSize}
      UsedSections :=  MinRecSize div SectionSize;
      {adjust used sections to account for overhead}
      while (UsedSections * SectionSize) <
            (MinRecSize + OverHead(UsedSections)) do Inc(UsedSections);
      UnusedBytes := BytesUnused(SectionSize, UsedSections, MinRecSize);
      WriteLn('    Minimum number of sections: ',UsedSections:2,
              '    Bytes unused: ',UnusedBytes);

      {Calculate sections used and unused bytes for MaxRecSize}
      UsedSections :=  MaxRecSize div SectionSize;
      {adjust used sections to account for VRec maintenance overhead}
      while (UsedSections * SectionSize) <
            (MaxRecSize + OverHead(UsedSections)) do Inc(UsedSections);
      UnusedBytes := BytesUnused(SectionSize, UsedSections, MaxRecSize);
      WriteLn('    Maximum number of sections: ',UsedSections:2,
              '    Bytes unused: ',UnusedBytes);

      {Average is computed assuming an equal usage distibution of the
       variable part of each record throughout your data file}
      UnusedAvg := 0;
      for L := MinRecSize to MaxRecSize do begin
        UsedSections :=  L div SectionSize;
        {adjust used sections to account for overhead}
        while (UsedSections * SectionSize) <
              (L + OverHead(UsedSections)) do Inc(UsedSections);
        UnusedAvg := UnusedAvg + BytesUnused(SectionSize,UsedSections,L);
      end;
      UnusedAvg := Round(UnusedAvg / (MaxRecSize-MinRecSize+1));
      WriteLn('     Average unused Bytes for an equal distibution: ',
              UnusedAvg:0:0);
      WriteLn;
    end;
  end;
  Close(Output);

  AssignCrt(Output);
  ReWrite(Output);
  WriteLn('  Done.');
end.  {BTVCalc}

