#include <windows.h>
#include "descript.h"

static char MS_DOS_STR[] = "MS-DOS";

//
// Return a far pointer to the LDT
//
unsigned short GetLDTAlias(void)
{
    unsigned short  LDT_alias;
    unsigned short  (far * dpmiproc)(void);

    //
    // Use INT 2Fh, fn. 168A to get the "DPMI extensions
    // entry point" function pointer
    //
    _asm     mov     si, offset MS_DOS_STR   // DS:SI = "MS-DOS"
    _asm     mov     ax, 168Ah
    _asm     int     2Fh
    _asm     cmp     al, 8Ah
    _asm     je      extensions_not_found

    //
    // The entry point is returned in ES:DI.  Save it
    //
    _asm     mov     word ptr [dpmiproc], di
    _asm     mov     word ptr [dpmiproc+2], es

    //
    // Call the extensions with AX == 0x100.  The LDT alias
    // selector is return in AX.  Carry flag is set on failure.
    //
    _asm     mov     ax, 100h
    LDT_alias = dpmiproc();
    _asm     jc      extensions_not_found;

    return  LDT_alias;

extensions_not_found:   // We get here if something failed
    return  0;
}

//
// Given a far 16:16 address, create a function pointer for it,
// and which uses a call gate.
//
GENERIC_PROC CreateCallgate(void far * func_address, unsigned params)
{
    CALLGATE_DESCRIPTOR far *callgate_desc;
    CODE_SEG_DESCRIPTOR far *ring0_desc;
    unsigned short ldt_alias;
    unsigned short ring_0_alias;
    unsigned short callgate_selector;

    if ( (ldt_alias = GetLDTAlias()) == 0 )
        return 0;

    //
    // Grab a selector from Windows that has the exact same
    // attributes as the CS portion of "func_address"
    //
    if ((ring_0_alias = AllocSelector(HIWORD(func_address)))==0)
        return 0;

    //
    // Change the DPL bits of the ring 0 alias descriptor to
    // be privilege 0.
    //
    ring0_desc = MAKELP( ldt_alias, ring_0_alias & 0xFFF8 );
    ring0_desc->dpl = 0;

    //
    // Allocate the selector that'll be used for the call gate
    //
    if ( (callgate_selector= AllocSelector(0)) == 0 )
    {
        FreeSelector( ring_0_alias );
        return 0;
    }

    //
    // Create a pointer to the call gate descriptor
    //
    callgate_desc = MAKELP( ldt_alias, callgate_selector & 0xFFF8 );

    //
    // Fill in the fields of the call gate descriptor with the
    // appropriate values for a 16 bit callgate.
    //
    callgate_desc->offset_0_15 = LOWORD( func_address );
    callgate_desc->selector = ring_0_alias;
    callgate_desc->param_count = params;
    callgate_desc->some_bits = 0;
    callgate_desc->type = 4;            // 286 call gate
    callgate_desc->app_system = 0;      // A system descriptor
    callgate_desc->dpl = 3;             // Ring 3 code can call
    callgate_desc->present = 1;
    callgate_desc->offset_16_31 = 0;

    return MAKELP( callgate_selector, 0 );
}

//
// Free a previously allocated call gate function pointer.
//
void FreeCallgate( GENERIC_PROC callgate_ptr )
{
    CALLGATE_DESCRIPTOR far *callgate_desc;
    unsigned short ldt_alias;

    if ( (ldt_alias = GetLDTAlias()) == 0 )
        return;

    //
    // Create a pointer to the call gate descriptor
    //
    callgate_desc = MAKELP( ldt_alias, HIWORD(callgate_ptr) & 0xFFF8 );

    //
    // First, free the ring 0 alias selector stored in the LDT
    // call gate descriptor, then free the call gate selector.
    //
    FreeSelector( callgate_desc->selector );
    FreeSelector( HIWORD(callgate_ptr) );
}
