/* 
DPMI_MEM.C -- walk DPMI memory allocations in Windows 3.1
Actually, the same as the VMM _PageAllocate chain
Andrew Schulman (CIS 76320,302)
Microsoft Systems Journal, February 1993
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <dos.h>
#ifdef DPMI_APP
#include "dpmishel.h"
#else
#include "windows.h"
#include "winio.h"
#endif
#include "protmode.h"
#include "vxdcalls.h"

// following is not really DPMI-specific: same as _PageAllocate chain
typedef struct {
    DWORD vm_owner;         // owner virtual machine
    DWORD sortof_linear;    // use _PageGetSizeAddr to get linear
    DWORD prev;
    DWORD next;
    // following are swapped in 3.0
    DWORD size;
    DWORD sign;     // 4152h ('RA') + 0, etc.
    DWORD unknown3;
    DWORD unknown4;
    DWORD unknown5;     // not present in Windows 3.0!
    } DPMI_ARENA;       // Windows 3.1 Enhanced mode only
        
// ... same VM_CB as in VMINFO.C in Figure 17

typedef struct {
    DWORD CB_VM_Status;
    DWORD CB_High_Linear;
    DWORD CB_Client_Pointer;
    DWORD CB_VMID;
    } VM_CB;    // Virtual Machine Control Block
        
BOOL dpmi_alloc(DWORD size, DWORD far *plin, DWORD far *phand);
BOOL dpmi_free(DWORD hand);

typedef BOOL (*WALKFUNC)(DWORD vm, VM_CB far *);        

int vmwalk(WALKFUNC walkfunc);
BOOL walkfunc(DWORD vm, VM_CB far *vm_cb);
void dpmi_mem(DWORD vm, DWORD vm_id, DWORD lin_arena);

BOOL GetVMPgCount(DWORD vm, DWORD *ptotal, DWORD *pnotmapped);
DWORD GetSysVMHandle(void);
DWORD GetCurVMHandle(void);
DWORD GetNextVMHandle(DWORD vm);
DWORD PageGetLinAddr(DWORD hmem);
DWORD PageGetNumPages(DWORD hmem);
WORD WindowsEnhancedMode(void);

void fail(const char *s)
{ 
    puts(s); 
    _dos_exit(1); 
}

#ifdef DPMI_APP
int v86_main(int, char*[])
{
fprintf(stderr,"DPMI_MEM version 1.2\n");
fprintf(stderr,
    "Walks DPMI memory allocation chain in Windows 3.1 Enhanced mode\n");
fprintf(stderr,
    "(Actually, the Windows VMM _PageAllocate chain for each VM)\n\n");

    if (WindowsEnhancedMode() != 0x0A03)    // 3.1 only
        fail("This program requires Windows 3.1 Enhanced mode");
    
    if (! GenericVxDInstalled())
        fail("This program requires the generic VxD (VXD.386)\n"
             "Put device=vxd.386 in [386Enh] section of SYSTEM.INI");
    
    return 0;   // okay to switch into protected mode
}
#endif

char *sig_str(DWORD sig)
{
    static char buf[16];
    if (sig & 0x4152)
        sprintf(buf, "RA-%04X", (WORD) (sig >> 16));
    else 
        sprintf(buf, "%08lX", sig);
    return buf;
}

// Given a prev pointer from an arbitrary arena in the list, walks
// back to the beginning of the list, by looking for the arena
// where arena->prev == -1
DWORD find_alloc_root(DWORD prev)
{
    DWORD lin_arena;
    DPMI_ARENA far *arena;
    for (;;)  
    {
        lin_arena = prev;
        arena = (DPMI_ARENA far *) map_linear(prev, sizeof(DPMI_ARENA));
        if ((prev = arena->prev) == -1)
            break;  // found first arena
        else if (prev == 0)
            fail("Problem!  Couldn't find root of allocation chain!");
        free_mapped_linear(arena);
    }
    free_mapped_linear(arena);
    return lin_arena;
}
    
WORD arena_root_offset = 0;

#ifdef DPMI_APP
int pmode_main(int argc, char *argv[])
#else
int main(int argc, char *argv[])
#endif
{
    // Temporarily allocate a block of memory just to get the
    // back pointer out of its arena header; DPMI handle is ptr
    // to arena header.  
    DWORD lin, hand;
    if (! dpmi_alloc(1, &lin, &hand))
        fail("Can't allocate any DPMI memory!");
    DPMI_ARENA far *arena = (DPMI_ARENA far *) map_linear(hand, 
        sizeof(DPMI_ARENA));
    DWORD prev = arena->prev;  // save away: about to free arena
    
    // Test PageGetLinAddr() against a known linear address.
    if (PageGetLinAddr(hand) != lin)
        fail("Can't get linear addresses");

    free_mapped_linear(arena);
    dpmi_free(hand);

    // Walk back along list until we get to beginning
    DWORD lin_arena = find_alloc_root(prev);
    
    // Find where root of arena list is stored in VM CB
    DWORD vm = GetCurVMHandle();
    DWORD far *tmp = (DWORD far *) map_linear(vm, 0x400 * 4);
    DWORD vm_id = ((VM_CB far *) tmp)->CB_VMID;
    DWORD far *tmp2;
    int i;
    for (i=0,tmp2=tmp; i<0x400; i++, tmp2++)
        if (*tmp2 == lin_arena)
            break;
    free_mapped_linear(tmp);
    if (i < 0x400)
        arena_root_offset = i << 2;
    
    if (arena_root_offset)
        vmwalk(walkfunc);
    else    // can only do our own VM
        dpmi_mem(vm,vm_id,lin_arena);
    
    return 0;
}

void dpmi_mem(DWORD vm, DWORD vm_id, DWORD lin_arena)
{
    static DWORD sys_vm = 0;
    if (! sys_vm) sys_vm = GetSysVMHandle();
    
    static DWORD cur_vm = 0;
    if (! cur_vm) cur_vm = GetCurVMHandle();
    
    // Go back until we find the arena where arena->prev == -1
    lin_arena = find_alloc_root(lin_arena);
    
    DPMI_ARENA far *arena;
    arena = (DPMI_ARENA far *) map_linear(lin_arena, sizeof(DPMI_ARENA));
    
    printf("VM %lu - %s - %lX\n", 
        vm_id,
        ((vm == sys_vm) ? "System VM" : 
         (vm == cur_vm) ? "Current VM" : "DOS Box"),
        vm);
    
    DWORD vm_num_pages = 0;
    printf("Arena    Prev     Next     #Pages  Linear  Owner VM Signature\n");
    for (;;)
    {
#ifdef DPMI_APP     
        if (ctrl_c_hit)
            fail("Ctrl-C detected");
#endif      

        DWORD num_pages = PageGetNumPages(lin_arena);
        vm_num_pages += num_pages;
        printf("%08lX %08lX %08lX %6lu %08lX %08lX %s\n",
            lin_arena, arena->prev, arena->next, 
            num_pages, PageGetLinAddr(lin_arena),
            arena->vm_owner, sig_str(arena->sign));
        if (arena->size != num_pages)
            puts("Warning: Size wrong?");
        if (arena->vm_owner != vm)
            puts("Warning: VM owner wrong?");
        lin_arena = arena->next;
        free_mapped_linear(arena);
        if (lin_arena == -1)    // reached end of list
            break;
        arena = (DPMI_ARENA far *) map_linear(lin_arena, sizeof(DPMI_ARENA));
    }
    
    printf("\n");
    DWORD total, notmapped;
    if (! GetVMPgCount(vm, &total, &notmapped))
        fail("Can't get current VM page count");

    if (total != vm_num_pages)
        printf("Warning: Page totals don't add up!\n");
    
    printf("Total pages: %6lu\n", total);
    printf("Unmapped:    %6lu\n", notmapped);
    printf("\n");
}

BOOL dpmi_alloc(DWORD size, DWORD far *plin, DWORD far *phand)
{
    _asm push si
    _asm push di
    _asm mov ax, 0501h
    /* size in bytes in BX:CX */        
    _asm mov bx, word ptr size+2
    _asm mov cx, word ptr size+0
    _asm int 31h
    _asm jc error
    /* linear address in BX:CX */
    _asm mov ax, bx
    _asm les bx, dword ptr plin
    _asm mov es:[bx], cx
    _asm mov es:[bx+2], ax
    /* handle in SI:DI */
    _asm les bx, dword ptr phand
    _asm mov es:[bx], di
    _asm mov es:[bx+2], si
    _asm pop di
    _asm pop si
    return TRUE;
error:
    _asm pop si
    _asm pop di
    return FALSE;
}

BOOL dpmi_free(DWORD hand)
{
    _asm push si
    _asm push di
    _asm mov ax, 0502h
    _asm mov si, word ptr hand+2
    _asm mov di, word ptr hand
    _asm int 31h
    _asm pop di
    _asm pop si
    _asm jc error
    return TRUE;
error:
    return FALSE;
}

// ... same vmwalk() as VMINFO.C, Figure 17

int vmwalk(WALKFUNC walkfunc)
{
    DWORD sys_vm = GetSysVMHandle();
    void far *vm_cb;
    VM_CB my_vm_cb;
    DWORD vm;
    int num_vm;
    for (vm=sys_vm, num_vm=0;; num_vm++)
    {
        vm_cb = map_linear(vm, sizeof(VM_CB));
        _fmemcpy(&my_vm_cb, vm_cb, sizeof(VM_CB));
        free_mapped_linear(vm_cb);
        if (! (*walkfunc)(vm, &my_vm_cb))
            return num_vm;
        if ((vm = GetNextVMHandle(vm)) == sys_vm)
            break;  // circular list
    }
    return num_vm;
}
        
BOOL walkfunc(DWORD vm, VM_CB far *vm_cb)
{
    DWORD far *fparena = (DWORD far *) 
        map_linear(vm + arena_root_offset, sizeof(DWORD));
    dpmi_mem(vm, vm_cb->CB_VMID, *fparena); // should do validity check
    free_mapped_linear(fparena);
    return TRUE;
}

DWORD PageGetSizeAddr(DWORD hmem, BOOL get_eax)
{
    VxDPushParams p;
    p.CallNum = _PageGetSizeAddr;
    p.NumP = 2;
    p.P[0] = hmem;
    p.P[1] = 0;
    if (! VxDPushCall(&p))
        return 0;
    if (! (p.OutEAX || p.OutEDX))
        return 0;
    else if (get_eax)
        return p.OutEAX;    // number of pages
    else
        return p.OutEDX;    // ring-0 linear address
}
        
DWORD PageGetNumPages(DWORD hmem) { return PageGetSizeAddr(hmem, TRUE); }

DWORD PageGetLinAddr(DWORD hmem) { return PageGetSizeAddr(hmem, FALSE); }

// ... same GetVMPgCount, GetSysVMHandle, GetCurVMHandle, 
// GetNextVMHandle, WindowsEnhancedMode as VMINFO.C, Figure 17

BOOL GetVMPgCount(DWORD vm, DWORD *ptotal, DWORD *pnotmapped)
{
    VxDPushParams p;
    p.CallNum = _GetVMPgCount;
    p.NumP = 2;
    p.P[0] = vm;
    p.P[1] = 0;     // flags=0
    if (! VxDPushCall(&p))
        return 0;
    if (! (p.OutEAX || p.OutEDX))
        return 0;
    else
    {
        *ptotal = p.OutEAX;
        *pnotmapped = p.OutEDX;
        return 1;
    }
}

DWORD GetSysVMHandle(void)
{
    VxDParams p;
    p.CallNum = Get_Sys_VM_Handle;
    return (VxDCall(&p)) ? p.OutEBX : 0;
}

DWORD GetCurVMHandle(void)
{
    VxDParams p;
    p.CallNum = Get_Cur_VM_Handle;
    return (VxDCall(&p)) ? p.OutEBX : 0;
}

DWORD GetNextVMHandle(DWORD vm)
{
    VxDParams p;
    p.CallNum = Get_Next_VM_Handle;
    p.InEBX = vm;
    return (VxDCall(&p)) ? p.OutEBX : 0;
}

WORD WindowsEnhancedMode(void)
{
    WORD vers;
    _asm mov ax, 1600h
    _asm int 2fh
    _asm mov vers, ax
    switch (vers & 0xFF)
    {
        case 0 :
        case 80 :
            return 0;   // not Windows Enhanced mode
        case 1 :
        case 0xFF :
            return 0;   // for now:  Windows/386 2.x
        default :
            return vers;
    }
}
