/* 
PGCOUNT.C
Andrew Schulman (CIS 76320,302)
Microsoft Systems Journal, February 1993

C++ program
Uses WINIO
Requires generic VxD (VXD.386)

for each block in global heap [heapwalk(), add_all()]
    find owner (module or task) [Module()]
    increment total bytes for owner [add()]
    determine how many pages in block [NumPages()]
    increment total pages for owner [add()]
    for each page in block [BytesPagedIn()]
        if page is present in memory
            increment paged-in total for owner [add()]

sort totals [print_totals()]
for each owner (module or task)
    display totals [print_totals()]

if user clicks on owner [clickfunc()]
    display new window
    for each block belonging to owner [heapwalk(), single_module()]
        for each page in block
            display information [display_pages()]
*/

#include <stdlib.h>
#include <string.h>
#include <dos.h>
#include "windows.h"
#include "toolhelp.h"
#include "winio.h"
#include "protmode.h"
#include "vxdcalls.h"

#define HMODULE_FROM_HTASK(hTask) \
    *((WORD far *) MK_FP(hTask, 0x1E))
        
#define FREE    0

typedef BOOL (*WALKFUNC)(GLOBALENTRY *pge);

char *GetModuleNameFromHandle(HANDLE handle);
HANDLE Module(HANDLE h);
DWORD BytesPagedIn(DWORD base, DWORD size, WORD *pnump, WORD *pnumpin);
BOOL CopyPageTable(DWORD pgnum, DWORD npages, 
    DWORD far *pagebuf, DWORD flags);

BOOL heapwalk(WALKFUNC walkfunc);
BOOL add_all(GLOBALENTRY *pge);
BOOL add(HANDLE h, DWORD base, DWORD size);
void init_totals(void);
void print_totals(void);

void fail(char *s) { puts(s); exit(1); }
    
void pgcount(HWND hwnd, int)
{
    winio_setcurrent(hwnd);
    winio_clear(hwnd);
    init_totals();
    heapwalk(add_all);
    print_totals();
}

void display_pages(DWORD base, DWORD size);

static char *typename[] = { 
    "Unknown", "DGROUP", "DATA", "CODE", "Task", "Resource",
    "Module", "FREE", "Internal", "Sentinel", "BurgerMaster"
    } ;

BOOL single_module(GLOBALENTRY *pge)
{
    if (Module(pge->hOwner) != GetProp(winio_current(), "HMOD"))
        return TRUE;    // keep going
    /* found one for specified module */
    printf("%04X %08lX %-7lu %s\n",
        pge->hBlock, pge->dwAddress, pge->dwBlockSize, 
        typename[pge->wType]);
    display_pages(pge->dwAddress, pge->dwBlockSize);
    return TRUE;
}

void do_module(HWND hwnd, int)
{
    winio_setcurrent(hwnd);
    winio_clear(hwnd);
    winio_setpaint(hwnd, FALSE);
    heapwalk(single_module);
    winio_setpaint(hwnd, TRUE);
    winio_home(hwnd);
}

void clickfunc(HWND hwnd, LPSTR line, int)
{
    static char buf[128];
    HMENU hmenu;
    
    _fstrcpy(buf, line);
    if (strncmp(buf, "Owner", 5) == 0)  // ignore clicks on banner
        return;
    
    HANDLE hmod;
    sscanf(buf, "%04X", &hmod);
    
    // need to test if is valid module handle still!!!!
        
    hwnd = winio_window(buf, 0, WW_HASMENU);
    winio_setcurrent(hwnd);
    winio_settitle(hwnd, buf);

    hmenu = winio_hmenumain(hwnd);
    AppendMenu(hmenu, MF_STRING | MF_ENABLED, 1, "&Refresh!");
    winio_setmenufunc(hwnd, 1, do_module);
    DrawMenuBar(hwnd);

    SetProp(hwnd, "HMOD", hmod);

    winio_setpaint(hwnd, FALSE);
    heapwalk(single_module);
    winio_setpaint(hwnd, TRUE);
    winio_home(hwnd);
}

#if 0
// this didn't work (produced very tiny window)
// bug in winio_defwindowsize??
// create object to run function before main
struct startup { 
    startup() { winio_defwindowsize(MAKELONG(12,40)); }
    } ;
static startup s;   
#endif  
    
main(int argc, char *argv[])
{
    HMENU hmenu;
    
    if (! (GetWinFlags() & WF_ENHANCED)) 
        fail("This program requires Windows Enhanced mode");
    
    hmenu = winio_hmenumain(__hMainWnd);
    AppendMenu(hmenu, MF_STRING | MF_ENABLED, 1, "&Refresh!");
    
    if (argc < 2)
    {
        winio_setmenufunc(__hMainWnd, 1, pgcount);
        DrawMenuBar(__hMainWnd);
        winio_setlinefn(__hMainWnd, clickfunc);
        pgcount(__hMainWnd,0);
    }
    else    // go straight to specific module
    {
        winio_setmenufunc(__hMainWnd, 1, do_module);
        DrawMenuBar(__hMainWnd);
        SetProp(__hMainWnd, "HMOD", GetModuleHandle(strupr(argv[1])));
        do_module(__hMainWnd,0);
    }

    return 0;
}

// maybe should change to return number of blocks found!
BOOL heapwalk(WALKFUNC walkfunc)
{
    GLOBALENTRY ge;
    BOOL ok;
    ge.dwSize = sizeof(ge);
    winio_setbusy();
    winio_setpaint(__hMainWnd, FALSE);
    ok = GlobalFirst(&ge, GLOBAL_ALL);
    while (ok)
    {
        if (! (*walkfunc)(&ge))
        {
            winio_setpaint(__hMainWnd, TRUE);
            return FALSE;
        }
        ok = GlobalNext(&ge, GLOBAL_ALL);
    }
    winio_setpaint(__hMainWnd, TRUE);
    winio_resetbusy();
    return TRUE;
}

BOOL add_all(GLOBALENTRY *pge)
{
    add(pge->hOwner, pge->dwAddress, pge->dwBlockSize);
    return TRUE;
}

// based on routine by Matt Pietrek, UndocWin, Chapter 10
char *GetModuleNameFromHandle(HANDLE handle)
{
    MODULEENTRY me;
    TASKENTRY   te;
    static char name[40];
    
    if (handle == FREE) return "FREE";

    me.dwSize = sizeof(me);
    te.dwSize = sizeof(te);
    
    if (ModuleFindHandle(&me, handle))
        strcpy(name, me.szModule);
    else if (TaskFindHandle(&te, handle))
        strcpy(name, te.szModule);
    else
        name[0] = 0;
    
    return name;
}

// combine all stuff for module in one entry
HANDLE Module(HANDLE h)
{
    // following was major performance bottleneck in program
    // until switched to one-time GetProcAddress
    static BOOL (FAR PASCAL *IsTask)(HANDLE h) = 
        (BOOL (FAR PASCAL *)(HANDLE)) 0;
    if (! IsTask)   // one-time initialization
        IsTask = (BOOL (FAR PASCAL *)(HANDLE)) 
            GetProcAddress(GetModuleHandle("KERNEL"), "ISTASK");
    return (IsTask(h)) ? HMODULE_FROM_HTASK(h) : h;
}

/**********************************************************************/

BOOL CopyPageTable(DWORD pgnum, DWORD npages, DWORD far *pagebuf, DWORD flags)
{
    VxDPushParams p;
    p.CallNum = _CopyPageTable;
    p.NumP = 4;
    p.P[0] = pgnum;
    p.P[1] = npages;
    p.P[2] = MapFlat(pagebuf);
    p.P[3] = flags;
    VxDPushCall(&p);
    UnmapFlat(pagebuf);
    return p.OutEAX;
}

DWORD NumPages(DWORD base, DWORD size)
{
    WORD offset = base & 0xFFFL;    // offset of seg on first page
    WORD first = 4096 - offset;     // room on first page
        
    if (size <= first)
        return 1;
    else
    {
        DWORD rest = size - first;  // bytes past first page (may be 0)
        WORD last = rest & 0xFFFL;  // bytes on last page
        DWORD num_pages = 1 + (rest >> 12);
        if (last)
            num_pages++;
        return num_pages;
    }
}
    
void display_pages(DWORD base, DWORD size)
{
    WORD num_pages = NumPages(base, size);
    if (num_pages == 0)
        return;
    
    DWORD *pagebuf;
    if (! (pagebuf = (DWORD *) calloc(num_pages, sizeof(DWORD))))
        return; // insufficient memory
    
    CopyPageTable(base >> 12, num_pages, pagebuf, 0);

    for (int i=0; i<num_pages; i++)
    {
        printf("   %08lX   ", (base & ~0xFFFL) + (4096L * i));
        if (pagebuf[i] & 1)
            printf("%08lX\n", pagebuf[i] & ~0xFFFL);
        else
            printf("Not Present (%08lX)\n", pagebuf[i]);
    }
    
    free(pagebuf);
}

/* Given linear base address and #bytes, return #bytes that are
   currently paged in.  Most of the code deals with the fact that
   the base may not start or end on a page boundary. */
DWORD BytesPagedIn(DWORD base, DWORD size, WORD *pnump, WORD *pnumpin)
{
    if (size == 0) 
        return 0;
    
    WORD offset = base & 0xFFFL;    // offset of seg on first page
    WORD first = 4096 - offset;     // room on first page
    DWORD rest = size - first;      // bytes past first page (may be 0)
    WORD last = rest & 0xFFFL;      // bytes on last page
        
    WORD num_pages;                 // number of pages (NOT size / 4k)
    *pnump = num_pages = NumPages(base, size);

    DWORD *pagebuf;
    if (! (pagebuf = (DWORD *) calloc(num_pages, sizeof(DWORD))))
        return 0;   // insufficient memory
    
    CopyPageTable(base >> 12, num_pages, pagebuf, 0);

    WORD num_paged_in = 0;
    for (int i=0; i<num_pages; i++)
        if (pagebuf[i] & 1) 
            num_paged_in++;
    *pnumpin = num_paged_in;

    DWORD paged_in = 0;
    if (size <= first)
    {
        if (pagebuf[0] & 1)
            paged_in = size;
    }
    else
    {
        if (pagebuf[0] & 1)                 // first page present?
            paged_in = first;
        WORD nump = rest >> 12;
        for (int i=1; i<=nump; i++)         // middle pages 
            if (pagebuf[i] & 1)             // present?
                paged_in += 4096;
        if (last)                           // if partial last page
            if (pagebuf[num_pages-1] & 1)   // present?
                paged_in += last;
    }

    free(pagebuf);
    return paged_in;
}

/**********************************************************************/

typedef struct {
    WORD owner; 
    WORD items;
    DWORD total; 
    DWORD paged_in;
    WORD num_pages;
    WORD num_paged_in;
    } TOTAL;
    
#define MAX_TOTAL   1024
    
static TOTAL totals[MAX_TOTAL] = {0} ;
static int num_totals = 0;

void init_totals(void)
{
    num_totals = 0;
}

BOOL add(HANDLE h, DWORD base, DWORD size)
{
    h = Module(h);
    
    /* Find the owner in the table */
    TOTAL *t;
    int i;
    for (i=0, t=totals; i<num_totals; i++, t++)
        if (t->owner == h)
            break;
    if (i == num_totals)    // not in table yet
    {
        if (num_totals >= MAX_TOTAL)
            return FALSE;
        t = &totals[num_totals];
        t->owner = h;
        t->items = t->total = t->paged_in = 0;
        num_totals++;
    }

    /* add this block to the totals for this owner */
    t->total += size;
    t->items++;
    
    /* still count # blocks with 0 bytes (e.g., discarded) */
    if (size == 0) 
        return TRUE;    
    
    /* get virtual-memory statistics for this block, and add to owner */
    WORD num_pages, num_paged_in;
    t->paged_in += BytesPagedIn(base, size, &num_pages, &num_paged_in);
    t->num_pages += num_pages;
    t->num_paged_in += num_paged_in;
    
    return TRUE;
}

int cmpfunc(const void *p1, const void *p2)
{ 
    DWORD t1p = ((TOTAL *) p1)->paged_in;
    DWORD t2p = ((TOTAL *) p2)->paged_in;
    // can't just return diff because DWORD -> int
    if (t1p < t2p) return 1;
    else if (t1p > t2p) return -1;
    else return 0;
}

void print_totals(void)
{
    DWORD total=0, free_total=0, paged_in_total=0;
    WORD items=0;
    
    qsort(totals, num_totals, sizeof(TOTAL), cmpfunc);
    
    winio_setpaint(__hMainWnd, FALSE);
    
    printf("Owner Items   PgIn/Bytes   (%In)   PgIn/Pgs Density  Owner\n");
    
    TOTAL *t;
    int i;
    for (i=0, t=totals; i<num_totals; i++, t++)
    {
        if (t->total && t->num_pages)   // avoid divide-by-zero
            printf("%04X %5u %7lu/%-7lu (%3u%%) %5u/%-5u %5u  %s\n", 
                t->owner,
                t->items, 
                t->paged_in, t->total,
                (int) ((t->paged_in * 100) / t->total),
                t->num_paged_in, t->num_pages,
                (WORD) (t->total / t->num_pages),
                GetModuleNameFromHandle(t->owner));
        
        /* get the grand total */
        if (t->owner == FREE)
            free_total = t->total;
        else
            total += t->total;
        items += t->items;
        paged_in_total += t->paged_in;
    }
    
    /* display the grand total */
    printf("\n");
    printf("Total:       %8lu bytes (%d items)\n", total, items);
//  following is meaningless, unfortunately
//  printf("Paged in:    %8lu bytes\n", paged_in_total);
    printf("Free:        %8lu bytes\n", free_total);
    winio_setpaint(__hMainWnd, TRUE);
    winio_home(__hMainWnd);
}

