/* 
VMINFO.C -- walk virtual machines in Windows Enhanced mode
Andrew Schulman (CIS 76320,302)
Microsoft Systems Journal, February 1993

Sample output:
ID   VM        Status      HiLinear  ClientPtr  Pages
 1   80481000  [ B P  VI]  81400000  80010D50  157/157  System VM
 3   8069B000  [  S     ]  81C00000  805F0F70   91/ 91  DOS Box
 2   805EA000  [   P    ]  81800000  804B0F70   93/ 93  Current VM

X=Exclusive  B=Background  S=Suspended  P=ProtMode
3=Use32      K=Blocked     I=Idle       V=PageableV86
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <dos.h>
#define DPMI_APP
#include "dpmishel.h"
#include "protmode.h"
#include "vxdcalls.h"

// from ddk vmm.inc
typedef struct {
    DWORD CB_VM_Status;
    DWORD CB_High_Linear;
    DWORD CB_Client_Pointer;
    DWORD CB_VMID;
    } VM_CB;    // Virtual Machine Control Block
        
#define VMStat_Exclusive        (1L << 0)
#define VMStat_Background       (1L << 1)
#define VMStat_Creating         (1L << 2)
#define VMStat_Suspended        (1L << 3)
#define VMStat_Not_Executeable  (1L << 4)
#define VMStat_PM_Exec          (1L << 5)
#define VMStat_PM_App           (1L << 6)
#define VMStat_PM_Use32         (1L << 7)
#define VMStat_VxD_Exec         (1L << 8)
#define VMStat_High_Pri_Back    (1L << 9)
#define VMStat_Blocked          (1L << 0x0a)
#define VMStat_Awakening        (1L << 0x0b)
#define VMStat_PageableV86      (1L << 0x0c)
#define VMStat_V86IntsLocked    (1L << 0x0d)
#define VMStat_TS_Sched         (1L << 0x0e)
#define VMStat_Idle             (1L << 0x0f)

typedef BOOL (*WALKFUNC)(DWORD vm, VM_CB far *);        

int vmwalk(WALKFUNC walkfunc);
BOOL walkfunc(DWORD vm, VM_CB far *vm_cb);

BOOL GetVMPgCount(DWORD vm, DWORD *ptotal, DWORD *pnotmapped);
DWORD GetSysVMHandle(void);
DWORD GetCurVMHandle(void);
DWORD GetNextVMHandle(DWORD vm);
WORD WindowsEnhancedMode(void);

void fail(const char *s)
{ 
    puts(s); 
    _dos_exit(1); 
}

int v86_main(int, char*[])
{
    if (! WindowsEnhancedMode())
        fail("This program requires Windows 3.x Enhanced mode");
    
    if (! GenericVxDInstalled())
        fail("This program requires the generic VxD (VXD.386)\n"
             "Put device=vxd.386 in [386Enh] section of SYSTEM.INI");
    
    return 0;   // okay to switch into protected mode
}

int pmode_main(int argc, char *argv[])
{
    printf("ID   VM        Status      HiLinear  ClientPtr  Pages\n");
    vmwalk(walkfunc);
    printf("\nX=Exclusive  B=Background  S=Suspended  P=ProtMode\n");
      printf("3=Use32      K=Blocked     I=Idle       V=PageableV86\n");
    return 0;
}

int vmwalk(WALKFUNC walkfunc)
{
    DWORD sys_vm = GetSysVMHandle();
    void far *vm_cb;
    VM_CB my_vm_cb;
    DWORD vm;
    int num_vm;
    for (vm=sys_vm, num_vm=0;; num_vm++)
    {
        vm_cb = map_linear(vm, sizeof(VM_CB));
        _fmemcpy(&my_vm_cb, vm_cb, sizeof(VM_CB));
        free_mapped_linear(vm_cb);
        if (! (*walkfunc)(vm, &my_vm_cb))
            return num_vm;
        if ((vm = GetNextVMHandle(vm)) == sys_vm)
            break;  // circular list
    }
    return num_vm;
}

char *status_str(DWORD vmstatus)
{
    static char *str = "        ";
    strnset(str, ' ', 8);
    if (vmstatus & VMStat_Exclusive)    str[0] = 'X' ;
    if (vmstatus & VMStat_Background)   str[1] = 'B' ;
    if (vmstatus & VMStat_Suspended)    str[2] = 'S' ;
    if (vmstatus & VMStat_PM_App)       str[3] = 'P' ;
    if (vmstatus & VMStat_PM_Use32)     str[4] = '3' ;
    if (vmstatus & VMStat_Blocked)      str[5] = 'K' ;
    if (vmstatus & VMStat_PageableV86)  str[6] = 'V' ;
    if (vmstatus & VMStat_Idle)         str[7] = 'I' ;
    return str;
}
        
BOOL walkfunc(DWORD vm, VM_CB far *vm_cb)
{
    DWORD total, notmapped;
    
    static DWORD sys_vm = 0;
    static DWORD cur_vm = 0;
    if (! sys_vm) sys_vm = GetSysVMHandle();
    if (! cur_vm) cur_vm = GetCurVMHandle();
    
    printf("%2lu   %08lX  [%s]  %08lX  %08lX  ",
        vm_cb->CB_VMID,
        vm,
        status_str(vm_cb->CB_VM_Status),
        vm_cb->CB_High_Linear,
        vm_cb->CB_Client_Pointer);
    
    GetVMPgCount(vm, &total, &notmapped);
    printf("%3lX/%3lX  ", total-notmapped, total);
    
    if (vm == sys_vm) printf("System VM");
    else if (vm == cur_vm) printf("Current VM");
    else printf("DOS Box");
    printf("\n");

#if 0
    // Example of how to use vm_cb->CB_High_Linear to get at memory
    // for other VMs.  The following peeks the BIOS data area in the
    // VM to see if any keystrokes are pending, by comparing the
    // WORD at 41Ah with the WORD at 41Ch.
    BYTE far *biosd = (BYTE far *) map_linear(vm_cb->CB_High_Linear, 10240);
    if (*((WORD far *) &biosd[0x41a]) != *((WORD far *) &biosd[0x41c]))
        printf("Keystrokes are pending\n");
    free_mapped_linear(biosd);
#endif
    
    return TRUE;
}
        
BOOL GetVMPgCount(DWORD vm, DWORD *ptotal, DWORD *pnotmapped)
{
    VxDPushParams p;
    p.CallNum = _GetVMPgCount;
    p.NumP = 2;
    p.P[0] = vm;
    p.P[1] = 0;     // flags=0
    if (! VxDPushCall(&p))
        return 0;
    if (! (p.OutEAX || p.OutEDX))
        return 0;
    else
    {
        *ptotal = p.OutEAX;
        *pnotmapped = p.OutEDX;
        return 1;
    }
}

DWORD GetSysVMHandle(void)
{
    VxDParams p;
    p.CallNum = Get_Sys_VM_Handle;
    return (VxDCall(&p)) ? p.OutEBX : 0;
}

DWORD GetCurVMHandle(void)
{
    VxDParams p;
    p.CallNum = Get_Cur_VM_Handle;
    return (VxDCall(&p)) ? p.OutEBX : 0;
}

DWORD GetNextVMHandle(DWORD vm)
{
    VxDParams p;
    p.CallNum = Get_Next_VM_Handle;
    p.InEBX = vm;
    return (VxDCall(&p)) ? p.OutEBX : 0;
}

WORD WindowsEnhancedMode(void)
{
    WORD vers;
    _asm mov ax, 1600h
    _asm int 2fh
    _asm mov vers, ax
    switch (vers & 0xFF)
    {
        case 0 :
        case 80 :
            return 0;       // not Windows Enhanced mode
        case 1 :
        case 0xFF :
            return 0;       // Windows/386 2.x
        default :
            return vers;    // Windows Enhanced mode
    }
}

