/*
 -------------------------------------
 Filename: x816b.c
 Version 1.013
 Portions are Copyright (c)1990 DataQue Software
 65816 Assembler
 -------------------------------------
 directory of functions in this module:

 main(argc, argv)     passed command line information
 initialize (ac, av)  initialize pass variables and files
 fileopen(fname,dir)  open file with passed filename and direction of open
 usage()              display cli option and parameter usage
 readline()           read a line of source code into line buffer
 include()            switch to include file
 error(code)          display error codes and messages
 warning(code)        display warning codes and messages
 stprnt()             print symbol table to list device

 -------------------------------------
*/

#include <stdio.h>
#include <string.h>
#include "x816a.h"
#include "x816b.h"

#define peek(x)   *((char *)x)
#define poke(x,y) *((char *)x)=y

static unsigned including;
static FILETYPE sviptr;            /* saved input file pointer */
static unsigned svline;
static char *progname;             /* name of this assembler */

/* 
 -------------------------------------
 main entry into assembler
 argc is a count of CL parameters
 argv is the base addr of the CL
 -------------------------------------
*/

main(argc, argv)
unsigned argc;
char *argv[];
{
  char c;
  unsigned cnt;
  unsigned i;
  progname = argv[0];

  printf("\nX816 - Version 1.013  (c)1990 DataQue Software\n");

  nflag = cflag = oflag = 1;
  aflag = errcnt = mflag = lflag = sflag = 0;
  obuff = lbuff = NULL;

  while (--argc && (*++argv)[0] == '-') {
    for (i = 1; (c = tolower((*argv)[i])) != '\0'; i++) {
           if (c == 'l') lflag = 1; /* list file         */
      else if (c == 'c') cflag = 0; /* conditional list  */
      else if (c == 'o') oflag = 0; /* object file       */
      else if (c == 's') sflag = 1; /* symbol listing    */
      else if (c == 'n') nflag = 0; /* string byte list  */
      else if (c == 'm') mflag = 1; /* map file          */
      else usage();
    }
  }

  abuff = (char *)malloc(ABSIZE);
  ibuff = (char *)malloc(IBSIZE);

  if (oflag) obuff = (char *)malloc(OBSIZE);
  if (!lflag) lbuff = (char *)malloc(LBSIZE);
  if (!argc) usage();      /* usage error */

  while (argc) {
    pass = FIRST_PASS;
    for (i = 0; i < HTSIZE; i++) hash_tbl[i] = NULL;
    initialize (argv);
    while (readline() != EOF)
    assemble();
    pass = LAST_PASS;
    if (!errcnt) {
      initialize (argv);
      while (readline() != EOF) {
        listed = 0;
        assemble();
        if (!listed) println();
      }
    }
    fclose(iptr);
    if (oflag) fclose(optr);
    stprnt();
    if (mflag) fclose(mptr);
    if (!lflag) {
      fprintf(lptr,"\nEnd of assembly, %d error(s) %d warning(s)\n",errcnt,warcnt);
      fclose(lptr);
    }
    printf("\nEnd of assembly, %d error(s) %d warning(s)\n",errcnt,warcnt);
    argc--;
    argv++;
  }
}

/*
  -------------------------------------
  initialize this pass 
  -------------------------------------
*/

VOID initialize (av)

char      *av[];
{
  unsigned lng;
  char *s,i,oname[30],lname[30],mname[30];

  s = *av;
  lng = strlen(s);                   /* look for end of filename */

  printf(" Pass: %d\n",pass+1);
  printf("  %s:\n", s);

  if (lng <= 2 || (strcmp(&s[lng-2],".a") && strcmp(&s[lng-2],".A"))) {
    error(28);                       /* bad filename */
    quit();
  }
  strncpy (filename, s, lng-2 );
  filename[lng-2] = '\0';            /* strncpy doesn't doesnt terminate */
  if (pass) {
    if(oflag) {
      strcpy(oname,filename);
      strcat(oname,".o");              /* create output filename & open */
      optr = fileopen(oname, "wb");
      setvbuf(optr, obuff, _IOFBF, OBSIZE-1);
      objloc = 0;
    }
    if (!lflag) {
      strcpy(lname,filename);
      strcat(lname,".l");              /* create list filename & open */
      lptr = fileopen(lname, "w");
      setvbuf(lptr, lbuff, _IOFBF, LBSIZE-1);
    }
    if (mflag) {
      strcpy(mname,filename);
      strcat(mname,".m");              /* create map filename & open */
      mptr = fileopen(mname, "w");
    }
  }
  
  iptr = fileopen(s, "r");           /* open source filename for read */
  setvbuf(iptr, abuff, _IOFBF, ABSIZE-1);

  ataflg = romsiz = accsiz = indsiz = offset = including = 0;
  rflag = datbnk = codbnk = datloc = codloc = slnum = 0;
  dflag = segnum = casmflg = 1;
}

/*
  -------------------------------------
  open file with attribute specified 
  -------------------------------------
*/

FILETYPE fileopen(fname,dir)         /* filename and direction */
char *fname, *dir;
{
  FILETYPE fd;
  char err;
  err = 0;
  if ((fd = fopen(fname,dir))==NULL) {
    printf("Can't open %s because DOS error type %d\n", fname, err );
    quit();
  }
  return fd;
}

/*
  -------------------------------------
  USAGE: a routine to instruct user
  on the assembler syntax and usage.
  -------------------------------------
*/

VOID usage()
{
    printf( "\nUsage: %s [-lcosn] file.a [file.a ...]\n",progname);
    printf( "\t\tcommand line options are: (* = default on)\n\n");
    printf( "\t\t[l] = * list file\n");
    printf( "\t\t[c] = * conditional list\n");
    printf( "\t\t[o] = * object file\n");
    printf( "\t\t[s] =   symbol listing\n");
    printf( "\t\t[n] = * short string byte list\n");
/*    printf( "\t\t[m] =   map file\n\n"); */
    quit();
}


/*
  -------------------------------------
  readline reads and formats an input 
  line, returns -1 at end of file.
  -------------------------------------
*/

int readline()
{
  unsigned i;              /* pointer into prlnbuf           */
  char cmnt;               /* comment line flag              */
  unsigned spcnt;          /* consecutive space counter      */
  char string;             /* ASCII string flag              */
  unsigned temp1;          /* temp used for line number conv */

  do {
    cpos = SFIELD;
    temp1 = ++slnum;

    if (peek(197)==63) quit();                        /* break key */

    for (i = 0; i < LAST_CH_POS; i++) prlnbuf[i] = ' ';
    i = 4;

    while (temp1) {                    /* put src line num into prlnbuf */
      prlnbuf[i--] = temp1 % 10 + '0';
      temp1 /= 10;
    }
    if (including) prlnbuf[4] = '#';   /* included line indicator */
    i = SFIELD;
    cmnt = string = 0;

    while ((ch = getc(iptr)) !='\n') {
      prlnbuf[i++] = ch;
      if (ch == EOF) {
        if (including) {
          prlnbuf[--i] = '\0';         /* erase EOF character */
          fclose(iptr);
          iptr = sviptr;               /* restore previous file */
          slnum = svline;
          including--;
        }
        else return(EOF);
      }
      else if ((ch == ';') && (!string)) {
        if (i == SFIELD + 1) ++cmnt;
        else if (prlnbuf[i - 2] != '\'') {
          ++cmnt;
          prlnbuf[i-1] = ' ';
          if (i < (SBOLSZ+10)) i = (SBOLSZ+10);
          prlnbuf[i++] = ';';
        }
      } else {
        if ((ch == '"') && (!cmnt)) string = string ^ 1;
        if (i >= LAST_CH_POS - 1) --i;
      }
    }
    prlnbuf[i] = 0;
    if (prlnbuf[SFIELD]=='#') include();
  } while (prlnbuf[SFIELD]=='#');
  return(0);
}

/*
  -------------------------------------
  Process include statement.
  -------------------------------------
*/

VOID include()
{
  static char *icldstr = "include ";

  char c,flnm[30];
  unsigned i;

  if (including) {
    error(29);         /* exceeded include level */
    quit();
  }
  i=0;
  while (nextch()!=' ' && ch==icldstr[i]) i++;
  if (i!=7) {
    error(30);         /* Bad include syntax */
    quit();
  }
  skip();
  if (!index("\"<",(c=ch))) {
    error(30);         /* Bad include syntax */
    quit();
  }
  if (pass) println();
  if (c=='<') c='>';
  i=0;
  while (nextch()!=c && ch) flnm[i++]=ch;
  flnm[i]='\0';
  svline = slnum;
  slnum = 0;
  sviptr = iptr;
  including++;
  if ((iptr=fileopen(flnm,"r")) == NULL) {
    error(31);         /* Bad include file open */
    quit();
  }
  setvbuf(iptr, ibuff, _IOFBF, IBSIZE-1);
  printf("  %s:\n", flnm);
}

/*
  -------------------------------------
  08/04/85 - Centralize error strings
  to avoid redundancy and eliminate
  'pseudo-xrefs' in other objects.
  -------------------------------------
*/

VOID error(code)
unsigned code;
{
static char *errmsg[] = {
/*  0 */ "UNKNOWN ERROR!!!",
/*  1 */ "Internal error",
/*  2 */ "Invalid operation code",
/*  3 */ "A,X,Y, and S are reserved symbols",
/*  4 */ "Symbol table overflow",
/*  5 */ "Duplicate definition",
/*  6 */ "Sync error",
/*  7 */ "Branch out of range",
/*  8 */ "Missing operand",
/*  9 */ "Invalid addressing mode",
/* 10 */ "Missing string terminator",
/* 11 */ "Value error",
/* 12 */ "Undefined symbol",
/* 13 */ "Symbol required",
/* 14 */ "Multiple definition",
/* 15 */ "Relocation error",
/* 16 */ "Conditional assembly nested too deeply",
/* 17 */ "Unbalanced parentheses",
/* 18 */ "Divide by zero",
/* 19 */ "Hi/Lo operator must be first",
/* 20 */ "Invalid operand field",
/* 21 */ "Out of memory",
/* 22 */ ".fi without .if",
/* 23 */ "Label required",
/* 24 */ "Label not allowed",
/* 25 */ "Bad addressing mode",
/* 26 */ "ROM directive must be before .CSEG",
/* 27 */ "Too many code segments",
/* 28 */ "Invalid assembler source filename, (.a expected)",
/* 29 */ "Only one include level supported",
/* 30 */ "Bad include file syntax",
/* 31 */ "Error in opening include file"
};

  if ((code == 15) && !aflag) return;
  listed = 1;
  printf("%s\n",prlnbuf);
  printf ("%*s^\n", cpos, "");
  printf("Error: %2d ",code);
  if (code<1 || code>31) code = 0;
  printf("%s\n", errmsg[code]);
  if (!lflag) {
    fprintf(lptr,"%s\n",prlnbuf);
    fprintf(lptr,"%*s^\n", cpos, "");
    fprintf(lptr,"Error: %2d ",code);
    if (code<1 || code>31) code = 0;
    fprintf(lptr,"%s\n", errmsg[code]);
  }
  errcnt++;
}

/*
  -------------------------------------
  Centralized warning strings
  only processed on pass 2
  -------------------------------------
*/

VOID warning(code)
unsigned code;
{

static char *warmsg[] = {
/*  0 */ "UNKNOWN Warning!!!",
/*  1 */ "Absolute use of a relative symbol",
/*  2 */ "specified rom space overflow",
/*  3 */ "ROM code space overlap"
};

  if (pass) {
    if ((code == 1) && !aflag) return;
    listed = 1;
    printf("%s\n",prlnbuf);
    printf ("%*s^\n", cpos, "");
    printf("Warning: %2d ",code);
    if (code<1 || code>3) code = 0;
    printf("%s\n", warmsg[code]);
    if (!lflag) {
      fprintf(lptr,"%s\n",prlnbuf);
      fprintf(lptr,"%*s^\n", cpos, "");
      fprintf(lptr,"Warning: %2d ",code);
      if (code<1 || code>3) code = 0;
      fprintf(lptr,"%s\n", warmsg[code]);
    }
    warcnt++;
  }
}

/*
  -------------------------------------
  symbol table print
  -------------------------------------
*/

VOID stprnt()
{
  static char symtp[] = {
    /* 0 INVALID */ 'I',
    /* 1 DEFNOT  */ 'U',
    /* 2 DEFMUL  */ 'M',
    /* 3 DEFZRO  */ 'Z',
    /* 4 DEFABS  */ 'A',
    /* 5 DEFLNG  */ 'L',
    /* 6 DEFREL  */ 'R',
    /* 7 DEFEXT  */ 'E'
  };

  static char pubfl[] = {
    /* LOCAL   */ ' ',
    /* PUBLIC  */ 'P'
  };

  char i,j,f,p;
  struct symtype *ptr, *nxtptr;

  if ((!lflag) && (sflag)) fprintf(lptr, "\nSYMBOL TABLE:\n");
  j = 0;
  for (i=0; i<HTSIZE; i++) {
    for (ptr = hash_tbl[i]; ptr ; ptr=nxtptr) {
      if ((!lflag) && (sflag)) {
        f = ptr->flag;
        if (f & 0x80) p = 1;
        else p = 0;
        f &= 0x07;
        fprintf(lptr,"  %c%c %02x:%04x %-*s", pubfl[p],
             symtp[f],ptr->valbnk, ptr->value,SBOLSZ,ptr->name);    
        j++;
        if (j > 2) j=0;
        if (!j) putc('\n',lptr);   /*  == 0 */
        else putc(' ',lptr);
      }
      if (mflag) {
        fprintf(mptr,"%-*s",SBOLSZ,ptr->name);    
        fputc(ptr->flag,mptr);
        fputc(ptr->value,mptr);
        fputc(ptr->valbnk,mptr);
      }
      nxtptr = ptr->next;
      free(ptr->name);
      free(ptr);
    }
  }
  if ((!lflag) && (sflag)) putc('\n',lptr);
}
