/*
 -------------------------------------
 Filename: x816c.c
 Version 1.013
 Portions are Copyright (c)1990 DataQue Software
 65816 Assembler
 -------------------------------------
 directory of functions in this module:

 nextch()            increment line buffer pointer
 getch()             get next char from line buffer
 skip()              skip past white space in line buffer
 cskip()             conditional skip
 assemble()          main assembler control
 println()           output contents of line buffer
 colsym()            collect all symbols in the line buffer
 islstrt ()          check for valid symbol character
 stlook()            symbol table lookup
 stinstal(hv)        symbol install into hash table
 oplook()            operation table lookup
 loadlc(val,f)       convert word value and place into line buffer
 loadlw(bnk,val)     convert long word and place into line buffer
 loadv(val,f)        convert byte value and place into line buffer
 hexcon(digit,num)   convert binary code to string of hex characters
 labldef(lval,ltyp)  assign value to label
                    
 -------------------------------------
*/

#include <stdio.h>
#include <string.h>
#include "x816a.h"
#include "x816b.h"

extern char *opmnem[];
extern unsigned optab[];

struct symtype *stlook();
struct symtype *stinstal();
static unsigned symlen;

/*
  -------------------------------------
  Advance line pointer to next character
  and store character in 'ch' char nextch()
  -------------------------------------
*/

char nextch()
{
  return((ch = prlnbuf[++cpos]));
}

/*
  -------------------------------------
  Insure we have the current character
  -------------------------------------
*/

char getch()
{
  return((ch = prlnbuf[cpos]));
}

/*
  -------------------------------------
  Unconditional skip to next non-blank
  -------------------------------------
*/

char skip()
{
  while (nextch() == ' ' || ch == '\t');   /* skip spaces or tabs */
  return (ch);
}

/* 
  -------------------------------------
  Conditional skip to next non-blank 
  -------------------------------------
*/

char cskip()
{
  if ((ch == ' ') || (ch == '\t')) skip();
  return (ch);
}

/* 
  -------------------------------------
  translate source line to machine language
  -------------------------------------
*/

VOID assemble()
{
  int flg;

  cpos = SFIELD;                               /* set starting position */
  getch();                                     /* get starting character */
  if ((ch == ';') || (!ch)) return;            /* comment or EOL */
  lablptr = NULL;
  if (colsym()) lablptr = stlook();
  cskip();
  
  if ((flg = oplook()) >= 0 && opflg == CONDIT) {
    cassm();
    return;
  }
  if (noasm()) {                               /* dont generate obj */
    listed |= !cflag;
    return;
  }
  if (flg < 0) {
    labldef(codbnk,codloc,DEFREL);             /* -1 = invalid operation */
    if (flg == -1) error(2);                   /* -2 = no opcode */
    if ((flg == -2) && (pass) && (lablptr)) loadlw(codbnk, codloc);
    return;                       
  }
  if (opflg == PSEUDO) pseudo();               /* directives */
  else {
    labldef(codbnk,codloc,DEFREL);
    if (opflg == CLASS1) class1();             /* one byte instructions */
    else if (opflg == CLASS2) class2();        /* two byte relatives */
    else if (opflg == CLASSW) class2();        /* three byte relatives */
    else class3();                             /* all other ops */
  }
  return;
}

/* 
  -------------------------------------
  printline prints the contents of prlnbuf
  -------------------------------------
*/

VOID println()
{
  if (!lflag) fprintf(lptr,"%s\n", prlnbuf);
}

/*
  -------------------------------------
  colsym() collects a symbol from prlnbuf
  into symbol[], leaves prlnbuf pointer 
  at first invalid symbol character,
  returns symbol length, or  0 => no
  symbol collected
  -------------------------------------
*/

unsigned colsym()
{
  unsigned i,valid;

  valid = 1;
  i = 0;
  while (valid) {
    if (islstrt());
    else if (i >= 1 && index("0123456789$",ch));
    else valid = 0;
    if (valid) {
      if (i < SBOLSZ ) symbol[i++] = ch;
      nextch();
    }
  }
  if (i == 1 && index("AaXxYySs", *symbol)) {
    error(3);                                  /* reserved symbol */
    i = 0;
  }
  symbol[i] = 0;
  return(i);
}

/*
  -------------------------------------
  ckeck for valid character
  -------------------------------------
*/

unsigned islstrt()
{
  char c;

  c = tolower( ch );
  return (index("_.abcdefghijklmnopqrstuvwxyz", c) != NULL);
}

/*
  -------------------------------------
  symbol table lookup.  if found, return
  pointer to symbol else, install symbol
  as undefined, and return pointer
  -------------------------------------
*/

struct symtype *stlook()
{
  char *sym;
  struct symtype *ptr;
  int hashv;

  hashv = 0;

  for (sym=symbol; *sym ; sym++) hashv += *sym;
  symlen = sym - symbol;

  hashv %= HTSIZE;
  for (ptr=hash_tbl[hashv]; ptr ; ptr = ptr->next) {
    if (!strcmp(symbol,ptr->name)) return(ptr);
  }
  return(stinstal(hashv));
}

/*
  -------------------------------------
  install symbol
  -------------------------------------
*/

struct symtype *stinstal(hv)
int hv;
{
  struct symtype *newsym;

  newsym = (struct symtype *) makerec(sizeof(struct symtype));
  newsym->flag = DEFNOT;
  newsym->name = (char *) makerec(symlen+1);
  strcpy(newsym->name,symbol);
  newsym->value = 0;
  newsym->valbnk = 0;
  newsym->next = hash_tbl[hv];
  hash_tbl[hv] = newsym;                       /* put at head of list */
  return(newsym);
}

/* 
  -------------------------------------
  operation code table lookup
  returns:
    index in 'opmnem' of symbol, if valid
    -1 if invalid
    -2 if no opcode collected
  -------------------------------------
*/

int oplook()
{
  unsigned c,h,i,j;
  int k;
  char temp[OPSZ+1];

  for (i=0;i<OPSZ;i++) temp[i] = ' ';
  i = j = 0;

  while(ch && !index(" ;\t", ch)) {
    c = tolower(ch);
    if ( index(".=abcdefghijklmnopqrstuvwxyz", c) ) temp[j] = c;
    else return(-1);

    if (++j > OPSZ) return(-1);     /* not found */
    if (c == '=') break;

    nextch();
  }
  if (!j) return(-2);             /* no opcode field */

/*
  -------------------------------
  use variables as follows:
  i = mnemonic table index (mid)
  j = lowest acceptable (low)
  h = highest acceptable (high)
  k = compare designator
  -------------------------------
*/

  temp[OPSZ] = '\0';                /* terminate with null */

  j = 0;
  h = NUMOPS-1;

  while (j <= h) {                 /* new binary search */
    i = ((j + h) >> 1);            /* divide by 2 */
	 k=strcmp(temp,opmnem[i]);
    if (k < 0) h = (i - 1);        /* go lower  */
    else if (k > 0) j = (i + 1);   /* go higher */
	 else {
	   j = (i << 1);                /* multiply by 2 */
      opoff = j;
      opflg = optab[j];            /* match */
      opval = optab[++j];
      return(i);
    } 
  }
  return(-1);
}

/*
  -------------------------------------
  load 16 bit value in printable form
  into prlnbuf
  -------------------------------------
*/

VOID loadlc(val, f)
unsigned val,f;
{
  char i;
  static char pos[] = {6, 13, 17};

  i = pos[f];
  hexcon(4, val);
  prlnbuf[i++] = hex[0];
  prlnbuf[i++] = hex[1];
  prlnbuf[i++] = hex[2];
  prlnbuf[i++] = hex[3];
}

/*
  -------------------------------------
  load 16 bit value in printable form
  into prlnbuf
  -------------------------------------
*/

VOID loadlw(bnk, val)
char bnk;
unsigned val;
{
  hexcon(2,bnk);
  prlnbuf[6] = hex[0];
  prlnbuf[7] = hex[1];
  hexcon(4,val);
  prlnbuf[8] = hex[0];
  prlnbuf[9] = hex[1];
  prlnbuf[10] = hex[2];
  prlnbuf[11] = hex[3];
}

/*
  -------------------------------------
  load value in hex into prlnbuf[contents[i]] 
  -------------------------------------
*/

VOID loadv(val,f)
unsigned  val,f;
{
  char i;

  i = (f << 1);                    /* fast multiply by 2 */
  hexcon(2, val);
  prlnbuf[13 + i] = hex[0];
  prlnbuf[14 + i] = hex[1];
}

/*
  -------------------------------------
  convert number supplied as argument
  to hexadecimal in hex[digit-1] (lsd)
  through hex[0] (msd)     
  -------------------------------------
*/

VOID hexcon(digit, num)
  unsigned digit,num;
{
  hex[digit] = 0;
  while (digit) {
    hex[--digit] = (num & 0x0f) + '0';
    if (hex[digit] > '9') hex[digit] += 'A' -'9' - 1;
    num >>= 4;
  }
}

/*
  -------------------------------------
  assign <value> to label pointed to by
  lablptr, checking for valid definition,
  etc. called with:
          value to be assigned
          label type (flag)
  returns:
          0 => success
          1 => multiply defined label
          2 => sync error
  -------------------------------------
*/

VOID labldef(lbnk,lval,ltyp)
unsigned lval,ltyp;
char lbnk;
{
  if (lablptr) {
    if (!pass) {
      if (lablptr->flag == DEFNOT ) {
        lablptr->value = lval;
        lablptr->valbnk = lbnk;
        lablptr->flag = ltyp;
      } else {
        lablptr->flag = DEFMUL;
        lablptr->value = 0;
        lablptr->valbnk = 0;
        error(5);                      /* multiply defined */
        return;
      }
    } else {
      if (lablptr->value != lval) {
        error(6);                      /* sync error */
        return;
      }
    }
  }
  return;
}
