(*----------------------------------------------------------------------*)
(*      Async_Carrier_Detect --- Check for modem carrier detect         *)
(*----------------------------------------------------------------------*)

FUNCTION Async_Carrier_Detect : BOOLEAN;

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*     Function:   Async_Carrier_Detect                                 *)
(*                                                                      *)
(*     Purpose:    Looks for modem carrier detect                       *)
(*                                                                      *)
(*     Calling Sequence:                                                *)
(*                                                                      *)
(*        Flag := Async_Carrier_Detect : BOOLEAN;                       *)
(*                                                                      *)
(*           Flag is set TRUE if carrier detected, else FALSE.          *)
(*                                                                      *)
(*     Calls:  None                                                     *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

BEGIN (* Async_Carrier_Detect *)

   Async_Carrier_Detect := ODD( Port[ UART_MSR + Async_Base ] SHR 7 ) OR
                           Async_Hard_Wired_On;

END   (* Async_Carrier_Detect *);

(*----------------------------------------------------------------------*)
(*      Async_Carrier_Drop --- Check for modem carrier drop/timeout     *)
(*----------------------------------------------------------------------*)

FUNCTION Async_Carrier_Drop : BOOLEAN;

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*     Function:   Async_Carrier_Drop                                   *)
(*                                                                      *)
(*     Purpose:    Looks for modem carrier drop/timeout                 *)
(*                                                                      *)
(*     Calling Sequence:                                                *)
(*                                                                      *)
(*        Flag := Async_Carrier_Drop : BOOLEAN;                         *)
(*                                                                      *)
(*           Flag is set TRUE if carrier dropped, else FALSE.           *)
(*                                                                      *)
(*     Calls:  None                                                     *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

BEGIN (* Async_Carrier_Drop *)

   Async_Carrier_Drop := NOT ( ODD( Port[ UART_MSR + Async_Base ] SHR 7 ) OR
                               Async_Hard_Wired_On );

END   (* Async_Carrier_Drop *);

(*----------------------------------------------------------------------*)
(*      Async_Term_Ready --- Set terminal ready status                  *)
(*----------------------------------------------------------------------*)

PROCEDURE Async_Term_Ready( Ready_Status : BOOLEAN );

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*     Procedure:  Async_Term_Ready                                     *)
(*                                                                      *)
(*     Purpose:    Sets terminal ready status                           *)
(*                                                                      *)
(*     Calling Sequence:                                                *)
(*                                                                      *)
(*        Async_Term_Ready( Ready_Status : BOOLEAN );                   *)
(*                                                                      *)
(*           Ready_Status --- Set TRUE to set terminal ready on,        *)
(*                            Set FALSE to set terminal ready off.      *)
(*                                                                      *)
(*     Calls:  None                                                     *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

VAR
   Mcr_Value: BYTE;

BEGIN (* Async_Term_Ready *)

   Mcr_Value := Port[ UART_MCR + Async_Base ];

   IF ODD( Mcr_Value ) THEN Mcr_Value := Mcr_Value - 1;

   IF Ready_Status THEN Mcr_Value := Mcr_Value + 1;

   Port[ UART_MCR + Async_Base ] := Mcr_Value;

   Async_Clear_Errors;

END   (* Async_Term_Ready *);

(*----------------------------------------------------------------------*)
(*          Async_Buffer_Check --- Check if character in buffer         *)
(*----------------------------------------------------------------------*)

FUNCTION Async_Buffer_Check : BOOLEAN;

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*     Function:   Async_Buffer_Check                                   *)
(*                                                                      *)
(*     Purpose:    Check if character in buffer                         *)
(*                                                                      *)
(*     Calling Sequence:                                                *)
(*                                                                      *)
(*        Flag := Async_Buffer_Check : BOOLEAN;                         *)
(*                                                                      *)
(*           Flag returned TRUE if character received in buffer,        *)
(*           Flag returned FALSE if no character received.              *)
(*                                                                      *)
(*     Calls:  None                                                     *)
(*                                                                      *)
(*     Remarks:                                                         *)
(*                                                                      *)
(*       This routine only checks if a character has been received      *)
(*       and thus can be read; it does NOT return the character.        *)
(*       Use Async_Receive to read the character.                       *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

BEGIN   (* Async_Buffer_Check *)

   Async_Buffer_Check := ( Async_Buffer_Head <> Async_Buffer_Tail );

END     (* Async_Buffer_Check *);

(*----------------------------------------------------------------------*)
(*          Async_Line_Error --- Check if line status error occurred    *)
(*----------------------------------------------------------------------*)

FUNCTION Async_Line_Error( VAR Error_Flags: BYTE ) : BOOLEAN;

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*     Function:   Async_Line_Error                                     *)
(*                                                                      *)
(*     Purpose:    Check if line status error occurred                  *)
(*                                                                      *)
(*     Calling Sequence:                                                *)
(*                                                                      *)
(*        Flag := Async_Line_Error(VAR Error_Flags: BYTE): BOOLEAN;   *)
(*                                                                      *)
(*           Error_Flags --- Current error flags                        *)
(*                                                                      *)
(*           Flag returned TRUE if line status error occurred,          *)
(*           Flag returned FALSE if no error.                           *)
(*                                                                      *)
(*     Calls:  None                                                     *)
(*                                                                      *)
(*     Remarks:                                                         *)
(*                                                                      *)
(*        The line status error flag is cleared here.                   *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

BEGIN   (* Async_Line_Error *)

   Async_Line_Error       := ( Async_Line_Error_Flags <> 0 );
   Error_Flags            := Async_Line_Error_Flags;
   Async_Line_Error_Flags := 0;

END     (* Async_Line_Error  *);

(*----------------------------------------------------------------------*)
(*            Async_Ring_Detect --- Check for phone ringing             *)
(*----------------------------------------------------------------------*)

FUNCTION Async_Ring_Detect : BOOLEAN;

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*     Function:   Async_Ring_Detect                                    *)
(*                                                                      *)
(*     Purpose:    Looks for phone ringing                              *)
(*                                                                      *)
(*     Calling Sequence:                                                *)
(*                                                                      *)
(*        Flag := Async_Ring_Detect : BOOLEAN;                          *)
(*                                                                      *)
(*           Flag is set TRUE if ringing detected, else FALSE.          *)
(*                                                                      *)
(*     Calls:  None                                                     *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

BEGIN (* Async_Ring_Detect *)

   Async_Ring_Detect := ODD( Port[ UART_MSR + Async_Base ] SHR 6 );

END   (* Async_Ring_Detect *);

(*----------------------------------------------------------------------*)
(*          Async_Send_Break --- Send break (attention) signal          *)
(*----------------------------------------------------------------------*)

PROCEDURE Async_Send_Break;

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*     Procedure:  Async_Send_Break                                     *)
(*                                                                      *)
(*     Purpose:    Sends break signal over communications port          *)
(*                                                                      *)
(*     Calling Sequence:                                                *)
(*                                                                      *)
(*        Async_Send_Break;                                             *)
(*                                                                      *)
(*     Calls:  None                                                     *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

VAR
   Old_Lcr   : BYTE;
   Break_Lcr : BYTE;

BEGIN (* Async_Send_Break *)

   Old_Lcr   := Port[ UART_LCR + Async_Base ];
   Break_Lcr := Old_Lcr;

   IF Break_Lcr >  127 THEN Break_Lcr := Break_Lcr - 128;
   IF Break_Lcr <=  63 THEN Break_Lcr := Break_Lcr +  64;

   Port[ UART_LCR + Async_Base ] := Break_Lcr;

   DELAY( Async_Break_Length * 10 );

   Port[ UART_LCR + Async_Base ] := Old_Lcr;

END   (* Async_Send_Break *);

(*----------------------------------------------------------------------*)
(*     Async_Send_String --- Send string over communications port       *)
(*----------------------------------------------------------------------*)

PROCEDURE Async_Send_String( S : AnyStr );

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*     Procedure:  Async_Send_String                                    *)
(*                                                                      *)
(*     Purpose:    Sends string out over communications port            *)
(*                                                                      *)
(*     Calling Sequence:                                                *)
(*                                                                      *)
(*        Async_Send_String( S : AnyStr );                              *)
(*                                                                      *)
(*           S --- String to send                                       *)
(*                                                                      *)
(*     Calls:  Async_Send                                               *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

VAR
   I : INTEGER;

BEGIN  (* Async_Send_String *)

  FOR I := 1 TO LENGTH( S ) DO
     Async_Send( S[I] )

END    (* Async_Send_String *);

(*----------------------------------------------------------------------*)
(*     Async_Send_String_With_Delays --- Send string with timed delays  *)
(*----------------------------------------------------------------------*)

PROCEDURE Async_Send_String_With_Delays( S          : AnyStr;
                                         Char_Delay : INTEGER;
                                         EOS_Delay  : INTEGER  );

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*     Procedure:  Async_Send_String_With_Delays                        *)
(*                                                                      *)
(*     Purpose:    Sends string out over communications port with       *)
(*                 specified delays for each character and at the       *)
(*                 end of the string.                                   *)
(*                                                                      *)
(*     Calling Sequence:                                                *)
(*                                                                      *)
(*        Async_Send_String_With_Delays( S          : AnyStr ;          *)
(*                                       Char_Delay : INTEGER;          *)
(*                                       EOS_Delay  : INTEGER );        *)
(*                                                                      *)
(*           S          --- String to send                              *)
(*           Char_Delay --- Number of milliseconds to delay after       *)
(*                          sending each character                      *)
(*           EOS_Delay  --- Number of milleseconds to delay after       *)
(*                          sending last character in string            *)
(*                                                                      *)
(*     Calls:  Async_Send                                               *)
(*             Async_Send_String                                        *)
(*             Length                                                   *)
(*             Delay                                                    *)
(*                                                                      *)
(*     Remarks:                                                         *)
(*                                                                      *)
(*        This routine is useful when writing routines to perform       *)
(*        non-protocol uploads.  Many computer systems require delays   *)
(*        between receipt of characters for correct processing.  The    *)
(*        delay for end-of-string usually applies when the string       *)
(*        represents an entire line of a file.                          *)
(*                                                                      *)
(*        If delays are not required, Async_Send_String is faster.      *)
(*        This routine will call Async_Send_String is no character      *)
(*        delay is to be done.                                          *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

VAR
   I : INTEGER;

BEGIN  (* Async_Send_String_With_Delays *)

   IF Char_Delay <= 0 THEN
      Async_Send_String( S )
   ELSE
      FOR I := 1 TO LENGTH( S ) DO
         BEGIN
            Async_Send( S[I] );
            Delay( Char_Delay );
         END;

   IF EOS_Delay > 0 THEN Delay( EOS_Delay );

END    (* Async_Send_String_With_Delays *);

(*----------------------------------------------------------------------*)
(*      Async_Percentage_Used --- Report Percentage Buffer Filled       *)
(*----------------------------------------------------------------------*)

FUNCTION Async_Percentage_Used : REAL;

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*     Function:   Async_Percent_Used                                   *)
(*                                                                      *)
(*     Purpose:    Reports percentage of com buffer currently filled    *)
(*                                                                      *)
(*     Calling Sequence:                                                *)
(*                                                                      *)
(*        Percentage := Async_Percentage_Used : Real;                   *)
(*                                                                      *)
(*           Percentage gets how much of buffer is filled;              *)
(*           value goes from 0.0 (empty) to 1.0 (totally full).         *)
(*                                                                      *)
(*     Calls:  None                                                     *)
(*                                                                      *)
(*     Remarks:                                                         *)
(*                                                                      *)
(*       This routine is helpful when incorporating handshaking into    *)
(*       a communications program.  For example, assume that the host   *)
(*       computer uses the XON/XOFF (DC1/DC3) protocol.  Then the       *)
(*       PC program should issue an XOFF  to the host when the value    *)
(*       returned by Async_Percentage_Used > .75 or so.  When the       *)
(*       utilization percentage drops below .25 or so, the PC program   *)
(*       should transmit an XON.                                        *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

BEGIN (* Async_Percentage_Used *)

   Async_Percentage_Used := Async_Buffer_Used / ( Async_Buffer_Size + 1 );

END   (* Async_Percentage_Used *);

(*----------------------------------------------------------------------*)
(*     Async_Purge_Buffer --- Purge communications input buffer         *)
(*----------------------------------------------------------------------*)

PROCEDURE Async_Purge_Buffer;

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*     Procedure:  Async_Purge_Buffer                                   *)
(*                                                                      *)
(*     Purpose:    Purges communications input buffer                   *)
(*                                                                      *)
(*     Calling Sequence:                                                *)
(*                                                                      *)
(*        Async_Purge_Buffer;                                           *)
(*                                                                      *)
(*     Calls:  Async_Receive                                            *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

VAR
   C: CHAR;
   L: INTEGER;

BEGIN  (* Async_Purge_Buffer *)

   L     := 10000 DIV Async_Baud_Rate;

   IF L <= 0 THEN
      L := 3;

   REPEAT
      DELAY( L )
   UNTIL ( NOT Async_Receive( C ) );

END    (* Async_Purge_Buffer *);

(*----------------------------------------------------------------------*)
(*          Async_Peek --- Peek ahead in communications buffer          *)
(*----------------------------------------------------------------------*)

FUNCTION Async_Peek( Nchars : INTEGER ) : CHAR;

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*     Function:   Async_Peek                                           *)
(*                                                                      *)
(*     Purpose:    Peeks ahead in comm buffer                           *)
(*                                                                      *)
(*     Calling Sequence:                                                *)
(*                                                                      *)
(*        Ch := Async_Peek( NChars: INTEGER) : CHAR;                    *)
(*                                                                      *)
(*           NChars --- # of characters to peek ahead                   *)
(*           Ch     --- returned (peeked) character                     *)
(*                                                                      *)
(*     Calls:  None                                                     *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

VAR
   I: INTEGER;

BEGIN (* Async_Peek *)

   I := ( Async_Buffer_Tail + NChars ) MOD Async_Buffer_Size;

   IF ( I > Async_Buffer_Head ) THEN
      Async_Peek := CHR( 0 )
   ELSE
      Async_Peek := Async_Buffer_Ptr^[ I ];

END   (* Async_Peek *);

(*----------------------------------------------------------------------*)
(*         Async_Setup_Port --- Setup port address and IRQ line         *)
(*----------------------------------------------------------------------*)

PROCEDURE Async_Setup_Port( ComPort       : INTEGER;
                            Base_Address  : INTEGER;
                            IRQ_Line      : INTEGER;
                            RS_232_Base   : INTEGER );

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*     Procedure:  Async_Setup_Port                                     *)
(*                                                                      *)
(*     Purpose:    Sets up port address and IRQ line                    *)
(*                                                                      *)
(*     Calling Sequence:                                                *)
(*                                                                      *)
(*        Async_Setup_Port( ComPort       : INTEGER;                    *)
(*                          Base_Address  : INTEGER;                    *)
(*                          IRQ_Line      : INTEGER;                    *)
(*                          RS_232_Base   : INTEGER );                  *)
(*                                                                      *)
(*           ComPort      --- which port (1 though MaxComPorts)         *)
(*           Base_Address --- Base address of port                      *)
(*           IRQ_Line     --- IRQ line for interrupts for port          *)
(*           RS_232_Base  --- RS 232 base address for port              *)
(*                                                                      *)
(*     Calls:  None                                                     *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

BEGIN  (* Async_Setup_Port *)

   IF ( ( ComPort > 0 ) AND ( ComPort <= MaxComPorts ) ) THEN
      BEGIN
         Com_Base [ComPort] := Base_Address;
         Com_Irq  [ComPort] := IRQ_Line;
         Com_RS232[ComPort] := RS_232_Base;
      END;

END    (* Async_Setup_Port *);

(*----------------------------------------------------------------------*)
(*         Async_Release_Buffers --- Release buffers for serial ports   *)
(*----------------------------------------------------------------------*)

PROCEDURE Async_Release_Buffers;

(*----------------------------------------------------------------------*)
(*                                                                      *)
(*     Procedure:  Async_Release_Buffers;                               *)
(*                                                                      *)
(*     Purpose:    Releases send and receive buffers                    *)
(*                                                                      *)
(*     Calling Sequence:                                                *)
(*                                                                      *)
(*        Async_Release_Buffers;                                        *)
(*                                                                      *)
(*     Calls:  None                                                     *)
(*                                                                      *)
(*----------------------------------------------------------------------*)

BEGIN  (* Async_Release_Buffers *)

                             (* If port open, close it down first.  *)
   IF Async_Open_Flag THEN
      Async_Close( FALSE );

   FREEMEM( Async_Buffer_Ptr  , Async_Buffer_Size  + 1 );
   FREEMEM( Async_OBuffer_Ptr , Async_OBuffer_Size + 1 );

END    (* Async_Release_Buffers *);
