/*

   Program: DEMO.PRG
   Purpose: Multi-window demonstration of a Clipper 5.0 application
            written to run under Microsoft Windows (with Clip-4-Win)
   Authors: John Skelton and Greg Lief
   Date:    November and December 1992

   This demo is Copyright (c) 1992 by John Skelton and Greg Lief.

   Clip-4-Win is Copyright (c) 1992 by John Skelton.

   The NTXREC() and NTXPOS() functions are based on Rick Spence's book,
   were written in Microsoft C and are provided by kind courtesy of
   Graham McKechnie @ RCM Software Pty. Ltd.  At the very least, Robert
   DiFalco and Don Caton have also contributed.  These functions make
   use of an internal function entitled _ntxhandle(), which was written
   in Assembler by Ted Means.  John Skelton has further modified them.

   Clip-4-Win is available from the following sources:

   *** North America, South America, Asia, and Australia ***
   (as well as contact point for dealers/resellers in those areas)

   Grumpfish, Inc.
   2450 Lancaster Drive, NE
   Salem, Oregon 97305
   USA
   Tel: +1 (503) 588-1815
   (USA Toll-free 800-367-7613)
   Fax: +1 (503) 588-1980
   BBS: +1 (503) 588-7572
   CompuServe: 71064,2543
*/

external descend  // for index files that might need it

// necessary in the event of pushbutton movement on data entry screen
// (see WREADER.PRG for complete structure of get:cargo array)
#define FORCE_FOCUS   6

// this one handles fields
#xcommand @ <row>, <col> SAY <prompt> GET [FIELD] <fname>		;
                         IN WINDOW <w>					;
                         [GETROW <grow>]				;
                         [GETCOL <gcol>]				;
                         [PICTURE <pic>]				;
                         [BUTTONS <buttons>]				;
                         [VALID <valid>]				;
                         [WHEN <when>]					;
                         [SAYCOLOR <saycolor>]				;
                         [GETCOLOR <getcolor>]				;
									;
      => AAdd( GetList, GetNew( <grow>, <gcol>, fieldblock(<fname>),	;
                                <"fname">, <pic>, <getcolor> )		;
             )								;
      ; ATail(GetList):reader := { | g, ev | WGetReader(g, ev, <w>,	;
      					  getlist,			;
                                          <buttons>, BUTTON_HEIGHT )	;
                                 }					;
      ; ATail(GetList):cargo := {					;
                  NIL , <prompt>, <row>, <col>, <saycolor>, .f., .f.	;
                                }					;
      [; ATail(GetList):postBlock := <{valid}>]				;
      [; ATail(GetList):preBlock := <{when}>]

// this one handles variables
#xcommand @ <row>, <col> SAY <prompt> GET <var>				;
                         IN WINDOW <w>					;
                         REDRAW WITH <redraw>				;
                         [GETROW <grow>]				;
                         [GETCOL <gcol>]				;
                         [PICTURE <pic>]				;
                         [BUTTONS <buttons>]				;
                         [VALID <valid>]				;
                         [WHEN <when>]					;
                         [SAYCOLOR <saycolor>]				;
                         [GETCOLOR <getcolor>]				;
									;
      => AAdd( GetList, GetNew( <grow>, <gcol>,				;
               { | _1 | if(_1 == NIL, <var>, <var> := _1) }, <(var)>,	;
               <pic>, <getcolor> )					;
             )								;
      ; ATail(GetList):reader := { | g, ev | WGetReader(g, ev, <w>,	;
      					  getlist,			;
                                          <buttons>, BUTTON_HEIGHT )	;
                                 }					;
      ; ATail(GetList):cargo := {					;
                  NIL , <prompt>, <row>, <col>, <saycolor>, .f., .f.	;
                                }					;
      [; ATail(GetList):postBlock := <{valid}>]				;
      [; ATail(GetList):preBlock := <{when}>]

#xtranslate StripPath( <f> ) => ;
           if("\" $ <f>, substr(<f>, rat("\", <f>) + 1), <f>)

#xtranslate StripExt( <fname> ) => ;
      if('.' $ <fname>, substr( <fname>, 1, at('.', <fname>) - 1), <fname> )

// if you have "Sound Explosion", uncomment the following statement
// and make sure that WAVE_DIRECTORY points to the directory where
// your Sound Explosion .WAV files reside
// #define SOUND_EXPLOSION
#define WAVE_DIRECTORY  "C:\WINDOWS\"

#define WIN_WANT_CLIPBOARD  // to make Clipboard directives accessible
#define WIN_WANT_LBS        // to make Listbox styles accessible
#define WIN_WANT_ALL

#include "dbstruct.ch"
#include "directry.ch"
#include "error.ch"
#include "getexit.ch"
#include "inkey.ch"
#include "windows.ch"
#include "setcaret.ch"
#include "font.ch"

#define	CR	chr(13)

// remove this if you do not want push buttons on data entry screen
#define BUTTONS

// manifest constants for GetClientRect() and GetDIBRect() arrays
// note that since Top and Left are always 0, they are unused
#define W_RIGHT  3
#define W_BOTTOM 4

// manifest constants for RGB() color combinations
#define C_RED        RGB(255,0,0)
#define C_BLUE       RGB(0,0,255)
#define C_GREEN      RGB(0,255,0)
#define C_MAGENTA    RGB(255,0,255)
#define C_BLACK      RGB(0,0,0)

#define APP_NAME     "Clip-4-Win"


static hWnd  // main (parent) window... must be visible in several places
static cText := ""   // text in main window if a file is opened... must be
                     // visible in several places for cut/copy to Clipboard
static cDIB          // handle for bitmapped logo in main window... must be
                     // visible in several places for copying to Clipboard
static nMainEvId     // ID of main event handler... visible throughout
                     // because it must be reset periodically


/*
   Function: Main()
*/
function main
local hMenu, nEvent, hCurrWnd
set scoreboard off     // don't even THINK about using it in Windows!!
hWnd := WinSetup(APP_NAME, "Clip-4-Win Demo")
hMenu := MenuSetup()
HideCaret(hWnd)
nMainEvId := AddHandler(hWnd, {|nEvent| MainEvent(nEvent)})
/*
   The C4W_AutoClose() function allows us to make the user confirm their
   decision to quit the app, even if they try to close the window
   with ALT-F4 (or via the System menu).  It will cause the EVENT_CLOSE
   event to be generated, which we can then react to (see below)
*/
C4W_AutoClose(.f.)
SetHandleCount(40)
do while .t.
   do while (nEvent := ChkEvent()) == EVENT_NONE
      // "background" processing could go here
   enddo
   HandleEvent(nEvent)
   do case
   case nEvent == EVENT_CLOSE
      // determine if main window is currently active
      hCurrWnd := _LasthWnd()
      if hWnd == hCurrWnd
         DoExit()
      else
         DestroyWindow(hCurrWnd)
         SetFocus(hWnd)
      endif
   case nEvent == EVENT_QUIT
      quit
   endcase
enddo
return nil


/*
   Function: MainEvent()
*/
static function MainEvent(nEvent)
local hDC
local aDIBRect, aClientRect
do case
case nEvent == EVENT_REDRAW
   hDC := GetDC(hWnd)
   if cDIB == NIL
      cDIB := ReadDIB("clip4win.bmp")
   endif
   aDIBRect := GetDIBRect(cDIB)
   aClientRect := GetClientRect(hWnd)
   ShowDIB(hDC, cDIB, (aClientRect[W_RIGHT] - aDIBRect[W_RIGHT]) / 2, ;
                      (aClientRect[W_BOTTOM] - aDIBRect[W_BOTTOM]) / 2)
   ReleaseDC(hWnd, hDC)
endcase
return nil


/*
   Function: Credits()
*/
static function credits
MessageBox( , "By John Skelton and Greg Lief (rev 17-June-93)", ;
              "About Clip-4-Win Demo", MB_ICONASTERISK + MB_OK)
return nil


/*
   Function: DoExit()
*/
static function DoExit()
// Note that MessageBox() returns a value based on the I.D. of
// the selected item, all of which have corresponding ID* manifest
// constants in the WINDOWS.CH header file
if MessageBox(0, "Are you sure you want to exit this demo?", ;
		 "Leaving so soon?", MB_OKCANCEL + MB_ICONQUESTION) == IDOK
   if IsWindow(hWnd)
      DestroyWindow(hWnd)
   endif
   UnregisterClass(APP_NAME)
   quit
endif
return nil


/*
   Function: DoAudio()
   Note:     If you have Sound Explosion, use it instead of the
             somewhat feeble stock Windows .WAV files
*/
static function DoAudio()
#ifdef SOUND_EXPLOSION
   static waves_ := { "WHISTLE1", "SCREAM5", "LAUGH4", "BELCH2" }
#else
   static waves_ := { "CHORD", "CHIMES", "TADA", "DING" }
#endif
local hLib, cSound, hWnd, n
hLib   := LoadLibrary("MMSYSTEM.DLL")
cSound := GetProcAddress(hLib, "SndPlaySound", "Pascal", "Int", "str, int")

// note that Alert() allows you to specify trigger letters by
// preceding them with an ampersand
#ifdef SOUND_EXPLOSION
   n := alert("Pick a sound", ;
	      { "&Whistle", "&Scream", "&Laugh" } )
#else
   n := alert("Pick a sound", ;
	      { "&Chord", "Chi&mes", "&Tada" } )
#endif

// if they escaped out, use sound #4
if n == 0
   n := 4
endif

// second parameter to SndPlaySound() is: 1 == return instantly,
// 0 == wait until finished playing before returning

n := CallDLL(cSound, WAVE_DIRECTORY + waves_[n] + ".WAV", 1)
if n == 0
   MessageBox(, "No audio hardware" + CR + "Or other error", ;
	      MB_ICONHAND + MB_OK)
endif
return nil


/*
   Function: DoColor()
*/
static function DoColor()
static  nX := 20, nY := 50
local   nColor := ChooseColor(), hWnd
if nColor >= 0         // else user chose cancel/close or hit Esc
   hWnd := WinNew("Color", nX += 40, nY += 60, 150, 100)
   AddHandler(hWnd, {|nEvent| ColorEvent(nEvent, hWnd, nColor)})
endif
return nil


/*
   Function: ColorEvent()
*/
static function ColorEvent(nEvent, hWnd, nColor)
local	hDC, hBrush
do case
case nEvent == EVENT_REDRAW
   hDC := GetDC(hWnd)
   hBrush := CreateSolidBrush(nColor)
   FillRect(hDC, 20, 20, 100, 50, hBrush)
   DeleteObject(hBrush)
   ReleaseDC(hWnd, hDC)
endcase
return nil


/*
   Function: DoOpen()
*/
static function DoOpen
local cFile := GetOpenFileName(, "*.txt", "Select a text file")
local hMenu
if cFile <> NIL
   if directory(cFile)[1][F_SIZE] < 65535
      // because we want this text file to be displayed in the
      // main window, we must first delete the primary event handler
      DelHandler(nMainEvId)
      cText := memoread(cFile)
      nMainEvId := AddHandler(hWnd, ;
		      { | nEvent | CBEventText(nEvent, hWnd, cText)})

      // enable the "Clear" and "Cut" menu items ("Copy" already enabled)
      hMenu := GetMenu(hWnd)       // retrieve reference to main menu
      EnableMenuItem(hMenu, "clear", MF_ENABLED)
      EnableMenuItem(hMenu, "cut", MF_ENABLED)

      // force main window to be redrawn immediately
      InvalidateRect(hWnd)
   else
     MessageBox(hWnd, cFile + " is too large to load", "Error", ;
		      MB_ICONEXCLAMATION + MB_OK)
   endif
endif
return nil


/*
   Function: DaDoRunRun()
   Note:     Long live Phil Spector!
*/
static function DaDoRunRun
local cFile := GetOpenFileName(, "*.exe;*.com;*.bat", "Run", ;
			       { {"programs", "*.exe;*.com;*.bat"} } )
local hCursor
local hOldcursor
if cFile <> NIL
   hCursor := LoadCursor(, IDC_WAIT)
   hOldcursor := SetCursor(hCursor)
   WinExec(cFile)
   SetCursor(hOldcursor)   // restore previous cursor
endif
return nil


/*
   Function: DoClear()
*/
static function DoClear
if MessageBox(hWnd, "Are you sure you want to clear the current text?", ;
	      "Question", MB_OKCANCEL + MB_ICONQUESTION) == IDOK
   cText := ''
   ResetMainEvent()
endif
return nil


/*
   Function: DoCutCopy()
*/
static function DoCutCopy(c)
if OpenClipboard(hWnd)
   EmptyClipboard()
   // if there is no text in the main window, copy the logo instead
   if empty(cText)
      SetClipboardData(CF_DIB, cDIB)
   else
      SetClipboardData(CF_TEXT, "*** Pasted From Clip-4-Win ***" + CR + cText)
   endif
   CloseClipboard()

   // if we just "cut", clear text and reset to show the logo
   if c == "cut"
      cText := ""
      ResetMainEvent()
   endif
else
   MessageBox( , "Clipboard not available", "Info", MB_ICONHAND + MB_OK)
endif
return nil


/*
   Function: DoPaste()
*/
static function DoPaste()
static  nX := 300, nY := 150
local hWnd
local cPasted
if OpenClipboard(hWnd)

   // retrieve text from Clipboard
   cPasted := GetClipbData(CF_TEXT)
   // if Clipboard contained text, create new window to hold it
   if ! empty(cPasted)
      hWnd := WinNew("Text from Clipboard", nX += 40, nY += 40, 250, 150)
      AddHandler(hWnd, { | nEvent | ;
			  CBEventText(nEvent, hWnd, cPasted)})
   else  // Clipboard did not contain text -- maybe it contains a bitmap?
      cPasted := GetClipbData(CF_DIB)
      if ! empty(cPasted)
         hWnd := WinNew("Bitmap from Clipboard", nX += 40, nY += 40, 250, 150)
         AddHandler(hWnd, {|nEvent| CBEventDIB(nEvent, hWnd, cPasted)})
      endif
   endif
   CloseClipboard()
   if empty(cPasted)
      MessageBox( , "Clipboard is either empty" + CR + ;
		    "or contains unknown data", "Info", MB_ICONHAND + MB_OK)
   endif
else
   MessageBox( , "Clipboard not available", "Info", MB_ICONHAND + MB_OK)
endif
return nil


/*
   Function: CBEventText()
*/
static function CBEventText(nEvent, hWnd, cText)
local hDC
do case
case nEvent == EVENT_REDRAW
   hDC := GetDC(hWnd)
   DrawText(hDC, cText, GetClientRect(hWnd))
   ReleaseDC(hWnd, hDC)
endcase
return nil


/*
   Function: CBEventDIB()
*/
static function CBEventDIB(nEvent, hWnd, cDIB)
local hDC
do case
case nEvent == EVENT_REDRAW
   hDC := GetDC(hWnd)
   ShowDIB(hDC, cDIB, 0, 0)
   ReleaseDC(hWnd, hDC)
endcase
return nil


/*
   Function: DoDLL()
*/
static function DoDLL()
static	nX := 20, nY := 200
local	hLib, cRectangle, hWnd
hLib := LoadLibrary("GDI.EXE")
cRectangle := GetProcAddress(hLib, "rectangle", "Pascal", "Int", ;
			     "int, int, int, int, int")
hWnd := WinNew("DLL", nX += 40, nY += 60, 200, 100)
AddHandler(hWnd, {|nEvent| DLLEvent(nEvent, hWnd, cRectangle)})
return nil


/*
   Function: DLLEvent()
*/
static function DLLEvent(nEvent, hWnd, cRectangle)
local	hDC, cText
do case
case nEvent == EVENT_REDRAW
   hDC := GetDC(hWnd)
   cText := "CallDLL(Rectangle, ...) --> " + ;
	     nstr(CallDLL(cRectangle, hDC, 10, 30, 100, 50))
   DrawText(hDC, cText, GetClientRect(hWnd))
   ReleaseDC(hWnd, hDC)
case nEvent == EVENT_LCLICK .or. nEvent == EVENT_RCLICK
   InvalidateRect(hWnd)
endcase
return nil


/*
   Function: DoFont()
*/
static function DoFont()
static  nX := 100, nY := 50
local   aFont := {40, 40, 450, 0, 400, .t., .f., .f., 1, 0, 0, 0, 0, "Arial"}
local   hWnd, nColor := C_RED
aFont := ChooseFont(aFont, , , @nColor)
if aFont <> NIL         // else user chose cancel/close or hit Esc
   hWnd := WinNew("Font", nX += 40, nY += 60, 300, 200)
   AddHandler(hWnd, {|nEvent| FontEvent(nEvent, hWnd, aFont, nColor)})
endif
return nil


/*
   Function: FontEvent()
*/
static function FontEvent(nEvent, hWnd, aFont, nColor)
local   hDC, hFont, hOldFont, i, j
static msg := "Clip-4-Win"
static  aShow := { {200, 200, 300},  ;     // {x coord, y coord, angle}
		   {0,  350, 1800},  ;     // angle is in 3600 gradiants
		   {10, 200,  800},  ;     // e.g., 900 points straight up
		   {20, 0,   2700},  ;     // 0 is horizontal left-to-right
		   {75, 400, 1350},  ;
		   {100, 20, 0},     ;
		   {100, 300, 450},  ;
		   {125, 425, 450},  ;     // 1800 is horizontal right-to-left
		   {200, 50, 3450},  ;
		   {250, 200, 0},    ;
		   {300, 15, 2100},  ;
		   {300,400, 1080},  ;
		   {375,200, 300},   ;
		   {470,300, 2700},  ;
		   {500,200, 1800},  ;
		   {550,100, 3500},  ;
		   {400, 50, 3150} }
do case
case nEvent == EVENT_REDRAW
   hDC := GetDC(hWnd)
   SetTextColor(hDC, nColor)
   j := len(aShow)
   for i := 1 to j
      aFont[LF_Escapement] := aShow[i, 3]
      hFont := CreateFont(aFont)
      hOldFont := SelectObject(hDC, hFont)
      TextOut(hDC, aShow[i, 1], aShow[i, 2], msg)
      // note that SelectObject() returns a handle to the
      // prior object, so you can delete it in one fell swoop
      DeleteObject( SelectObject(hDC, hOldFont) )
   next
   ReleaseDC(hWnd, hDC)
endcase
return nil


/*
   Function: DoPie()
*/
static function DoPie()
static  nX := 400, nY := 50
local   hWnd := WinNew("Pie", nX += 40, nY += 60, 100, 100)
AddHandler(hWnd, {|nEvent| PieEvent(nEvent, hWnd)})
return nil


/*
   Function: PieEvent()
*/
static function PieEvent(nEvent, hWnd)
local hDC, aRect, hBrush, hOldbrush
do case
case nEvent == EVENT_REDRAW
   hBrush := CreateSolidBrush(C_RED)
   aRect := GetClientRect(hWnd)
   hDC := GetDC(hWnd)
   hOldbrush := SelectObject(hDC, hBrush)
   pie(hDC, 0, 0, aRect[W_RIGHT], aRect[W_BOTTOM], ;
	    aRect[W_RIGHT] / 2, 0, aRect[W_RIGHT], aRect[W_BOTTOM])
   hBrush := CreateSolidBrush(C_MAGENTA)
   // Just as it is important to close each open file handle, you
   // should always delete each object when you are done with it.
   // Note that this can be done with the following syntax, which
   // lets you re-use the same variable name for multiple objects.
   DeleteObject( SelectObject(hDC, hBrush) )

   pie(hDC, 0, 0, aRect[W_RIGHT], aRect[W_BOTTOM], aRect[W_RIGHT], ;
	    aRect[W_BOTTOM], aRect[W_RIGHT] * .75, aRect[W_BOTTOM] * .25)
   hBrush := CreateSolidBrush(C_BLUE)
   DeleteObject( SelectObject(hDC, hBrush) )
   pie(hDC, 0, 0, aRect[W_RIGHT], aRect[W_BOTTOM], aRect[W_RIGHT], ;
	    aRect[W_BOTTOM] / 2, aRect[W_RIGHT] * .75, aRect[W_BOTTOM] * .25)
   hBrush := CreateSolidBrush(C_GREEN)
   DeleteObject( SelectObject(hDC, hBrush) )
   pie(hDC, 0, 0, aRect[W_RIGHT], aRect[W_BOTTOM], ;
       aRect[W_RIGHT] * .75, aRect[W_BOTTOM] * .25, aRect[W_RIGHT] / 2, 0)
   DeleteObject( SelectObject(hDC, hOldbrush) )
   ReleaseDC(hWnd, hDC)
endcase
return nil


/*
   Function: DoBar()
*/
static function DoBar()
static  nX := 150, nY := 175
local hWnd := WinNew("Bar Graph", nX += 40, nY += 60, 250, 125)
AddHandler(hWnd, {|nEvent| BarEvent(nEvent, hWnd)})
return nil


/*
   Function: BarEvent()
*/
static function BarEvent(nEvent, hWnd)
local hDC, aRect, hBrush, hOldbrush, nWidth
do case
case nEvent == EVENT_REDRAW
   hBrush := CreateHatchBrush(HS_BDIAGONAL, C_RED)
   aRect := GetClientRect(hWnd)
   nWidth := aRect[W_RIGHT] / 6
   hDC := GetDC(hWnd)
   hOldbrush := SelectObject(hDC, hBrush)
   TextOut(hDC, nWidth + 10, aRect[W_BOTTOM] * .15 - 20, "U.S.A.")
   rectangle(hDC, nWidth, aRect[W_BOTTOM] * .15, ;
		  nWidth * 2, aRect[W_BOTTOM])
   hBrush := CreateHatchBrush(HS_VERTICAL, C_MAGENTA)
   DeleteObject( SelectObject(hDC, hBrush) )
   TextOut(hDC, nWidth * 2 + 10, aRect[W_BOTTOM] * .4 - 20, "England")
   rectangle(hDC, nWidth * 2, aRect[W_BOTTOM] * .4, ;
		  nWidth * 3, aRect[W_BOTTOM])
   hBrush := CreateHatchBrush(HS_FDIAGONAL, C_BLUE)
   DeleteObject( SelectObject(hDC, hBrush) )
   TextOut(hDC, nWidth * 3 + 10, aRect[W_BOTTOM] * .8 - 20, "Australia")
   rectangle(hDC, nWidth * 3, aRect[W_BOTTOM] * .8, ;
		  nWidth * 4, aRect[W_BOTTOM])
   hBrush := CreateHatchBrush(HS_CROSS, C_GREEN)
   DeleteObject( SelectObject(hDC, hBrush) )
   TextOut(hDC, nWidth * 4 + 10, aRect[W_BOTTOM] * .55 - 20, "Germany")
   rectangle(hDC, nWidth * 4, aRect[W_BOTTOM] * .55, ;
		  nWidth * 5, aRect[W_BOTTOM])
   DeleteObject( SelectObject(hDC, hOldbrush) )
   ReleaseDC(hWnd, hDC)
endcase
return nil


/*
   Function: DoPrint()
*/
static function DoPrint()
local hPrintDC, hIcon, hBrush, hOldBrush, hCursor, hOldCursor
local nBlack := C_BLACK
local i, nWidth, nHeight

// display printer dialog box, so the user can choose the settings
hPrintDC := GetPrintDC()
if empty(hPrintDC)
   // cancelled by the user
   return nil
endif

// print a test page

hCursor := LoadCursor( , IDC_WAIT)
hOldCursor := SetCursor(hCursor)

nWidth := GetDeviceCaps(hPrintDC, HORZRES)
nHeight := GetDeviceCaps(hPrintDC, VERTRES)

StartDoc(hPrintDC, "TestOutput")
StartPage(hPrintDC)

TextOut(hPrintDC, 100, 50, "Clip-4-Win Printer Test Page")
Rectangle(hPrintDC, 0, 0, nWidth, nHeight)
MoveTo(hPrintDC, 0, 0)
LineTo(hPrintDC, nWidth, nHeight)
MoveTo(hPrintDC, nWidth, 0)
LineTo(hPrintDC, 0, nHeight)

Arc(hPrintDC,   1000, 1000,   1300, 1200,   1250, 1190,   1100, 1100)

hBrush := CreateHatchBrush(HS_HORIZONTAL, nBlack)
hOldBrush := SelectObject(hPrintDC, hBrush)
Chord(hPrintDC, 1500, 1200,   2000, 1350,   1550, 1340,   1400, 1200)
DeleteObject( SelectObject(hPrintDC, hOldBrush) )

hBrush := CreateHatchBrush(HS_BDIAGONAL, nBlack)
hOldBrush := SelectObject(hPrintDC, hBrush)
Pie(hPrintDC,   100, 1200,   700, 1500,   650, 1490,   120, 1280)
DeleteObject( SelectObject(hPrintDC, hOldBrush) )

hBrush := CreateHatchBrush(HS_FDIAGONAL, nBlack)
hOldBrush := SelectObject(hPrintDC, hBrush)
Polygon(hPrintDC, { {1000, 250}, {1600, 500}, {1800, 100} })
DeleteObject( SelectObject(hPrintDC, hOldBrush) )
PolyLine(hPrintDC, { {300, 700}, {100, 900}, {500, 1000} })
for i := 100 to 500 step 100
   TextOut(hPrintDC, i + 400, i + 100, nstr(i))
next i
EndPage(hPrintDC)
EndDoc(hPrintDC)
DeleteDC(hPrintDC)
SetCursor(hOldCursor)
return nil


/*
   Function: DoTimer()
*/
static function DoTimer()
static  lTimer := .F., hWnd
if ! lTimer
   hWnd := WinNew("Timer", 400, 150, 135, 75)
   SetTimer(hWnd, 1, 1000)     // every 1000 millisecs (= one second)
   lTimer := .T.
   AddHandler(hWnd, ;
	      {|nEvent| TimerEvent(nEvent, hWnd, @lTimer)})
endif
return nil


/*
   Function: TimerEvent()
*/
static function TimerEvent(nEvent, hWnd, lTimer)
local hDC, hFont, hOldFont, aRect
static aFont := { -55, -17, 0, 0, FW_SEMIBOLD, .F., .F., .F., 0, 3, 2, 1, 18, ;
		  "Times New Roman" }
do case
case nEvent == EVENT_TIMER
   InvalidateRect(hWnd)
case nEvent == EVENT_CLOSE .or. nEvent == EVENT_DESTROY
   KillTimer(hWnd, 1)
   lTimer := .F.
   DestroyWindow(hWnd)
case nEvent == EVENT_WINSIZE
   // adjust font height/width based on new size of this window
   aRect := GetClientRect(hWnd)
   aFont[LF_Height] := - aRect[W_BOTTOM] * .85
   aFont[LF_Width] := - aRect[W_RIGHT] * .11

case nEvent == EVENT_REDRAW
   hDC := GetDC(hWnd)
   SetTextColor(hDC, C_BLUE)
   hFont := CreateFont(aFont)
   hOldFont := SelectObject(hDC, hFont)
   DrawText(hDC, time(), GetClientRect(hWnd))
   DeleteObject( SelectObject(hDC, hOldFont) )
   ReleaseDC(hWnd, hDC)
endcase
return nil


/*
   Function: NStr()
*/
static function nstr(n)
return alltrim(str(n)) + " "


/*
   Function: asString()
*/
static function asString(x)
local	v := valtype(x)
do case
   case v == "C"
   case v == "N"
      return nstr(x)
   case v == "L"
      if x
         return ".T."
      else
         return ".F."
      endif
   case v == "D"
      return "date"
   case v == "U"
      return "NIL"
   case v $ "AOB"
      return ""
   otherwise
      return ""
end case
return x


/*
   Function: MenuSetup()
*/
static function MenuSetup()
local hWnd := SelectWindow()
local hMenu
local hPopupMenu

if (hMenu := GetMenu(hWnd)) <> NIL
   DestroyMenu(hMenu)
endif

// note the grayed out entries (search for MF_GRAYED)
// also note that ampersand indicates the trigger letter
hMenu := CreateMenu()
hPopupMenu := CreatePopupMenu()
AppendMenu(hMenu, "file", MF_ENABLED + MF_POPUP, "&File", hPopupMenu)
AppendMenu(hPopupMenu, "open", MF_ENABLED + MF_STRING, "&Open...", {|| DoOpen()})
AppendMenu(hPopupMenu, "clear", MF_GRAYED + MF_STRING, "&Clear", {|| DoClear()})
AppendMenu(hPopupMenu, "run", MF_ENABLED + MF_STRING, "&Run...", {|| DaDoRunRun()})
AppendMenu(hPopupMenu, "save", MF_GRAYED + MF_STRING, "&Save", {|| qout("save")})
AppendMenu(hPopupMenu, "saveas", MF_GRAYED + MF_STRING, "Save &As...", {|| qout("save as")})
AppendMenu(hPopupMenu, "", MF_SEPARATOR)
AppendMenu(hPopupMenu, "print", MF_ENABLED + MF_STRING, "&Print...", {|| DoPrint()})
AppendMenu(hPopupMenu, "", MF_SEPARATOR)
AppendMenu(hPopupMenu, "exit", MF_ENABLED + MF_STRING, "E&xit", {|| DoExit() })

hPopupMenu := CreatePopupMenu()
AppendMenu(hMenu, "edit", MF_ENABLED + MF_POPUP, "&Edit", hPopupMenu)
AppendMenu(hPopupMenu, "cut", MF_GRAYED + MF_STRING, "Cu&t", {|c| DoCutCopy(c)})
AppendMenu(hPopupMenu, "copy", MF_ENABLED + MF_STRING, "&Copy", {|c| DoCutCopy(c)})
AppendMenu(hPopupMenu, "paste", MF_ENABLED + MF_STRING, "&Paste", {|| DoPaste()})
AppendMenu(hPopupMenu, "", MF_SEPARATOR)
AppendMenu(hPopupMenu, "toggle", MF_ENABLED + MF_STRING, "&Toggle", {|c| ToggleItem(c) })
AppendMenu(hPopupMenu, "find", MF_GRAYED + MF_STRING, "&Find...", {|c| qout(c)})
AppendMenu(hPopupMenu, "replace", MF_GRAYED + MF_STRING, "&Replace...", {|c| qout(c)})

hPopupMenu := CreatePopupMenu()
AppendMenu(hMenu, "stuff1", MF_ENABLED + MF_POPUP, "&Stuff", hPopupMenu)
AppendMenu(hPopupMenu, "color", MF_ENABLED + MF_STRING, "&Color...", {|c| DoColor()})
AppendMenu(hPopupMenu, "dll", MF_ENABLED + MF_STRING, "&DLL", {|c| DoDLL()})
AppendMenu(hPopupMenu, "font", MF_ENABLED + MF_STRING, "&Font...", {|c| DoFont()})
AppendMenu(hPopupMenu, "", MF_SEPARATOR)
AppendMenu(hPopupMenu, "pie chart", MF_ENABLED + MF_STRING, "&Pie Chart", {|c| DoPie()})
AppendMenu(hPopupMenu, "bar graph", MF_ENABLED + MF_STRING, "&Bar Graph", {|c| DoBar()})

hPopupMenu := CreatePopupMenu()
AppendMenu(hMenu, "stuff2", MF_ENABLED + MF_POPUP, "&More Stuff", hPopupMenu)
AppendMenu(hPopupMenu, "audio", MF_ENABLED + MF_STRING, "&Audio...", {|c| DoAudio()})
AppendMenu(hPopupMenu, "cursor", MF_ENABLED + MF_STRING, "&Cursor", {|c| DoCursor()})
AppendMenu(hPopupMenu, "icons", MF_ENABLED + MF_STRING, "&Icons", {|c| DoIcons()})
AppendMenu(hPopupMenu, "printer", MF_ENABLED + MF_STRING, "&Printer...", {|c| DoPrint()})
AppendMenu(hPopupMenu, "timer", MF_ENABLED + MF_STRING, "&Timer", {|c| DoTimer()})

hPopupMenu := CreatePopupMenu()
AppendMenu(hMenu, "dataentry", MF_ENABLED + MF_POPUP, "&Data Entry", hPopupMenu)
AppendMenu(hPopupMenu, "modelessgets", MF_ENABLED + MF_STRING, "&Modeless", {|| ModelessGets()})
AppendMenu(hPopupMenu, "valtest", MF_ENABLED + MF_STRING, "&Validation", {|| ValidTest()})

hPopupMenu := CreatePopupMenu()
AppendMenu(hMenu, "browsemenu", MF_ENABLED + MF_POPUP, "&Browses", hPopupMenu)
AppendMenu(hPopupMenu, "browse", MF_ENABLED + MF_STRING, "&Browse...", {|| DoBrowse()})

hPopupMenu := CreatePopupMenu()
AppendMenu(hMenu, "help", MF_ENABLED + MF_POPUP, "&Help", hPopupMenu)
AppendMenu(hPopupMenu, "about", MF_ENABLED + MF_STRING, "&About", {|| credits()})
SetMenu(hWnd, hMenu)
return hMenu


/*
   Function: DoCursor()
*/
static function DoCursor()
static cursors_ := { IDC_ARROW, IDC_IBEAM, IDC_WAIT, IDC_CROSS, ;
		     IDC_UPARROW, IDC_SIZE, IDC_ICON, IDC_SIZENWSE, ;
		     IDC_SIZENESW, IDC_SIZEWE, IDC_SIZENS }
local y := len(cursors_)
local x
local hCursor
local hOldcursor := SetCursor()
MessageBox(hWnd, "Press spacebar to cycle through the stock cursors", "Info", ;
		 MB_OK)
for x := 1 to y
   hCursor := LoadCursor( , cursors_[x])
   if hCursor <> 0
      SetCursor(hCursor)
   endif
   inkey(3)
next
SetCursor(hOldcursor)
return nil



/*
   Function: DoIcons()
*/
static function DoIcons()
static  nX := 250, nY := 100
local hWnd := WinNew("Icons", nX += 40, nY += 60, 300, 100)
AddHandler(hWnd, {|nEvent| IconEvent(nEvent, hWnd)})
return nil


/*
   Function: IconEvent()
*/
static function IconEvent(nEvent, hWnd)
static icons_ := { IDI_EXCLAMATION, IDI_HAND, IDI_ASTERISK, ;
		   IDI_QUESTION, IDI_APPLICATION }
local hDC
local x
local hIcon
local aRect
do case
case nEvent == EVENT_REDRAW
   aRect := GetClientRect(hWnd)
   hDC := GetDC(hWnd)
   if hDC <> 0
      for x := 1 to 5
         if ( hIcon := LoadIcon(, icons_[x]) ) <> 0
            DrawIcon(hDC, aRect[W_RIGHT] * x / 6, ;
                          aRect[W_BOTTOM] * .4, hIcon)
         endif
      next
      ReleaseDC(hWnd, hDC)
   endif
endcase
return nil


/*
   Function: ModelessGets()
*/
static function ModelessGets
local cFile
local getlist := {}
local x
local f
local nFields
local hCWnd
local nHand
local nArea
local nOldarea := select()
local bOldinsert
#ifdef BUTTONS
   local nWidth
   local aRect
   local nY
   local hInst
   local lContinue := .t.
   local aButtons := {}
   local nButton
#endif
cFile := GetOpenFileName(, "*.dbf", "Select a database")
if cFile <> NIL
   if select(StripPath(StripExt(cFile))) == 0
      use (cFile) new
   else
      // trying to browse/edit a dbf that's already open

      // the following would probably just confuse
      //dbSelectArea(StripPath(StripExt(cFile)))
      MessageBox( , "Already in use", "Error", MB_ICONHAND + MB_OK)
      return nil
   endif
   x := alias()

   // I realize that this is not particularly graceful, but the
   // purpose of this demo is not necessarily to show locking
   // techniques... i.e., puh-leeze feel free to substitute your own!
   if ! rlock()
      MessageBox(, "Cannot edit " + x + " database at this time", ;
		   MB_ICONHAND + MB_OK)
      dbSelectArea(nOldarea)
      return nil
   endif

   // the WS_CLIPCHILDREN makes sure anything drawn with WS_CHILD is visible
   // (e.g. the buttons)
   hCWnd := WinNew("Editing " + left(x, 1) + lower(substr(x, 2)) + ;
		   " Database", 0, 0, 350, 250,		    ;
		   WS_OVERLAPPEDWINDOW + WS_CLIPCHILDREN)
   #ifdef BUTTONS

   #define BUTTON_HEIGHT  20
   #define B_NEXT   1
   #define B_PREV   2
   #define B_FIRST  3
   #define B_LAST   4
   #define B_ADD    5

   #define TOTAL_BUTTONS  5

   aRect := GetClientRect(hCWnd)
   hInst := _GetInstance()
   nWidth := aRect[W_RIGHT] / TOTAL_BUTTONS
   nY := aRect[W_BOTTOM] - BUTTON_HEIGHT
   aadd(aButtons, CreateWindow("button",         ;       // button class name
                               "Next",           ;       // title text
                               WS_CHILD          ;       // child window
                               + WS_VISIBLE      ;       // ... that can be seen
                               + BS_PUSHBUTTON,  ;       // ... a push button
                               0, nY,            ;       // x,y position
                               nWidth,           ;       // width
                               BUTTON_HEIGHT,    ;       // height
                               hCWnd,            ;       // parent window
                               B_NEXT,           ;       // unique child id
                               hInst)  )                 // parent's instance

   aadd(aButtons, CreateWindow("button",         ;       // button class name
                               "Prev",           ;       // title text
                               WS_CHILD          ;       // child window
                               + WS_VISIBLE      ;       // ... that can be seen
                               + BS_PUSHBUTTON,  ;       // ... a push button
                               nWidth, nY,       ;       // x,y position
                               nWidth,           ;       // width
                               BUTTON_HEIGHT,    ;       // height
                               hCWnd,            ;       // parent window
                               B_PREV,           ;       // unique child id
                               hInst)   )                // parent's instance

   aadd(aButtons, CreateWindow("button",         ;       // button class name
                               "First",          ;       // title text
                               WS_CHILD          ;       // child window
                               + WS_VISIBLE      ;       // ... that can be seen
                               + BS_PUSHBUTTON,  ;       // ... a push button
                               nWidth*2, nY,     ;       // x,y position
                               nWidth,           ;       // width
                               BUTTON_HEIGHT,    ;       // height
                               hCWnd,            ;       // parent window
                               B_FIRST,          ;       // unique child id
                               hInst)   )                // parent's instance

   aadd(aButtons, CreateWindow("button",         ;       // button class name
                               "Last",           ;       // title text
                               WS_CHILD          ;       // child window
                               + WS_VISIBLE      ;       // ... that can be seen
                               + BS_PUSHBUTTON,  ;       // ... a push button
                               nWidth*3, nY,     ;       // x,y position
                               nWidth,           ;       // width
                               BUTTON_HEIGHT,    ;       // height
                               hCWnd,            ;       // parent window
                               B_LAST,           ;       // unique child id
                               hInst)   )                // parent's instance

   aadd(aButtons, CreateWindow("button",         ;       // button class name
                               "Add",            ;       // title text
                               WS_CHILD          ;       // child window
                               + WS_VISIBLE      ;       // ... that can be seen
                               + BS_PUSHBUTTON,  ;       // ... a push button
                               nWidth*4, nY,     ;       // x,y position
                               nWidth,           ;       // width
                               BUTTON_HEIGHT,    ;       // height
                               hCWnd,            ;       // parent window
                               B_ADD,            ;       // unique child id
                               hInst)   )                // parent's instance

   #endif

   // create Getlist array and load it with GET objects
   nFields := fcount()
   for x := 1 to nFields
      f := field(x)
      #ifdef BUTTONS
         @ x, 2 say lower(f) get field f in window hCWnd buttons aButtons ;
                             picture '@K' saycolor 'n/w' getcolor 'n/bg,+w/b'
      #else
         @ x, 2 say lower(f) get field f in window hCWnd picture '@K' ;
                             saycolor 'n/w' getcolor 'n/bg,+w/b'
      #endif
   next

   ShowCaret(hCWnd)

//   EnableMenuItem(GetMenu(hWnd), "modelessgets", MF_GRAYED)  // one at a time
   nArea := select()
#ifdef BUTTONS
   nHand := AddHandler(hCWnd, {|nEvent| ModelessEvent(GetList, nEvent, hCWnd, nHand, nArea, aButtons)})
#else
   nHand := AddHandler(hCWnd, {|nEvent| ModelessEvent(GetList, nEvent, hCWnd, nHand, nArea, {})})
#endif // BUTTONS
endif

dbSelectArea(nOldarea)
return nil


/*
   Function: DoBrowse()
*/
static function DoBrowse
static nX := 10, nY := 10
local b
local hWnd
local x
local f
local c
local e
local nFields
local fields_
local cFile
local cNtxfile
local bOldhandler
local nArea
cFile := GetOpenFileName(, "*.dbf", "Select a database")
if cFile <> NIL
   if select(StripPath(StripExt(cFile))) == 0
      cNtxfile := GetOpenFileName(, "*.ntx", "Select an index file")
      use (cFile) new shared
      if cNtxfile <> NIL
         bOldhandler := errorblock( { | e | bogusindex(e, bOldhandler) } )
         dbsetindex(cNtxfile)
         // verify the validity of this index key
         begin sequence
            x := eval( &("{ || " + indexkey(0) + "}") )
         recover using e
            MessageBox(, "Invalid index (missing " + e:operation + " "  + ;
			 if(e:genCode == EG_NOVAR, "field", "") +	  ;
			 if(e:genCode == EG_NOFUNC, "function", "") +	  ;
			 if(e:genCode == EG_NOALIAS, "alias", "") +	  ;
			 ")" + CR + "Browsing " + StripPath(cFile) +      ;
			 " in natural order", "Bogus Index",		  ;
			 MB_ICONHAND + MB_OK)
            dbclearindex()
         end sequence
         errorblock(bOldhandler)
      endif
   else
      // trying to browse a dbf that's already open

      // the following would probably just confuse
      //dbSelectArea(StripPath(StripExt(cFile)))
      MessageBox( , "Already in use", "Error", MB_ICONHAND + MB_OK)
      return nil
   endif
   x := alias()
   hWnd := WinNew("Browsing " + left(x, 1) + lower(substr(x, 2)) + ;
                  " Database", nX += 20, nY += 20, 525, 200, ;
                  WS_OVERLAPPEDWINDOW + WS_VSCROLL + WS_HSCROLL)
   b := TBrowseDB(0, 0, maxrow(), maxcol())
   b:headSep := ""
   b:colSep  := "  "
   b:colorSpec := "+w/b, +gr/n, +w/r, +r/w"

   nFields := fcount()

   #define B_FIELDNAMES    1
   #define B_RECNO         2
   #define B_ALIAS         3
   #define B_CARGO_LEN     3

   b:cargo := array(B_CARGO_LEN)

   // we are going to store an array of fieldnames in the TBrowse
   // cargo instance variable (to be used when inserting new columns)
   b:cargo[B_FIELDNAMES] := array(nFields)
   b:cargo[B_RECNO] := recno()     // record pointer for this browse window
   b:cargo[B_ALIAS] := alias()     // needed when editing specific cells
   fields_ := dbstruct()
   for x := 1 to nFields
      b:cargo[B_FIELDNAMES][x] := field(x)

      // memos must be treated differently than "regular" fields
      if fields_[x][DBS_TYPE] == "M"
         c := TBColumnNew(b:cargo[B_FIELDNAMES][x], &("{ || if(empty(" + ;
                          b:cargo[B_FIELDNAMES][x] + "), '<memo>', '<MEMO>') }"))
         c:cargo := { || ShowMemo(b) }
      else
         c := TBColumnNew(b:cargo[B_FIELDNAMES][x], fieldwblock(b:cargo[B_FIELDNAMES][x], select()))
      endif

      // set color block to highlight negative numbers
      // dumb example, I know, but it will have to do for now...
      if valtype(eval(c:block)) == "N"
         c:colorBlock := { | x | if( x < 0, { 3, 4 }, { 1, 2 } ) }
      endif

      b:AddColumn(c)
   next

   // set range of horizontal scrollbar to match # of columns
   // note that if you later decide to add or delete columns,
   // you should do this again to set the new range
   SetScrollRange(hWnd, SB_HORZ, 1, b:colCount, .f.)

   nArea := select()
   AddHandler(hWnd, {|nEvent| BrowseEvent(nEvent, hWnd, nArea, b)})
   InvalidateRect(hWnd)
endif
return nil


/*
   Function: BogusIndex()
   Purpose:  To catch any errors due to invalid indexes
*/
static function bogusindex(oError, bOldhandler)
if oError:genCode == EG_NOFUNC		;
   .or. oError:genCode == EG_NOVAR	;
   .or. oError:genCode == EG_NOALIAS
   break oError
endif
return eval(bOldhandler, oError)


/*
   Function: WinNew()
*/
static function WinNew(cTitle, nX, nY, nWidth, nHeight, nStyle)
local hInst := _GetInstance()
local nCmdShow := _GetnCmdShow()
local hWin
if nStyle == NIL
   nStyle := WS_OVERLAPPEDWINDOW
endif
hWin := CreateWindow(APP_NAME,           ;       // window class
                    cTitle,              ;       // caption for title bar
                    nStyle,              ;       // window style
                    nX,                  ;       // x co-ordinate
                    nY,                  ;       // y co-ordinate
                    nWidth,              ;       // width
                    nHeight,             ;       // height
                    hWnd,                ;       // hWnd of parent
                    0,                   ;       // hMenu of menu (none yet)
                    hInst)                       // our own app instance
if hWin == 0
   // probably out of resources
   MessageBox( , "Can't create window", "Error", MB_ICONEXCLAMATION + MB_OK)
   return nil
endif
HideCaret(hWin)
// make sure it's displayed ...
ShowWindow(hWin, nCmdShow)
// ... and up to date
UpdateWindow(hWin)
return hWin


/*
   Function: ResetMainEvent()
   Purpose:  Reset event handler for main window (i.e., delete the
             primary event handler, which had been changed in DoOpen())
*/
static function ResetMainEvent
local hMenu
DelHandler(nMainEvId)
nMainEvId := AddHandler(hWnd, {|nEvent| MainEvent(nEvent)})
// disable the "Clear" and "Cut" menu items
hMenu := GetMenu(hWnd)       // retrieve reference to main menu
EnableMenuItem(hMenu, "clear", MF_GRAYED)
EnableMenuItem(hMenu, "cut", MF_GRAYED)
// force main window to be redrawn immediately
InvalidateRect(hWnd)
return nil


/*
   Function: ValidTest()
*/
static function ValidTest
local cFruit := space(10)
local hWnd := WinNew("Sample Listbox Validation", 0, 0, 300, 100)
local getlist := {}
local nHand
ShowCaret(hWnd)
@ 2,5 say "Type in a fruit" color 'n/w' ;
      get cFruit color '+w/b,+w/b' valid FruitLook()
nHand := AddHandler(hWnd, {|nEvent| ValidEvent(GetList, nEvent, hWnd, nHand, cFruit)})
return nil


/*
   Procedure: ValidEvent()
*/
static procedure ValidEvent(GetList, nEvent, hWnd, nHand, cFruit)
local	hOldWnd
local	lDone := (GetList[1] == nil)	// only do the read until it finishes
if !lDone
	if nEvent == EVENT_REDRAW
		hOldWnd = SelectWindow(hWnd)
		@ 2,5 say "Type in a fruit" color 'n/w'
		SelectWindow(hOldWnd)
	endif
	if ReadModeless(GetList, nEvent) == NIL
		// not yet done
		return				// wait for next event
	endif
endif
// getting here means the read has completed (or been abandoned)

if nEvent == EVENT_DESTROY
	// abandoned
	return
endif
if !lDone
	// get here exactly once
	// (immediately after ReadModeless() returns non-NIL)
	GetList[1] := nil		// cancel the GET
	HideCaret(hWnd)
	InvalidateRect(hWnd)		// trigger a re-draw (below)
endif

do case
case nEvent == EVENT_REDRAW
	hOldWnd = SelectWindow(hWnd)
	@ 2,5 say "Fruit selected: " + cFruit color '+r/w'
	SelectWindow(hOldWnd)
endcase
return


/*
   Function: FruitLook()
*/
static function FruitLook
static aChoices := {"Apple", "Banana", "Coconut", "Grape", "Grapefuit", ;
		    "Lemon", "Lime", "Orange", "Papaya", "Pineapple",   ;
		    "Raisin", "Raspberry", "Strawberry" }
local g := getactive()
local v := upper( g:varGet() )
static aDlg
if ascan(aChoices, { | f | upper(f) = upper(v) } ) == 0
   if aDlg == NIL
      aDlg := CreateDialog("Fruit List",				  ;
                          WS_CAPTION + WS_SYSMENU + WS_GROUP + WS_TABSTOP ;
                          + WS_THICKFRAME + WS_VISIBLE + WS_POPUP,	  ;
                          100, 30, 100, 100)
      aDlg := AppendDialog(aDlg, "listbox", DLG_LISTBOX,		  ;
                          LBS_STANDARD + WS_TABSTOP + WS_CHILD + WS_VISIBLE, ;
                          10, 10, 80, 60,				  ;
                          aChoices)
      aDlg := AppendDialog(aDlg, "ok", DLG_BUTTON,			  ;
                          BS_DEFPUSHBUTTON + WS_TABSTOP + WS_CHILD + WS_VISIBLE,;
                          10, 75, 35, 15,				  ;
                          "&Ok")
      aDlg := AppendDialog(aDlg, "cancel", DLG_BUTTON,			  ;
                          BS_PUSHBUTTON + WS_TABSTOP + WS_CHILD + WS_VISIBLE, ;
                          55, 75, 35, 15,				  ;
                          "&Cancel")
   endif
   if ModalDialog(aDlg) <> 0 .and. ! GetDialogResult(aDlg, "cancel")
      g:varPut( padr(GetDialogResult(aDlg, "listbox"), len(g:buffer)) )
   endif
endif
return .t.


/*
   Function: ToggleItem()
   Purpose:  Demonstrate how to check and uncheck any menu item
*/
static function ToggleItem(cItem)
local hMenu := GetMenu(hWnd)       // retrieve reference to main menu
local nCheck := CheckMenuItem(hMenu, cItem)
CheckMenuItem(hMenu, cItem, if(nCheck == MF_CHECKED, MF_UNCHECKED, MF_CHECKED))
return nil

// end of file DEMO.PRG
