/*****************************************************************************
** BROWSE.PRG
**
** ProVision:Windows v1.21 browse window sample
**
** by J. David Reynolds
**
** Copyright 1992 SofDesign International, Inc.
** All rights reserved
**
** tab spacing = 3
**
*****************************************************************************/

#include "inkey.ch"
#include "pw.ch"
#include "tutor.ch"



/*****************************************************************************
** BrowseWindow() --> oBrowseWnd
**
** This function creates a browse window and returns a reference to it.
**
*****************************************************************************/
FUNCTION BrowseWindow()

   LOCAL bColor

   LOCAL i

   LOCAL oBrowseWnd, ;
         oMenu, ;
         oTB


   // Turn the mouse cursor into an hourglass (or display a wait message).
   WaitMsg(.T.)

   // Create a browse window--it should probably have scrollbars.
   CREATE TBROWSE WINDOW oBrowseWnd ;
      AT 03, 07 ;
      SIZE 12, 50 ;
      VIRTUAL SIZE pwMaxRow(), pwMaxCol() ;
      TITLE "Customers" ;
      STYLE PWSTYLE_PRIMARY + PWSTYLEINFO_KEEPVSCROLLBAR + ;
         PWSTYLEINFO_KEEPHSCROLLBAR ;
      WINDOW HANDLER { | idMsg, _1, _2, _3, _4, oWnd | ;
         BrowseWndHandler(idMsg, oWnd) } ;
      EVENT HANDLER { | idMsg, _1, _2, _3, _4, oWnd | ;
         BrowseEventHandler(idMsg, _1, _2, _3, oWnd) }

   // Create a menu for this window.
   MENU oMenu

      POPUP PROMPT "~File"
         MENUITEM oBrowseWnd:cargo[BROWSE_ITEMSEARCH] ;
            PROMPT "~Search..." ;
            ACTION { || SearchWindow(oBrowseWnd) }
         SEPARATOR
         MENUITEM PROMPT "E~xit\tAlt+F4" ;
            ACTION { || oBrowseWnd:close() }
      END POPUP // File

      POPUP PROMPT "~Record"
         MENUITEM oBrowseWnd:cargo[BROWSE_ITEMEDIT] ;
            PROMPT "~Edit...\tEnter" ;
            ACTION { || EditWindow(RECORD_EDIT, oBrowseWnd) }
         MENUITEM PROMPT "~Add..." ;
            ACTION { || EditWindow(RECORD_APPEND, oBrowseWnd) }
         SEPARATOR
         MENUITEM oBrowseWnd:cargo[BROWSE_ITEMDELETE] ;
            PROMPT "~Delete" ;
            ACTION { || DeleteRecallRecord(RECORD_DELETE, oBrowseWnd) }
         MENUITEM oBrowseWnd:cargo[BROWSE_ITEMRECALL] ;
            PROMPT "~Recall" ;
            ACTION { || DeleteRecallRecord(RECORD_RECALL, oBrowseWnd) }
      END POPUP // Record

   END MENU

   ATTACH MENU oMenu TO oBrowseWnd

   // Create your own TBROWSE--it doesn't really matter what coordinates
   // you specify, and the data source doesn't have to be a DBF.  Also,
   // if you use Class(y) to inherit from TBROWSE, an instance of your
   // new subclass can be attached to a window.
   oTB   := TBROWSENEW()
   oTB:goBottomBlock := { || Customer->(BrowseGoBottom(oBrowseWnd)) }
   oTB:goTopBlock    := { || Customer->(BrowseGoTop(oBrowseWnd)) }
   oTB:skipBlock     := { | nToSkip | ;
      Customer->(BrowseSkip(nToSkip, oBrowseWnd)) }
   oTB:colorSpec     := IIF(ISCOLOR(), "+W/B, N/W, W/B", "+W/N, N/W, W/N")
   oTB:headSep       := ""
   oTB:colSep        := "  "
   oTB:addColumn(TBCOLUMNNEW("Cust #",    { || Customer->CustNmbr }))
   oTB:addColumn(TBCOLUMNNEW("Company",   { || Customer->Company }))
   oTB:addColumn(TBCOLUMNNEW("City",      { || Customer->City }))
   oTB:addColumn(TBCOLUMNNEW("State",     { || Customer->State }))
   oTB:addColumn(TBCOLUMNNEW("Contact",   { || Customer->Contact }))
   bColor   := { || Customer->(IIF(DELETED(), { 3, 2 }, { 1, 2 })) }
   FOR i := 1 TO 5
      oTB:getColumn(i):colorBlock   := bColor
   NEXT i 

   // Attach the TBROWSE to the window.
   ATTACH TBROWSE oTB TO oBrowseWnd


   //
   // If you have a means of obtaining the relative position of the current
   // physical data row in your TBROWSE, this method shows how to make the
   // vertical scrollbar operate smoothly.  In this example, NTXPos() is
   // assumed to return the relative position of the current physical data
   // row, and NTXRec() is assumed to return the physical data row at the
   // current relative position.
   //
   // oBrowseWnd:setBrowseBlocks( ;
   //    { | oTB, nRowCount, nRowNumber | ;
   //       nRowCount   := Customer->(LASTREC()), ;
   //       nRowNumber  := Customer->(NTXPos(RECNO())) }, ;
   //    { | oTB, nRowNumber | ;
   //       Customer->(DBGOTO(NTXRec(nRowNumber))) } )
   //


   // Open the window.
   OPEN WINDOW oBrowseWnd

   // We need to set the initial states of the Record menuitems.
   UpdateMenuitems(oBrowseWnd)

   // Restore the mouse cursor (or remove the wait message).
   WaitMsg(.F.)

   // Return a reference to the window.
   RETURN(oBrowseWnd)
   // END BrowseWindow()



/*****************************************************************************
** STATIC BrowseWndHandler( idMsg, oBrowseWnd ) --> lHandled
**
** This implements the window handler and is responsible for saving and
** restoring the RECNO().  If the search window is open when the browse
** window is closed, the search window must also be closed.
**
*****************************************************************************/
STATIC FUNCTION BrowseWndHandler( idMsg, oBrowseWnd )

   LOCAL lHandled := .T.


   DO CASE
   CASE idMsg == PWMSG_CLOSE
      IF .NOT. oBrowseWnd:cargo[BROWSE_DESTROYED]

         oBrowseWnd:cargo[BROWSE_DESTROYED]  := .T.
         oBrowseWnd:destroy()

         // If the edit window exists, destroy it.
         IF oBrowseWnd:cargo[BROWSE_EDITWND] != NIL
            oBrowseWnd:cargo[BROWSE_EDITWND]:destroy()
         END IF // BrowseWnd:cargo[BROWSE_EDITWND] != NIL

         // If the search window exists, destroy it.
         IF oBrowseWnd:cargo[BROWSE_SEARCHWND] != NIL
            oBrowseWnd:cargo[BROWSE_SEARCHWND]:destroy()
         END IF // BrowseWnd:cargo[BROWSE_SEARCHWND] != NIL

      END IF // .NOT. oBrowseWnd:cargo[BROWSE_DESTROYED]

   CASE idMsg == PWMSG_CREATE
      // Initialize cargo values for the window.
      oBrowseWnd:cargo  := ARRAY(BROWSECARGOLEN)
      oBrowseWnd:cargo[BROWSE_RECNO]      := Customer->(RECNO())
      oBrowseWnd:cargo[BROWSE_MINIMIZED]  := .F.
      oBrowseWnd:cargo[BROWSE_DESTROYED]  := .F.

   CASE idMsg == PWMSG_KILLFOCUS
      // Save the current record number.
      oBrowseWnd:cargo[BROWSE_RECNO]   := Customer->(RECNO())

   CASE idMsg == PWMSG_MINIMIZE
      // Keep track of the browse window's state.
      oBrowseWnd:cargo[BROWSE_MINIMIZED]  := .T.

      // If the search window is open, close it, and make a note that it
      // needs to be reopened.
      IF (oBrowseWnd:cargo[BROWSE_SEARCHWND] != NIL) .AND. ;
            pvGetBit(oBrowseWnd:cargo[BROWSE_SEARCHWND]:getState(), ;
            PWSTATE_OPEN)
         CLOSE WINDOW oBrowseWnd:cargo[BROWSE_SEARCHWND]
         oBrowseWnd:cargo[BROWSE_RESTSEARCH] := .T.
      ELSE
         oBrowseWnd:cargo[BROWSE_RESTSEARCH] := .F.
      END IF // (oBrowseWnd:cargo[BROWSE_SEARCHWND] != NIL) .AND. ...

   CASE idMsg == PWMSG_RESTORE
      IF oBrowseWnd:cargo[BROWSE_MINIMIZED]

         // Keep track of the browse window's state.
         oBrowseWnd:cargo[BROWSE_MINIMIZED]  := .F.

         // If the search window was open before, reopen it.
         IF oBrowseWnd:cargo[BROWSE_RESTSEARCH]
            OPEN WINDOW oBrowseWnd:cargo[BROWSE_SEARCHWND]
            oBrowseWnd:cargo[BROWSE_RESTSEARCH] := .F.
         END IF // oBrowseWnd:cargo[BROWSE_RESTSEARCH]

      END IF // oBrowseWnd:cargo[BROWSE_MINIMIZED]

   CASE idMsg == PWMSG_SETFOCUS
      // Restore the proper workarea and record number.
      SELECT Customer
      GO oBrowseWnd:cargo[BROWSE_RECNO]

   END CASE

   RETURN(lHandled)
   // END BrowseWndHandler( idMsg, oBrowseWnd )



/*****************************************************************************
** STATIC BrowseEventHandler( idMsg, _1, _2, _3, oBrowseWnd ) --> lHandled
**
** This implements an event handler for the window and can be used for
** trapping keystrokes and/or mouse events.  This sample allows a record
** to be edited by either pressing Enter to edit the current record or by
** doubleclicking on a record.
**
** Note that if you opt to use Enter as a NEXTFIELD key, it will _not_ be
** passed to the window's event handler because it will be processed by the
** default window behaviour.  It _will_, however, be passed to any keypress
** filters you have added to the window.
**
*****************************************************************************/
STATIC FUNCTION BrowseEventHandler( idMsg, _1, _2, _3, oBrowseWnd )

   LOCAL lHandled := .T.

   LOCAL nCol, ;
         nRow


   DO CASE
   CASE (idMsg == PWMSG_KEYPRESS) .AND. (_1 == K_ENTER) .AND. ;
         (.NOT. DELETED()) .AND. (.NOT. oBrowseWnd:cargo[BROWSE_MINIMIZED])
      // If the key was Enter, the record is not deleted, and the window is
      // not minimized, edit the record.
      EditWindow(RECORD_EDIT, oBrowseWnd)

   CASE (idMsg == PWMSG_MOUSEDOUBLE) .AND. (_1 == PMBUTTON_LEFT) .AND. ;
         (.NOT. DELETED()) .AND. (.NOT. oBrowseWnd:cargo[BROWSE_MINIMIZED])
      // If the left mouse button was doubleclicked, the record is not
      // deleted, and the window is not minimized, edit the record.  The
      // first click will have already moved the highlight to the proper
      // position.

      // Convert screen coordinates to window coordinates.
      oBrowseWnd:screenToWnd(_2, _3, @nRow, @nCol)

      // Make sure the doubleclick was in the window's viewport.
      IF oBrowseWnd:isPosVisible(nRow, nCol)
         EditWindow(RECORD_EDIT, oBrowseWnd)
      END IF // oBrowseWnd:isPosVisible(nRow, nCol)

   END CASE

   RETURN(lHandled)
   // END BrowseEventHandler( idMsg, _1, _2, _3, oBrowseWnd )

