/***
*  Errorsys.prg
*  Standard Clipper 5.0 error handler
*  Copyright (c) 1990 Nantucket Corp.  All rights reserved.
*
*  Compile:  /m/n/w
*/

#include "error.ch"
#include "pw.ch"


// Used below.
#define NTRIM( n )                     LTRIM( STR( n ) )
#define CRLF                           CHR(13) + CHR(10)


// Put messages to STDERR.
#command ? <list,...> ;
         => ;
         ?? CRLF ;;
         ?? <list>

#command ?? <list,...> ;
         => ;
         OutErr( <list> )



/***
*  ErrorSys()
*
*  Note:  automatically executes at startup
*/
PROCEDURE ErrorSys()

   ERRORBLOCK({ | e | DefError(e) })

   RETURN
   // END ErrorSys()



/***
*  DefError()
*/
STATIC FUNCTION DefError( e )

   LOCAL GetList  := {}, ;
         aOptions

   LOCAL cMessage, ;
         cProcName

   LOCAL lSysError

   LOCAL i, ;
         nChoice, ;
         nCol, ;
         nLines

   LOCAL oWnd


   // By default, division by zero yields zero.
// IF e:genCode == EG_ZERODIV
//    RETURN(0)
// END IF // e:genCode == EG_ZERODIV


   // For network open error, set NETERR() and subsystem default.
   IF (e:genCode == EG_OPEN) .AND. (e:osCode == 32) .AND. e:canDefault
      NETERR(.T.)
      RETURN(.F.) // Note this!
   END IF // (e:genCode == EG_OPEN) .AND. (e:osCode == 32) .AND. e...


   // For lock error during APPEND BLANK, set NETERR() and subsystem default.
   IF (e:genCode == EG_APPENDLOCK) .AND. e:canDefault
      NETERR(.T.)
      RETURN(.F.) // Note this!
   END IF // (e:genCode == EG_APPENDLOCK) .AND. e:canDefault


   // Build error message.
   cMessage := ErrorMessage(e)


   // Build options array.
   aOptions := { "Quit" }
   IF e:canRetry
      AADD(aOptions, "Retry")
   END IF // e:canRetry
   IF e:canDefault
      AADD(aOptions, "Default")
   END IF // e:canDefault


   // Save and set error handler state.
   lSysError   := pw():sysError(.T.)

   WaitMsg(.T.)

   nChoice  := 1
   IF .NOT. EMPTY(e:osCode)
      cMessage += CRLF + "(DOS Error " + NTRIM(e:osCode) + ")"
   END IF // .NOT. EMPTY(e:osCode)
   nLines   := MLCOUNT(cMessage, 46, i)

   CREATE WINDOW oWnd ;
      CENTERED ;
      SIZE nLines + 6, 50 ;
      STYLE PWSTYLE_ERROR + PWSTYLEINFO_MOVABLE ;
      TITLE "Clipper Runtime Error" ;
      WINDOW HANDLER { | idMsg | IIF(idMsg == PWMSG_CLOSE, ;
         pw():endModal(), ), .T. }

   FOR i := 01 TO nLines
      @ i, 00 SAY PADC(RTRIM(MEMOLINE(cMessage, 46, i)), 46)
   NEXT i 

   IF LEN(aOptions) == 1
      nCol  := 20
   ELSEIF LEN(aOptions) == 2
      nCol  := 13
   ELSE
      nCol  := 07
   END IF // LEN(aOptions) == 1

   @ nLines + 1, nCol GET AS PUSHBUTTON ;
      SIZE 03, 09 ;
      PROMPT aOptions[1] ;
      ACTION { || nChoice := 1, oWnd:close() }

   IF LEN(aOptions) >= 2
      nCol  += 12
      @ nLines + 1, nCol GET AS PUSHBUTTON ;
         SIZE 03, 09 ;
         PROMPT aOptions[2] ;
         ACTION { || nChoice := 2, oWnd:close() }
   END IF // LEN(aOptions) >= 2

   IF LEN(aOptions) >= 3
      nCol  += 12
      @ nLines + 1, nCol GET AS PUSHBUTTON ;
         SIZE 03, 09 ;
         PROMPT aOptions[3] ;
         ACTION { || nChoice := 3, oWnd:close() }
   END IF // LEN(aOptions) >= 3

   ATTACH CONTROLS TO oWnd

   OPEN WINDOW oWnd MODAL

   WaitMsg(.F.)

   IF pw():curFocus() == oWnd
      pw():beginModal()
   END IF // pw():curFocus() == oWnd

   DESTROY WINDOW oWnd

   // Restore error handler state.
   pw():sysError(lSysError)


   // If possible, recover or ignore.
   IF .NOT. EMPTY(nChoice)
      // Do as instructed.
      IF aOptions[nChoice] == "Break"
         BREAK(e)
      ELSEIF aOptions[nChoice] == "Retry"
         RETURN(.T.)
      ELSEIF aOptions[nChoice] == "Default"
         RETURN(.F.)
      END IF // aOptions[nChoice] == "Break"
   END IF // .NOT. EMPTY(nChoice)


   // Display message and traceback.
   ? cMessage

   // Start with the fourth activation to skip the ProVision:Windows error
   // handler.
   i := 4

   WHILE .NOT. EMPTY(PROCNAME(i))

      cProcName := PROCNAME(i)

      //
      // Comment out this IF statement if you want to include
      // ProVision:Windows-related procedures in the display of the
      // traceback.  This can make the traceback harder to read, thereby
      // making it more difficult to track down the real source of the
      // error.
      //
      IF .NOT. (("obj:PW" $ cProcName) .OR. ("PWSYS" $ cProcName) .OR. ;
            ("PWWND" $ cProcName))
         ? "Called from", TRIM(cProcName) + "(" + NTRIM(PROCLINE(i)) + ")"
      END IF // .NOT. (("obj:PW" $ cProcName) .OR. ("PWSYS" $ cPro...

      ++i

   END WHILE // .NOT. EMPTY(PROCNAME(i))

   // Give up.
   ERRORLEVEL(1)
   QUIT

   RETURN(.F.)
   // END DefError( e )



/***
*  ErrorMessage()
*/
STATIC FUNCTION ErrorMessage( e )

   LOCAL cMessage


   // Start error message.
   cMessage := IIF(e:severity > ES_WARNING, "Error ", "Warning ")

   // Add subsystem name if available.
   IF VALTYPE(e:subsystem) == "C"
      cMessage += e:subsystem
   ELSE
      cMessage += "???"
   END IF // VALTYPE(e:subsystem) == "C"

   // Add subsystem's error code if available.
   IF VALTYPE(e:subCode) == "N"
      cMessage += "/" + NTRIM(e:subCode)
   ELSE
      cMessage += "/???"
   END IF // ValType(e:subCode) == "N"

   // Add error description if available.
   IF VALTYPE(e:description) == "C"
      cMessage += "  " + e:description
   END IF // VALTYPE(e:description) == "C"

   // add either filename or operation
   IF .NOT. EMPTY(e:filename)
      cMessage += ": " + e:filename
   ELSEIF .NOT. EMPTY(e:operation)
      cMessage += ": " + e:operation
   END IF // .NOT. EMPTY(e:filename)

   RETURN(cMessage)
   // END ErrorMessage( e )

