/*****************************************************************************
** FILEBOX.PRG
**
** ProVision:Windows v1.21 file selection box
**
** by J. David Reynolds
**
** Copyright 1992 SofDesign International, Inc.
** All rights reserved
**
** tab spacing = 3
**
*****************************************************************************/

#include "directry.ch"
#include "inkey.ch"
#include "pw.ch"


STATIC   cFileName, ;
         cPattern

STATIC   oDir, ;
         oFile, ;
         oFileSpec, ;
         oFileWnd



/*****************************************************************************
** FileBox( cTitle, cFileSpec, lReadOnly, lNew ) --> cFileName
*****************************************************************************/
FUNCTION FileBox( cTitle, cFileSpec, lReadOnly, lNew )

   LOCAL GetList  := {}

   LOCAL cCurDir  := pvGetCWD(), ;
         cLastDir := pvGetCWD()

   LOCAL oOK


   // change mouse cursor to an hourglass
   WaitMsg(.T.)

   IIF(cTitle == NIL, cTitle := "Select File", )
   IIF(cFileSpec == NIL, cFileSpec := cCurDir + ;
      IIF(RIGHT(cCurDir, 1) == "\", "", "\") + "*.*", )
   IIF(lReadOnly == NIL, lReadOnly := .F., )
   IIF(lNew == NIL, lNew := .F., )

   cFileSpec   := UPPER(cFileSpec)
   cFileName   := ""


   CREATE WINDOW oFileWnd ;
      CENTERED ;
      SIZE 17, 52 ;
      STYLE PWSTYLE_DIALOG ;
      TITLE cTitle ;
      WINDOW HANDLER { | idMsg | IIF(idMsg == PWMSG_CLOSE, ;
         (pvCD(cLastDir), pw():endModal()), ), .T. }


   ParseFileSpec(@cFileSpec, @cCurDir)
   ChangeDir(cCurDir)
   cCurDir  := pvGetCWD()


   @ 01, 01 GET AS TEXTFIELD oFileSpec ;
      INITIAL PADR(cFileSpec, 64) ;
      PROMPT "File~name: " ;
      PICTURE "@!KS38"

   @ 06, 01 GET AS LISTBOX oFile ;
      USING GetFiles() ;
      SIZE 09, 16 ;
      PROMPT "~Files:" ;
      FILTER { | idMsg, xParam1, xParam2, xParam3 | ;
         oFileBlock(@idMsg, @xParam1, @xParam2, @xParam3) } ;
      ACTION { || oFileBlock(PWMSG_SETFOCUS,,,) }

   @ 06, 19 GET AS LISTBOX oDir ;
      USING GetDirectories() ;
      SIZE 09, 18 ;
      PROMPT "~Directories:" ;
      FILTER { | idMsg, xParam1, xParam2, xParam3 | ;
         oDirBlock(@idMsg, @xParam1, @xParam2, @xParam3) } ;
      ACTION { || oDirBlock(PWMSG_SETFOCUS,,,) }

   @ 08, 40 GET AS PUSHBUTTON oOK ;
      PROMPT "  OK  " ;
      ACTION { || IIF(oOKBlock(lNew), oFileWnd:close(), ) }

   @ 11, 40 GET AS PUSHBUTTON ;
      PROMPT "Cancel" ;
      ACTION { || oFileWnd:close() } ;
      SHORTCUT K_ESC

   ATTACH CONTROLS TO oFileWnd INITIAL oFile DEFAULT ACTION oOK

   IF lReadOnly .AND. (.NOT. lNew)
      oFileSpec:disable()
   END IF // lReadOnly .AND. (.NOT. lNew)

   OPEN WINDOW oFileWnd MODAL

   // get rid of the hourglass
   WaitMsg(.F.)

   IF pw():curFocus() == oFileWnd
      pw():beginModal()
   END IF // pw():curFocus() == oFileWnd

   oFileWnd:destroy()

   RETURN(cFileName)
   // END FileBox( cTitle, cFileSpec, lReadOnly, lNew )



/*****************************************************************************
** STATIC GetDirectories() --> aDirs 
*****************************************************************************/
STATIC FUNCTION GetDirectories()

   LOCAL aDirs := {}, ;
         aDirectory

   LOCAL cVolume

   LOCAL i


   cVolume     := "no label"
   aDirectory  := DIRECTORY("*.*", "V")
   IF LEN(aDirectory) > 0
      cVolume  := aDirectory[1, F_NAME]
   END IF // LEN(aDirectory) > 0

   aDirectory  := DIRECTORY("*.*", "D")
   IF LEN(aDirectory) > 0
      FOR i := 1 TO LEN(aDirectory)
         IF "D" $ aDirectory[i, F_ATTR]
            AADD(aDirs, "[" + aDirectory[i, F_NAME] + "]")
         END IF // "D" $ aDirectory[i, F_ATTR]
      NEXT i 
   END IF // LEN(aDirectory) > 0

   i  := ASCAN(aDirs, "[.]")
   IF i != 0
      ADEL(aDirs, i)
      ASIZE(aDirs, LEN(aDirs) - 1)
   END IF // i != 0

   ASORT(aDirs)

   FOR i := 65 TO 90
      IF pvIsDrive(CHR(i))
         AADD(aDirs, "[-" + CHR(i) + "-]")
      END IF // pvIsDrive(CHR(i))
   NEXT i 

   @ 03, 01 SAY PADR("Directory of " + LEFT(pvGetCWD(), 2) + " (" + ;
      cVolume + ")", 48)
   IF LEN(pvGetCWD()) > 50
      @ 04, 01 SAY LEFT(SUBSTR(pvGetCWD(), 3), 22) + "" + ;
         RIGHT(SUBSTR(pvGetCWD(), 2), 23)
   ELSE
      @ 04, 01 SAY PADR(SUBSTR(pvGetCWD(), 3), 48)
   END IF // LEN(pvGetCWD()) > 50

   IF LEN(aDirs) == 0
      AADD(aDirs, SPACE(14))
   END IF // LEN(aDirs) == 0

   RETURN(aDirs)
   // END GetDirectories()



/*****************************************************************************
** STATIC GetFiles() --> aFiles
*****************************************************************************/
STATIC FUNCTION GetFiles()

   LOCAL aDirectory, ;
         aFiles   := {}

   LOCAL i


   aDirectory  := DIRECTORY(cPattern)
   IF LEN(aDirectory) > 0
      FOR i := 1 TO LEN(aDirectory)
         AADD(aFiles, aDirectory[i, F_NAME])
      NEXT i 
   END IF // LEN(aDirectory) > 0

   ASORT(aFiles)

   FOR i := 1 TO LEN(aFiles)
      aFiles[i]   := PADR(aFiles[i], 12)
   NEXT i 

   RETURN(aFiles)
   // END GetFiles()



/*****************************************************************************
** STATIC oDirBlock( @idMsg, @xParam1, @xParam2, @xParam3 ) --> NIL
*****************************************************************************/
STATIC FUNCTION oDirBlock( idMsg, xParam1, xParam2, xParam3 )

   LOCAL cDir


   DO CASE
   CASE idMsg == PWMSG_SETFOCUS
      ParsecDir(@cDir)
      IF ":" $ cDir
         oFileSpec:setValue(PADR(cDir + cPattern, 64))
      ELSE
         oFileSpec:setValue(PADR(cDir + cPattern, 64))
      END IF // ":" $ cDir

   CASE (idMsg == PWMSG_KEYPRESS) .AND. (xParam1 == K_ENTER)
      ParsecDir(@cDir)
      UpdateFiles(cDir)

      // signal message handled
      idMsg := PWMSG_NONE

   CASE (idMsg == PWMSG_MOUSEDOUBLE) .AND. (xParam1 == PMBUTTON_LEFT)
      // was the doubleclick on an item?
      IF TestListbox(oDir, xParam2, xParam3)

         ParsecDir(@cDir)
         UpdateFiles(cDir)

         // signal message handled
         idMsg := PWMSG_NONE

      END IF // TestListbox(oDir, xParam2, xParam3)

   END CASE

   RETURN(NIL)
   // END oDirBlock( idMsg, xParam1, xParam2, xParam3 )



/*****************************************************************************
** STATIC oFileBlock( @idMsg, @xParam1, @xParam2, @xParam3 ) --> NIL
*****************************************************************************/
STATIC FUNCTION oFileBlock(   idMsg, xParam1, xParam2, xParam3 )

   LOCAL cFile


   DO CASE
   CASE idMsg == PWMSG_SETFOCUS
      IF oFile:getValue() != 0
         cFile := oFile:getArray()[oFile:getValue()]
      ELSE
         cFile := cPattern
      END IF // oFile:getValue() != 0
      oFileSpec:setValue(PADR(cFile, 64))

   CASE (idMsg == PWMSG_MOUSEDOUBLE) .AND. (xParam1 == PMBUTTON_LEFT)
      // was the doubleclick on an item?
      IF TestListbox(oFile, xParam2, xParam3)

         IF LEN(pw():k_defaultAction) > 0
            oFileWnd:event(PWMSG_KEYPRESS, pw():k_defaultAction[1])
         END IF // LEN(pw():k_defaultAction) > 0

         // signal message handled
         idMsg := PWMSG_NONE

      END IF // TestListbox(oFile, xParam2, xParam3)

   END CASE

   RETURN(NIL)
   // END oFileBlock(   idMsg, xParam1, xParam2, xParam3 )



/*****************************************************************************
** STATIC oOKBlock( lNew ) --> lValid
*****************************************************************************/
STATIC FUNCTION oOKBlock( lNew )

   LOCAL cCurDir, ;
         cFileSpec

   LOCAL lValid


   cFileSpec   := RTRIM(oFileSpec:getValue())
   lValid      := lNew

   IF lValid

      cFileName   := cFileSpec
      lValid      := .T.

   ELSEIF .NOT. (("*" $ cFileSpec) .OR. ("?" $ cFileSpec))

      IF FILE(cFileSpec) .AND. ((":" $ cFileSpec) .OR. ("\" $ cFileSpec))

         cFileName   := cFileSpec
         lValid      := .T.

      ELSE

         cFileSpec   := pvGetCWD() + IIF(RIGHT(pvGetCWD(), 1) == "\", ;
            "", "\") + cFileSpec
         IF FILE(cFileSpec)
            cFileName   := cFileSpec
            lValid      := .T.
         ELSE
            cFileSpec   := RTRIM(oFileSpec:getValue()) + ;
               IIF(RIGHT(cFileSpec, 1) $ ":\", "", "\") + cPattern
         END IF // FILE(cFileSpec)

      END IF // FILE(cFileSpec) .AND. ((":" $ cFileSpec) .OR. ("\" $ ...

   END IF // lValid

   IF .NOT. lValid

      ParseFileSpec(@cFileSpec, @cCurDir)
      UpdateFiles(cCurDir)
      oFileWnd:select(oFile)

   END IF // .NOT. lValid

   RETURN(lValid)
   // END oOKBlock( lNew )



/*****************************************************************************
** STATIC ParsecDir( @cDir )
*****************************************************************************/
STATIC FUNCTION ParsecDir( cDir )


   cDir  := oDir:getArray()[oDir:getValue()]

   IF LEFT(cDir, 2) == "[-"
      cDir  := SUBSTR(cDir, 3, 1) + ":"
   ELSE
      cDir  := SUBSTR(cDir, 2, LEN(cDir) - 2) + "\"
   END IF // LEFT(cDir, 2) == "[-"

   RETURN(NIL)
   // END ParsecDir( @cDir )



/*****************************************************************************
** STATIC ParseFileSpec( @cFileSpec, @cCurDir ) --> NIL
*****************************************************************************/
STATIC FUNCTION ParseFileSpec( cFileSpec, cCurDir )


   cFileSpec   := LTRIM(RTRIM(cFileSpec))

   DO CASE
   CASE "\" $ cFileSpec
      cCurDir  := LEFT(cFileSpec, RAT("\", cFileSpec))
      cPattern := SUBSTR(cFileSpec, RAT("\", cFileSpec) + 1)

   CASE ":" $ cFileSpec
      cCurDir  := LEFT(cFileSpec, 2) + ;
         SUBSTR(pvGetCWD(LEFT(cFileSpec, 2)), 3)
      cPattern := SUBSTR(cFileSpec, 3)

   OTHERWISE
      cCurDir  := ""
      cPattern := cFileSpec

   END CASE

   IF EMPTY(cCurDir)
      cCurDir  := pvGetCWD()
   END IF // EMPTY(cCurDir)
   IF (RIGHT(cCurDir, 1) == "\") .AND. (LEN(cCurDir) > 1)
      cCurDir  := LEFT(cCurDir, LEN(cCurDir) - 1)
   END IF // (RIGHT(cCurDir, 1) == "\") .AND. (LEN(cCurDir) > 1)
   IF RIGHT(cCurDir, 1) == ":"
      cCurDir  := cCurDir + "\"
   END IF // RIGHT(cCurDir, 1) == ":"

   IF EMPTY(cPattern)
      cPattern := "*.*"
   END IF // EMPTY(cPattern)
   IF (RIGHT(cPattern , 1) == "*") .AND. ;
         (.NOT. (RIGHT(cPattern , 2) == ".*")) .AND. ;
         (LEN(cPattern) < 8)
   cPattern += ".*"
   END IF // (RIGHT(cPattern , 1) == "*") .AND. ...

   cFileSpec   := PADR(cPattern, 64)

   RETURN(NIL)
   // END ParseFileSpec( cFileSpec, cCurDir )



/*****************************************************************************
** STATIC UpdateFiles( cDir )
*****************************************************************************/
STATIC FUNCTION UpdateFiles( cDir )

   LOCAL cNewDir


   // change mouse cursor to an hourglass
   WaitMsg(.T.)

   IF RIGHT(cDir, 1) == ":"
      cNewDir  := cDir + "."
   ELSEIF RIGHT(cDir, 2) == ":\"
      cNewDir  := cDir
   ELSEIF (RIGHT(cDir, 1) == "\") .AND. (LEN(cDir) > 1)
      cNewDir  := LEFT(cDir, LEN(cDir) - 1)
   ELSE
      cNewDir  := cDir
   END IF // SUBSTR(cDir, 2, 1) == ":"

   IF ChangeDir(cNewDir)
      oFile:setArray(GetFiles())
      oDir:setArray(GetDirectories())
   END IF // ChangeDir(cNewDir)

   // get rid of the hourglass
   WaitMsg(.F.)

   RETURN(NIL)
   // END UpdateFiles( cDir )



/*****************************************************************************
** STATIC ChangeDir( cDir ) --> lSuccess
*****************************************************************************/
STATIC FUNCTION ChangeDir( cDir )

   LOCAL cCurDir  := pvGetCWD(), ;
         cDrive

   LOCAL lRetry   := .T., ;
         lSuccess := pvCD(cDir)


   WHILE (.NOT. lSuccess) .AND. lRetry

      cDrive   := LEFT(cDir, 2)
      IF .NOT. (SUBSTR(cDrive, 2) == ":")
         cDrive   := LEFT(pvGetCWD(), 2)
      END IF // .NOT. (SUBSTR(cDrive, 2) == ":")

      lRetry   := DriveError(cDrive)
      IF lRetry
         lSuccess := pvCD(cDir)
      END IF // lRetry

   END WHILE // (.NOT. lSuccess) .AND. lRetry

   IF .NOT. lSuccess
      pvCD(cCurDir)
   END IF // .NOT. lSuccess

   RETURN(lSuccess)
   // END ChangeDir( cDir )



/*****************************************************************************
** STATIC DriveError( cDrive ) --> lRetry
*****************************************************************************/
STATIC FUNCTION DriveError( cDrive )

   LOCAL GetList  := {}

   LOCAL lRetry   := .F.

   LOCAL oErrorWnd


   CREATE WINDOW oErrorWnd ;
      CENTERED ;
      SIZE 08, 34 ;
      STYLE PWSTYLE_ERROR ;
      TITLE "Disk Error" ;
      WINDOW HANDLER { | idMsg | IIF(idMsg == PWMSG_CLOSE, ;
         pw():endModal(), ), .T. }

   @ 01, 01 SAY "Unable to access the specified"
   @ 02, 01 SAY "file or directory on drive " + cDrive

   @ 03, 06 GET AS PUSHBUTTON ;
      PROMPT "~Retry" ;
      ACTION { || lRetry := .T., oErrorWnd:close() }

   @ 03, 18 GET AS PUSHBUTTON ;
      PROMPT "Cancel" ;
      ACTION { || oErrorWnd:close() } ;
      SHORTCUT K_ESC

   ATTACH CONTROLS TO oErrorWnd

   OPEN WINDOW oErrorWnd MODAL

   // get rid of the hourglass
   WaitMsg(.F.)

   IF pw():curFocus() == oErrorWnd
      pw():beginModal()
   END IF // pw():curFocus() == oErrorWnd

   oErrorWnd:destroy()

   // change mouse cursor to an hourglass
   WaitMsg(.T.)

   RETURN(lRetry)
   // END DriveError( cDrive )



/*****************************************************************************
** STATIC TestListbox( oListbox, nScreenRow, nScreenCol ) --> lOnTopOf
**
** This will determine if screen coordinates (as passed with mouse messages)
** are in the "items" area of a listbox.
**
*****************************************************************************/
STATIC FUNCTION TestListbox( oListbox, nScreenRow, nScreenCol )

   LOCAL lOnTopOf := .F.

   LOCAL nCol, ;
         nRow, ;
         nWndCol, ;
         nWndRow

   LOCAL oCtrlRef, ;
         oWnd     := oListbox:wnd


   // Convert screen coordinates to window coordinates.
   oWnd:screenToWnd(nScreenRow, nScreenCol, @nRow, @nCol)

   // Get the input control at those window coordinates.
   oCtrlRef := oWnd:ctrlAtPos(nRow, nCol)

   // Are the coordinates on the listbox?
   IF oCtrlRef == oListbox

      // Convert window coordinates to control coordinates.
      oListbox:wndToCtrl(nRow, nCol, @nRow, @nCol)

      // Are the coordinates in the "items" area?
      IF (nCol < (oListbox:width - 3)) .AND. (nRow > 1) .AND. ;
            (nRow <= (oListbox:height - 2))
         lOnTopOf := .T.
      END IF // (nCol < (oListbox:width - 3)) .AND. (nRow > 1) .AN...

   END IF // oCtrlRef == oDir

   // lOnTopOf will be .T. if the screen position nScreenRow, nScreenCol
   // is in the "items" area of oListbox.
   RETURN(lOnTopOf)
   // END TestListbox( oListbox, nScreenRow, nScreenCol )

