/*****************************************************************************
** FILES.PRG
**
** ProVision:Windows v1.21 functions to generate, open, browse, and close
** the sample database files
**
** by J. David Reynolds
**
** Copyright 1991, 1992 SofDesign International, Inc.
** All rights reserved
**
** tab spacing = 3
**
*****************************************************************************/

#include "inkey.ch"
#include "pw.ch"
#include "tutor.ch"


STATIC   aEditList   := {}             // records currently being edited



/*****************************************************************************
** OpenFiles() --> NIL
**
** This function opens the files and creates them, if necessary.
**
*****************************************************************************/
FUNCTION OpenFiles()


   // Open the files.  Create them if they don't exist.
   IF FILE("CUSTOMER.DBF") .AND. FILE("INVOICES.DBF") .AND. ;
         FILE("PAYMENTS.DBF")

      // They're all there.
      USE Invoices NEW EXCLUSIVE

      USE Payments NEW EXCLUSIVE

      USE Customer NEW EXCLUSIVE

   ELSE

      // Generate some pseudorandom records.  This will also force new
      // index files to be built.
      GenData()   // Reminds me of the old NetWare days!

   END IF // FILE("CUSTOMER.DBF") .AND. FILE("INVOICES.DBF") .A...

   // Open the index files.  Create them if they don't exist.
   IF FILE("CUSTOMER.NTX") .AND. FILE("INVOICES.NTX") .AND. ;
         FILE("PAYMENTS.NTX")

      SELECT Invoices
      SET INDEX TO Invoices

      SELECT Payments
      SET INDEX TO Payments

      SELECT Customer
      SET INDEX TO Customer

   ELSE

      // Because the files already exist, this will only build
      // new index files.
      GenData()   // Reminds me of the old NetWare days!

   END IF // FILE("CUSTOMER.NTX") .AND. FILE("INVOICES.NTX") .A...

   // Relate Customer to its children.
   SET RELATION   TO Customer->CustNmbr INTO Invoices, ;
                  TO Customer->CustNmbr INTO Payments
   GO TOP

   RETURN(NIL)
   // END OpenFiles()



/*****************************************************************************
** CloseFiles() --> NIL
**
** This function closes the files.
**
*****************************************************************************/
FUNCTION CloseFiles()


   // Turn the mouse cursor into an hourglass (or display a wait message).
   WaitMsg(.T.)

   // Close the files.
   SELECT Invoices
   CLOSE

   SELECT Payments
   CLOSE

   SELECT Customer
   CLOSE

   // Restore the mouse cursor (or remove the wait message).
   WaitMsg(.F.)

   RETURN(NIL)
   // END CloseFiles()



/*****************************************************************************
** STATIC GenData() --> NIL
**
** This function generates 100 pseudorandom customer records.  It displays
** a progress gauge in a window and, through the use of pw():yield(), allows
** the user to cancel the generation.  It then builds an index for the
** generated data, also displaying a progress gauge, but not allowing the
** user to cancel the operation.
**
*****************************************************************************/
STATIC FUNCTION GenData()

   LOCAL GetList  := {}

   LOCAL lQuit    := .F.

   LOCAL i, ;
         j, ;
         nLastRec, ;
         nPrefix

   LOCAL oMsgWnd, ;
         oOutput


   // Save the current output window.
   oOutput  := pw():getOutput()

   // Create the window.
   CREATE WINDOW oMsgWnd ;
      CENTERED ;
      SIZE 08, 30 ;
      STYLE PWSTYLE_MESSAGE - PWSTYLEINFO_SYSTEMMENU ;
      WINDOW HANDLER { | idMsg | IIF(idMsg == PWMSG_CLOSE, ;
         lQuit := .T., ), .T. }

   // Create a 'Cancel' pushbutton.
   @ 03, 10 GET AS PUSHBUTTON ;
      PROMPT "Cancel" ;
      ACTION { || oMsgWnd:close() } ;
      SHORTCUT K_ESC

   // Attach the controls to the window.
   ATTACH CONTROLS TO oMsgWnd

   IF .NOT. (FILE("CUSTOMER.DBF") .AND. FILE("INVOICES.DBF") .AND. ;
         FILE("PAYMENTS.DBF"))

      // Restore the mouse cursor (or remove the wait message).
      WaitMsg(.F.)

      // Set the window title.
      oMsgWnd:setTitle("Generating data...")

      // Open the window modally.
      OPEN WINDOW oMsgWnd MODAL

      // Initialize the progress gauge.
      Gauge(0, 100)

      // Create the files.
      DBCREATE("INVOICES.DBF", ;
         {  { "CUSTNMBR",  "N",  5, 0 }, ;
            { "INVNMBR",   "N",  5, 0 }, ;
            { "AMOUNT",    "N", 15, 2 } })

      DBCREATE("PAYMENTS.DBF", ;
         {  { "CUSTNMBR",  "N",  5, 0 }, ;
            { "PAYMENT",   "D",  8, 0 }, ;
            { "AMOUNT",    "N", 15, 2 } })

      DBCREATE("CUSTOMER.DBF", ;
         {  { "CUSTNMBR",  "N",  5, 0 }, ;
            { "ACTIVE",    "L",  1, 0 }, ;
            { "COMPANY",   "C", 35, 0 }, ;
            { "DIVISION",  "C", 15, 0 }, ;
            { "CONTACT",   "C", 35, 0 }, ;
            { "SALUTE",    "N",  1, 0 }, ;
            { "TITLE",     "C", 10, 0 }, ;
            { "ADDR1",     "C", 35, 0 }, ;
            { "ADDR2",     "C", 35, 0 }, ;
            { "CITY",      "C", 25, 0 }, ;
            { "STATE",     "C", 20, 0 }, ;
            { "COUNTRY",   "C", 22, 0 }, ;
            { "ZIP",       "C", 10, 0 }, ;
            { "HOMEPHONE", "N", 10, 0 }, ;
            { "WORKPHONE", "N", 10, 0 }, ;
            { "CARPHONE",  "N", 10, 0 }, ;
            { "FAXPHONE",  "N", 10, 0 }, ;
            { "MAXCREDIT", "N", 15, 2 }, ;
            { "LASTPMT",   "N", 15, 2 }, ;
            { "TERMS",     "N",  2, 0 }, ;
            { "DISCOUNT",  "N",  2, 0 }, ;
            { "NOTES",     "M", 10, 0 } })

      // Open the files.
      USE Invoices NEW EXCLUSIVE

      USE Payments NEW EXCLUSIVE

      USE Customer NEW EXCLUSIVE

      // Add 100 pseudorandom customer records.
      i  := 1
      WHILE (i <= 100) .AND. (.NOT. lQuit)

         // Update the progress gauge.
         Gauge(i, 100)

         // Add a record.
         APPEND BLANK
         REPLACE Customer->CustNmbr    WITH NewNumber(10000, 99999)

         // Add 1 to 15 pseudorandom invoice records.
         SELECT Invoices
         FOR j := 1 TO NewNumber(1, 15)
            APPEND BLANK
            REPLACE Invoices->CustNmbr WITH Customer->CustNmbr
            REPLACE Invoices->InvNmbr  WITH NewNumber(10000, 99999)
            REPLACE Invoices->Amount   WITH NewNumber(100, 9999)
         NEXT j 

         // Add 1 to 15 pseudorandom payment records.
         SELECT Payments
         FOR j := 1 TO NewNumber(1, 15)
            APPEND BLANK
            REPLACE Payments->CustNmbr WITH Customer->CustNmbr

            REPLACE Payments->Payment  WITH CTOD(STRZERO(NewNumber(1, ;
               12), 2) + "/" + STRZERO(NewNumber(1, 28), 2) + "/" + ;
               STRZERO(NewNumber(90, 92), 2))
            REPLACE Payments->Amount   WITH NewNumber(100, 9999)
         NEXT j 

         SELECT Customer
         REPLACE Customer->Active      WITH (NewNumber(1, 2) == 1)
         REPLACE Customer->Company     WITH NewCompany()
         REPLACE Customer->Division    WITH { "", "Electronics", ;
            "Maintenance", "Engineering", "Administration", ;
            "Development" }[NewNumber(1, 6)]
         REPLACE Customer->Contact     WITH NewName()
         REPLACE Customer->Salute      WITH NewNumber(1, 3)
         REPLACE Customer->Title       WITH { "", "V.P.", "Pres.", ;
            "C.E.O.", "C.I.O.", "C.F.O.", "Director" }[NewNumber(1, 7)]
         REPLACE Customer->Addr1       WITH NewAddress()
         REPLACE Customer->Addr2       WITH IIF(NewNumber(1, 4) == 1, ;
            "deliver to " + NewAddress(), "")
         REPLACE Customer->City        WITH NewCity()
         REPLACE Customer->State       WITH NewState()
         REPLACE Customer->Country     WITH "USA"
         REPLACE Customer->Zip         WITH LTRIM(STR(NewNumber(10000, ;
            99999)))
         nPrefix  := NewNumber(100, 999) * 10000000
         REPLACE Customer->HomePhone   WITH nPrefix + NewNumber(1000000, ;
            9999999)
         REPLACE Customer->WorkPhone   WITH nPrefix + NewNumber(1000000, ;
            9999999)
         REPLACE Customer->CarPhone    WITH nPrefix + NewNumber(1000000, ;
            9999999)
         REPLACE Customer->FAXPhone    WITH nPrefix + NewNumber(1000000, ;
            9999999)
         REPLACE Customer->MaxCredit   WITH NewNumber(10, NewNumber(15, ;
            1000)) * 1000
         REPLACE Customer->LastPmt     WITH NewNumber(500, 2000)
         REPLACE Customer->Terms       WITH { 10, 30, 90 }[NewNumber(1, 3)]
         REPLACE Customer->Discount    WITH NewNumber(0, 33)

         // Allow ProVision:Windows to process any waiting events.
         pw():yield()

         // Increment the record count.
         ++i

      END WHILE // (i <= 100) .AND. (.NOT. lQuit)

      // Close the window.
      CLOSE WINDOW oMsgWnd

      // Turn the mouse cursor into an hourglass (or display a wait message).
      WaitMsg(.T.)

   END IF // .NOT. (FILE("CUSTOMER.DBF") .AND. FILE("INVOICES.DBF"...

   // Disable the 'Cancel' pushbutton.
   oMsgWnd:getGetList()[1]:disable()

   // Set the window title.
   oMsgWnd:setTitle("Generating CUSTOMER.NTX...")

   // Don't let that window close!
   oMsgWnd:windowHandler   := { | idMsg | IIF(idMsg == PWMSG_CLOSE, ;
      .F., .T.) }

   // Reopen the window modally.
   OPEN WINDOW oMsgWnd MODAL

   // Initialize the progress gauge.
   nLastRec := Customer->(LASTREC())
   Gauge(0, nLastRec)

   SELECT Customer
   DBCREATEINDEX("CUSTOMER.NTX", "(CustNmbr)", ;
      { || Gauge(RECNO(), nLastRec), pw():yield(), ;
      Customer->CustNmbr }, .F.)

   // Set the window title.
   oMsgWnd:setTitle("Generating INVOICE.NTX...")

   // Initialize the progress gauge.
   nLastRec := Invoices->(LASTREC())
   Gauge(0, nLastRec)

   // Build the indexes, updating the progress gauge.  The yield() calls
   // are only necessary if you want timer, timeout, idle, and background
   // processing to occur while the indexes are being built.  Removing the
   // yield() calls speeds up the indexing process a great deal.
   SELECT Invoices
   DBCREATEINDEX("INVOICES.NTX", "(CustNmbr)", ;
      { || Gauge(RECNO(), nLastRec), pw():yield(), ;
      Invoices->CustNmbr }, .F.)

   // Set the window title.
   oMsgWnd:setTitle("Generating PAYMENTS.NTX...")

   // Initialize the progress gauge.
   nLastRec := Payments->(LASTREC())
   Gauge(0, nLastRec)

   SELECT Payments
   DBCREATEINDEX("PAYMENTS.NTX", "(CustNmbr)", ;
      { || Gauge(RECNO(), nLastRec), pw():yield(), ;
      Payments->CustNmbr }, .F.)

   SELECT Customer

   // It's okay to close the window now.
   oMsgWnd:windowHandler   := bTRUE

   // Close and destroy the message window.
   DESTROY WINDOW oMsgWnd

   // Restore the previous output window.
   pw():setOutput(oOutput)

   // Reset the timeout value in case the building of the files and the
   // indexing took longer than the timeout value.
   pw():resetTimeout()

   RETURN(NIL)
   // END GenData()



/*****************************************************************************
** STATIC Gauge( nCurrent, nMax ) --> NIL
**
** This displays a progress indicator in the current window.
**
*****************************************************************************/
STATIC FUNCTION Gauge( nCurrent, nMax )


   IF nCurrent == 0
      @ 01, 03 SAY "Ĵ"
      @ 02, 03 SAY "0   25   50   75   100"
   END IF // nCurrent == 0

   IF nCurrent <= nMax
      @ 01, 03 SAY SPACE(INT((nCurrent / nMax) * 21)) COLOR "N/W"
   END IF // nCurrent <= nMax

   RETURN(NIL)
   // END Gauge( nCurrent, nMax )



/*****************************************************************************
** BrowseGoBottom( oBrowseWnd ) --> NIL
**
** This function implements the goBottomBlock for the TBROWSE.  It also
** updates the states of the Record menuitems.
**
*****************************************************************************/
FUNCTION BrowseGoBottom( oBrowseWnd )


   // Go to the end of the file.
   GO BOTTOM

   // Update the window's current record number, and update the states of
   // the Record menuitems.
   UpdateMenuitems(oBrowseWnd)

   RETURN(NIL)
   // END BrowseGoBottom( oBrowseWnd )



/*****************************************************************************
** BrowseGoTop( oBrowseWnd ) --> NIL
**
** This function implements the goTopBlock for the TBROWSE.  It also
** updates the states of the Record menuitems.
**
*****************************************************************************/
FUNCTION BrowseGoTop( oBrowseWnd )


   // Go to the beginning of the file.
   GO TOP

   // Update the window's current record number, and update the states of
   // the Record menuitems.
   UpdateMenuitems(oBrowseWnd)

   RETURN(NIL)
   // END BrowseGoTop( oBrowseWnd )



/*****************************************************************************
** BrowseSkip( nToSkip, oBrowseWnd ) --> nSkipped
**
** This function implements the skipBlock for the TBROWSE.  It also updates
** the states of the Record menuitems when the data pointer moves.
**
*****************************************************************************/
FUNCTION BrowseSkip( nToSkip, oBrowseWnd )

   LOCAL nSkipped := 0


   DO CASE
   CASE nToSkip > 0
      // moving forward
      WHILE (nSkipped < nToSkip) .AND. (.NOT. EOF())
         SKIP +1
         ++nSkipped
      END WHILE // (nSkipped < nToSkip) .AND. (.NOT. EOF())
      IF EOF()
         SKIP -1
         --nSkipped
      END IF // EOF()

   CASE nToSkip < 0
      // moving backward
      WHILE (nSkipped > nToSkip) .AND. (.NOT. BOF())
         SKIP -1
         --nSkipped
      END WHILE // (nSkipped > nToSkip) .AND. (.NOT. BOF())
      IF BOF()
         ++nSkipped
      END IF // BOF()

   CASE nToSkip == 0
      // no movement
      SKIP 0

   END CASE

   // If the data pointer moved, update the window's current record number,
   // and update the states of the Record menuitems.
   IF nSkipped != 0
      UpdateMenuitems(oBrowseWnd)
   END IF // nSkipped != 0

   RETURN(nSkipped)
   // END BrowseSkip( nToSkip, oBrowseWnd )



/*****************************************************************************
** RefreshTBROWSE( oBrowseWnd ) --> NIL
**
** This refreshes all rows in the TBROWSE display and updates the browse
** window's scrollbars.
**
*****************************************************************************/
FUNCTION RefreshTBROWSE( oBrowseWnd )

   LOCAL oOutput


   // Make sure Clipper's output is directed to the browse window.
   oOutput  := pw():setOutput(oBrowseWnd)

   // All rows may need refreshing if the index key was changed.
   oBrowseWnd:getTBROWSE():refreshAll()

   // This is necessary to refresh the scrollbars of the browse window.
   // It will also stabilize the TBROWSE.
   oBrowseWnd:scroll(PWSCROLL_VERTICAL, PWMSG_SCROLLREFRESH, NIL)
   oBrowseWnd:scroll(PWSCROLL_HORIZONTAL, PWMSG_SCROLLREFRESH, NIL)

   // Restore the previous output window.
   pw():setOutput(oOutput)

   RETURN(NIL)
   // END RefreshTBROWSE( oBrowseWnd )



/*****************************************************************************
** UpdateMenuitems( oBrowseWnd ) --> NIL
**
** This function updates the states of the Record menuitems.
**
*****************************************************************************/
FUNCTION UpdateMenuitems( oBrowseWnd )


   // Update the states of the Record menuitems.
   IF DELETED()
      oBrowseWnd:cargo[BROWSE_ITEMEDIT]:disable()
      oBrowseWnd:cargo[BROWSE_ITEMDELETE]:disable()
      oBrowseWnd:cargo[BROWSE_ITEMRECALL]:enable()
   ELSE
      oBrowseWnd:cargo[BROWSE_ITEMEDIT]:enable()
      oBrowseWnd:cargo[BROWSE_ITEMDELETE]:enable()
      oBrowseWnd:cargo[BROWSE_ITEMRECALL]:disable()
   END IF // DELETED()

   RETURN(NIL)
   // END UpdateMenuitems( oBrowseWnd )



/*****************************************************************************
** SearchWindow( oBrowseWnd ) --> NIL
**
** This opens a window with which you can search for a record.
**
*****************************************************************************/
FUNCTION SearchWindow( oBrowseWnd )

   LOCAL GetList  := {}

   LOCAL oSearchFor


   // Turn the mouse cursor into an hourglass (or display a wait message).
   WaitMsg(.T.)

   // Create one instance of a search window for the browse window.
   IF oBrowseWnd:cargo[BROWSE_SEARCHWND] == NIL

      // Create the window if it doesn't exist.
      CREATE WINDOW oBrowseWnd:cargo[BROWSE_SEARCHWND] ;
         AT 01, 02 ;
         SIZE 08, 26 ;
         TITLE "Customer Search" ;
         STYLE PWSTYLE_DIALOG ;
         WINDOW HANDLER { | idMsg | SearchWndHandler(idMsg, oBrowseWnd) }

      // Store the browse window in the search window's cargo.
      oBrowseWnd:cargo[BROWSE_SEARCHWND]:cargo  := oBrowseWnd

      // Now let's create the input controls for the window.
      @ 01, 01 SAY "Search for Customer #:"
      @ 02, 09 GET AS TEXTFIELD oSearchFor ;
         PROMPT "" ;
         PICTURE "#####" ;
         INITIAL 0

      @ 03, 08 GET AS PUSHBUTTON ;
         PROMPT "Search" ;
         ACTION { || SearchRecord(oSearchFor:getValue(), oBrowseWnd), ;
            oBrowseWnd:cargo[BROWSE_SEARCHWND]:select(oSearchFor) }

      ATTACH CONTROLS TO oBrowseWnd:cargo[BROWSE_SEARCHWND] ;
         DEFAULT ACTION ATAIL(GetList)

   END IF // oBrowseWnd:cargo[BROWSE_SEARCHWND] == NIL

   // Open the window.  Everything to this point has been invisible to the
   // user.
   OPEN WINDOW oBrowseWnd:cargo[BROWSE_SEARCHWND]

   // Restore the mouse cursor (or remove the wait message).
   WaitMsg(.F.)

   RETURN(NIL)
   // END SearchWindow( oBrowseWnd )



/*****************************************************************************
** STATIC SearchWndHandler( idMsg, oBrowseWnd ) --> lHandled
**
** This implements the window handler for the search window and is
** responsible for disabling the File/Search menuitem when the search
** window is opened, and reenabling it when the search window is closed.
** It must also restore the proper workarea and record number when it
** receives focus.
**
*****************************************************************************/
STATIC FUNCTION SearchWndHandler( idMsg, oBrowseWnd )

   LOCAL lHandled := .T.


   DO CASE
   CASE idMsg == PWMSG_CLOSE
      // Reenable the File/Search menuitem.
      oBrowseWnd:cargo[BROWSE_ITEMSEARCH]:enable()

   CASE idMsg == PWMSG_KILLFOCUS
      // Save the current record number.
      oBrowseWnd:cargo[BROWSE_RECNO]   := RECNO()

   CASE idMsg == PWMSG_OPEN
      // Disable the File/Search menuitem.
      oBrowseWnd:cargo[BROWSE_ITEMSEARCH]:disable()

   CASE idMsg == PWMSG_SETFOCUS
      // Restore the proper workarea and record number.
      SELECT Customer
      GO oBrowseWnd:cargo[BROWSE_RECNO]

   END CASE

   RETURN(lHandled)
   // END SearchWndHandler( idMsg, oBrowseWnd )



/*****************************************************************************
** STATIC SearchRecord( nKey, oBrowseWnd ) --> NIL
**
** This searches for a record by key and updates the display if a match is
** found.
**
*****************************************************************************/
STATIC FUNCTION SearchRecord( nKey, oBrowseWnd )

   LOCAL nRecNo


   // Search for the key.
   SEEK nKey

   // If it is found, refresh the browse window, and save this new record
   // number as the browse window's current record.
   IF FOUND()

      // Save the found record.
      nRecNo   := RECNO()

      // Turn off display updating.
      DISPBEGIN()

      // Stabilize the TBROWSE.
      Customer->(RefreshTBROWSE(oBrowseWnd))

      // This is to handle the "anomaly" of TBROWSE that when you're near
      // the beginning of the data source, greater importance is placed on
      // keeping the highlight bar in the same place than on keeping the
      // data source positioned in the same place.  Because of this, after
      // stabilizing the TBROWSE we may have moved forward off of the
      // record we just found.
      IF RECNO() != nRecNo

         // The TBROWSE moved our record pointer forward, so we need to
         // move back to the original record.  We have to follow TBROWSE's
         // rules and send it up() messages, since if we just did a
         // GO nRecNo and refreshed the TBROWSE, we'd be right back where
         // we started.  Here, we're actually going to send scroll up
         // messages to the browse window.  This will, in turn, send the
         // up() messages to the TBROWSE automatically, in addition to
         // refreshing the vertical scroll bar for us.
         WHILE RECNO() != nRecNo
            oBrowseWnd:scroll(PWSCROLL_VERTICAL, PWMSG_SCROLLUP, 1)
         END WHILE // RECNO() != nRecNo

      END IF // RECNO() != nRecNo

      // Turn on display updating.
      DISPEND()

   ELSE

      // A matching record wasn't found, so return to the previous record.
      GO oBrowseWnd:cargo[BROWSE_RECNO]

   END IF // FOUND()

   RETURN(NIL)
   // END SearchRecord( nKey, oBrowseWnd )



/*****************************************************************************
** DeleteRecallRecord( lDelete, oBrowseWnd ) --> NIL
**
** This either deletes or recalls the current record, updates the display,
** and 
**
*****************************************************************************/
FUNCTION DeleteRecallRecord( lDelete, oBrowseWnd )

   LOCAL oOutput


   // Delete or recall the record and update the states of the Record
   // menuitems.
   IF lDelete
      DELETE
   ELSE
      RECALL
   END IF // lDelete

   UpdateMenuitems(oBrowseWnd)

   // Because of the :refreshCurrent() call below, make sure Clipper's
   // output is directed to the browse window.
   oOutput  := pw():setOutput(oBrowseWnd)

   // The old row may need refreshing if it will still be visible.
   oBrowseWnd:getTBROWSE():refreshCurrent()

   // This is necessary to refresh the scrollbar of the browse window.
   // It will also stabilize the TBROWSE.
   oBrowseWnd:scroll(PWSCROLL_VERTICAL, PWMSG_SCROLLREFRESH, NIL)

   // Restore the previous output window.
   pw():setOutput(oOutput)

   RETURN(NIL)
   // END DeleteRecallRecord( lDelete, oBrowseWnd )



/*****************************************************************************
** AddEditList( nRecNo ) --> lSuccess
**
** This adds the specified record number to the list of records currently
** being edited.  If the record is already being edited by another
** procedure, .F. is returned.
**
*****************************************************************************/
FUNCTION AddEditList( nRecNo )

   LOCAL nPos  := ASCAN(aEditList, nRecNo)


   IF nPos == 0
      AADD(aEditList, nRecNo)
   END IF // nPos == 0

   RETURN(nPos == 0)
   // END AddEditList( nRecNo )



/*****************************************************************************
** DelEditList( nRecNo ) --> lSuccess
**
** This removes the specified record number from the list of records
** currently being edited.  If the record is not being edited by any
** procedure, .F. is returned.
**
*****************************************************************************/
FUNCTION DelEditList( nRecNo )

   LOCAL nPos  := ASCAN(aEditList, nRecNo)


   IF nPos != 0
      ADEL(aEditList, nPos)
      ASIZE(aEditList, LEN(aEditList) - 1)
   END IF // nPos != 0

   RETURN(nPos != 0)
   // END DelEditList( nRecNo )

