/*****************************************************************************
** MULTI.PRG
**
** ProVision:Windows v1.20 related browse windows sample
**
** by J. David Reynolds
**
** Copyright 1991, 1992 SofDesign International, Inc.
** All rights reserved
**
** tab spacing = 3
**
*****************************************************************************/

#include "inkey.ch"
#include "pw.ch"
#include "tutor.ch"

FIELD    CustNmbr


STATIC   idIdle                        // id of idle procedure

STATIC   lMinimized        := .F., ;   // keeps track of the window's state
         lRefreshChildren  := .F.      // .T. if child browse windows need refreshing

STATIC   nBrowseRecNo                  // keeps track of proper record number

STATIC   oBrowseWnd, ;                 // the window that contains the TBROWSE
         oChild1TB, ;                  // the first child TBROWSE object
         oChild1Wnd, ;                 // first child browse window
         oChild2TB, ;                  // the second child TBROWSE object
         oChild2Wnd, ;                 // second child browse window
         oTB                           // the TBROWSE object itself



/*****************************************************************************
** MultiWindow( oItemMulti ) --> oBrowseWnd
**
** This function creates a browse window and returns a reference to it.
**
*****************************************************************************/
FUNCTION MultiWindow( oItemMulti )

   LOCAL bColor

   LOCAL i

   LOCAL oMenu


   // Turn the mouse cursor into an hourglass (or display a wait message).
   WaitMsg(.T.)

   // Create the window if it doesn't exist.
   IF oBrowseWnd == NIL

      // Create your own TBROWSE--it doesn't really matter what coordinates
      // you specify, and the data source doesn't have to be a DBF.  Also,
      // if you use Class(y) to inherit from TBROWSE, an instance of your
      // new class can be attached to a window.
      oTB   := TBROWSENEW()
      oTB:goBottomBlock := { || Customer->(DBGOBOTTOM()) }
      oTB:goTopBlock    := { || Customer->(DBGOTOP()) }
      oTB:skipBlock     := { | nToSkip | ;
         Customer->(BrowseSkip(nToSkip)) }
      oTB:colorSpec     := IIF(ISCOLOR(), "+W/B, N/W, W/B", "+W/N, N/W, W/N")
      oTB:headSep       := ""
      oTB:colSep        := "  "
      oTB:addColumn(TBCOLUMNNEW("Cust #",    { || Customer->CustNmbr }))
      oTB:addColumn(TBCOLUMNNEW("Company",   { || Customer->Company }))
      oTB:addColumn(TBCOLUMNNEW("City",      { || Customer->City }))
      oTB:addColumn(TBCOLUMNNEW("State",     { || Customer->State }))
      oTB:addColumn(TBCOLUMNNEW("Contact",   { || Customer->Contact }))
      bColor   := { || Customer->(IIF(DELETED(), { 3, 2 }, { 1, 2 })) }
      FOR i := 1 TO 5
         oTB:getColumn(i):colorBlock   := bColor
      NEXT i 


      //
      // If you have a means of obtaining the relative position of the current
      // physical data row in your TBROWSE, this method shows how to make the
      // vertical scrollbar operate smoothly.  In this example, NTXPos() is
      // assumed to return the relative position of the current physical data
      // row, and NTXRec() is assumed to return the physical data row at the
      // current relative position.
      //
      // oBrowseWnd:setBrowseBlocks( ;
      //    { | oTB, nRowCount, nRowNumber | ;
      //       nRowCount   := Customer->(LASTREC()), ;
      //       nRowNumber  := Customer->(NTXPos(1, RECNO())) }, ;
      //    { | oTB, nRowNumber | ;
      //       Customer->(DBGOTO(NTXRec(1, nRowNumber))) } )
      //


      // Create a window--it should probably have scrollbars.
      CREATE TBROWSE WINDOW oBrowseWnd ;
         USING oTB ;
         AT 00, 00 ;
         SIZE 12, 80 ;
         VIRTUAL SIZE pwMaxRow(), pwMaxCol() ;
         TITLE "Customers" ;
         STYLE PWSTYLE_PRIMARY + PWSTYLEINFO_KEEPVSCROLLBAR + ;
            PWSTYLEINFO_KEEPHSCROLLBAR ;
         WINDOW HANDLER { | idMsg, _1 | ;
            BrowseWndHandler(idMsg, _1, oItemMulti) }

      // Create a menu for this window.
      MENU oMenu

         MENUITEM PROMPT "~Quit!" ;
               ACTION { || oBrowseWnd:close() }

      END MENU

      ATTACH MENU oMenu TO oBrowseWnd

      // Open the windows.
      OPEN WINDOW oBrowseWnd
      OPEN WINDOW oChild1Wnd
      OPEN WINDOW oChild2Wnd
      pw():select(oBrowseWnd)


   ELSE

      // Open the windows.
      OPEN WINDOW oBrowseWnd
      OPEN WINDOW oChild1Wnd
      OPEN WINDOW oChild2Wnd
      pw():select(oBrowseWnd)

      // Stabilize the TBROWSE.
      Customer->(RefreshTBROWSE(oBrowseWnd))

      // Stabilize the child browse windows.
      Invoices->(RefreshTBROWSE(oChild1Wnd))
      Payments->(RefreshTBROWSE(oChild2Wnd))

   END IF // oBrowseWnd == NIL

   // Restore the mouse cursor (or remove the wait message).
   WaitMsg(.F.)

   // Return a reference to the window.
   RETURN(oBrowseWnd)
   // END MultiWindow( oItemMulti )



/*****************************************************************************
** STATIC BrowseWndHandler( idMsg, _1, oItemMulti ) --> lHandled
**
** This implements the window handler and is responsible for saving and
** restoring the SELECT() area and the RECNO().  It also opens the files
** when the window is opened, and closes the file if the window is closed.
**
*****************************************************************************/
STATIC FUNCTION BrowseWndHandler( idMsg, _1, oItemMulti )

   LOCAL lHandled := .T.


   DO CASE
   CASE idMsg == PWMSG_CLOSE
      oItemMulti:enable()

      // Remove the idle procedure.
      pw():delIdleProc(idIdle)

      // Close the child browse windows.
      oChild1Wnd:close()
      oChild2Wnd:close()

   CASE idMsg == PWMSG_KILLFOCUS
      // Save the current record number.
      nBrowseRecNo   := Customer->(RECNO())

   CASE idMsg == PWMSG_MINIMIZE
      // Keep track of the browse window's state.
      lMinimized  := .T.

      // Close the child browse windows.
      oChild2Wnd:close()
      oChild1Wnd:close()

   CASE idMsg == PWMSG_OPEN
      oItemMulti:disable()

      // Create the child windows.
      oChild1Wnd  := Child1Window()
      oChild2Wnd  := Child2Window()

      // Install an idle procedure to refresh the child browse windows when
      // necessary.
      idIdle   := pw():addIdleProc({ || RefreshChildren() })

   CASE idMsg == PWMSG_RESTORE
      IF lMinimized

         // Keep track of the browse window's state.
         lMinimized  := .F.

         // Reopen the child browse windows.
         oChild2Wnd:open()
         oChild1Wnd:open()

         // Select the main browse window.
         pw():select(oBrowseWnd)

      END IF // lMinimized

   CASE (idMsg == PWMSG_SCROLL) .AND. (_1 == PWSCROLL_VERTICAL)
      // If the browse is scrolled, set the refresh flag.
      lRefreshChildren  := .T.

   CASE idMsg == PWMSG_SETFOCUS
      // Restore the proper workarea and record number.
      SELECT Customer
      IF nBrowseRecNo != NIL
         GO nBrowseRecNo
      END IF // nBrowseRecNo != NIL

   END CASE

   RETURN(lHandled)
   // END BrowseWndHandler( idMsg, oItemMulti )



/*****************************************************************************
** STATIC BrowseSkip( nToSkip ) --> nSkipped
**
** This function implements the skipBlock for the TBROWSE.  It also updates
** the states of the Record menuitems when the data pointer moves.
**
*****************************************************************************/
STATIC FUNCTION BrowseSkip( nToSkip )

   LOCAL nSkipped := 0


   DO CASE
   CASE nToSkip > 0
      // moving forward
      WHILE (nSkipped < nToSkip) .AND. (.NOT. EOF())
         SKIP +1
         ++nSkipped
      END WHILE // (nSkipped < nToSkip) .AND. (.NOT. EOF())
      IF EOF()
         SKIP -1
         --nSkipped
      END IF // EOF()

   CASE nToSkip < 0
      // moving backward
      WHILE (nSkipped > nToSkip) .AND. (.NOT. BOF())
         SKIP -1
         --nSkipped
      END WHILE // (nSkipped > nToSkip) .AND. (.NOT. BOF())
      IF BOF()
         ++nSkipped
      END IF // BOF()

   CASE nToSkip == 0
      // no movement
      SKIP 0

   END CASE

   // If the data pointer moved, update the window's current record number,
   // update the states of the Record menuitems, and set the refresh flag.
   IF nSkipped != 0
      nBrowseRecNo      := RECNO()
      lRefreshChildren  := .T.
   END IF // nSkipped != 0

   RETURN(nSkipped)
   // END BrowseSkip( nToSkip )



/*****************************************************************************
** STATIC Child1Window() --> oChild1Wnd
**
** This function creates a child browse window and returns a reference to it.
**
*****************************************************************************/
STATIC FUNCTION Child1Window()

   LOCAL bColor

   LOCAL i


   // Only do this once.
   IF oChild1Wnd == NIL

      // Turn the mouse cursor into an hourglass (or display a wait message).
      WaitMsg(.T.)

      // Create a TBROWSE.
      oChild1TB               := TBROWSENEW()
      oChild1TB:goTopBlock    := { || Invoices->(DBSEEK(Customer->CustNmbr)) }
      oChild1TB:goBottomBlock := { || Invoices->(DBSEEK(Customer->CustNmbr + ;
         1)), Invoices->(DBSKIP(-1)) }
      oChild1TB:skipBlock     := { | n | Invoices->(ChildSkipper(n)) }
      oChild1TB:addColumn(TBCOLUMNNEW("Cust #",    { || Invoices->CustNmbr }))
      oChild1TB:addColumn(TBCOLUMNNEW("Invoice #", { || Invoices->InvNmbr }))
      oChild1TB:addColumn(TBCOLUMNNEW("Amount",    { || Invoices->Amount }))
      oChild1TB:colorSpec  := "+W/B, N/W, W/B"
      oChild1TB:headSep    := ""
      oChild1TB:colSep     := "  "
      bColor   := { || Customer->(IIF(DELETED(), { 3, 2 }, { 1, 2 })) }
      FOR i := 1 TO 3
         oChild1TB:getColumn(i):colorBlock   := bColor
      NEXT i 

      // Create a window--it should probably have scrollbars.
      CREATE TBROWSE WINDOW oChild1Wnd ;
         USING oChild1TB ;
         AT 12, 00 ;
         SIZE 09, 40 ;
         VIRTUAL SIZE pwMaxRow(), pwMaxCol() ;
         TITLE "Invoices" ;
         STYLE PWSTYLE_SECONDARY - PWSTYLEINFO_MENU + ;
            PWSTYLEINFO_KEEPVSCROLLBAR + PWSTYLEINFO_KEEPHSCROLLBAR

      // Restore the mouse cursor (or remove the wait message).
      WaitMsg(.F.)

   END IF // oChild1Wnd == NIL

   // Return a reference to the window.  It will be needed by whoever
   // called Child1Window() to open the window.
   RETURN(oChild1Wnd)
   // END Child1Window()



/*****************************************************************************
** STATIC Child2Window() --> oChild2Wnd
**
** This function creates a child browse window and returns a reference to it.
**
*****************************************************************************/
STATIC FUNCTION Child2Window()

   LOCAL bColor

   LOCAL i


   // Only do this once.
   IF oChild2Wnd == NIL

      // Turn the mouse cursor into an hourglass (or display a wait message).
      WaitMsg(.T.)

      // Create a TBROWSE.
      oChild2TB               := TBROWSENEW()
      oChild2TB:goTopBlock    := { || Payments->(DBSEEK(Customer->CustNmbr)) }
      oChild2TB:goBottomBlock := { || Payments->(DBSEEK(Customer->CustNmbr + ;
         1)), Payments->(DBSKIP(-1)) }
      oChild2TB:skipBlock     := { | n | Payments->(ChildSkipper(n)) }
      oChild2TB:addColumn(TBCOLUMNNEW("Cust #",    { || Payments->CustNmbr }))
      oChild2TB:addColumn(TBCOLUMNNEW("Payment",   { || Payments->Payment }))
      oChild2TB:addColumn(TBCOLUMNNEW("Amount",    { || Payments->Amount }))
      oChild2TB:colorSpec  := "+W/B, N/W, W/B"
      oChild2TB:headSep    := ""
      oChild2TB:colSep     := "  "
      bColor   := { || Customer->(IIF(DELETED(), { 3, 2 }, { 1, 2 })) }
      FOR i := 1 TO 3
         oChild2TB:getColumn(i):colorBlock   := bColor
      NEXT i 

      // Create a window--it should probably have scrollbars.
      CREATE TBROWSE WINDOW oChild2Wnd ;
         USING oChild2TB ;
         AT 12, 40 ;
         SIZE 09, 40 ;
         VIRTUAL SIZE pwMaxRow(), pwMaxCol() ;
         TITLE "Payments" ;
         STYLE PWSTYLE_SECONDARY - PWSTYLEINFO_MENU + ;
            PWSTYLEINFO_KEEPVSCROLLBAR + PWSTYLEINFO_KEEPHSCROLLBAR

      // Restore the mouse cursor (or remove the wait message).
      WaitMsg(.F.)

   END IF // oChild2Wnd == NIL

   // Return a reference to the window.  It will be needed by whoever
   // called Child2Window() to open the window.
   RETURN(oChild2Wnd)
   // END Child2Window()



/*****************************************************************************
** STATIC ChildSkipper( n ) --> nSkipped
**
** This function implements the skip block for the child browse windows.
**
*****************************************************************************/
STATIC FUNCTION ChildSkipper( n )

   LOCAL i  := 0


   DO CASE
   CASE n > 0
      // moving forward
      WHILE (i < n) .AND. (CustNmbr == Customer->CustNmbr)
         SKIP +1
         ++i
      END WHILE // (i < n) .AND. (CustNmbr == Customer->CustNmbr)
      IF CustNmbr != Customer->CustNmbr
         SKIP -1
         --i
      END IF // CustNmbr != Customer->CustNmbr

   CASE n < 0
      // moving backward
      WHILE (i > n) .AND. (CustNmbr == Customer->CustNmbr)
         SKIP -1
         IF BOF()
            EXIT
         END IF // BOF()
         --i
      END WHILE // (i > n) .AND. (CustNmbr == Customer->CustNmbr)
      IF CustNmbr != Customer->CustNmbr
         SKIP +1
         ++i
      END IF // CustNmbr != Customer->CustNmbr

   CASE n == 0
      // no movement
      SKIP 0

   END CASE

   RETURN(i)
   // END ChildSkipper( n )



/*****************************************************************************
** STATIC RefreshChildren() --> NIL
**
** This function refreshes the child browse windows.
**
*****************************************************************************/
STATIC FUNCTION RefreshChildren()


   IF lRefreshChildren

      // Stabilize the child browse windows and reset the refresh flag.
      Invoices->(RefreshTBROWSE(oChild1Wnd))
      Payments->(RefreshTBROWSE(oChild2Wnd))
      lRefreshChildren  := .F.

   END IF // lRefreshChildren

   RETURN(NIL)
   // END RefreshChildren()

