// ****************************************************************************
// *  PWBLOCK.PRG - sample game written in Clipper                            *
// *  Written in September 1991 By Skip Key CIS 73130,2102                    *
// *  Compiled with Clipper 5.01                                              *
// ****************************************************************************

// ****************************************************************************
// *  This game was written a few months ago when a friend of mine            *
// *  was putting down Clipper with a comment like, "Yeah, you can write      *
// *  database programs with it, but can you write games?"  So, just to       *
// *  show him I sat down and wrote this simple little game in about two      *
// *  hours.  It uses quite a few of Clipper 5 features, and in fact would    *
// *  have taken much longer to write in Clipper Summer 87.  I hope that      *
// *  you have as much fun playing it as I did writing it.                    *                                   *
// *                                                                          *
// *  Skip Key 73130,2102                                                     *
// ****************************************************************************

// ****************************************************************************
// * August 1, 1992.  Rewrote event loop and display to work with PVW         *
// * To get it working it only took changes to two functions, the main event  *
// * loop, and the function to draw the pieces.  In addition I rewrote the    *
// * rest of the display stuff to use PVW's low-level writing functions,      *
// * although this really wasn't necessary.
// ****************************************************************************

#include "pw.ch"

#include "inkey.ch"
#include "error.ch"

#define NUMSHAPES 7
#define RND_DIVISOR                    10000000
#define RND_FACTOR                     3236713

#define PWCHAR_LSCROLLUP               12    // 9-pixel
#define PWCHAR_RSCROLLUP               13    // 8-pixel
#define PWCHAR_LSCROLLDOWN        15    // 9-pixel
#define PWCHAR_RSCROLLDOWN        16    // 8-pixel
#define PWCHAR_LSCROLLRIGHT            20    // 9-pixel
#define PWCHAR_RSCROLLRIGHT            21    // 8-pixel
#define PWCHAR_LSCROLLLEFT        24    // 9-pixel
#define PWCHAR_RSCROLLLEFT        25    // 8-pixel

//#define TEST

STATIC aShapes             // shape table
STATIC idCurShape          // current shape
STATIC nCurRow             // current row position  on board zero based
STATIC nCurCol             // current column position on board zero based
STATIC idOrient            // orientation
STATIC aBoard              // board contents
STATIC lGameOver := .F.    // is game over?
STATIC lNotPaused := .T.   // are we paused?
STATIC nScore:=0           // adjust upwards for cheaters
STATIC nLevel:=1           // level
STATIC nRows:=0            // rows removed
STATIC nTimeConst          // value used to time operations

STATIC oGameWnd            // game window
STATIC idMoveProc          // handle to background procedure that moves pieces

#ifdef TEST
#define WaitMsg(<lVal>) => IIF(<lVal>,pw():mouseCursor(PMCURSOR_HOUR),pw():mouseCursor(PMCURSOR_ARROW))

FUNCTION Main()
LOCAL oWnd, oMenu, oBlocks
   INIT WINDOWS

   WaitMsg(.T.)

   CREATE WINDOW oWnd CENTERED SIZE 20, 50 STYLE PWSTYLE_PRIMARY ;
      TITLE "Test Window"

   OPEN WINDOW oWnd

   MENU oMenu
      POPUP PROMPT "~Games"
         MENUITEM oBlocks PROMPT "~Blocks" ACTION {||pwBlocks(oBlocks)}
      ENDPOPUP
   ENDMENU

   ATTACH MENU oMenu TO oWnd

   WaitMsg(.F.)

   ACTIVATE WINDOWS
   DEINIT WINDOWS

RETURN NIL
#endif TEST


FUNCTION pwBlocks(oItemBlocks)
LOCAL oGameMenu, oPaused

   WaitMsg(.T.)

   IF oItemBlocks != NIL
      oItemBlocks:disable()   // disables menuitem or control if passed
   ENDIF

   GetTime()            // sets static nTimeConst

//   CREATE WINDOW oGameWnd AT 1, 1 ;
//      SIZE 22, 47 STYLE PWSTYLE_DIALOG+PWSTYLEINFO_MENU ;
//      TITLE "BLOCKS" ;
//      COLOR IIF(ISCOLOR() .AND. pw():isTMR(PWTMR_BORDERS), ;
//      "N/W, W/*N, +W/N, W/*N, N/W, N/W", ) ;
//      EVENT HANDLER {|idMsg, nKey| IIF(idMsg==PWMSG_KEYPRESS, ProcessKey(nKey),), .T.} ;
//      WINDOW HANDLER {|idMsg| GameWndHandler(idMsg, oItemBlocks)}
   CREATE WINDOW oGameWnd AT 1, 1 ;
      SIZE 22, 47 STYLE PWSTYLE_DIALOG+PWSTYLEINFO_MENU ;
      TITLE "Blocks" ;
      COLOR IIF(ISCOLOR() .AND. pw():isTMR(PWTMR_BORDERS), ;
         "N/W, W/*N, +W/N, W/*N, W/*N", ) ;
      EVENT HANDLER {|idMsg, nKey| IIF(idMsg==PWMSG_KEYPRESS, ProcessKey(nKey),), .T.} ;
      WINDOW HANDLER {|idMsg| GameWndHandler(idMsg, oItemBlocks)}

   MENU oGameMenu ;
      COLOR IIF(ISCOLOR() .AND. pw():isTMR(PWTMR_BORDERS), ;
         "W/*N, +W/N, +W/*N, +N/W, +N/N", )

      POPUP PROMPT "~Game"

         MENUITEM PROMPT " ~New Game" ACTION {||StartNewGame() }

         MENUITEM oPaused PROMPT "~Paused" UNCHECKED  ;
            ACTION {|| lNotPaused := !oPaused:getValue() }

      END POPUP

      MENUITEM PROMPT "~Help!" ACTION {||AboutWnd()}

   END MENU

   ATTACH MENU oGameMenu TO oGameWnd

   pwFancyBox(0,   1, 18, 22, "W, W, +N, +N, +W, +W")
   pwFancyBox(4,  31,  6, 42, "W, W, +N, +N, +W, +W")
   pwFancyBox(7,  31,  9, 42, "W, W, +N, +N, +W, +W")
   pwFancyBox(10, 31, 12, 42, "W, W, +N, +N, +W, +W")

   pwWriteStr(PWRW_TEXT,  5, 25, "SCORE")
   pwWriteStr(PWRW_TEXT,  8, 25, "ROWS")
   pwWriteStr(PWRW_TEXT, 11, 25, "LEVEL")

   DisplayStats()

   OPEN WINDOW oGameWnd
   SETCOLOR("N/W")

   WaitMsg(.F.)

RETURN NIL

// ****************************************************************************
// *  FUNCTION MovePiece()                                                    *
// *  PARAMETERS  NONE                                                        *
// *  RETURNS     NIL                                                         *
// *  This function is set up as a background procedure.  Its purpose in life *
// *  is to see if enough time has elapsed, and if so and the game is not     *
// *  paused or over, move the current shape down one row.                    *
// ****************************************************************************

STATIC FUNCTION MovePiece()
STATIC nLastMove := 0
   IF lNotPaused .AND. !lGameOver .AND. (SECONDS()-nLastMove)>nTimeConst
      MoveShape()
      nLastMove := seconds()
   ENDIF
RETURN NIL

// ****************************************************************************
// *  FUNCTION GameWndHandler()                                               *
// *  PARAMETERS  idMsg - Message received                                    *
// *              oItemBlocks - menuitem or control to re-enable              *
// *  RETURNS     .T.                                                         *
// *  This function is the window handler for the game window.  It contains   *
// *  an example of how to handle the recursion when you want to destroy a    *
// *  window when you close it.                                               *
// ****************************************************************************

STATIC FUNCTION GameWndHandler(idMsg, oItemBlocks)
STATIC  lDestroyed         // for recursion
   DO CASE
      CASE idMsg == PWMSG_CLOSE
         IF lDestroyed
            pw():delBackProc(idMoveProc)   // remove the background procedure
            idMoveProc := NIL
            IF oItemBlocks != NIL
               oItemBlocks:enable()   // enables menuitem or control if passed
            ENDIF
         ELSE
            lDestroyed := .T.
            oGameWnd:destroy()       // destroy the window
         ENDIF
      CASE idMsg == PWMSG_CREATE
         lDestroyed := .F.
      CASE idMsg == PWMSG_SETFOCUS
         IF idMoveProc != NIL
            idMoveProc := pw():addBackProc({||MovePiece()})
         ENDIF
      CASE idMsg == PWMSG_KILLFOCUS
         pw():delBackProc(idMoveProc)   // remove the background procedure
      CASE idMsg==PWMSG_CREATE
         lGameOver := .F.
   ENDCASE
RETURN .T.

// ****************************************************************************
// *  FUNCTION StartNewGame()                                                 *
// *  PARAMETERS  NONE                                                        *
// *  RETURNS     NIL                                                         *
// *  This function initializes everything to start a new game.               *
// ****************************************************************************

STATIC FUNCTION StartNewGame()

   InitTables()
   lGameOver := .F.
   nScore := 0
   nRows := 0
   nLevel := 1
   idCurShape := NIL

   IF idMoveProc == NIL
      idMoveProc := pw():addBackProc({||MovePiece()})
   ELSE
      SCROLL(1, 2, 17, 21)      // Only clear the field if not new
      DisplayStats()
   ENDIF

RETURN NIL

// ****************************************************************************
// *  FUNCTION    InitTables()                                                *
// *  PARAMETERS  none                                                        *
// *  RETURNS     NIL                                                         *
// *  PURPOSE     This function draws the board and sets up the shape tables  *
// *  NOTES       The shapetable is a nested array that contains the positions*
// *              of the blocks in their four rotations.  The innermost array *
// *              is an array of eight numbers.  These eight numbers are      *
// *              really four coordinates that are relative to the center of  *
// *              the piece.  So, for example, the very first one             *
// *              {0,-1, 0,0, 0,1, 0,2}   translates to:                      *
// *                                                                          *
// *                                  X   (0, -1)                             *
// *                                  X   (0, 0)  CENTER                      *
// *                                  X   (0, 1)                              *
// *                                  X   (0, 2)                              *
// *                                                                          *
// *              those arrays are grouped by fours into the next higher      *
// *              array.  Each of the four is for the four different          *
// *              orientations.  The outermost array is an array of each of   *
// *              the different shapes.  So aShapes[1] would be the first     *
// *              shape, aShapes[1][1] would be the first orientation of      *
// *              the first piece, and aShapes[1][1][1] would be the          *
// *              relative row position of the first block in the first       *
// *              orientation of the first piece.                             *
// *                                                                          *
// *              The board is a nested array of characters that will contain *
// *              an 'X' if there is a block there and a space if not.        *
// ****************************************************************************

STATIC FUNCTION InitTables()
LOCAL m, n

   aBoard := {}
   FOR n := 1 TO 16
      AADD(aBoard, {})
      FOR m := 1 TO 10
          AADD(aBoard[n], ' ')
      NEXT
   NEXT

   aShapes := {{{0,-1, 0,0, 0,1, 0,2},;
                  {-2,0, -1,0, 0,0, 1,0},;
                  { 0,-1, 0,0, 0,1, 0,2},;
                  {-2,0, -1,0, 0,0, 1,0}},;
                 {{-1,0, -1,1, 0,0, 0,1},;
                  {-1,0, -1,1, 0,0, 0,1},;
                  {-1,0, -1,1, 0,0, 0,1},;
                  {-1,0, -1,1, 0,0, 0,1}},;
                 {{0,-1, 0,0, 0,1, 1,0},;
                  {-1,0, 0,0, 0,1, 1,0},;
                  {-1,0, 0,-1, 0,0, 0,1},;
                  {-1,0, 0,-1, 0,0, 1,0}},;
                 {{-1,-1, -1,0, 0,0, 0,1},;
                  {-1,1, 0,0, 0,1, 1,0},;
                  {-1,-1, -1,0, 0,0, 0,1},;
                  {-1,1, 0,0, 0,1, 1,0}},;
                 {{0,0, 0,1, 1,-1, 1,0},;
                  {-1,0, 0,0, 0,1, 1,1},;
                  {0,0, 0,1, 1,-1, 1,0},;
                  {-1,0, 0,0, 0,1, 1,1}},;
                 {{-1,-1, 0,-1, 0,0, 0,1},;
                  {-1,0, 0,0, 1,-1, 1,0},;
                  {0,-1, 0,0, 0,1, 1,1},;
                  {-1,0, -1,1, 0,0, 1,0}},;
                 {{-1,1, 0,-1, 0,0, 0,1},;
                  {-1,-1, -1,0, 0,0, 1,0},;
                  {0,-1, 0,0, 0,1, 1,-1},;
                  {-1,0, 0,0, 1,0, 1,1}}}
RETURN NIL

// ****************************************************************************
// *  FUNCTION    GetTime()                                                   *
// *  PARAMETERS  none                                                        *
// *  RETURNS     NIL                                                         *
// *  This function sets nTimeConst, a static variable.  It can be changed    *
// *  to change the speed of the game                                         *
// ****************************************************************************

STATIC FUNCTION GetTime()
   nTimeConst := 0.80         // adjust downward to speed up
RETURN NIL

// ****************************************************************************
// *  FUNCTION    MoveShape()                                                 *
// *  PARAMETERS  none                                                        *
// *  RETURNS     NIL                                                         *
// *  This function determines whether or not a shape can descend a row, and  *
// *  if so, causes it to do so.  It also initiates new pieces.               *
// ****************************************************************************

STATIC FUNCTION MoveShape()

   IF idCurShape == NIL
      idCurShape := NewNumber(1, NUMSHAPES)
      nCurRow   := 1
      nCurCol   := 5
      idOrient := 1
   ENDIF

   DISPBEGIN()
   DrawShape(.F.)

   IF NotBlocked()
      nCurRow++
      DrawShape(.T.)
   ELSE
      FallRow()
      DrawShape(.T.)
      RemoveLines()
   ENDIF

   IF (idCurShape == NIL) .AND. (!lGameOver)
      idCurShape := NewNumber(1, NUMSHAPES)
      nCurRow   := 1
      nCurCol   := 5
      idOrient := 1
      DrawShape(.T.)
   ENDIF

   DISPEND()

RETURN NIL

// ****************************************************************************
// *  FUNCTION    ProcessKey(nKey)                                            *
// *  PARAMETERS  nKey - value of key pressed                                 *
// *  RETURNS     NIL                                                         *
// *              This function processes the keys that the game accepts.     *
// *              note that an escape will exit the game at any time          *
// ****************************************************************************

STATIC FUNCTION ProcessKey(nKey)
   IF idCurShape != NIL
      DO CASE

         CASE nKey == 52   // numeric 4
            IF NotBlocked(nKey)
               DrawShape(.F.)
               nCurCol--
               DrawShape(.T.)
            END

         CASE nKey == 54   // numeric 6
            IF NotBlocked(nKey)
               DrawShape(.F.)
               nCurCol++
               DrawShape(.T.)
            END

         CASE nKey == 53   // numeric 5
            IF NotBlocked(nKey)
               DrawShape(.F.)
               idOrient := IIF(++idOrient>4, 1, idOrient)
               DrawShape(.T.)
            END

         CASE nKey == 50   // numeric 2
            DrawShape(.F.)
            FallRow()
            DrawShape(.T.)
            RemoveLines()
      END
   ENDIF
RETURN NIL

// ****************************************************************************
// *  FUNCTION    DrawShape(lVisible)                                         *
// *  PARAMETERS  lVisible - logical indicating whether to draw or erase      *
// *  RETURNS     NIL                                                         *
// *              This function either draws the shape at the current         *
// *              row and column or erases it depending on visible            *
// ****************************************************************************

STATIC FUNCTION DrawShape(lVisible)
STATIC aColors := {"N", "B", "G", "BG", "R", "RB", "RG", "W"}
LOCAL row := nCurRow+1
LOCAL col := 2*(nCurCol)
LOCAL cFillString
LOCAL n

// This section of code requires a little bit of explanation.  In order to
// draw the blocks using the TMR characters, I am calling the PVW internal
// function pwWndChar().  pwWndChar() takes a numeric constant and returns the
// current character used to draw that character.

   IF lVisible
      IF pw():isTMR(PWTMR_CONTROLS)
         DO CASE
            CASE idOrient == 1
               cFillString := pwWndChar(PWCHAR_LSCROLLUP) + ;
                              pwWndChar(PWCHAR_RSCROLLUP)
            CASE idOrient == 2
                cFillString := pwWndChar(PWCHAR_LSCROLLLEFT) + ;
                              pwWndChar(PWCHAR_RSCROLLLEFT)
            CASE idOrient == 3
               cFillString := pwWndChar(PWCHAR_LSCROLLDOWN) + ;
                              pwWndChar(PWCHAR_RSCROLLDOWN)
            CASE idOrient == 4
               cFillString := pwWndChar(PWCHAR_LSCROLLRIGHT) + ;
                              pwWndChar(PWCHAR_RSCROLLRIGHT)
         ENDCASE

      ELSE
         cFillString := "[]"
      ENDIF

   ELSE
      cFillString := '  '
   ENDIF

   FOR n := 1 TO 8
      pwWriteStr(PWRW_TEXTATTR, aShapes[idCurShape][idOrient][n]+row, ;
         2*aShapes[idCurShape][idOrient][++n]+col, cFillString, ;
         IIF(lVisible, "N/"+aColors[idCurShape+1], "N/W+") )
   NEXT

RETURN NIL

// ***************************************************************************
// *   FUNCTION   NotBlocked(direction)                                      *
// *  PARAMETERS direction - inkey value of arrow pressed, or down if NIL    *
// *  RETURNS    whether or not the shape can move in that direction         *
// *             This function tests whether or not the shape would overlap  *
// *             any current blocks if it were moved in this direction.      *
// *             The only interesting thing about this function is that      *
// *             instead of testing for array out of bounds, I use a begin   *
// *             sequence ... end sequence construct and install my own      *
// *             error handler and assume that any array out of bounds       *
// *             errors in this section are caused by attempting to move the *
// *             piece off of the screen.                                    *
// ***************************************************************************

STATIC FUNCTION NotBlocked(direction)
LOCAL lRetVal := .T.
LOCAL nTestRow := nCurRow
LOCAL nTestCol := nCurCol
LOCAL idTestOrient := idOrient
LOCAL n
LOCAL nTest2Row
LOCAL nTest2Col
LOCAL bOldErr
LOCAL ObjErr

   // install my error handler
   bOldErr := ERRORBLOCK({|errObj| ArrErr(errObj)})

   DO CASE
      CASE direction == NIL
         nTestRow++
      CASE direction == 52
         nTestCol--
      CASE direction == 53
         idTestOrient := IIF(++idTestOrient>4, 1, idTestOrient)
      CASE direction == 54
         nTestCol++
   ENDCASE

   FOR n := 1 TO 8
      nTest2Row := nTestRow
      nTest2Col := nTestCol
      nTest2Row += aShapes[idCurShape][idTestOrient][n]
      nTest2Col += aShapes[idCurShape][idTestOrient][++n]

      // if any of the array accesses here are out of bounds my error
      // handler will be invoked which will break to the recover

      BEGIN SEQUENCE

         IF aBoard[nTest2Row][nTest2Col] == 'X'
            lRetVal := .F.
         ENDIF

      RECOVER USING objErr

         IF objErr:genCode == EG_BOUND   // if out of bounds return false
            lRetVal := .F.
         ELSE
            EVAL(bOldErr, objErr)     // else chain to other error handler
         ENDIF

      END SEQUENCE
   NEXT

   ERRORBLOCK(bOldErr)     // reinstall old handler

RETURN lRetVal

// ****************************************************************************
// *  FUNCTION    ArrErr()                                                    *
// *  PARAMETERS  oBjErr - error Oblect                                       *
// *  RETURNS     NIL                                                         *
// *  this function simply breaks so that the begin sequence recover end      *
// *  sequence construct can be used                                          *
// ****************************************************************************

STATIC FUNCTION ArrErr(objErr)
   BREAK objErr
RETURN NIL

// ****************************************************************************
// *  FUNCTION    FallRow()DrawShape()                                        *
// *  PARAMETERS  None                                                        *
// *  RETURNS     NIL                                                         *
// *  This function is used to move a piece as far down the board as it       *
// *  can possibly go.  A local error handler determines whether or not the   *
// * game is over.                                                            *
// ****************************************************************************

STATIC FUNCTION FallRow()
LOCAL n
LOCAL bOldErr
LOCAL objErr
LOCAL nScoreRow := 0

   WHILE NotBlocked()
      nCurRow++
      nScoreRow++
   ENDDO

   nScore += (nScoreRow*nLevel)
   DisplayStats()

   bOldErr := ERRORBLOCK({|errObj| ArrErr(errObj)})

   BEGIN SEQUENCE

      FOR n := 1 TO 8
         aBoard[aShapes[idCurShape][idOrient][n]+nCurRow][aShapes[idCurShape][idOrient][++n]+nCurCol]:='X'
      NEXT

   RECOVER USING objErr

      IF objErr:genCode == EG_BOUND
         lGameOver := .T.
      ENDIF

   END SEQUENCE

   ERRORBLOCK(bOldErr)

RETURN NIL

// ****************************************************************************
// *  FUNCTION    RemoveLines(visible)                                        *
// *  PARAMETERS  None                                                        *
// *  RETURNS     NIL                                                         *
// *  This function removes all of the full lines.                            *
// ****************************************************************************

STATIC FUNCTION RemoveLines()
LOCAL n
LOCAL m

   FOR n := 1 TO 16
      IF (ASCAN(aBoard[n], ' ') == 0)

         nScore+=25
         nRows++
         DisplayStats()
         SCROLL(1, 2, n+1, 21, -1)
         ADEL(aBoard, n)
         AINS(aBoard, 1)
         aBoard[1] := {}

         FOR m := 1 TO 10
            AADD(aBoard[1], ' ')
         NEXT

      ENDIF
   NEXT

   idCurShape := NIL
RETURN NIL

// ****************************************************************************
// *  FUNCTION    DisplayStats()                                              *
// *  PARAMETERS  None                                                        *
// *  RETURNS     NIL                                                         *
// *  This function updates the statistics.                                   *
// ****************************************************************************

STATIC FUNCTION DisplayStats()
STATIC nDeltaTime

   IF nDeltaTime == NIL
      nDeltaTime := nTimeConst/10
   ENDIF

   IF INT(nRows/10)>(nLevel-1)
      nLevel++
      IF nLevel<10
         nTimeConst-=nDeltaTime
      ENDIF
   ENDIF

   pwWriteStr(PWRW_TEXT,  5, 32, STR(nScore))
   pwWriteStr(PWRW_TEXT,  8, 32, STR(nRows))
   pwWriteStr(PWRW_TEXT, 11, 32, STR(nLevel))

RETURN NIL

// ****************************************************************************
// * STATIC FUNCTION AboutWnd()                                               *
// * PARAMETERS   None                                                        *
// * RETURNS      NIL                                                         *
// * This function uses a modal window to display some information about      *
// * ProVision:Windows.                                                       *
// ****************************************************************************

STATIC FUNCTION AboutWnd()
STATIC  oAboutWnd
LOCAL GetList := {}


   // If the window has not yet been created, do so.
   IF oAboutWnd == NIL

      // Create a window for the message.
      CREATE WINDOW oAboutWnd ;
         CENTERED ;
         SIZE 16, 48 ;
         STYLE PWSTYLE_DIALOG ;
         TITLE "About Blocks"

      // Display the about message.
      @ 01, 01 SAY "                 Wow, Cool!                 "
      @ 02, 01 SAY "                                            "
      @ 03, 01 SAY "   ProVision:Windows is flexible enough to  "
      @ 04, 01 SAY "   allow you to do anything you want!       "
      @ 05, 01 SAY "   And since development time will be cut   "
      @ 06, 01 SAY "   down enough to allow you lots of free    "
      @ 07, 01 SAY "   time, this game will keep you busy.      "
      @ 09, 01 SAY "   Note:  NumLock must be ON to play this   "
      @ 10, 01 SAY "          game!                             "

      // Create an 'OK' pushbutton.
      @ 11, 19 GET AS PUSHBUTTON ;
         PROMPT "  OK  " ;
      ACTION {|| oAboutWnd:close() }

      // Attach the controls to the window.
      ATTACH CONTROLS TO oAboutWnd


   END IF // oAboutWnd == NIL

   // Open the window modally.
   OPEN WINDOW oAboutWnd MODAL

   RETURN(NIL)
   // END AboutWindow()


#ifdef TEST
/*****************************************************************************
** NewNumber( nLRange, nURange ) --> nNumber
**
** This function is a variation on a pseudorandom number generator found
** in the NanForum Toolkit.
**
*****************************************************************************/
FUNCTION NewNumber( nLRange, nURange )

   STATIC   nSeed


   // init seed
   IIF(nSeed == NIL, nSeed := SECONDS(), )

   RETURN(INT((nURange - nLRange + 1) * ((nSeed := MOD(nSeed * ;
      RND_FACTOR + 1, RND_DIVISOR)) / RND_DIVISOR)) + nLRange)
   // END NewNumber( nLRange, nURange )
#endif
