/*****************************************************************************
** TUTOR.PRG
**
** ProVision:Windows v1.20 tutorial and demo
**
** by J. David Reynolds
**
** Copyright 1991, 1992 SofDesign International, Inc.
** All rights reserved
**
** tab spacing = 3
**
*****************************************************************************/

#include "inkey.ch"
#include "pw.ch"
#include "set.ch"


#define TOPIC_INITIALIZING             1
#define TOPIC_USING                    2
#define TOPIC_WRITING                  3
#define TOPIC_CONTROLS                 4
#define TOPIC_MANIPULATING             5
#define TOPIC_MESSAGES                 6

#define MAXMSGLEN                      9
#define MAXLINELEN                     60

#define CRLF                           CHR(13) + CHR(10)


STATIC   nFrame, ;                     // current frame number
         nTopic                        // current topic id

STATIC   oMainWnd, ;                   // the window for the menu
         oMsgWnd, ;                    // the message box window
         oPopTopics, ;                 // the Topics popup
         oTopicWnd                     // the window for the current topic



/*****************************************************************************
** TutorMain( cParam1, cParam2 ) --> NIL
**
** This is the startup module for the tutorial.  If /STD is passed on the
** command line, the program will be in standard mode instead of TMR mode.
** If /43 or /50 is passed on the command line, the video will be switched
** to 43 or 50 lines.
**
*****************************************************************************/
FUNCTION TutorMain( cParam1, cParam2 )

   LOCAL nCols, ;
         nRows := 0


   // Validate the command line parameters, if any.
   IIF(cParam1 == NIL, cParam1 := "", )
   IIF(cParam2 == NIL, cParam2 := "", )
   cParam1  := UPPER(cParam1 + cParam2)

   // Switch video modes if requested and supported.
   IF "/43" $ cParam1
      nRows := MAXROW() + 1
      nCols := MAXCOL() + 1
      SETMODE(43, 80)
   END IF // "/43" $ cParam1
   IF "/50" $ cParam1
      nRows := MAXROW() + 1
      nCols := MAXCOL() + 1
      SETMODE(50, 80)
   END IF // "/50" $ cParam1

   // Init ProVision:Windows.
   IF "/STD" $ cParam1
      INIT WINDOWS ;
         STANDARD MOUSE ;
         STANDARD BORDERS ;
         STANDARD CONTROLS ;
         SETBLINK .T.
   ELSE
      INIT WINDOWS ;
         GRAPHIC MOUSE ;
         GRAPHIC BORDERS ;
         GRAPHIC CONTROLS
   END IF // "/STD" $ cParam1

   IF pw():isTMR(PWTMR_BORDERS)
      // Load new wallpaper.
      IF (pwMaxRow() == 42) .OR. (pwMaxRow() == 49)
         pw():wallpaper(PWWP_GRID, "W/N")
      ELSE
         pw():wallpaper(PWWP_RINGS, IIF(ISCOLOR(), "W/*N", "W/N"))
      END IF // (pwMaxRow() == 42) .OR. (pwMaxRow() == 49)
   ELSE
      // Clear the screen.
      SETCOLOR("+W/N")
      CLEAR SCREEN
   END IF // pw():isTMR(PWTMR_BORDERS)

   // Turn the mouse cursor into an hourglass (or display a wait message).
   WaitMsg(.T.)

   // By default, you cannot use Alt+C or or Alt+D as accelerator keys or
   // shortcut keys since Clipper traps these.  To use these keys,
   // uncomment the two lines below.  Note that by doing so, you will not
   // be able to abort the application with Alt+C nor invoke the debugger
   // with Alt+D.
   SETCANCEL(.F.)
// ALTD(0)

   // Turn on insert mode for new text entry.
   SET(_SET_INSERT, .T.)

   // Turn on softseeking.
   SET(_SET_SOFTSEEK, .T.)

   // Install a timer to periodically display an "advertisment" for
   // ProVision:Windows.  Comment out this line when it gets too annoying.
   pw():addTimer(PWTIMER_PERIODIC, 300, { || Advertisment() })

   // Install a system filter for keypresses.  This will be used for F1
   // help and to allow Alt+X to exit.
   pw():addFilter({ | idMsg, nKey | SysKeyFilter(@idMsg, nKey) }, ;
      PWMSG_KEYPRESS)

   // Set a system timeout value of five minutes.  This means that after
   // five minutes of inactivity (no keyboard or mouse events have occurred)
   // each open window will receive a PWMSG_TIMEOUT message through its
   // custom window message handler, if any.
   pw():setTimeout(5 * 60)

   // Open the About window as an opening screen.
   AboutWindow()

   // Open the main window (behind the About window).
   OPEN WINDOW MainWnd() ;
      BACK

   // Restore the mouse cursor (or remove the wait message).
   WaitMsg(.F.)

   // Activate the ProVision:Windows event system.
   ACTIVATE WINDOWS

   // Turn the mouse cursor into an hourglass (or display a wait message).
   WaitMsg(.T.)

   // Close the files needed for the browse samples.
   CloseFiles()

   // Restore the mouse cursor (or remove the wait message).
   WaitMsg(.F.)

   // Deinit ProVision:Windows.
   DEINIT WINDOWS

   // Try to switch back to the original mode, if necessary.
   IF nRows != 0
      SETMODE(nRows, nCols)
   END IF // nRows != 0

   RETURN(NIL)
   // END TutorMain( cParam1, cParam2 )



/*****************************************************************************
** STATIC MainWnd() --> oMainWnd
**
** This function creates a window in which the ProVision logo and copyright
** message are displayed.  It also contains a menu from which the user can
** select the part of the tutorial they want to see.
**
*****************************************************************************/
STATIC FUNCTION MainWnd()

   LOCAL oItemBlocks, ;
         oItemCalc, ;
         oItemClock, ;
         oItemMDI, ;
         oItemMemory, ;
         oItemMulti, ;
         oItemText, ;
         oMenu


   // Turn the mouse cursor into an hourglass (or display a wait message).
   WaitMsg(.T.)

   // Create the main window.
   CREATE WINDOW oMainWnd ;
      AT 01, 02 ;
      SIZE 20, 54 ;
      STYLE PWSTYLE_DIALOG + PWSTYLEINFO_MENU + ;
         PWSTYLEINFO_MINIMIZE ;
      TITLE "ProVision:Windows tutorial" ;
      ICON TITLE "Tutorial" ;
      WINDOW HANDLER { | idMsg | MainWndHandler(idMsg) }

   // Create the menu for the window.
   MENU oMenu

      POPUP PROMPT "~File"
         MENUITEM PROMPT "~About..." ;
            ACTION { || AboutWindow() }
         SEPARATOR
         MENUITEM PROMPT "E~xit\tAlt+F4" ;
            ACTION { || oMainWnd:close() }
      END POPUP // File

      POPUP oPopTopics PROMPT "~Topics"
         MENUITEM PROMPT "~Initializing ProVision:Windows..." ;
            ACTION { || TopicsWindow(TOPIC_INITIALIZING, ;
            "Initializing ProVision:Windows") }
         MENUITEM PROMPT "~Using windows..." ;
            ACTION { || TopicsWindow(TOPIC_USING, "Using windows") }
         MENUITEM PROMPT "~Writing to windows..." ;
            ACTION { || TopicsWindow(TOPIC_WRITING, "Writing to windows") }
         MENUITEM PROMPT "Using ~controls..." ;
            ACTION { || TopicsWindow(TOPIC_CONTROLS, "Using controls") }
         MENUITEM PROMPT "~Manipulating windows..." ;
            ACTION { || TopicsWindow(TOPIC_MANIPULATING, ;
               "Manipulating windows") }
         MENUITEM PROMPT "~Events and messages..." ;
            ACTION { || TopicsWindow(TOPIC_MESSAGES, "Events and messages") }
      END POPUP // Topics

      POPUP PROMPT "~Samples"

         MENUITEM oItemBlocks PROMPT "B~locks..." ;
            ACTION { || pwBlocks(oItemBlocks) }

         POPUP PROMPT "~Browse windows"
            MENUITEM PROMPT "~Single browse window..." ;
               ACTION { || BrowseWindow() }
            MENUITEM oItemMulti PROMPT "~Related browse windows..." ;
               ACTION { || MultiWindow(oItemMulti) }
         END POPUP // Browse windows

         MENUITEM oItemCalc PROMPT "~Calculator..." ;
            ACTION { || Calculator(oItemCalc) }
         MENUITEM oItemClock PROMPT "Cloc~k..." ;
            ACTION { || Clock(oItemClock) }
         MENUITEM PROMPT "Generate an ~error..." ;
            ACTION { || MsgBox("This option demonstrates using " + ;
               "ProVision:Windows with the Clipper error system by " + ;
               "generating a runtime error.  Select 'Default' to " + ;
               "continue using the tutorial."), ;
               DBUSEAREA(.T.,, "ERROR.DBF") }
         MENUITEM oItemMDI PROMPT "~MDI window..." ;
            ACTION { || MDIWindow(oItemMDI) }
         MENUITEM oItemMemory PROMPT "Memory ~status..." ;
            ACTION { || MemoryWindow(oItemMemory) }
         MENUITEM oItemText PROMPT "~Textedit window..." ;
            ACTION { || TextWindow(oItemText) }
      END POPUP // Samples

   END MENU // oMenu

   // Attach the menu to the window.
   ATTACH MENU oMenu TO oMainWnd

   // Draw the ProVision logo.

   // Draw the 'O' shadow.
   IF ISCOLOR()
      SETCOLOR("+W/W")
      @ 02, 14 SAY SPACE(6)
      @ 03, 18 SAY "    "
      @ 04, 20 SAY "   "
      @ 05, 21 SAY " "
   END IF // ISCOLOR()

   // Draw the 'O.'
   SETCOLOR(IIF(ISCOLOR(), "+W/N", "N/W"))
   @ 03, 12 SAY SPACE(6)
   @ 04, 10 SAY SPACE(10)
   @ 05, 09 SAY SPACE(12)
   @ 06, 10 SAY SPACE(10)
   @ 07, 12 SAY SPACE(6)

   // Draw the 'V' shadow.
   IF ISCOLOR()
      SETCOLOR("+W/W")
      @ 01, 25 SAY SPACE(17)
      @ 02, 40 SAY " "
      @ 03, 39 SAY " "
      @ 04, 38 SAY " "
      @ 05, 37 SAY " "
      @ 06, 36 SAY " "
      @ 07, 35 SAY " "
      @ 08, 34 SAY " "
      @ 09, 33 SAY " "
   END IF // ISCOLOR()

   // Draw the 'V'.
   IF ISCOLOR()
      SETCOLOR("+W/W")
      @ 03, 27 SAY "   "
      @ 03, 33 SAY "   "
      @ 04, 28 SAY "   "
      @ 04, 32 SAY "   "
      @ 05, 29 SAY "     "
      @ 06, 30 SAY "   "
      @ 07, 31 SAY " "
   END IF // ISCOLOR()

   // Draw the triangle.
   SETCOLOR(IIF(ISCOLOR(), "+W/R", "N/W"))
   @ 02, 23 SAY SPACE(17)
   @ 03, 24 SAY "   "
   @ 03, 30 SAY "   "
   @ 03, 36 SAY "   "
   @ 04, 25 SAY "   "
   @ 04, 31 SAY " "
   @ 04, 35 SAY "   "
   @ 05, 26 SAY "   "
   @ 05, 34 SAY "   "
   @ 06, 27 SAY "   "
   @ 06, 33 SAY "   "
   @ 07, 28 SAY "   "
   @ 07, 32 SAY "   "
   @ 08, 29 SAY "     "
   @ 09, 30 SAY "   "
   @ 10, 31 SAY " "

   // Draw the copyright message.
   SETCOLOR(IIF(ISCOLOR(), "+W/B", "+W/N"))
   @ 12, 14 SAY "ProVision:Windows v1.20"
   @ 13, 01 SAY "Copyright 1991, 1992 SofDesign International, Inc."
   @ 14, 16 SAY "All rights reserved"
   @ 16, 08 SAY "Press F1 for help or Alt+X to exit."

   // Restore the mouse cursor (or remove the wait message).
   WaitMsg(.F.)

   // Return a reference to the new window.
   RETURN(oMainWnd)
   // END MainWnd()



/*****************************************************************************
** STATIC MainWndHandler( idMsg ) --> lHandled
**
** This function is a custom window message handler for oMainWnd.  It
** attempts to terminate the application when the main window is closed.
**
*****************************************************************************/
STATIC FUNCTION MainWndHandler( idMsg )

   STATIC   lClosed  := .F.

   LOCAL lHandled := .T.


   IF idMsg == PWMSG_CLOSE
      //
      // The use of lClosed is necessary because pw():terminate() also tries
      // to close the window.  Without this precaution, an infinite loop
      // forms with the close logic trying to terminate the application and
      // the terminate logic trying to close the window.
      //
      IF .NOT. lClosed
         IF ExitWindow()
            lClosed  := .T.
            pw():terminate()
         ELSE
            lHandled := .F.
         END IF // ExitWindow()
      END IF // .NOT. lClosed
   END IF // idMsg == PWMSG_CLOSE

   RETURN(lHandled)
   // END MainWndHandler( idMsg )



/*****************************************************************************
** STATIC AboutWindow() --> NIL
**
** This function uses a modal window to display some information about
** ProVision:Windows.
**
*****************************************************************************/
STATIC FUNCTION AboutWindow()

   STATIC   oAboutWnd

   LOCAL GetList  := {}


   // If the window has not yet been created, do so.
   IF oAboutWnd == NIL

      // Turn the mouse cursor into an hourglass (or display a wait message).
      WaitMsg(.T.)

      // Create a window for the message.
      CREATE WINDOW oAboutWnd ;
         CENTERED ;
         SIZE 18, 48 ;
         STYLE PWSTYLE_DIALOG ;
         TITLE "About ProVision:Windows" ;
         WINDOW HANDLER { | idMsg | AboutWndHandler(idMsg) }

      // Display the about message.
      @ 01, 01 SAY "     ProVision:Windows is an innovative     "
      @ 02, 01 SAY "user interface library for text mode Clipper"
      @ 03, 01 SAY "       applications from the software       "
      @ 04, 01 SAY "               architects at                "
      @ 06, 01 SAY "       SofDesign International, Inc.        "
      @ 07, 01 SAY "       1303 Columbia Drive, Suite 209       "
      @ 08, 01 SAY "       Richardson, Texas 75081 USA          "
      @ 10, 01 SAY "           Phone: 214-644-0098              "
      @ 11, 01 SAY "       Toll-free: 1-800-755-7344            "
      @ 12, 01 SAY "             FAX: 214-644-4286              "

      // Create an 'OK' pushbutton.
      @ 13, 19 GET AS PUSHBUTTON ;
         PROMPT "  OK  " ;
         ACTION { || oAboutWnd:close() }

      // Attach the controls to the window.
      ATTACH CONTROLS TO oAboutWnd

      // Restore the mouse cursor (or remove the wait message).
      WaitMsg(.F.)

   END IF // oAboutWnd == NIL

   // Open the window modally.
   OPEN WINDOW oAboutWnd MODAL

   RETURN(NIL)
   // END AboutWindow()



/*****************************************************************************
** STATIC AboutWndHandler( idMsg ) --> lHandled
**
** This function is a custom window message handler for oAboutWnd.
**
*****************************************************************************/
STATIC FUNCTION AboutWndHandler( idMsg )

   STATIC   lFirstTime  := .T.

   LOCAL lHandled := .T.


   IF idMsg == PWMSG_CLOSE

      // This is kind of weird, but the first time the About window is
      // closed (at application startup), we'll open the sample files
      // for the browses.
      IF lFirstTime

         lFirstTime  := .F.

         // Turn the mouse cursor into an hourglass (or display a wait
         // message).
         WaitMsg(.T.)

         // Open the files needed for the browse samples.
         OpenFiles()

         // Restore the mouse cursor (or remove the wait message).
         WaitMsg(.F.)

      END IF // lFirstTime

   END IF // idMsg == PWMSG_CLOSE

   RETURN(lHandled)
   // END AboutWndHandler( idMsg )



/*****************************************************************************
** STATIC ExitWindow() --> lOkToExit
**
** This function uses a modal window to verify that the user wants to exit
** the application.  It returns .T. if it is okay to exit.
**
*****************************************************************************/
STATIC FUNCTION ExitWindow()

   STATIC   lOkToExit

   STATIC   oExitWnd, ;
            oOK

   LOCAL GetList  := {}


   // If the window has not yet been created, do so.
   IF oExitWnd == NIL

      // Turn the mouse cursor into an hourglass (or display a wait message).
      WaitMsg(.T.)

      // Create a window for the message.
      CREATE WINDOW oExitWnd ;
         CENTERED ;
         SIZE 08, 33 ;
         STYLE PWSTYLE_DIALOG ;
         TITLE "Exit tutorial" ;
         WINDOW HANDLER { | idMsg | IIF(idMsg == PWMSG_CLOSE, ;
            pw():endModal(), ), .T. }

      // Display the exit message.
      @ 01, 07 SAY "This will end the"
      @ 02, 02 SAY "ProVision:Windows tutorial."

      // Create 'OK' and 'Cancel' pushbuttons.
      @ 03, 05 GET AS PUSHBUTTON oOK ;
         PROMPT "  OK  " ;
         ACTION { || lOkToExit := .T., oExitWnd:close() }

      @ 03, 18 GET AS PUSHBUTTON ;
         PROMPT "Cancel" ;
         SHORTCUT K_ESC ;
         ACTION { || oExitWnd:close() }

      // Attach the controls to the window.
      ATTACH CONTROLS TO oExitWnd

      // Restore the mouse cursor (or remove the wait message).
      WaitMsg(.F.)

   END IF // oExitWnd == NIL

   // Make sure 'OK' is selected and the exit flag is .F.
   oExitWnd:select(oOK)
   lOkToExit   := .F.

   // Open the window modally.
   OPEN WINDOW oExitWnd MODAL

   // If the window was successfully opened, begin a local event loop.
   IF pw():curFocus() == oExitWnd
      pw():beginModal()
   END IF // pw():curFocus() == oExitWnd

   // Return the result.
   RETURN(lOkToExit)
   // END ExitWindow()



/*****************************************************************************
** OKBox( cMsg ) --> NIL
**
** This function uses a modal window to display a message and wait until
** the user presses the OK pushbutton.
**
*****************************************************************************/
FUNCTION OKBox( cMsg )

   LOCAL GetList  := {}

   LOCAL i, ;
         nLines   := MLCOUNT(cMsg, 36)

   LOCAL oOKWnd
         

   // Turn the mouse cursor into an hourglass (or display a wait message).
   WaitMsg(.T.)

   // Create a window for the message.
   CREATE WINDOW oOKWnd ;
      CENTERED ;
      SIZE nLines + 5, 40 ;
      STYLE PWSTYLE_THIN ;
      WINDOW HANDLER { | idMsg | IIF(idMsg == PWMSG_CLOSE, ;
         pw():endModal(), ), .T. }

   // Display the message.
   FOR i := 1 TO nLines
      @ i - 1, 00 SAY PADC(RTRIM(MEMOLINE(cMsg, 36, i)), 36)
   NEXT i 

   // Create an OK pushbutton.
   @ i - 1, 15 GET AS PUSHBUTTON ;
      PROMPT "  OK  " ;
      ACTION { || oOKWnd:close() }

   // Attach the controls to the window.
   ATTACH CONTROLS TO oOKWnd

   // Restore the mouse cursor (or remove the wait message).
   WaitMsg(.F.)

   // Open the window modally.
   OPEN WINDOW oOKWnd MODAL

   // If the window was successfully opened, begin a local event loop.
   IF pw():curFocus() == oOKWnd
      pw():beginModal()
   END IF // pw():curFocus() == oOKWnd

   // Destroy the window.
   oOKWnd:destroy()

   RETURN(NIL)
   // END OKBox( cMsg )



/*****************************************************************************
** STATIC SysKeyFilter( idMsg, nKey ) --> NIL
**
** This function implements the system keypress filter, allowing F1 to invoke
** help and Alt+X to exit from anywhere in the application.
**
*****************************************************************************/
STATIC FUNCTION SysKeyFilter( idMsg, nKey )


   IF nKey == K_F1

      // Display help only if the message window is not already open.
      IF (oMsgWnd == NIL) .OR. ;
            (.NOT. pvGetBit(oMsgWnd:getState(), PWSTATE_OPEN))

         MsgBox("ProVision:Windows v1.20" + CRLF + ;
            "Copyright 1991, 1992 SofDesign International, Inc." + CRLF + ;
            "All rights reserved" + CRLF + ;
            CRLF + ;
            "Use a mouse or Alt+Esc or Alt+F6 to switch between windows, " + ;
            "Tab to switch between controls, and F10 to access menus or " + ;
            "a window's system menu.  Press a key or a mouse button to " + ;
            "remove message windows, such as this one.")

         idMsg := PWMSG_NONE

      END IF // (oMsgWnd == NIL) .OR. ...

   ELSEIF nKey == K_ALT_X

      IF oMainWnd:close()
         idMsg := PWMSG_NONE
      END IF // oMainWnd:close()

   ELSEIF nKey == K_ALT_F11

      _pvTxt2Grph()

   END IF // nKey == K_F1

   RETURN(NIL)
   // END SysKeyFilter( idMsg, nKey )



/*****************************************************************************
** STATIC TopicsWindow( idTopic, cTitle ) --> oTopicWnd
**
** This function creates a window in which sample code for the topics will
** be shown.  The window also contains pushbuttons with which the user can
** control the tutorial.
**
*****************************************************************************/
STATIC FUNCTION TopicsWindow( idTopic, cTitle )

   STATIC   nCurrentTopic

   STATIC   oBtnContinue, ;
            oBtnMenu, ;
            oListing

   LOCAL GetList     := {}

   LOCAL nViewHeight := pwMaxRow() - (MAXMSGLEN + 6)


   // Turn the mouse cursor into an hourglass (or display a wait message).
   WaitMsg(.T.)

   // If the window has not yet been created, do so.
   IF oTopicWnd == NIL

      // Create a window in which sample code will be shown.
      CREATE WINDOW oTopicWnd ;
         AT MAXMSGLEN + 4, 02 ;
         SIZE pwMaxRow() - (MAXMSGLEN + 4), pwMaxCol() - 3 ;
         STYLE PWSTYLE_DIALOG ;
         TITLE "" ;
         WINDOW HANDLER { | idMsg | TopicWndHandler(idMsg, oListing, ;
            oBtnContinue, oBtnMenu) }

      // The code will be shown inside this listbox.
      @ 00, 01 GET AS LISTBOX oListing ;
         SIZE nViewHeight, pwMaxCol() - 19 ;
         PROMPT "" ;
         USING {}

      // Create these pushbuttons to control the tutorial.
      @ INT(((nViewHeight - 1) / 2) - 2), pwMaxCol() - 16 ;
         GET AS PUSHBUTTON oBtnContinue ;
         PROMPT "Continue" ;
         ACTION { || Frames(+1, oListing, oBtnContinue, oBtnMenu) }

      @ INT(((nViewHeight - 1) / 2) + 1), pwMaxCol() - 16 ;
         GET AS PUSHBUTTON oBtnMenu ;
         PROMPT "  Menu  " ;
         ACTION { || oTopicWnd:close() }

      // Attach the controls to the window.
      ATTACH CONTROLS TO oTopicWnd

   END IF // oTopicWnd == NIL

   // Minimize the main window.
   oMainWnd:minimize()

   // Set the current topic id and reset the frame number.
   nTopic   := idTopic
   nFrame   := 0

   // Disable the Topics menuitem.
   oPopTopics:disable()

   // Set the listing box to an empty array.
   oListing:setArray({})

   // Make sure the continue pushbutton is enabled and has focus.
   oBtnContinue:enable()
   oTopicWnd:select(oBtnContinue)

   // Set the title for the topics window.
   oTopicWnd:setTitle(cTitle)

   // Display the introductory topic message.
   MsgBox(cTitle + CRLF + ;
      CRLF + ;
      "Press 'Continue' to step through the topic, or press " + ;
      "'Menu' to return to the menu window.  You can manipulate " + ;
      "the windows with the keyboard or the mouse at any time in " + ;
      "the tutorial.  Use Alt+Esc to switch between windows, Tab " + ;
      "to switch between controls, and F10 to access menus or " + ;
      "a window's system menu.  Press a key or a mouse button to " + ;
      "remove message windows, such as this one.")

   // Open the new window.
   OPEN WINDOW oTopicWnd

   // Restore the mouse cursor (or remove the wait message).
   WaitMsg(.F.)

   // Return a reference to the new window.
   RETURN(oTopicWnd)
   // END TopicsWindow( idTopic, cTitle )



/*****************************************************************************
** STATIC TopicWndHandler( idMsg, oListing, oBtnContinue, ;
**                         oBtnMenu ) --> lHandled
**
** This function is the window message handler for oTopicWnd.
**
*****************************************************************************/
STATIC FUNCTION TopicWndHandler( idMsg, oListing, oBtnContinue, ;
                                 oBtnMenu )

   LOCAL lHandled := .T.


   DO CASE
   CASE idMsg == PWMSG_CLOSE
      // Notify the current topic that the window is closing.
      Frames(-1000000, oListing, oBtnContinue, oBtnMenu)

      // Enable the Topics menuitem.
      oPopTopics:enable()

      // Restore and select the main window.
      IF pvGetBit(oMainWnd:getState(), PWSTATE_OPEN)
         oMainWnd:restore()
         pw():select(oMainWnd)
      END IF // pvGetBit(oMainWnd:getState(), PWSTATE_OPEN)

   END CASE

   RETURN(lHandled)
   // END TopicWndHandler( idMsg, oListing, oBtnContinue, ;



/*****************************************************************************
** STATIC Frames( nStep, oListing, oBtnContinue, oBtnMenu ) --> NIL
**
** This function dispatches requests for the next frame to the current
** topic.  The topic function returns .F. if the tutorial should return to
** the main window.
**
*****************************************************************************/
STATIC FUNCTION Frames( nStep, oListing, oBtnContinue, oBtnMenu )

   LOCAL lContinue


   // Set the counter for the next or previous frame.
   nFrame   += nStep

   // Execute the next frame for the current topic.
   DO CASE
   CASE nTopic == TOPIC_INITIALIZING
      lContinue   := InitFrames(nFrame, oTopicWnd, oListing)

   CASE nTopic == TOPIC_USING
      lContinue   := UsingFrames(nFrame, oTopicWnd, oListing)

   CASE nTopic == TOPIC_WRITING
      lContinue   := WritingFrames(nFrame, oTopicWnd, oListing)

   CASE nTopic == TOPIC_CONTROLS
      lContinue   := CtrlFrames(nFrame, oTopicWnd, oListing)

   CASE nTopic == TOPIC_MANIPULATING
      lContinue   := ManipFrames(nFrame, oTopicWnd, oListing)

   CASE nTopic == TOPIC_MESSAGES
      lContinue   := EventsFrames(nFrame, oTopicWnd, oListing)

   END CASE

   // If the last frame is shown, disable the Continue pushbutton and
   // select the Menu pushbutton.
   IF (.NOT. lContinue) .AND. (nFrame > 0)
      oBtnContinue:disable()
      oTopicWnd:select(oBtnMenu)
   END IF // (.NOT. lContinue) .AND. (nFrame > 0)

   // Return NIL.
   RETURN(NIL)
   // END Frames( nStep, oListing, oBtnContinue, oBtnMenu )



/*****************************************************************************
** MsgBox( cMsg ) --> NIL
**
** This function opens a window in which the passed message is displayed
** until any event except mouse movement occurs.  The window is then closed.
**
*****************************************************************************/
FUNCTION MsgBox( cMsg )

   STATIC   nFilter

   LOCAL i, ;
         nMsgLen  := MLCOUNT(cMsg, MAXLINELEN), ;
         nRow


   // If the window has not yet been created, create it.
   IF oMsgWnd == NIL

      // Turn the mouse cursor into an hourglass (or display a wait message).
      WaitMsg(.T.)

      // Create a simple window with a thin border.
      CREATE WINDOW oMsgWnd ;
         AT 01, 02 ;
         SIZE MAXMSGLEN + 2, MAXLINELEN + 4 ;
         STYLE PWSTYLE_THIN ;
         WINDOW HANDLER { | idMsg | IIF(idMsg == PWMSG_CLOSE, ;
            pw():delFilter(nFilter), ), .T. }

      // Restore the mouse cursor (or remove the wait message).
      WaitMsg(.F.)

   END IF // oMsgWnd == NIL

   // Open the window.
   DISPBEGIN()
   OPEN WINDOW oMsgWnd
   CLEAR SCREEN
   DISPEND()

   // Display the message centered in the window.
   nRow  := INT((MAXMSGLEN - nMsgLen) / 2) - 1
   FOR i := 1 TO nMsgLen
      @ nRow + i, 01 SAY PADC(RTRIM(MEMOLINE(cMsg, MAXLINELEN, i)), ;
         MAXLINELEN)
   NEXT i 

   // Add a system filter that will close the window when any event
   // except mouse movement occurs.
   nFilter  := pw():addFilter({ || oMsgWnd:close() }, PWMSG_EVENT - ;
      PWMSG_MOUSEMOVEMENT)

   // Return NIL.
   RETURN(NIL)
   // END MsgBox( cMsg )



/*****************************************************************************
** WaitMsg( lToggle ) --> NIL
**
** If a mouse is in use and in TMR mode, this function turns the mouse
** cursor to an hourglass when lToggle is .T.  Otherwise it displays
** "WAIT" in the upper-right corner of the wallpaper.  When lToggle is
** .F., either the mouse cursor is restored to an arrow, or the wait
** message is removed.
**
*****************************************************************************/
FUNCTION WaitMsg( lToggle )

   STATIC   cScreen

   STATIC   lUseMouse

   STATIC   nOpen := 0

   LOCAL oOutput


   // Determine if ProVision:Windows initialized a mouse driver and
   // if the mouse cursor is in TMR mode.
   IF lUseMouse == NIL
      lUseMouse   := pw():isMouse() .AND. pw():isTMR(PWTMR_MOUSE)
   END IF // lUseMouse == NIL

   IF lToggle

      // Keep track of the number of calls to allow nesting.
      ++nOpen

      // Only change the state if this is the beginning of an outer-level
      // call.
      IF nOpen == 1

         // If a mouse was initialized in TMR mode, turn the cursor to
         // an hourglass, otherwise display a wait message on the
         // wallpaper.
         IF lUseMouse
            pw():mouseCursor(PMCURSOR_HOUR)
         ELSE
            oOutput  := pw():setOutput(NIL)
            cScreen  := SAVESCREEN(00, MAXCOL() - 05, 00, MAXCOL())
            SETCOLOR("*N/W")
            @ 00, MAXCOL() - 05 SAY " WAIT "
            pw():setOutput(oOutput)
         END IF // lUseMouse

      END IF // nOpen == 1

   ELSE

      // Keep track of the number of calls to allow nesting.
      --nOpen

      // Only change the state if this is the end of an outer-level call.
      IF nOpen == 0

         // If a mouse was initialized in TMR mode, restore the cursor to
         // an arrow, otherwise remove the wait message from the
         // wallpaper.
         IF lUseMouse
            pw():mouseCursor(PMCURSOR_ARROW)
         ELSE
            oOutput  := pw():setOutput(NIL)
            RESTSCREEN(00, MAXCOL() - 05, 00, MAXCOL(), cScreen)
            pw():setOutput(oOutput)
         END IF // lUseMouse

      END IF // nOpen == 0

   END IF // lToggle

   RETURN(NIL)
   // END WaitMsg( lToggle )

