// Symmetry in Chaos, the Program.
// By Dan Farmer
// Based upon code from the book "Symmetry in Chaos"
//
// NOTE: Graphics requires SVGABGI from  Jordan Hargraphix Software
// (Not included. Available from the Borland forum on CompuServe)
//
#include <bios.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <dos.h>
#include <dir.h>
#include <stdio.h>
#include <alloc.h>
#include <math.h>
#include <conio.h>
#include <graphics.h>
#include "svga16.h"
#include "svga256.h"
#include "symm.h"

#define DEFAULT_MAP ".\\default.map"

#define ICONS         1
#define AFFINE        2
#define SQUARE_QUILTS 3
#define MAX_FRACTAL_TYPES 3

#define D_SYMMETRY  0
#define Z_SYMMETRY  1
#define MAX_TRANSIENTS 2000;

char far *Imagebuffer;                      // Screensave buffer
double x, y, xNew, yNew, Scale, Xcenter, Ycenter, Aspect;

// Note: These following lists are the variable declarations that were new
// for each type as I was putting them in. Thus, just because I've got them
// grouped by type, don't assume that they aren't used by other types, and
// that they don't use variables from other types!

// Parameters for the ICONS type
double lambda, alpha, beta, gamma, omega;

// Parameters for the AFFINE type
double a11, a12, a21, a22, b1, b2;
char conj;

// Parameters for the SQUARE_QUILTS type
int nperiod, ncount, ma;
double shift, mxNew, myNew;
double P2 = M_PI * 2;

// sin/cos tables for AFFINE transformations
#define TRIG_TABLE_ENTRIES   100
double cos_table[TRIG_TABLE_ENTRIES],
       sin_table[TRIG_TABLE_ENTRIES];


int n, Maxx, Maxy, Maxcolors, WhichFormula;

unsigned int transients;
char MapFlag = FALSE;          // Has mapped palette been loaded yet?

int Fractal_Type = ICONS;      // Start off with the Icon type
unsigned int Iters, High_hit;

int main()
{
	int px, py, c;
	int keypress;

    Scale = 1.0;       // This will be changed below for some types.

	Init_Graphics();
    Maxx = getmaxx();
    Maxy = getmaxy()-12;  // Small offset for text on bottom.
    Maxcolors = getmaxcolor();

    WhichFormula = 0;                     // Index to built-in formula types
    next_formula(&WhichFormula,1);        // Set parameters for first formula
    init_stats();                         // Set up statistics variables

    transients = MAX_TRANSIENTS;          // Let initial transients settle down

	switch(Fractal_Type) {
	case ICONS:
        x = 0.01; y = 0.003;              // Give the pendulum a shove
		break;

	case AFFINE:
        x = 0.1; y = -0.01;               // Throw things off balance a tad
        break;

    case SQUARE_QUILTS:
        x = 0.1; y = 0.334;               // Butterfly wind
        break;
	 }

	do {
		iterate();
		if(transients > 0)
			transients--;
		else {                                   // start plotting
            switch(Fractal_Type) {
            case ICONS:
            case AFFINE:
                plot_points1(x, y);
                break;
            case SQUARE_QUILTS:
                plot_points2(x, y);
                break;
            }
		}
		keypress = getkey();
		if (keypress) dokeypress(keypress);
	}  while (1);

	restorecrtmode();
	return(0);
}

// Used by Icon and Affine types
void plot_points1(double x, double y)
{
    int c, px, py;

    px = (int) Ycenter * (x + Scale);  // Yes, I know.  Ycenter, not Xcenter.
    py = (int) Ycenter * (y + Scale);

    c = getpixel(px, py) +1;
    if(c > Maxcolors) c = Maxcolors;
        putpixel(px, py, c);
    Iters++;
    if( c > High_hit) High_hit = c;
}

// Used by the Square Quilts method
void plot_points2(double x, double y)
{
    int px, py, c, mc;
    register int i,j;

    for (i = 0; i < nperiod; i++){
        for (j = 0; j < nperiod; j++){

            px = Maxx * (x+i) / nperiod;
            py = Maxy * (y+j) / nperiod;

            c = getpixel(px, py) +1;
            if(c > Maxcolors) c = Maxcolors;
                putpixel(px, py, c);
        }
    }
    Iters++;
    if( c > High_hit) High_hit = c;
}


void reset_scale(void)
{
    int i, j;
    double Aspect = (double) Maxy / Maxx;
    Ycenter = Maxy / (2.0 * Scale );
    Xcenter = Maxx / (2.0 * Scale);
    transients = MAX_TRANSIENTS;
    cleardevice();
}

void init_stats(void)
{
    Iters = 0;
    High_hit = 0;
}

// Simply a dispatching routine to call the correct interation routine
void iterate(void)
{
	switch(Fractal_Type){
	case ICONS:
		iterate1();
        break;
	case AFFINE:
        iterate2();
        break;
    case SQUARE_QUILTS:
        iterate3();
        break;
	}
}

// ICONS iteration method
void iterate1(void)
{
    double zzbar, zreal, zimag, za, zb, zn, p, xNew, yNew;
	register int i;
	double px, py;
	int c;

	zzbar = x*x + y*y;
    zreal = x; zimag = y;
    for(i=1; i <= n-2; i++) {
			za = zreal * x - zimag * y;
            zb = zimag * x + zreal * y;
			zreal = za;
            zimag = zb;
	}
    zn = x * zreal - y * zimag;
	p = lambda + alpha * zzbar + beta * zn;
    xNew = p * x + gamma * zreal - omega * y;
    yNew = p * y - gamma * zimag + omega * x;

    x = xNew; y = yNew;
}

// AFFINE iteration method
void iterate2(void)
{
	double x1,y1;
	int m;

    xNew = a11 * x + a12 * y + b1;
    yNew = a21 * x + a22 * y + b2;

    m = (int)(random(n));
    x1 = xNew; y1 = yNew;
    xNew = cos_table[m] * x1 - sin_table[m] * y1;
    yNew = sin_table[m] * x1 + cos_table[m] * y1;

    if(conj == D_SYMMETRY) {
        x = xNew; y = yNew;
        return;
    }

    m = (int) random(2);
    if(m==1) yNew = -yNew;

    x = xNew; y = yNew;
}

// Square Quilts iteration method
void iterate3(void)
{
    double sx,sy;

    sx = sin(P2*x); sy = sin(P2*y);

    xNew = (lambda + alpha * cos(P2 * y)) * sx - omega * sy + beta *
      sin(2.0 * P2 * x) + gamma * sin(3.0 * P2 * x) * cos(2 * P2 * y) + ma *
      x + shift;

    yNew = (lambda + alpha * cos(P2 * x)) * sy + omega * sx + beta *
      sin(2.0 * P2 * y) + gamma * sin(3.0 * P2 * y) * cos(2.0 * P2 * x) + ma *
      y + shift;

    if(xNew > 1.0) xNew = xNew -(int)xNew;
    if(yNew > 1.0) yNew = yNew -(int)yNew;
    if(xNew < 0) xNew = xNew + (int) -xNew +1.0;
    if(yNew < 0) yNew = yNew + (int) -yNew +1.0;

    x = xNew; y = yNew;
}

void clear_trig_tables(void)
{
	int i;
	for(i=0; i < TRIG_TABLE_ENTRIES; i++) {
		cos_table[i] = 0.0;
		sin_table[i] = 0.0;
	}
}

void init_trig_tables(void)
{
	int i;

    clear_trig_tables();

    for(i=0; i < n; i++) {
		cos_table[i] = (double) cos(2.0 * M_PI * i / n);
		sin_table[i] = (double) sin(2.0 * M_PI * i / n);
	}
}

void next_formula(int *WhichFormula, int direction)
{
	switch(Fractal_Type){

	case ICONS:
        next_formula_1(WhichFormula, direction);
		x = 0.0234; y = 0.1234;
		break;

	case AFFINE:
        next_formula_2(WhichFormula, direction);
		x = 0.1; y = -0.01;
        init_trig_tables();
		break;

    case SQUARE_QUILTS:
        next_formula_3(WhichFormula, direction);
		x = 0.0234; y = 0.1234;
		break;
	}
}

// Currently, this routine simply selects from a list of hard-coded
// formulas.  Eventually it should select and load from a saved file.

// ICON formulas
void next_formula_1(int *WhichFormula, int direction)
{
	int maxformulas=13;
    int  formula;
	char formula_name[80];

    *WhichFormula += direction;

    if (*WhichFormula > maxformulas)
        *WhichFormula=1;

    if (*WhichFormula < 1)
       *WhichFormula=maxformulas;

    formula = *WhichFormula;

    // Omega = 0 gives D_n symmetery, omega != 0 gives Z_n symmetry
    // n := is degree of symmetery
    switch(formula) {
    case 1: // -fig 1.1
        lambda = -2.7; alpha = 5.0; beta = 1.5; gamma = 1.0; omega = 0.0;
        n = 6; Scale = 0.95;
		sprintf(formula_name,"Halloween");
		break;

    case 2: // -fig 1.2
        lambda = -2.08; alpha = 1.0; beta = -0.1; gamma = 0.167; omega = 0.0;
        n = 7; Scale = 1.35;
		sprintf(formula_name,"Mayan Bracelet");
		break;

    case 3: // -fig 1.18(b)
        lambda = -2.195; alpha = 10.0; beta = -12.0; gamma = 1.0; omega = 0.0;
        n = 3; Scale = 0.5;
		sprintf(formula_name,"Fish & Eye");
		break;

	case 4: // --fig 5.12a
        lambda = 1.455; alpha = -1.0; beta = 0.03; gamma = -0.8; omega = 0.0;
        n = 3; Scale = 1.5;
		sprintf(formula_name,"Fig. 5.12(a)");
		break;

    case 5: // --fig 5.11a
        lambda = 2.409; alpha = -2.5; beta = 0.0; gamma = 0.9; omega = 0.0;
        n = 23; Scale = 1.1;
		sprintf(formula_name,"Kachina Dolls");
		break;

    case 6: // --fig 3.14
        lambda = -2.05; alpha = 3.0; beta = -16.79; gamma = 1.0; omega = 0.0;
        n = 9; Scale = 0.75;
		sprintf(formula_name,"French Glass");
		break;

    case 7: // --fig 3.5(b)
        lambda = -2.34; alpha = 2.0; beta = 0.2; gamma = 0.1; omega = 0.0;
        n = 5; Scale = 1.25;
		sprintf(formula_name,"Sand Dollar");
		break;

    case 8: // --fig 1.17
        lambda = 1.56; alpha = -1.0; beta = 0.1; gamma = -0.82; omega = 0.0;
        n = 3; Scale = 1.45;
		sprintf(formula_name,"Trampoline");
		break;

    case 9: // --fig 1.13
        lambda = -1.806; alpha = 1.806; beta = 0.0; gamma = 1.0; omega = 0.0;
        n = 5; Scale = 0.9;
		sprintf(formula_name,"Emperor's Cloak");
		break;

    case 10: // --fig 3.6(b)
        lambda = 2.6; alpha = -2.0; beta = 0.0; gamma = -0.5; omega = 0.0;
        n = 5; Scale = 1.45;
		sprintf(formula_name,"Pentagon Attractor");
		break;

    case 11: // --fig 3.15(a)
        lambda = -2.32; alpha = 2.32; beta = 0.0; gamma = 0.75; omega = 0.0;
        n = 5; Scale = 1.15;
		sprintf(formula_name,"The Pentangle");
		break;

    case 12: // --fig 5.5
		sprintf(formula_name,"Golden Flintstone");
        lambda = 2.5; alpha = -2.5; beta = 0.0; gamma = 0.9; omega = 0.0;
        n = 3; Scale = 1.45;
		break;
// Icons with non-polynomial terms
    case 13: // --fig 5.5
		sprintf(formula_name,"Test");
        lambda = 2.5; alpha = -2.5; beta = 0.0; gamma = 0.45; omega = 0.45;
        n = 3; Scale = 1.45;
		break;


	}
	reset_scale();
    print_formula_name(formula_name);
}

// AFFINE type formulas
void next_formula_2(int *WhichFormula, int direction)
{
	double a1,a2;
    int formula;
	char formula_name[80];

    int maxformulas = 14;
    *WhichFormula += direction;

    if (*WhichFormula > maxformulas)
        *WhichFormula=1;

    if (*WhichFormula < 1)
       *WhichFormula=maxformulas;

    formula = *WhichFormula;


	switch(formula) {

    case 1: // -fig 7.3
		a11 = 0.5; a12 = 0.0; a21 = 0.0;  a22 = 0.5;
        b1  = 0.5;  b2 = 0.0; n = 3; conj = Z_SYMMETRY; Scale = 1.0;
		sprintf(formula_name,"Sierpinksi Triangle");
		break;

    case 2: // -fig 7.4(a)
		a11 = 0.5; a12 = 0.0; a21 = 0.0;  a22 = 0.5;
        b1  = 0.5;  b2 = 0.0; n = 5; conj = Z_SYMMETRY; Scale = 1.0;
		sprintf(formula_name,"Sierpinksi Pentagon");
		break;

    case 3: // -fig 7.4(b)
		a11 = 0.5; a12 = 0.0; a21 = 0.0;  a22 = 0.5;
        b1  = 0.5;  b2 = 0.0; n = 6; conj = Z_SYMMETRY; Scale = 1.0;
		sprintf(formula_name,"Sierpinksi Hexagon");
		break;

    case 4: // -fig 7.7(a)
        a11 = 0.45; a12 = 0.0; a21 = 0.0;  a22 = 0.45;
        b1  = 0.55;  b2 = 0.0; n = 3; conj = Z_SYMMETRY; Scale = 1.0;
        sprintf(formula_name,"Strict Fractal");
		break;

    case 5: // -fig 7.7(c)
        a11 = 0.55; a12 = 0.0; a21 = 0.0;  a22 = 0.55;
        b1  = 0.45;  b2 = 0.0; n = 3; conj = Z_SYMMETRY; Scale = 1.0;
        sprintf(formula_name,"Overlaid Fractal");
		break;

    case 6: // -fig 8.0
        a11 = 0.45; a12 = -0.1; a21 = -0.31;  a22 = 0.45;
        b1  = 0.1;  b2 = 0.2; n = 11; conj = D_SYMMETRY; Scale = 0.45;
        sprintf(formula_name,"Circular Saw");
		break;

    case 7: // -fig 7.9
        a11 = 0.4; a12 = -0.1; a21 = -0.35;  a22 = 0.4;
        b1  = 0.01;  b2 = 0.2; n = 9; conj = D_SYMMETRY; Scale = 0.45;
        sprintf(formula_name,"Catherine Wheel");
		break;

    case 8: // -fig 7.10
        a11 = -0.1; a12 = 0.35; a21 = 0.2;  a22 = 0.5;
        b1  = 0.5;  b2 = 0.4; n = 3; conj = Z_SYMMETRY; Scale = 1.45;
        sprintf(formula_name,"Bee");
		break;

    case 9: // -fig 7.12
		a11 = 0.46; a12 = 0.0; a21 = 0.0;  a22 = 0.46;
        b1  = 0.54;  b2 = 0.0; n = 4; conj = Z_SYMMETRY; Scale = 1.0;
		sprintf(formula_name,"Sierpinski Square Variation");
		break;

    case 10: // -fig 7.13
		a11 = -0.25; a12 = -0.3; a21 = 0.3;  a22 = -0.26;
        b1  = 0.5;  b2 = 0.5; n = 8; conj = Z_SYMMETRY; Scale = 1.25;
		sprintf(formula_name,"Doily");
		break;

    case 11: // -fig 7.14
		a11 = -0.25; a12 = -0.3; a21 = 0.3;  a22 = -0.34;
        b1  = 0.5;  b2 = 0.5; n = 4; conj = Z_SYMMETRY; Scale = 1.0;
		sprintf(formula_name,"Astigmatism");
		break;

    case 12: // -fig 7.15
        a11 = -0.25; a12 = -0.3; a21 = 0.14;  a22 = -0.26;
        b1  = 0.5;  b2 = 0.5; n = 12; conj = D_SYMMETRY; Scale = 1.4;
        sprintf(formula_name,"Paisley");
		break;

    case 13: // -fig 7.16(a)
        a11 = 0.45; a12 = -0.1; a21 = 0.3;  a22 = -0.4;
        b1  = 0.15;  b2 = 0.1; n = 8; conj = D_SYMMETRY; Scale = 0.45;
        sprintf(formula_name,"Symmetric Coloring");
		break;

    case 14: // -fig 7.18
		a11 = -0.15; a12 = 0.75; a21 = 0.2;  a22 = -0.3;
        b1  = 0.075;  b2 = 0.4; n = 50; conj = D_SYMMETRY; Scale = 1.0;
		sprintf(formula_name,"Cashmere");
		break;
	}

	a1 = a11 * a11 + a21 * a21;
	a2 = a21 * a21 + a22 * a22;
    if(a1 > 1 || a2 > 1 || a1+a2 > 1+(a11*a22 - a12*a21) * (a11*a22 - a12*a21)) {
         printf("Warning:          \n");
         printf("affine mapping is \n");
         printf("NOT a contraction!\n");
             getch();
         printf("                  \n");
         printf("                  \n");
         printf("                  \n");

    }

	reset_scale();
    print_formula_name(formula_name);
    return;
}

// Square Quilts formulas
void next_formula_3(int *WhichFormula, int direction)
{
    int maxformulas = 13;
    int formula;
	char formula_name[80];

    *WhichFormula += direction;

    if (*WhichFormula > maxformulas)
        *WhichFormula=1;

    if (*WhichFormula < 1)
       *WhichFormula=maxformulas;

    formula = *WhichFormula;
    nperiod=3;

    switch(formula) {
    case 1:
        lambda = -0.65; alpha = 0.35; beta = 0.25; gamma = -0.25; omega = 0.65;
        ma = 3; shift = 0.5;
        sprintf(formula_name,"Square Quilt");
		break;
    case 2:
        lambda = -0.59; alpha = 0.2; beta = 0.1; gamma = -0.33; omega = 0.0;
        ma = 2; shift = 0.0;
        sprintf(formula_name,"Emerald Mosaic");
		break;
    case 3:
        lambda = -0.59; alpha = 0.2; beta = 0.1; gamma = -0.27; omega = 0.0;
        ma = 0; shift = 0.5;
        sprintf(formula_name,"Sugar and Spice");
		break;
    case 4:
        lambda = -0.2; alpha = -0.1; beta = 0.1; gamma = -0.25; omega = 0.0;
        ma = 0; shift = 0.0;
        sprintf(formula_name,"Sicilian Tile");
		break;
    case 5:
        lambda = 0.25; alpha = -0.3; beta = 0.2; gamma = 0.3; omega = 0.0;
        ma = 1; shift = 0.0;
        sprintf(formula_name,"Roses");
		break;
    case 6:
        lambda = -0.28; alpha = 0.25; beta = 0.05; gamma = -0.24; omega = 0.0;
        ma = -1; shift = 0.0;
        sprintf(formula_name,"Wagonwheels");
		break;
    case 7:
        lambda = -0.12; alpha = -0.36; beta = 0.18; gamma = -0.14; omega = 0.0;
        ma = 1; shift = 0.5;
        sprintf(formula_name,"Victorian Tiles");
		break;
    case 8:
        lambda = 0.1; alpha = 0.2; beta = 0.1; gamma = 0.39; omega = 0.0;
        ma = -1; shift = 0.0;
        sprintf(formula_name,"Mosque");
		break;
    case 9:
        lambda = -0.589; alpha = 0.2; beta = 0.04; gamma = -0.2; omega = 0.0;
        ma = 0; shift = 0.5;
        sprintf(formula_name,"Red Tiles");
		break;
    case 10:
        lambda = -0.28; alpha = 0.08; beta = 0.45; gamma = -0.05; omega = 0.0;
        ma = 0; shift = 0.5;
        sprintf(formula_name,"Cathedral Attractor");
		break;
    case 11:
        lambda = -0.59; alpha = 0.2; beta = 0.2; gamma = 0.3; omega = 0.0;
        ma = 2; shift = 0.0;
        sprintf(formula_name,"Gyroscope");
		break;
    case 12:
        lambda = -0.28; alpha = 0.25; beta = 0.05; gamma = -0.24; omega = 0.0;
        ma = -1; shift = 0.5;
        sprintf(formula_name,"Cat's Eyes");
		break;
    case 13:
        lambda = -0.11; alpha = -0.26; beta = 0.19; gamma = -0.059; omega = 0.007;
        ma = 2; shift = 0.5;
        sprintf(formula_name,"Flowers with Ribbons");
		break;
    }
    reset_scale();
    print_formula_name(formula_name);
}

// Assumes that a small (12 pixel) clear area has already been
// reserved in Maxy for the actual display region.  This routine
void print_formula_name(char *thisformula)
{
    int printy = Maxy-textheight(thisformula)+12;
    switch(Fractal_Type) {
    case ICONS:
        outtextxy(0, printy, "Type: Icon         ");
        break;
    case AFFINE:
        outtextxy(0, printy, "Type: Symmetrical  ");
        break;
    case SQUARE_QUILTS:
        outtextxy(0, printy, "Type: Square Quilts");
        break;
    }
    outtextxy(Maxx/2, printy, thisformula);
}


int getkey(void)
{
    int key, lo, hi;
	if(!bioskey(1))
		return(0);              /* No Keypress waiting */
	key = bioskey(0);           /* Find out what key it is */
	lo  = key & 0X00FF;
	hi  = (key & 0XFF00) >> 8;
	return((lo == 0) ? hi + 256 : lo);
}
// For some reason, isalpha is shifting the left cursor key.  The
// following re=#define works, but I'm suspicious, since I had it working
// before.  I did add some more of the standard include files this morning.
// Perhaps that changed something.  At any rate, the scan code for
// LEFTKEY *should* be 331.
#undef LEFTKEY
#define LEFTKEY 75

int dokeypress(int key)
{
    char buf[200];

    printf("");
    if(isalpha(key)) key = toupper(key);

	switch(key)
	{
        case 0 :
             return(0);

        case UPKEY :
            Fractal_Type++;
			if(Fractal_Type > MAX_FRACTAL_TYPES)
				Fractal_Type=1;
            WhichFormula=0;
            next_formula(&WhichFormula,1);
			break;

        case DOWNKEY :
            Fractal_Type--;
			if(Fractal_Type < 1)
				Fractal_Type= MAX_FRACTAL_TYPES;
            WhichFormula=0;
            next_formula(&WhichFormula,1);
			break;

        case LEFTKEY :
            next_formula(&WhichFormula,-1);
			break;

        case RIGHTKEY :
            next_formula(&WhichFormula,1);
			break;

        case HOMEKEY:          // Reset Scale, period
             if(Fractal_Type==SQUARE_QUILTS)
                 nperiod = 3;
             else
                 Scale = 1.0;

             reset_scale();
             break;

        case PGUPKEY:     // Increase image size
             // Note that scale works backwards from what you'd think.
             // This is the way it was in the book, so I just kept it.
            if(Fractal_Type==SQUARE_QUILTS)
                nperiod+=1;
            else
                Scale *= 0.75;

            reset_scale();
            break;

        case PGDNKEY:        // Decrease image size
            if(Fractal_Type==SQUARE_QUILTS && nperiod > 1)
                nperiod-=1;
            else
                Scale *= 1.25;

            reset_scale();
            break;

        case '?': case '/':
            gotoxy(1,1);
            printf("Stats:\n");
            printf("Iters : %u\n", Iters);
            printf("High : %u\n", High_hit);
            if(Iters > 0)
                 printf("Ratio : %g\n", (double)High_hit / Iters);
            break;

        case 'C':
            color_mode();
            break;

		case 27:
			restorecrtmode();
			exit(0);
	}
	return(0);
}

void color_mode(void)
{
    int key = 0;
    char exit_flag = FALSE;

    // Need a map loaded in order to color-cycle.  Load default map
    // if flag not set.  Set flag.
    if(!MapFlag) {
        MapFlag = loadpalette();
    }


    setcolor(Maxcolors-1);             // for rectangle border

    while (!exit_flag) {
        rectangle(0,0,Maxx,Maxy);      // border
        key=getkey();
        if(isalpha(key)) key = toupper(key);

        switch(key)
        {
            case ' ':
                rotate_palette(1,1);      // forward, one-step
                break;

            case UPKEY:
                // cycle continuous, up
                rotate_palette(1,0);
                break;

            case DOWNKEY:
                // cycle continuous, down
                rotate_palette(-1,0);
                break;

            case 'L':
                loadpalette();
                break;

            case ESC:
                exit_flag = TRUE;
                break;
        }
    }
    setcolor(0);
    rectangle(0,0,Maxx,Maxy);
    return;
}

int matherr(struct exception *e)
{
	gotoxy(1,1);
    printf("SPLAT!! : ");
    switch (e->type) {
    case DOMAIN   : printf("DOMAIN error in '%s'\n", e->name);
                    x=0.0234; y=0.1234;
                    break;
    case SING     : printf("SING   error in '%s'\n", e->name); break;
    case OVERFLOW : printf("OVERFLOW error in '%s'\n", e->name); break;
	case UNDERFLOW: printf("UNDERFLOW error in '%s'\n", e->name); break;
    case TLOSS    : printf("TLOSS error in '%s'\n", e->name); break;
    case PLOSS    : printf("PLOSS error in '%s'\n", e->name); break;
    case STACKFAULT: printf("STACKFAULT error in %s \n", e->name); break;
    case EDOM     : printf("EDOM error in '%s'\n", e->name); break;
	case ERANGE   : printf("ERANGE error in '%s'\n", e->name); break;
    default       : printf("Unknown math error in '%s'\n",e->name);
                    return(0);        /* indicate error uncorrected */
	}
    return (1);  /* Indicate the math error was corrected... */
}

int loadpalette(void)
{
	FILE *mapfile;
    int r, g, b;
	int index=0;
	char mapname[81];
	char buf[160];
	unsigned char filename[FILENAME_MAX+1];

    if(!MapFlag) {
    	strcpy(mapname,DEFAULT_MAP);
    }
    else {
        gotoxy(1,1);
        printf("Map: ");
       	scanf("%80s", mapname);
        fflush(stdin);
        gotoxy(1,1);
        printf("                                \n");
    }


	strcpy(filename,mapname);
	if (strchr(filename,'.') == NULL) strcat(filename,".map");
	findpath(filename,buf);

	if((mapfile=fopen(buf,"rt"))==NULL) {
		gotoxy(1,1);
		printf("\07Cannot open input file: %s\n", mapname);
		printf("    Press any key to continue");
		getch();
		gotoxy(1,1);
		printf("                                                         \n");
		printf("                             ");
		gotoxy(1,1);
		return(-1);
	}


	// Read map into new palette
	for( index = 0; index < 256; index++ ) {
		if (fgets(buf,160,mapfile) == NULL)
			break;
		if(sscanf(buf,"%d %d %d", &r, &g, &b)) {
			Palette256[index][0]=r >>2;
			Palette256[index][1]=g >>2;
			Palette256[index][2]=b >>2;
		}
	}
	fclose(mapfile);
	while (index < 256) {
		Palette256[index][0]=40;
		Palette256[index][1]=40;
		Palette256[index][2]=40;
   }
    // Black background.  This is also found in rotate_palette()
    // and may go away.
//	Palette256[0][0]=0;
//  Palette256[0][1]=0;
//	Palette256[0][2]=0;


   // Set new palette
   setvgapalette256(&Palette256);
   MapFlag = TRUE;
   return(1);
}

// Extremely rudimentary color-cycling routine
// Really needs to be done with asm, but at least should be
// rewritten to use pointers.
// This routine is called once per keystroke.
// direction: -1 or 1  ******[ NOT IMPLEMENTED YET! ]*******
// single_step_flag: 0 or 1 (TRUE/FALSE)

void rotate_palette(int direction, int single_step_flag)
{
	int index=0;
	int r,g,b;

    while(!kbhit()) {


    if(direction == 1) {

        r=Palette256[0][0];
        g=Palette256[0][1];
        b=Palette256[0][2];

        for( index = 0; index < Maxcolors; index++ ) {
            Palette256[index][0]=Palette256[index+1][0];
            Palette256[index][1]=Palette256[index+1][1];
            Palette256[index][2]=Palette256[index+1][2];
        }

        Palette256[Maxcolors][0]=r;
        Palette256[Maxcolors][1]=g;
        Palette256[Maxcolors][2]=b;

    } else {

        r=Palette256[Maxcolors][0];
        g=Palette256[Maxcolors][1];
        b=Palette256[Maxcolors][2];
        for( index = Maxcolors; index > 0; index-- ) {
            Palette256[index][0]=Palette256[index-1][0];
            Palette256[index][1]=Palette256[index-1][1];
            Palette256[index][2]=Palette256[index-1][2];
        }
        Palette256[0][0]=r;
        Palette256[0][1]=g;
        Palette256[0][2]=b;
    }

	setvgapalette256(&Palette256);
    if(single_step_flag) return;
    }
    return;
}


void findpath(char *filename, char *fullpathname) /* return full pathnames */
{
	if (filename[0] == '\\'
	  || (filename[0] && filename[1] == ':')) {
	   strcpy(fullpathname,filename);
	   return;
       }
    fullpathname[0] = 0;
    strcpy(fullpathname,searchpath(filename));
    if (fullpathname[0] != 0)
        if (strncmp(&fullpathname[2],"\\\\",2) == 0)
            strcpy(&fullpathname[3],filename);

    return;
}


// Save screen region to imagebuffer
// (Not currently used in SYMM)
int save_screen( void far *Imagebuffer, int x1, int y1, int x2, int y2)
{
    unsigned imsize = getimgsize(x1, y1, x2, y2);
    if((Imagebuffer = (void far *)malloc(imsize)) == (void far *)NULL)
        return(FALSE);

    getimage(x1, x2, y1, y2, Imagebuffer);
    return (TRUE);
}

// Restore saved screen region
void restore_screen(void far *Imagebuffer, int x, int y)
{
    putimage(x, y, Imagebuffer, COPY_PUT);
    farfree(Imagebuffer);
}

unsigned far getimgsize(int x1, int y1, int x2, int y2)
{
    int xwid = x2 - x1;     // Assumes no programmer screwups!!
    int ywid = y2 - y1;

    switch(Maxcolors){
	case 15:
        return (unsigned) ((xwid >> 1)+1) * ywid +4;
	case 255:
        return (unsigned) xwid * ywid +4;
	case 32767:
        return (unsigned) 2 * xwid * ywid +4;
	}
    return(-1);
}

