            page    55,132
            title   "PERUSE - Capture and Replay Screen"

; ---------------------------------------------------------------------
;
; PERUSE 1.00  Copyright (c) 1994, Bob Flanders and Michael Holmes"
; First Published in PC Magazine April 12, 1994
;
; PERUSE - Capture scrolled lines to EMS or XMS memory
;          for future replay.
;
; ---------------------------------------------------------------------

; ---------------------------------------------------------------------
;
; Maintenance Log:
;
; Version    Date   Description                        Who
; -------  -------  ---------------------------------- ----------------
;   1.00            Under Developement                 Flanders/Holmes
;
; ---------------------------------------------------------------------
;
; $lgb$
; 1.0      01/17/94      RVF      Initial Check-in         
; 1.1      01/18/94      RVF      Coding Complete          
; 1.2      01/20/94      RVF      Removed 5 key, start w/ ScrLk, XMS
;                               + driver present but not tested
; 1.3      01/20/94      RVF      Added log request        
; 1.4      01/21/94      RVF      Started implementation of 43/50 line
;                               + support
; 1.5      01/25/94      RVF      43/50 support complete. Still needs
;                               + F1 key and Windows Compatibility.
; 1.6      01/26/94      RVF      Windows Compatibility Complete.
;                               + Still requires F1 screen restore.
; 1.7      01/26/94      RVF      Changed utility's name to PERUSE.
;                               + Still requires F1 screen restore.
; 1.8      01/28/94      MDH      Fixed problems associated with
;                               + changing scrolling directions and
;                               + when returning from F1 Help menu.
; 1.9      01/31/94      RVF      PERUSE Complete. Must make
;                               + compatible with Microsoft Assembler
; 1.10     02/09/94      RVF      Final Beta Complete. Change header
;                               + before release.
; 1.11     02/11/94      RVF      Final Beta #2 Complete. ESC key
;                               + exits. On screen signal shows in
;                               + popup. Change header before release.
; 1.12     02/12/94      RVF      Final Beta #3 Complete. Added
;                               + blinking star during Help. Changed
;                               + help message. Change header before
;                               + release.
; 1.13     03/21/94      RVF      Allows spaces after the memory type
;                               + argument, reduces number of beeps
; $lge$
;
; ---------------------------------------------------------------------
; $nokeywords$


; -------------------------------------------------------------------
;
; PERUSE uses a 1k record structure that contains as many compressed
; lines as will fit. As lines scroll off the screen, they are added
; to the end of the current record.
;
; When the record fills, it is written to the buffer memory in EMS or
; XMS. Finally, when the buffer memory fills, the oldest record in the
; buffer is overlayed with the new information. When this occurs, the
; oldest lines are lost.
;
; In any case, the records proceed from the lowest address to the
; highest address. The memory drivers serve to retrieve or write the
; records as needed. The following structure defines the format of a
; record.
;
; Terminology:
;
;     Line    - A single line from the video buffer. Each line
;               comprises 80 characters and 80 attribute bytes.
;
;     Record  - A series of compressed lines. Lines never span
;               records. Record 0 is the oldest record and record
;               max-1 is the most recent.
;
;     Buffer  - The EMS or XMS memory that holds a series of records.
;               With the exception of the interface routine, PERUSE
;               "forgets" the type of memory selected for the buffer.
;
; -------------------------------------------------------------------


; -------------------------------------------------------------------
; Record header
; -------------------------------------------------------------------

Rec         struc

RecNext     dw      0                       ; next byte in record
RecCount    dw      0                       ; lines in record
RecData     db      1020 dup (?)            ; space for data

Rec         ends

RecSize     equ     1024                    ; record size



; -------------------------------------------------------------------
;
; As each line scrolls off the screen, PERUSE compresses and copies
; the lines to the scroll record. For speed purposes, PERUSE
; compresses the data using a simple Run Length Encoding (RLE)
; scheme.
;
; Actually, there are two possibilities:
;
;   1. Repeating runs of characters and/or attributes.
;
;      If there are three or more repeating characters or attributes,
;      PERUSE uses a sentinel bit to indicate that compressed data
;      follows. The next byte contains the repeating value. PERUSE
;      always encodes the attributes first followed by the characters.
;
;   2. Non-repeating runs of characters or attributes.
;
;      When PERUSE encounters a non-repeating run of characters or
;      attributes, it creates a sentinel indicating the length of
;      the run directly followed by the characters in the run.
;
; The format of a line is:
;
;       line length
;       character runs
;       attribute runs
;       line length (repeated for backward scrolling.)
;
; Example:
;
;      A line containing all blanks with attribute 07 would compress
;      into the following string of bytes:
;
;      06 D0 20 D0 07 06
;      -- -- -- -- -- --
;       |  |  |  |  |  |
;       |  |  |  |  |  +- Number of bytes in compressed line
;       |  |  |  |  |
;       |  |  |  |  +---- Repeat Attribute
;       |  |  |  |
;       |  |  |  +------- x... .... (80h) Repeat flag
;       |  |  |           .xxx xxxx (50h) Repeat count - 80 decimal
;       |  |  |
;       |  |  +---------- Repeat Character (blank)
;       |  |
;       |  +------------- x... .... (80h) Repeat flag
;       |                 .xxx xxxx (50h) Repeat count - 80 decimal
;       |
;       +---------------- Number of bytes in compressed line
;
; -------------------------------------------------------------------


; -------------------------------------------------------------------
; Line header
; -------------------------------------------------------------------

Line        struc

LineLen     db      0                       ; length of this record
LineData    db      0                       ; first data byte

Line        ends


; --------------------------------------------------------------------
;
; PERUSE encodes the character runs using a sentinel bit. This
; bit indicates if the run is repeating or non-repeating
; and a count of the bytes in the run. For repeating runs, the
; sentinel is followed by the repeating byte. For non-repeating runs,
; the sentinel is followed by the number of bytes indicated.
;
; --------------------------------------------------------------------

SentRpt     equ     80h                     ; 1... .... Repeating
                                            ; 0... .... Non-Repeating
SentCnt     equ     07fh                    ; .xxx xxxx Count



; -------------------------------------------------------------------
;  BIOS communications area definitions
; -------------------------------------------------------------------

BIOSCom     segment at 40h

            org     49h
VideoMode   db      ?                       ; current video mode

            org     50h
VideoPos    db      ?                       ; current column
VideoLine   db      ?                       ; current line (page 0)

            org     62h
VideoPage   db      ?                       ; current page displayed

BIOSCom     ends



; -------------------------------------------------------------------
;
;  Main entry point
;
; -------------------------------------------------------------------

            IFDEF   Borland                 ; group for TASM assemblies
CGroup      group   MainLine, EmsInt, XmsInt
            ENDIF

MainLine    segment para public 'code'      ; mainline
            assume  cs:MainLine, ds:MainLine, es:MainLine, ss:MainLine
            org     100h

Begin:      jmp     Start                   ; start processing



; --------------------------
;
;   interrupt vector control
;
; --------------------------

IntBlkCnt   equ     4                       ; number of int routines
IntBlkLen   equ     7                       ; length of blocks
IntBlk1     equ     offset OldInt08         ; first to fill in

OldInt08    dd      0                       ; old cs:ip
            db      08h                     ; interrupt to take over
            dw      offset Int08            ; new interrupt offset

OldInt09    dd      0                       ; old cs:ip
            db      09h                     ; interrupt to take over
            dw      offset Int09            ; new interrupt offset

OldInt10    dd      0                       ; old cs:ip
            db      10h                     ; interrupt to take over
            dw      offset Int10            ; new interrupt offset

OldInt28    dd      0                       ; old cs:ip
            db      28h                     ; interrupt to take over
            dw      offset Int28            ; new interrupt offset

ScrFlg      db      0                       ; flags
ScrFlgRes   equ     80h                     ; 1... .... Already resident
ScrFlgStr   equ     40h                     ; .1.. .... Star displayed

StarCnt     equ     9                       ; ticks for star character
BeepAt      equ     9                       ; ticks for beep 

StarEqu     equ     15                      ; Star character ''
StarChr     db      StarEqu                 ; Work area for star character

TimeCnt     dw      0                       ; Time Counter
BeepCnt     dw      BeepAt                  ; Time since last beep

Int10Flg    db      0                       ; Int 10 usage counter
Int28Flg    db      0                       ; Int 28 use flag
ResSeg      dw      ?                       ; resident segment
MemType     db      ?                       ; type of memory requested

MemRtn      dd      0                       ; memory routine
MemRtnLen   dw      0                       ; ..length of routine
MemRtnSeg   equ     word ptr MemRtn+2       ; ..segment of routine

InDosFlg    dd      ?                       ; address of InDos Flag

InWin       db      0                       ; Windows flag @ startup

WorkLine    label   byte                    ; Start of work area
            Line    <>                      ; Build line header
            db      164 dup (?)             ; Work area for data

NxtAvail    dw      MemHndlr                ; next available byte

LineCnt     db      0                       ; lines being scrolled
LinesScr    db      25                      ; lines per screen
LinesScrM1  db      24                      ; lines per screen - 1
LastScrLine dw      24 * 160                ; last line in screen buffer
AppMode     db      0                       ; appropriate mode flag

VideoSeg    dw      0                       ; video buffer segment
BIOSSeg     dw      40h                     ; Bios Segment

TsrActive   db      0                       ; tsr active flag
TsrReq      db      0                       ; tsr request flag

WorkHdr     db      0                       ; header record flag
                                            ;  -1 = TOF
                                            ;   0 = in file
                                            ;   1 = EOF reached
                                            ;   2 = ..and displayed

WorkDir     db      0                       ; direction flag
                                            ;   0 = up, 1 = down
WorkOffset  dw      0                       ; processing point in record
WorkRNbr    dw      0                       ; last record number processed
WorkMNbr    dw      0                       ; max record number

WorkTOF     db      00                      ; top of scroll buffer message
            db      1, "", 80h + 30, "", 17, "  Top of Buffer  "
            db      80h + 31, "", 1, "", 0d0h
WorkTOFa    db      1fh

WorkEOF     db      00                      ; end of scroll buffer message
            db      1, "", 80h + 30, "", 17, "  End of Buffer  "
            db      80h + 31, "", 1, "", 0d0h
WorkEOFa    db      1fh


   Comment  *           Layout of Help Message
                        ----------------------

 PERUSE v1.00  Keys:  PgUp PgDn Home End Esc ScrLock  Pop-Up Indicator--> 


           Note: The help message is RLE compressed to save space.

   EndComment *            

WorkHlp1    db      80, " PERUSE v1.00  Keys:  PgUp PgDn Home End "
            db          "Esc ScrLock  Pop-Up Indicator--> ", 0d0h
WorkHlp1a   db      1fh

WorkHlp2    db      1, "", 80h + 78, "", 1, "", 0d0h
WorkHlp2a   db      1fh

KbFlag      dd      00400017h               ; BIOS kbd shift flags
KbFlagSL    equ     10h                     ; scroll lock flag
KbFlagC     equ     NOT KbFlagSL            ; scroll lock led clear



; ----------------------------------------------------------------------
; Key Command Table - Used by TSR command processor
; ----------------------------------------------------------------------

CmdTable    label   byte
            db      4fh                     ; end - must be first
            dw      CmdEnd

            db      47h                     ; home
            dw      CmdHome

            db      48h                     ; up arrow
            dw      CmdUp

            db      49h                     ; page up
            dw      CmdPgUp

            db      50h                     ; down arrow
            dw      CmdDown

            db      51h                     ; page down
            dw      CmdPgDn

            db      3bh                     ; f1 help
            dw      CmdHelp

            db      1                       ; esc
            dw      CmdEsc

            db      0ffh                    ; end of table marker



; ----------------------------------------------------------------------
;
;   Int08 - Interrupt 8 (timer) processor
;
;   This routine intercepts the timer interrupt (08h) and checks
;   if the user has requested the TSR. If so, control is passed to the
;   TSR routine.
;
; ----------------------------------------------------------------------

Int08       proc                            ; timer - check for TSR

            pushf                           ; save flags
            call    cs:OldInt08             ; ..handle the interrupt

            inc     cs:TimeCnt              ; increment time counter
            inc     cs:BeepCnt              ; .. and the beep timer

            call    Tsr                     ; Otherwise .. go TSR now

            iret                            ; return to caller

Int08       endp



; ----------------------------------------------------------------------
;
;   Int09 - Check for the hot key being pressed.
;
;   This routine intercepts the hardware keyboard interrupt (09h) and
;   checks to see if the scroll lock key has been pressed. If so,
;   it simply sets a flag indicating that the TSR has been requested
;   and continues processing.  The popup occurs at the next timer tick.
;
; ----------------------------------------------------------------------

Int09       proc
            pushf                           ; call old int 9
            call    cs:OldInt09             ; ..to finish reading key

            sti                             ; allow interrupts
            push    bx                      ; save work registers
            push    es                      ; . . . .

            les     bx, cs:KbFlag           ; es:bx -> kbd shift flag
            mov     bl, es:[bx]             ; bl = shift flags
            and     bl, KbFlagSL            ; bl = scroll lock flag
            mov     cs:TsrReq, bl           ; set TSR request if made

            pop     es                      ; restore registers
            pop     bx                      ; . . . .
            iret                            ; go back where we came from

Int09       endp



; ----------------------------------------------------------------------
;
;   Int10 - Interrupt 10 processor
;
;   This routine intercepts the BIOS display interrupt 10h and checks for
;   a scroll request (function 06) or a write TTY request (function 0Eh).
;   If either of these requests causes data to be scrolled off the screen,
;   the line or lines are placed in the scrollback buffer. Control is then
;   then passed to the BIOS. Proper checks are made to prevent re-entry or
;   saving of scrolled lines when the TSR is active.
;
;   Additionally, this routine notes calls to the BIOS change video mode
;   function to determine if the new mode is approprate for capturing
;   or redisplaying scrolled information.
;
; ----------------------------------------------------------------------

Int10       proc    far                     ; check for scroll requests

            cmp     ah, 0                   ; q. mode change?
            jne     int10_10                ; a. no .. continue checking

            call    RunInt10                ; run mode change thru BIOS

            call    WinCheck                ; q. Win status change?
            jc      Int10Jmp                ; a. yes .. don't check now

            call    ModeChange              ; check new parameters

            iret                            ; ..return to caller

Int10_10:   cmp     ax, 0B0BFh              ; q. PERUSE active query?
            jne     Int10_20                ; a. no .. continue checking

;
;  Show PERUSE is loaded
;

            push    cs                      ; push our segment
            pop     dx                      ; dx = our segment
            mov     ax, 0FB0Bh              ; ax = our signature
            sub     al, cs:InWin            ; modify with InWin flag

            iret                            ; ..return to caller

Int10_20:   cmp     cs:AppMode, 0           ; q. appropriate video mode?
            je      Int10Jmp                ; a. no .. go to old BIOS

            call    WinCheck                ; q. windows status change
            jc      Int10Jmp                ; a. yes .. go to old bios

            cmp     cs:TsrActive, 0         ; q. TSR active now?
            jne     Int10Jmp                ; a. yes.. don't check

            cmp     cs:Int10Flg, 0          ; q. Int 10 already in use?
            jne     Int10Jmp                ; a. yes .. go to old int 10

            push    ds                      ; save ds
            mov     ds, cs:BIOSSeg          ; ds -> BIOS segment
            assume  ds:BIOSCom              ; ..tell assembler about it
            cmp     VideoPage, 0            ; q. page 0?
            pop     ds                      ; ..restore ds
            assume  ds:MainLine             ; ..tell assembler about it
            jne     Int10Jmp                ; a. no .. go to old Int 10

            cmp     ah, 06h                 ; q. scroll up request?
            je      Int10ChkS               ; a. yes .. check it

            cmp     ah, 0Eh                 ; q. write tty request?
            je      Int10ChkW               ; a. yes .. check it

Int10Jmp:   jmp     cs:OldInt10             ; go directly to original Int10

;
;  Test scroll request (ah = 06)
;

Int10ChkS:  cmp     cx, 0                   ; q. top line?
            jne     Int10Jmp                ; a. no .. skip this scroll

            cmp     dl, 80-1                ; q. whole line?
            jne     Int10Jmp                ; a. no .. skip this scroll

            mov     cs:LineCnt, al          ; save number of lines
            jmp     Int10Call               ; ..scroll the lines

;
; Test Write TTY request
;

Int10ChkW:  push    ds                      ; save registers

            mov     ds, cs:BIOSSeg          ; ds -> BIOS Com segment
            assume  ds:BIOSCom              ; ..tell the assembler

            push    ax                      ; save ax
            mov     al, VideoLine           ; al = current line

            cmp     al, cs:LinesScrM1       ; q. on last line?
            pop     ax                      ; ..restore ax
            jnb     Int10ChkW2              ; a. yes .. continue

Int10ChkW1: pop     ds                      ; restore regs

            jmp     Int10Jmp                ; Run old int 10


Int10ChkW2: cmp     al, 0dh                 ; q. write carriage return?
            je      Int10ChkW1              ; a. yes .. don't check

            cmp     al, 07                  ; q. bell?
            je      Int10ChkW1              ; a. yes .. don't check

            cmp     al, 08                  ; q. backspace?
            je      Int10ChkW1              ; a. yes .. don't check

            cmp     al, 0ah                 ; q. line feed?
            je      Int10ChkW9              ; a. yes .. save a line

            cmp     VideoPos, 80-1          ; q. cursor in last position?
            jne     Int10ChkW1              ; a. no .. go to old int 10

Int10ChkW9: pop     ds                      ; restore ds
            mov     cs:LineCnt, 1           ; save one line

Int10Call:  call    SaveLines               ; save reqested lines

            call    RunInt10                ; call Video BIOS
            iret

            assume  ds:MainLine             ; tell assembler about ds

Int10       endp



; ----------------------------------------------------------------------
;
;   Int28 - Interrupt 28 - Dos keyboard busy routine
;
;   This routine intercepts the DOS keyboard busy interrupt and checks
;   if the user has requested the TSR. If so, control is passed to the
;   TSR routine. Ultimately, control is passed to the next interupt 28
;   handler
;
; ----------------------------------------------------------------------

Int28       proc                            ; timer - check for TSR

            mov     cs:Int28Flg, 1          ; show we're in int 28
            call    Tsr                     ; check if tsr needed
            mov     cs:Int28Flg, 0          ; ..and that we're done

            jmp     cs:OldInt28             ; let next 28 routine run

Int28       endp



; ----------------------------------------------------------------------
;
;   RunInt10 - Invoke original Int10
;
;   This routine invokes the original Video BIOS handler.
;
;   ENTRY: Normal Video BIOS registers
;
;   EXIT: Normal Video BIOS return
;
; ----------------------------------------------------------------------

RunInt10    proc

            pushf                           ; push flags
            inc     cs:Int10Flg             ; ..increment the call flag
            call    cs:OldInt10             ; ..call the old int 10
            dec     cs:Int10Flg             ; ..release int 10

            ret                             ; return to caller

RunInt10    endp



; ----------------------------------------------------------------------
;
;   ModeChange - Check after BIOS mode change
;
;   ModeChange monitors calls to BIOS change mode requests and notes
;   the screen metrics resulting from the new mode. Applicable modes
;   are determined as follows:
;
;   - If the current mode is 2, 3, or 7, the screen is assumed to be in
;     an appropriate 80x25 video mode.
;
;   - If the screen width is not 80, it is not an applicable mode.
;
;   - The routine retrieves the character occupying the first byte of
;     video memory (B000:0 for mode 7, b800:0 for all others) by a direct
;     memory read and by Video BIOS get character. If the characters do
;     not match, the adapter is probably in a graphics mode, and lines
;     will not be captured nor will rediplay requests be honored. The
;     video is not in an appropriate mode.
;
;   - The number of lines on the screen are retrieved and saved.
;
;   ENTRY: No special entry parameters.
;
;   EXIT: The number of lines on screen are recorded.
;         AppMode is set to true if the new mode will work with PERUSE.
;
; ----------------------------------------------------------------------

ModeChange  proc                            ; check mode compatibility
            push    ax
            push    bx
            push    cx
            push    dx
            push    ds
            push    es

            push    cs                      ; save our segment
            pop     ds                      ; ds -> our segment
            mov     es, BIOSSeg             ; es -> BIOS segment
            assume  ds:MainLine, es:BIOSCom ; ..tell the assembler

            mov     VideoSeg, 0b800h        ; set default video segment
            mov     LinesScr, 25            ; ..lines per screen
            mov     LinesScrM1, 24          ; ..lines per screen - 1
            mov     LastScrLine, 24 * 160   ; ..last line offset
            mov     al, 1fh                 ; al = color attribute
            mov     WorkTOFa, al            ; ..set up top
            mov     WorkEOFa, al            ; ..and bottom
            mov     WorkHlp1a, al           ; ..set up help ..
            mov     WorkHlp2a, al           ; ..menu lines
            mov     AppMode, 1              ; ..show appropriate mode

            cmp     es:VideoMode, 2         ; q. mode 2 set?
            je      ModeCh90                ; a. yes .. return to caller

            cmp     es:VideoMode, 3         ; q. mode 3 set?
            je      ModeCh90                ; a. yes .. return to caller

            cmp     es:VideoMode, 7         ; q. mode 7 set?
            jne     ModeCh10                ; a. no .. continue

            mov     VideoSeg, 0b000h        ; reset default video segment
            mov     al, 7                   ; al = black and white attr
            mov     WorkTOFa, al            ; ..set up top
            mov     WorkEOFa, al            ; ..and bottom
            mov     WorkHlp1a, al           ; ..set up help ..
            mov     WorkHlp2a, al           ; ..menu lines
            jmp     short ModeCh90          ; ..and return to caller

ModeCh10:   call    CmpChar0                ; q. match on first character?
            jc      ModeCh80                ; a. no .. inappropriate mode

            mov     ah, 0fh                 ; ah = return video status
            call    RunInt10                ; ..get the current status

            cmp     ah, 80                  ; q. 80 column mode?
            jne     ModeCh80                ; a. no .. inappropriate mode

            push    es                      ; save registers
            push    bp

            mov     ax, 1130h               ; ax = get char gen info
            xor     bh, bh                  ; bh = get int 1fh vector
            call    RunInt10                ; ..get the info

            pop     bp                      ; restore changed registers
            pop     es

            mov     al, dl                  ; al = lines per screen - 1
            mov     LinesScrM1, dl          ; save lines per screen - 1
            inc     dl                      ; dl = lines per screen
            mov     LinesScr, dl            ; save lines per screen

            xor     ah, ah                  ; clear upper bits
            mov     dl, 160                 ; dl = line length multiplier
            mul     dl                      ; ax = last line offset
            mov     LastScrLine, ax         ; ..save the offset
            jmp     short ModeCh90          ; ..return to caller

ModeCh80:   mov     AppMode, 0              ; mode is not appropriate

ModeCh90:   pop     es                      ; restore registers
            pop     ds
            pop     dx
            pop     cx
            pop     bx
            pop     ax

            ret

ModeChange  endp



; ----------------------------------------------------------------------
;
;   WinCheck - Check if in same mode as when installed
;
;   This routine check if Windows has been started since PERUSE was
;   installed. If so, this routine returns with the carry flag set.
;
;   ENTRY: InWin set to installation value
;
;   EXIT: carry set if windows installed since PERUSE was installed.
;
; ----------------------------------------------------------------------

WinCheck    proc                            ; check if windows installed     

            push    ax                      ; save ax     
            
            mov     ax, 1600h               ; ax = get installed flag
            int     2fh                     ; hey bud, Windows in?

            cmp     al, cs:InWin            ; q. Win status changed?
            pop     ax                      ; .. restore ax
            je      WinCheck90              ; a. no .. continue

            stc                             ; changed
            mov     cs:AppMode, 0           ; .. inappropriate mode
            ret                             ; return to caller

WinCheck90: clc                             ; no change      
            ret                             ; return to caller

WinCheck    endp



; ----------------------------------------------------------------------
;
;   CmpChar0 - Compare first characters
;
;   This routine compares the character in the upper-left corner to the
;   character returned by BIOS. If the characters do not match, carry is
;   set showing the screen is in a graphics mode and the lines should
;   not be saved.
;
;   ENTRY: VideoSet must contain segment of current video buffer.
;
;   EXIT:  Carry indicates inappropriate video mode for PERUSE.
;
; ----------------------------------------------------------------------

CmpChar0    proc                            ; compare first video char

            push    ax                      ; save registers
            push    bx
            push    cx
            push    dx
            push    ds

            mov     ah, 3                   ; ah = get cursor position
            xor     bh, bh                  ; ..on page 0
            int     10h                     ; ..ask BIOS about it

            push    dx                      ; save cursor position

            mov     ah, 2                   ; ah = set cursor position
            xor     dx, dx                  ; dx = position (0,0)
            int     10h                     ; ..set cursor's postion

            mov     ah, 8                   ; ah = get char & attribute
            int     10h                     ; ..get the character

            pop     dx                      ; dx = original cursor pos
            push    ax                      ; save retrieved char & attr

            mov     ah, 2                   ; ah = set cursor position
            int     10h                     ; ..reset the cursor

            mov     ds, cs:VideoSeg         ; ds -> video segment
            pop     ax                      ; ax = char & attr

            cmp     word ptr ds:[0], ax     ; q. same char/attribute?

            pop     ds                      ; restore registers
            pop     dx
            pop     cx
            pop     bx
            pop     ax

            je      CmpChar90               ; a. yes .. return

            stc                             ; else .. show not equal

CmpChar90:  ret                             ; return to caller

CmpChar0    endp



; ----------------------------------------------------------------------
;
;   SaveLines - Save lines about to scroll off the screen.
;
;   This routine compresses and saves the lines before they roll off the
;   screen.
;
;   ENTRY: VideoSeg must contain the segment of the current video buffer.
;          LineCnt must contain the number of lines to save.
;
;   EXIT:  No registers changed.
;
; ----------------------------------------------------------------------

SaveLines   proc                            ; save requested nbr of lines

            push    ax                      ; save registers
            push    bx
            push    cx
            push    dx
            push    si
            push    di
            push    es
            push    ds

            call    WinCheck                ; q. Windows status change?
            jc      SaveLines9              ; a. yes .. don't save

            mov     ds, cs:VideoSeg         ; ds = video segment

            push    cs                      ; save our segment
            pop     es                      ; es = our segment

            mov     si, 0                   ; si -> first line
            mov     cl, cs:LineCnt          ; cx = number of lines to save
            xor     ch, ch                  ; ..upper bits off

            cmp     cl, cs:LinesScr         ; q. too many lines?
            ja      SaveLines0              ; a. yes .. use max

            or      cl, cl                  ; q. any lines?
            jnz     SaveLines1              ; a. yes .. continue

SaveLines0: mov     cl, cs:LinesScr         ; cl = max lines

SaveLines1: push    cs                      ; save our segment
            pop     es                      ; es -> our segment
            mov     di, offset WorkLine     ; di -> WorkLine area
            call    Compress                ; compress a video line

            call    Line2Rec                ; Send the line to the record

            add     si, 160                 ; si -> next line
            loop    SaveLines1              ; save the next line

SaveLines9: pop     ds                      ; restore registers
            pop     es
            pop     di
            pop     si
            pop     dx
            pop     cx
            pop     bx
            pop     ax
            ret                             ; return to caller

SaveLines   endp



; ----------------------------------------------------------------------
;
;   Line2Rec - Add a line to the record
;
;   This routine adds the line in WorkLine to the current record. If the
;   record will not hold this line, Line2Rec wries the record to scroll
;   memory and starts a new record.
;
;   ENTRY: WorkLine contains the line to save
;          WorkRec  contains the current record being built.
;
;   EXIT:  No registers changed.
;
; ----------------------------------------------------------------------

Line2Rec    proc                            ; save a line in the record

            push    cx                      ; save registers
            push    dx
            push    si
            push    di
            push    ds
            push    es

            push    cs                      ; save our segment
            pop     ds                      ; ds = our segment

            push    cs                      ; same for ..
            pop     es                      ; es = our segment

            mov     di, WorkRec.RecNext     ; di -> next available byte
            mov     si, RecSize             ; si = record length
            sub     si, di                  ; si = bytes remaining

            mov     cl, WorkLine.LineLen    ; cl = length of line
            xor     ch, ch                  ; ..upper bits off

            cmp     cx, si                  ; q. enough space left?
            jna     Line2Rec10              ; a. yes .. move it in

            call    Rec2Buf                 ; else .. save, start new buf
            mov     di, WorkRec.RecNext     ; bx -> next byte

Line2Rec10: add     WorkRec.RecNext, cx     ; point at next available byt

            lea     di, WorkRec[di]         ; es:di -> output position
            lea     si, WorkLine            ; ds:si -> input line
            cld                             ; incrementally
       rep  movsb                           ; move line to record

            inc     WorkRec.RecCount        ; Increment line count

            pop     es                      ; restore registers
            pop     ds
            pop     di
            pop     si
            pop     dx
            pop     cx
            ret                             ; return to caller

Line2Rec    endp



; ----------------------------------------------------------------------
;
;   Rec2Buf - Move a record to memory scroll buffer
;
;   This routine adds a completed record to the memory scroll buffer.
;
;   ENTRY: WorkRec  contains the current record being built.
;
;   EXIT:  No registers changed.
;
; ----------------------------------------------------------------------

Rec2Buf     proc                            ; move record to buffer memory

            push    ax                      ; save registers
            push    bx

            mov     ax, 02h                 ; ah = write record to buffer
            mov     bx, offset WorkRec      ; ds:bx -> work record
            call    MemRtn                  ; ..tell the memory routine

            call    ClearRec                ; re-init the record

            pop     bx                      ; restore registers
            pop     ax

            ret                             ; return to caller

Rec2Buf     endp



; ----------------------------------------------------------------------
;
;   ClearRec - Initialize the record buffer
;
;   This routine clears the record buffer and sets the RecNext field to
;   the value 4.
;
;   ENTRY: No special entry conditions
;
;   EXIT:  WorkRec initialized
;
; ----------------------------------------------------------------------

ClearRec    proc                            ; move record to buffer memory

            push    ax                      ; save registers
            push    cx
            push    di

            lea     di, WorkRec             ; di -> work record area
            mov     ax, 0                   ; ax = value to store
            mov     cx, RecSize/2           ; cx = words to store
       rep  stosw                           ; clear the buffer

            mov     WorkRec.RecNext, 4      ; reset next byte pointer

            pop     di                      ; restore registers
            pop     cx
            pop     ax

            ret                             ; return to caller

ClearRec    endp



; ----------------------------------------------------------------------
;
;  Compress - Compress a line from the video buffer
;
;  This routine compresses a single line of 80 character (160 bytes)
;  into the requested area.
;
;  Entry:   DS:SI -> line to compress
;           ES:DI -> output area
;
;  Exit:    AL    =  length of compressed line
;
; ----------------------------------------------------------------------

Compress    proc
            push    bx                      ; save registers
            push    si

            mov     bx, di                  ; bx -> output area

            mov     es:[di].LineLen, 1      ; reset length of line
            lea     di, [di].LineData       ; es:di -> output area

            push    si                      ; save line pointer
            call    CmpData                 ; ..compress the characters
            pop     si                      ; restore line pointer


            add     es:[bx].LineLen, dh     ; add to length of line

            inc     si                      ; si -> attribute
            call    CmpData                 ; ..compress the attributes

            add     es:[bx].LineLen, dh     ; add to length of line

            inc     es:[bx].LineLen         ; Increment the line length
            mov     al, es:[bx].LineLen     ; al = length of the line
            stosb                           ; ..append length to line

            pop     si                      ; restore registers
            pop     bx
            ret                             ; ..return to caller

Compress    endp



; ----------------------------------------------------------------------
;
;  CmpData - Compress 80 alternating bytes
;
;  This routine scans the requested line and compresses the data.
;
;  Entry: DS:SI     -> first byte of line to scan (scans every other byte)
;         ES:DI     -> next output byte
;
;  Exit:  ES:DI     -> next output byte to use
;         DH        =  bytes added to output string
;
; ----------------------------------------------------------------------

CmpData     proc
            push    bx                      ; save registers
            push    si
            push    cx

            mov     cx, 80                  ; max is 80 bytes per line
            mov     bx, si                  ; bx -> starting byte
            xor     dx, dx                  ; reset non-repeat and o/p len

CmpData10:  mov     ah, byte ptr [bx]       ; ah = byte to scan
            xor     al, al                  ; al = count

CmpData20:  add     bx, 2                   ; bx -> next byte to check
            inc     al                      ; ..and increment the count

            cmp     ah, [bx]                ; q. same byte?
            loope   CmpData20               ; a. yes .. continue

            cmp     al, 3                   ; q. RLE string?
            jb      CmpData30               ; a. no.. treat as normal string

            or      dl, dl                  ; q. any non-RLE characters?
            jz      CmpData25               ; a. no .. continue

            add     dh, dl                  ; add in non-repeat length
            inc     dh                      ; ..and count byte

            xchg    dl, cl                  ; cx = count, dl = index
            mov     es:[di], cl             ; save the count
            inc     di                      ; di -> next byte

CmpData23:  movsb                           ; move a byte
            inc     si                      ; bx -> next byte
            loop    CmpData23               ; ..loop 'til string moved

            xchg    dl, cl                  ; restore index, reset count

;
; String repeats .. encode as RLE
;

CmpData25:  or      al, SentRpt             ; Set repeat flag
            stosw                           ; Save the RLE value
            add     dh, 2                   ; add to output length
            mov     si, bx                  ; si -> next byte

            or      cl, cl                  ; q. end of string?
            jz      CmpData90               ; a. yes .. exit now
            jmp     short CmpData10         ; else ... get next byte

;
; Non-repeating string
;

CmpData30:  add     dl, al                  ; add in the non-repeat count

            or      cl, cl                  ; q. end of string?
            jnz     short CmpData10         ; get next byte

;
; End of line
;

            or      dl, dl                  ; q. any non-repeat characters?
            jz      CmpData90               ; a. no .. continue

            add     dh, dl                  ; add in non-repeat length
            inc     dh                      ; ..and count byte

            xchg    dl, cl                  ; cx = count, dl = index
            mov     es:[di], cl             ; save the count
            inc     di                      ; di -> next byte

CmpData80:  movsb                           ; move a byte
            inc     si                      ; bx -> next byte
            loop    CmpData80               ; ..loop 'til string moved

            xchg    dl, cl                  ; restore index, reset count

CmpData90:  pop     cx                      ; restore regs
            pop     si
            pop     bx

            ret                             ; return to caller

CmpData     Endp



; ----------------------------------------------------------------------
;
;  Expand - Reconstitute compressed data
;
;  This routine expands a single line of 80 character (160 bytes)
;  into the requested area
;
;  Entry:   DS:SI -> Line Structure to decompress
;           ES:DI -> output area
;
;  Exit:    DS:SI -> byte after compressed data
;
; ----------------------------------------------------------------------

Expand      proc

            push    di                      ; save register

            call    ExpData                 ; expand the characters

            inc     di                      ; di -> first attribute byte
            call    ExpData                 ; expand the attributes

            pop     di                      ; restore register
            ret

Expand      endp



; ----------------------------------------------------------------------
;
;  ExpData - Reconstitute 80 alternating bytes of data
;
;  This routine expands 80 character or attribute bytes
;  into the requested area
;
;  Entry:   DS:SI -> data to decompress
;           ES:DI -> output area
;
;  Exit:    DS:SI -> byte after compressed data
;
; ----------------------------------------------------------------------

ExpData     proc

            push    ax                      ; save registers
            push    cx
            push    dx
            push    di

            xor     dl, dl                  ; dl = output count

ExpData10:  mov     cx, [si]                ; get the sentinel & char

            test    cl, SentRpt             ; q. repeater?
            jz      ExpData50               ; a. no .. non-repeating

            mov     al, ch                  ; al = character
            and     cx, SentCnt             ; cx = length of string
            add     dl, cl                  ; dl = output length

ExpData20:  stosb                           ; add byte to output area
            inc     di                      ; ..skip alternating byte
            loop    ExpData20               ; ..loop 'til done

            add     si, 2                   ; si -> next sentinel

            cmp     dl, 80                  ; q. 80 bytes yet?
            jb      ExpData10               ; a. no .. get next sentinel
            jmp     short ExpData90         ; else .. return to caller

ExpData50:  inc     si                      ; si -> first byte to move
            and     cx, SentCnt             ; cx = bytes to move
            add     dl, cl                  ; dl = output length

ExpData60:  movsb                           ; move a byte
            inc     di                      ; ..skip a byte
            loop    ExpData60               ; ..'til all moved

            cmp     dl, 80                  ; q. 80 bytes yet?
            jb      ExpData10               ; a. no .. get next sentinel

ExpData90:  pop     di                      ; restore registers
            pop     dx
            pop     cx
            pop     ax
            ret                             ; return to caller

ExpData     endp



; ----------------------------------------------------------------------
;
;   Tsr - Try to pop up the scroll back screen
;
;   If a pop-up occurs, this routine saves everything upon entry and
;   restores everything before exiting.
;
; ----------------------------------------------------------------------

Tsr         proc

            cmp     cs:TsrActive, 0         ; q. TSR Active?
            jne     Tsr00                   ; a. yes .. don't test

            cmp     cs:TsrReq, 0            ; q. Requested?
            jne     Tsr02                   ; a. yes .. try and start it

Tsr00:      ret

Tsr02:      push    ax                      ; save registers
            push    bx
            push    cx
            push    dx
            push    si
            push    di
            push    bp
            push    ds
            push    es
            pushf                           ; save flags

            mov     ax, cs                  ; ax = our segment
            mov     ds, ax                  ; ds -> our segment
            assume  ds:MainLine             ; tell assembler

            call    WinCheck                ; q. Win status change?
            jnc     Tsr03                   ; a. yes .. don't check

            mov     TsrReq, 0               ; set off TSR requeted
            jmp     Tsr99                   ; Can't use previous instance

Tsr03:      cmp     Int28Flg, 0             ; q. called from int 28?
            jne     Tsr04                   ; a. yes .. skip test

            les     bx, InDosFlg            ; es:bx -> InDosFlg

            cmp     byte ptr es:[bx], 0     ; q. DOS in use?
            je      Tsr04                   ; a. no .. pop up ok
            jmp     Tsr99                   ; else .. don't even try

Tsr04:      mov     es, ax                  ; es -> our segment
            assume  es:MainLine             ; tell assembler

            mov     TsrActive, 1            ; Show the TSR active

            cld                             ; set the direction

            cmp     AppMode, 0              ; q. valid mode?
            je      Tsr05                   ; a. no .. reject the request

            push    ds                      ; save ds
            mov     ds, BIOSSeg             ; ds -> BIOS com area
            assume  ds:BIOSCom              ; ..tell assembler about it
            cmp     VideoPage, 0            ; q. Page 0?
            pop     ds                      ; ..restore ds
            assume  ds:MainLine             ; ..tell assembler about it
            jne     Tsr05                   ; a. no .. stop now.

            call    CmpChar0                ; q. first chars match?
            jnc     Tsr10                   ; a. yes .. continue

Tsr05:      mov     ax, 0e07h               ; ax = write a beep
            call    RunInt10                ; call BIOS
            jmp     short Tsr95             ; ..and return to caller

Tsr10:      cmp     WorkRec.RecCount, 0     ; q. any records in buffer?
            je      Tsr15                   ; a. no .. don't write it

            call    Rec2Buf                 ; else .. save current record

Tsr15:      mov     ax, 5                   ; ah = save record pointers
            call    MemRtn                  ; ..call the memory routine

            mov     al, LinesScr            ; al = max lines on screen
            mov     LineCnt, al             ; save a screenful
            call    SaveLines               ; ..in memory file

            cmp     WorkRec.RecCount, 0     ; q. any records in buffer?
            je      Tsr40                   ; a. no .. don't write it

            call    Rec2Buf                 ; else .. send record to buffer

Tsr40:      mov     ah, 0fh                 ; ah = get current info
            int     10h                     ; call BIOS

            push    bx                      ; save current video page

            mov     ah, 3                   ; ah = get cursor position
            int     10h                     ; call BIOS

            push    dx                      ; save cursor position
            push    cx                      ; ..and cursor mode

            mov     ah, 1                   ; ah = set cursor type
            mov     cx, 2020h               ; cx = no cursor
            int     10h                     ; call BIOS

            sti                             ; allows interrupts
            call    CmdProc                 ; ..handle commands
            cli                             ; ..kill interrupts

            mov     ax, 6                   ; ah = restore record pointers
            call    MemRtn                  ; ..ask the memory routine

            call    ClearRec                ; Start with a clear record

            mov     cx, ax                  ; cx = records written

Tsr90:      call    Rec2Buf                 ; kill written records ..
            loop    Tsr90                   ; ..in save buffer

            mov     ax, 6                   ; ax = restore record pointers
            call    MemRtn                  ; ..new records are gone.

            pop     cx                      ; cx = original mode
            mov     ah, 1                   ; ah = set cursor mode
            int     10h                     ; ..do it

            pop     dx                      ; dx = original position
            pop     bx                      ; bx = video page nbr
            mov     ah, 2                   ; ah = set positin
            int     10h                     ; ..do it

Tsr95:      les     bx, KbFlag                  ; es:bx -> shift flags
            and     byte ptr es:[bx], KbFlagC   ; clear scroll led/flag
            mov     cs:TsrActive, 0             ; reset active flag

Tsr99:      popf                            ; restore flags
            pop     es                      ; restore registers
            pop     ds
            pop     bp
            pop     di
            pop     si
            pop     dx
            pop     cx
            pop     bx
            pop     ax

            ret                             ; ..and return to caller

Tsr         endp



; ----------------------------------------------------------------------
;
;   Beep - Give an audible tone
;
; ----------------------------------------------------------------------

Beep        proc
            cmp     cs:BeepCnt, BeepAt      ; time to beep yet?
            jb      Beep90                  ; a. no .. continue

            mov     ax, 0e07h               ; ax = write a beep
            int     10h                     ; call BIOS

            mov     cs:BeepCnt, 0           ; zero the beep counter

Beep90:     ret                             ; ..and return to caller

Beep        endp



; ----------------------------------------------------------------------
;
;   CmdProc - Console command processor
;
;   This routine process the user's console commands.  This includes
;   all commands to manipulate and display the saved scroll data.
;
;   Exit:  Only ES and DS are saved.  All others are saved by caller.
;
; ----------------------------------------------------------------------

CmdProc     proc
            push    ds                      ; save registers
            push    es                      ; ..and es

            mov     StarChr, StarEqu        ; restore the star
            and     ScrFlg, NOT ScrFlgStr   ; .. shut off flag
            mov     TimeCnt, 0              ; .. reset the timer

            call    CmdEnd                  ; fake an "End" key
            jc      CmdProc20               ; ..on error, return

CmdProc09:

CmdProc10:  call    CmdStar                 ; If you wish upon a star...

            cmp     TsrReq, 0               ; q. end scrolling?
            jne     CmdProc30               ; a. no .. wait some more

CmdProc20:  call    CmdEnd                  ; Go to end of save area
            call    CmdUp                   ; ..and up one line
            pop     es                      ; restore entry seg regs
            pop     ds                      ; . . . .
            ret                             ; and just return to caller

CmdProc30:  mov     ah, 1                   ; ah = check for keys fnc
            int     16h                     ; q. anything there?
            jz      CmdProc10               ; a. no ..  get key

            mov     ah, 10h                 ; ah = read key extended fnc
            int     16h                     ; get a key

            lea     si, CmdTable            ; si -> commands table

CmdProc40:  cmp     ah, [si]                ; q. 1st key entry?
            je      CmdProc50               ; a. yes .. exit loop

            add     si, 3                   ; si -> next entry

            cmp     byte ptr [si], 0ffh     ; q. end of table?
            jne     CmdProc40               ; a. no .. check next entry

            call    Beep                    ; give error beep
            jmp     short CmdProc10         ; ..and loop back again

CmdProc50:  call    ForceStar               ; Remove the star

            call    1[si]                   ; goto requested routine
            jmp     short CmdProc10         ; ..and loop back again

CmdProc     endp



; ----------------------------------------------------------------------
;
;   CmdStar - Display the flashing star
;
;  This routine swaps the character in the upper right corner with a 
;  star character. The blinking character indicates that PERUSE is 
;  popped-up and active.
;
; ----------------------------------------------------------------------

CmdStar     proc    

            cmp     TimeCnt, StarCnt        ; q. timeout?
            jb      CmdStar90               ; a. no .. continue

            xor     ScrFlg, ScrFlgStr       ; invert the star bit
            mov     es, VideoSeg            ; es -> video segment
            mov     bx, 79 * 2              ; es:bx -> upper right corner
            mov     al, StarChr             ; al = character
            xchg    al, es:[bx]             ; exchange the characters
            mov     StarChr, al             ; .. replace char in memory
            mov     TimeCnt, 0              ; .. restart the counter

CmdStar90:  ret                             ; return to caller

CmdStar     endp



; ----------------------------------------------------------------------
;
;  ForceStar - Force char back to star
;
;  This routine check if the star character is displayed. If so, it  
;  sets the TimeCnt to force CmdStar to reset the characters.      
;
; ----------------------------------------------------------------------

ForceStar   proc    

            test    ScrFlg, ScrFlgStr       ; q. star displayed?
            jz      ForceStar9              ; a. no .. continue

            mov     TimeCnt, StarCnt        ; Show timeout
            call    CmdStar                 ; .. swap the chars

ForceStar9: ret                             ; return to caller

ForceStar   endp



; ----------------------------------------------------------------------
;
;   CmdEnd - Command key processor: End
;
;   The End key processor causes the last lines of the saved scroll
;   data to be displayed.
;
;   Exit:   carry set if no records are available in scroll buffer
;
; ----------------------------------------------------------------------

CmdEnd      proc

            mov     ax, 3                   ; ax = get max nbr of records
            call    MemRtn                  ; ..call memory handler

            mov     bx, ax                  ; bx = max record nbr
            mov     WorkMNbr, bx            ; ..save for later
            dec     bx                      ; bx = record nbr to read
            mov     WorkRNbr, bx            ; ..also save for later
            push    ds                      ; save our data segment
            pop     es                      ; es = ds
            lea     di, WorkRec             ; di -> work record area
            mov     ax, 1                   ; ax = read a record
            call    MemRtn                  ; call memory handler
            jc      CmdEnd90                ; ..on error, return to caller

            mov     WorkDir, 0              ; set direction flag to up
            mov     WorkHdr, 1              ; ..and header flags
            lea     si, WorkEOF             ; si -> top of file message
            sub     si, offset WorkRec      ; si => TOF message
            mov     WorkOffset, si          ; save new offset

            call    CmdDispUp               ; display a page from bottom up
            clc                             ; clear return code

CmdEnd90:   ret                             ; ..then return to caller

CmdEnd      endp



; ----------------------------------------------------------------------
;
;   CmdHome - Command key processor: Home
;
;   The Home key processor causes the oldest lines of the saved scroll
;   data to be displayed.
;
; ----------------------------------------------------------------------

CmdHome     proc

            xor     bx, bx                  ; bx = record nbr to read
            mov     WorkRNbr, bx            ; ..also save for later
            push    ds                      ; save our data segment
            pop     es                      ; es = ds
            lea     di, WorkRec             ; di -> work record area
            mov     ax, 1                   ; ax = read a record
            call    MemRtn                  ; ..call memory handler

            mov     WorkDir, 1              ; set direction flag to down
            mov     WorkHdr, -1             ; ..and header flags
            lea     si, WorkTOF             ; si -> top of file message
            sub     si, offset WorkRec      ; si => TOF message
            mov     WorkOffset, si          ; save new offset

            call    CmdDispDn               ; display a page from top down
            ret                             ; return to caller

CmdHome     endp



; ----------------------------------------------------------------------
;
;   CmdPgDn - Command key processor: PgDn
;
;   The PgDn key processor causes the next screenful of saved scroll
;   data to be displayed.
;
; ----------------------------------------------------------------------

CmdPgDn     proc

            mov     al, 1                   ; al = down direction
            call    CmdPosition             ; try to position for display
            jc      CmdPgDn10               ; ..on error, just return

            call    CmdDispDn               ; display a page from top down
            jnc     CmdPgDn90               ; ..if ok, return to caller

CmdPgDn10:  call    CmdEnd                  ; display last page

CmdPgDn90:  ret                             ; return to caller

CmdPgDn     endp



; ----------------------------------------------------------------------
;
;   CmdPgUp - Command key processor: PgUp
;
;   The PgDn key processor causes the previous screenful of saved scroll
;   data to be displayed.
;
; ----------------------------------------------------------------------

CmdPgUp     proc

            mov     al, 0                   ; al = up direction
            call    CmdPosition             ; try to position for display
            jc      CmdPgUp10               ; ..on error, just do "home"

            call    CmdBack                 ; backup one more line
            jc      CmdPgUp10               ; ..on error, just do "home"

            call    CmdDispUp               ; display a page from bottom up
            jnc     CmdPgUp90               ; ..if ok, return to caller

CmdPgUp10:  call    CmdHome                 ; display first page

CmdPgUp90:  ret                             ; return to caller

CmdPgUp     endp



; ----------------------------------------------------------------------
;
;   CmdUp - Command key processor: Up
;
;   The Up key processor causes the previous line of saved scroll
;   data to be displayed.
;
; ----------------------------------------------------------------------

CmdUp       proc
            mov     al, 0                   ; al = up direction
            call    CmdPosition             ; try to position for display
            jc      CmdUp10                 ; ..on error, just return

            call    CmdBack                 ; back up one line
            jc      CmdUp10                 ; ..on error, also return

            mov     ax, 0701h               ; ax = scroll screen down fnc
            mov     bh, 0                   ; bh = attribute
            xor     cx, cx                  ; cx = upper left (0,0)
            mov     dh, LinesScrM1          ; dh = lowest line
            mov     dl, 79                  ; dl = lowest right column
            int     10h                     ; call BIOS to scroll screen

            mov     di, 0                   ; di => first line on screen
            call    CmdExpand               ; expand one line onto screen

CmdUp10:    ret                             ; return to caller

CmdUp       endp



; ----------------------------------------------------------------------
;
;   CmdDown - Command key processor: Down
;
;   The Down key processor causes the next line of saved scroll
;   data to be displayed.
;
; ----------------------------------------------------------------------

CmdDown     proc
            mov     al, 1                   ; al = down direction
            call    CmdPosition             ; try to position for display
            jc      CmdDown10               ; ..on error, just return

            mov     ax, 0601h               ; ax = scroll screen up fnc
            mov     bh, 0                   ; bh = attribute
            xor     cx, cx                  ; cx = upper left (0,0)
            mov     dh, LinesScrM1          ; dh = lowest line
            mov     dl, 79                  ; dl = lowest right column
            int     10h                     ; call BIOS to scroll screen

            mov     di, LastScrLine         ; di => last line on screen
            call    CmdExpand               ; expand one line onto screen

            call    CmdFwd                  ; move to next line
CmdDown10:  ret                             ; return to caller

CmdDown     endp



; ----------------------------------------------------------------------
;
;   CmdDispUp - Display a page from the bottom up
;
;   This routine handles common display for End and PgUp keys.
;
;   Exit:   carry set if at top of memory file.
;
; ----------------------------------------------------------------------

CmdDispUp   proc

            mov     di, LastScrLine         ; di => start of last line

CmdDispUp1: call    CmdExpand               ; expand one line onto screen

            or      di, di                  ; q. at top of screen?
            je      short CmdDispUp2        ; a. yes .. exit loop

            call    CmdBack                 ; try to position back a line
            jc      CmdDispUp2              ; ..on error, just return

            sub     di, 160                 ; di => previous line on screen
            jmp     short CmdDispUp1        ; ..and loop back for next line

CmdDispUp2: ret                             ; ..finally return to caller

CmdDispUp   endp



; ----------------------------------------------------------------------
;
;   CmdDispDn - Display a page from the top down
;
;   This routine handles common display for Home and PgDn keys.
;
;   Exit:   carry set if at top of memory file.
;
; ----------------------------------------------------------------------

CmdDispDn   proc

            xor     di, di                  ; di => first line

CmdDispD1:  call    CmdExpand               ; expand one line onto screen

            cmp     di, LastScrLine         ; q. at end of screen?
            ja      short CmdDispD2         ; a. yes .. exit loop

            call    CmdFwd                  ; try to position for next line
            jc      CmdDispD2               ; ..on error, just return

            add     di, 160                 ; di => next line on screen
            jmp     short CmdDispD1         ; ..and loop back for next line

CmdDispD2:  ret                             ; return to caller

CmdDispDn   endp



; ----------------------------------------------------------------------
;
;   CmdEsc - Alternate END key                 
;
; ----------------------------------------------------------------------

CmdEsc      proc
            push    bx                      ; save work registers
            push    es                      ; . . . .

            les     bx, cs:KbFlag           ; es:bx -> kbd shift flag
            mov     al, es:[bx]             ; al = shift flags
            and     al, KbFlagC             ; al = scroll lock flag
            mov     cs:TsrReq, al           ; set TSR request if made
            mov     es:[bx], al             ; .. reset scroll lock 

            pop     es                      ; restore registers
            pop     bx                      ; . . . .
            ret                             ; just return to caller
CmdEsc      endp



; ----------------------------------------------------------------------
;
;   CmdHelp - Display help message at top of the screen
;
;   This routine handles the display of a short help menu and when
;   finished, the redisplay of the current screen.
;
; ----------------------------------------------------------------------

CmdHelp     proc

            lea     si, WorkHlp1            ; si -> 1st display message
            mov     es, VideoSeg            ; es = video memory segment
            xor     di, di                  ; di -> top of screen
            call    Expand                  ; call routine to unpack line

            mov     di, 160                 ; di -> 2nd line on display
            lea     si, WorkHlp2            ; si -> 2nd display message
            call    Expand                  ; call routine to unpack line

CmdHelp10:  call    CmdStar                 ; Look for the flashing star...

            cmp     TsrReq, 0               ; q. end scrolling?
            je      CmdHelp20               ; a. yes .. exit here

            mov     ah, 1                   ; ah = check for keys fnc
            int     16h                     ; q. anything there?
            jz      CmdHelp10               ; a. no .. try again

CmdHelp20:  call    ForceStar               ; forcibly remove the star

            mov     al, WorkDir             ; al = current direction

            or      al, al                  ; q. backward?
            jne     CmdHelp30               ; a. no .. continue

            mov     WorkDir, 1              ; change directions
            call    CmdDispDn               ; display from top of screen
            jmp     short CmdHelp90         ; ..and then return

CmdHelp30:  mov     WorkDir, 0              ; change directions

            call    CmdBack                 ; back up a line
            call    CmdDispUp               ; ..then display from bottom

CmdHelp90:  ret                             ; finally, return to caller

CmdHelp     endp



; ----------------------------------------------------------------------
;
;   CmdPosition - Handle positioning when scrolling directions changes
;
;   This routine handles the positioning of the memory file to the
;   next line to be displayed.  In the changing of directions, the
;   memory file is positioned a screenfull of lines.
;
;   Entry:  al = direction (0=up, 1=down)
;
;   Exit:   carry is set at file boundaries (TOF/EOF)
;
; ----------------------------------------------------------------------

CmdPosition proc

            xor     ch, ch                  ; clear upper part ..
            mov     cl, LinesScr            ; cx = potential loop count

            cmp     al, 1                   ; q. moving forward?
            jne     CmdPos10                ; a. no .. continue

            cmp     WorkHdr, 2              ; q. already display end msg?
            je      CmdPos90                ; a. yes .. escape now

CmdPos10:   cmp     al, WorkDir             ; q. changing directions?
            mov     WorkDir, al             ; ..first save new direction
            je      CmdPos80                ; a. no .. just continue

            or      al, al                  ; q. moving up lines?
            jz      CmdPos20                ; a. yes .. continue

            lea     si, CmdFwd              ; else .. si -> next line rtn
            jmp     short CmdPos30          ; ..and do loop

CmdPos20:   lea     si, CmdBack             ; si -> back one line routine

CmdPos30:   call    si                      ; move pointers one line
            jc      CmdPos90                ; ..on error, beep and return
            loop    CmdPos30                ; ..loop till movement done

CmdPos80:   clc                             ; show all went well
            ret                             ; ..and return to caller

CmdPos90:   call    Beep                    ; ..beep
            stc                             ; ..set error flag
            ret                             ; ..and return to caller

CmdPosition endp



; ----------------------------------------------------------------------
;
;   CmdBack - Back up one line in the record buffer
;
;   This routine handles logically moving up one line in the record
;   buffer.  If necessary, a previous record will be retrieved.
;
;   Exit:   carry set if attempting to move before top of file.
;
; ----------------------------------------------------------------------

CmdBack     proc

            push    si                      ; save registers
            push    di
            push    es
            push    ax
            push    bx

            cmp     WorkHdr, 2              ; q. past file extremes?
            jne     CmdBack10               ; a. no .. continue

            mov     WorkHdr, 1              ; set up for EOF
            jmp     short CmdBack90         ; ..and return to caller

CmdBack10:  cmp     WorkHdr, 0              ; q. at file extremes?
            jg      CmdBack50               ; a. if EOF, use last record
            je      CmdBack20               ; a. if not, continue

            call    Beep                    ; give error beep
            stc                             ; set error return
            jmp     short CmdBack95         ; ..and return to caller

CmdBack20:  mov     si, WorkOffset          ; si = current offset

            cmp     si, 4                   ; q. at start of buffer?
            ja      CmdBack60               ; a. no .. continue

CmdBack30:  mov     bx, WorkRNbr            ; bx = current record nbr

            or      bx, bx                  ; q. at top of file?
            jnz     CmdBack40               ; a. no .. continue

            mov     WorkHdr, -1             ; show using TOF header message
            lea     si, WorkTOF             ; si -> top of file message
            sub     si, offset WorkRec      ; si => TOF message
            jmp     short CmdBack70         ; ..then just continue

CmdBack40:  dec     bx                      ; bx = previous record number
            mov     WorkRNbr, bx            ; ..save for later
            mov     ax, 1                   ; ax = read a record function
            push    ds                      ; save our data segment
            pop     es                      ; ..and use in es register
            lea     di, WorkRec             ; di -> work record buffer
            call    MemRtn                  ; get record from memory file
            jc      CmdBack95               ; ..on error, quit quickly

CmdBack50:  cmp     WorkRec.RecCount, 0     ; q. any lines in this record?
            je      CmdBack30               ; a. no .. back up another

            mov     si, WorkRec.RecNext     ; si => end of record
            mov     WorkHdr, 0              ; clear header flag

CmdBack60:  mov     al, WorkRec[si - 1]     ; al = previous length code
            xor     ah, ah                  ; ..clear upper bits
            sub     si, ax                  ; si -> previous record
CmdBack70:  mov     WorkOffset, si          ; ..and save for later

CmdBack90:  clc                             ; show ok return code
CmdBack95:  pop     bx                      ; restore registers
            pop     ax
            pop     es
            pop     di
            pop     si
            ret                             ; ..and return to caller

CmdBack     endp



; ----------------------------------------------------------------------
;
;   CmdFwd - Move forward one line in the record buffer
;
;   This routine handles logically moving down one line in the record
;   buffer.  If necessary, the next record will be retrieved.
;
;   Exit:   carry set if attempting to move past end of file.
;
; ----------------------------------------------------------------------

CmdFwd      proc

            push    si                      ; save registers
            push    di
            push    es
            push    ax
            push    bx

            cmp     WorkHdr, 0              ; q. at file extremes?
            jl      CmdFwd50                ; a. if TOF, use first record
            je      CmdFwd20                ; a. if not, continue

            cmp     WorkHdr, 2              ; q. at very end?
            jne     CmdFwd10                ; a. no .. continue

            call    Beep                    ; give error beep

CmdFwd10:   mov     WorkHdr, 2              ; flag EOF msg given
            stc                             ; set error return
            jmp     short CmdFwd90          ; ..and return to caller

CmdFwd20:   mov     si, WorkOffset          ; si = current offset

            cmp     si, WorkRec.RecNext     ; q. past end of buffer?
            jb      CmdFwd60                ; a. no .. continue

CmdFwd30:   mov     bx, WorkRNbr            ; bx = current record nbr
            inc     bx                      ; bx = next record nbr

            cmp     bx, WorkMNbr            ; q. at end of file?
            jb      CmdFwd40                ; a. no .. continue

            mov     WorkHdr, 1              ; show using EOF header message
            lea     si, WorkEOF             ; si -> top of file message
            sub     si, offset WorkRec      ; si => TOF message
            mov     WorkOffset, si          ; save new offset
            jmp     short CmdFwd70          ; ..then just continue

CmdFwd40:   mov     WorkRNbr, bx            ; ..save for later
            mov     ax, 1                   ; ax = read a record function
            push    ds                      ; save our data segment
            pop     es                      ; ..and use in es register
            lea     di, WorkRec             ; di -> work record buffer
            call    MemRtn                  ; get record from memory file
            jc      CmdFwd90                ; ..on error, quit quickly

CmdFwd50:   cmp     WorkRec.RecCount, 0     ; q. any lines in this record?
            je      CmdFwd30                ; a. no .. goto next record

            mov     si, 4                   ; si => beginning of record
            mov     WorkOffset, si          ; save new offset
            mov     WorkHdr, 0              ; clear header flag
            jmp     short CmdFwd70          ; ..and return ok

CmdFwd60:   mov     al, WorkRec[si]         ; al = previous length code
            xor     ah, ah                  ; ..clear upper bits
            add     si, ax                  ; si -> next record
            mov     WorkOffset, si          ; save new offset

            cmp     si, WorkRec.RecNext     ; q. past end of buffer?
            jae     CmdFwd30                ; a. yes .. read next record

CmdFwd70:   clc                             ; show ok return code

CmdFwd90:   pop     bx                      ; restore registers
            pop     ax
            pop     es
            pop     di
            pop     si
            ret                             ; ..and return to caller

CmdFwd      endp



; ----------------------------------------------------------------------
;
;   CmdExpand - Expand a line onto the screen
;
;   This routine handles expanding one line onto the screen.  The
;   CmdExpand routine also handles repositioning of the memory file
;   after display.
;
;   Entry:  di = screen offset into VideoSeg
;
; ----------------------------------------------------------------------

CmdExpand   proc
            push    di

            mov     si, WorkOffset          ; si = offset in record
            lea     si, WorkRec[si + 1]     ; si -> first compress byte

            mov     es, VideoSeg            ; es = video memory segment
            call    Expand                  ; call routine to unpack line

            pop     di                      ; restore registers
            ret                             ; ..and return to caller

CmdExpand   endp



; ----------------------------------------------------------------------
;
;   Start routine - startup processsing
;
;   This routine performs the following functions:
;       - Call Cmd to perform initial processing
;       - Move the MemHndlr
;       - Set all interrupt vectors
;       - Go TSR
;
; ----------------------------------------------------------------------

start       proc

            push    ds                      ; save entry ds
            cld                             ; clear direction flag!
            call    cmd                     ; initialize the system
     rep    movsb                           ; ..move the MemHndlr rtn
            pop     ds                      ; restore ds

            call    ClearRec                ; initialize work record area

            mov     MemRtnSeg, dx           ; setup new MemHndlr

            mov     dx, NxtAvail            ; dx = bytes used
            add     dx, MemRtnLen           ; ..add length of routine
            add     dx, 15                  ; ..next para if needed

            mov     cl, 4                   ; cl = shift count
            shr     dx, cl                  ; dx = paragraph count

            push    dx                      ; save memory to keep

            mov     si, IntBlk1             ; si -> 1st interrupt block
            mov     cx, intBlkCnt           ; cx = nbr of ints handled

start60:    mov     ah, 35h                 ; ah = get interrupt vector
            mov     al, [si+4]              ; al = interrupt number
            int     21h                     ; ..get the current setting

            mov     [si+2], es              ; save segment of old int
            mov     [si], bx                ; ..and offset

            mov     dx, [si+5]              ; dx -> new interrupt
            mov     ah, 25h                 ; ah = set interrupt vector
            int     21h                     ; ..set up new vector

            add     si, intblklen           ; si -> next entry
            loop    start60                 ; ..set next interrupt

            call    ModeChange              ; ..test mode setting

            pop     dx                      ; restore memory to keep

            mov     ax, 3100h               ; ax = TSR, rc = 0
            int     21h                     ; ..exit/stay resident

start       endp


            align   16                      ; align to a paragraph

; ----------------------------------------------------------------------
;
;   Note: Everything beyond this is overlaid after initialization.
;
; ----------------------------------------------------------------------

; ---------------------------------------------------------------------
;   Uninitialized data areas
; ---------------------------------------------------------------------

WorkRec     label   byte                    ; work record

MemHndlr    equ     WorkRec + RecSize       ; start of buffer driver

; ----------------------------------------------------------------------
;   The memory handler will be moved to MemRtn after initialization
; ----------------------------------------------------------------------

; ----------------------------------------------------------------------
;   Initialization code data areas --  available during init only
; ----------------------------------------------------------------------

HdrMsg      db  "PERUSE 1.00  Copyright (c) 1994, Bob Flanders and "
            db  "Michael Holmes"
            db  13,10
            db  "First Published in PC Magazine, April 12, 1994"
            db  13,10
Dollar      db  "$"

Help        db  10
            db  "Usage:  PERUSE /En|/Xn",13,10
            db  "        PERUSE /U",13,10,10
            db  "Where:  /En  tells PERUSE to use nK of EMS memory *",13,10
            db  "        /Xn  tells PERUSE to use nK of XMS memory",13,10
            db  "        /U   removes PERUSE from memory",13,10,10
            db  "        n must be in the range 16 to 8192.",13,10,10
            db  "* Note: You may not use EMS in a DOS box under Windows."
            db  13,10
            db  "$"

NumRecs     dw  ?                               ; size of scroll area in K

InvReq      db  10
            db  "Invalid request - "
InvReqCd    db  " ",13,10,"$"

NotUp       db  10
            db  "PERUSE not installed",13,10,"$"

UpAlrdy     db  10
            db  "PERUSE already installed",13,10,"$"

InvMemz     db  10
            db  "Invalid memory size specified; "
            db  "Must between 16 and 8192.",13,10,"$"

FreeOK      db  10
            db  "Uninstalled successfully",13,10,"$"

CantInit    db  10
            db  "Unable to allocate memory buffer",13,10,"$"

CantFree    db  10
            db  "Cannot release resident copy at this time.",13,10,"$"

Windows     db  10
            db  "You cannot use EMS under Windows. Use XMS.",13,10,"$"

; ----------------------------------------------------------------------
;
;   Cmd - Command processor & Initialization procedure
;
;    Entry: cs, ds, es -> our segment
;
;   Return: ds:si -> buffer interface routine
;           es:di -> new address for interface routine
;              cx =  len to move
;              dx =  new interface routine segment
;
;   1. Scan for arguments
;       - if error, issue error message & exit
;
;   2. Call appropriate command processor
;
;   3. Setup regs for move of memory handler routine
;
; ----------------------------------------------------------------------

Cmd         proc                            ; find/process command
            push    es                      ; save regs

            mov     dx, offset HdrMsg       ; ds:dx -> header message
            mov     ah, 9                   ; ah = print ASCII$ string
            int     21h                     ; ..display header

            mov     ax, 1600h               ; ax = windows active request
            int     2fh                     ; see if windows active

            mov     InWin, al               ; save in windows flag

            mov     ax, 0B0BFh              ; ah = query PERUSE active?
            xor     dx, dx                  ; dx = zero
            int     10h                     ; ..ask int 10h
            add     al, InWin               ; ..modify with InWin flag

            cmp     ax, 0FB0Bh              ; q. running?
            jne     Cmd0                    ; a. no .. continue process

            or      ScrFlg, ScrFlgRes       ; show we're resident
            mov     ResSeg, dx              ; ..and save resident CS

Cmd0:       mov     bx, 81h                 ; bx -> command line

            call    NxtOp                   ; q. any operands?
            jnc     Cmd00                   ; a. yes .. process

            mov     dx, offset Dollar       ; dx -> null message
            mov     al, 1                   ; al = give help
            call    Die                     ; terminal w/help

Cmd00:      cmp     al, 'E'                 ; q. EMS startup request?
            jnz     Cmd10                   ; a. no .. check next

            cmp     InWin, 0                ; q. in windows?
            je      Cmd02                   ; a no .. continue

            mov     dx, offset Windows      ; dx -> no EMS in Windows
            mov     al, 1                   ; al = give help
            call    Die                     ; ..die gracefully

Cmd02:      call    Init                    ; call init code
            jmp     short Cmd90             ; return to caller

Cmd10:      cmp     al, 'X'                 ; q. XMS startup request?
            jnz     Cmd20                   ; a. no .. check next

            call    Init                    ; call init code
            jmp     short Cmd90             ; return to caller

Cmd20:      cmp     al, 'U'                 ; q. uninstall?
            jnz     Cmd60                   ; a. no .. check next

            jmp     UnInstall               ; else.. call  uninstall

Cmd60:      mov     InvReqCd, al            ; save error code

            mov     dx, offset InvReq       ; ds:dx -> error message
            mov     al, 1                   ; al = give help
            call    Die                     ; Message & Die

Cmd90:      pop     es                      ; restore regs

            lea     di, MemHndlr            ; di -> MemHndlr rtn

            mov     dx, di                  ; dx = offset also
            mov     cl, 4                   ; cl = shift amount
            shr     dx, cl                  ; dx = segment offset
            mov     ax, cs                  ; ax = program segment
            add     dx, ax                  ; dx = MemHndlr segment

            xor     si, si                  ; si = 0 offset
            mov     cx, MemRtnLen           ; cx = size of routine
            cld                             ; direction = up

            mov     ds, MemRtnSeg           ; ds = current seg of rtn

            ret                             ; return to caller

Cmd         endp                            ; end of scan routine



; ----------------------------------------------------------------------
;   NxtOp: find next operand;  bx -> current position
;
;                      Returns: bx -> 1st char (after slash) of next op
;                               al =  1st letter of opnd (capitalized)
;                               carry = end of line
; ----------------------------------------------------------------------

NxtOp       proc
            cmp     byte ptr [bx], 0dh      ; q. end of line?
            je      NxtOp80                 ; a. yes .. scan no more

            cmp     byte ptr [bx], '/'      ; q. slash?
            je      NxtOp50                 ; a. yes .. return next char

            cmp     byte ptr [bx], ' '      ; q. whitespace?
            jna     NxtOp20                 ; a. yes .. skip whitespace

; skip non-white space

NxtOp10:    inc     bx                      ; bx -> next char

            cmp     byte ptr [bx], 0dh      ; q. end of line?
            je      NxtOp80                 ; a. yes .. scan no more

            cmp     byte ptr [bx], '/'      ; q. slash?
            je      NxtOp50                 ; a. yes .. return next char

            cmp     byte ptr [bx], ' '      ; q. whitespace?
            ja      NxtOp10                 ; a. no .. skip it

NxtOp20:    inc     bx                      ; bx -> next char

            cmp     byte ptr [bx], 0dh      ; q. end of line?
            je      NxtOp80                 ; a. yes .. scan no more

            cmp     byte ptr [bx], '/'      ; q. slash?
            je      NxtOp50                 ; a. yes .. return next char

            cmp     byte ptr [bx], ' '      ; q. whitespace?
            jna     NxtOp20                 ; a. yes .. get next char
            jmp     short NxtOp60           ; else .. return char

NxtOp50:    inc     bx                      ; bx -> byte after slash

NxtOp60:    mov     al, [bx]                ; al = byte of operand

            cmp     al, 'a'                 ; q. char >= lower case a?
            jna     NxtOp90                 ; a. no .. return as is

            cmp     al, 'z'                 ; q. char <= lower case z?
            jnb     NxtOp90                 ; a. no .. return as is

            and     al, not 20h             ; xlate to upper case
            jmp     short NxtOp90           ; return to caller, ok

NxtOp80:    stc                             ; carry = end of line
            ret                             ; return to caller

NxtOp90:    clc                             ; no carry = al contains char
            ret                             ; return to caller
NxtOp       endp



; ----------------------------------------------------------------
;
;   Initialize buffer;   bx -> next char to process
;                        al =  memory type flag
;
; ----------------------------------------------------------------

Init        proc

            test    ScrFlg, ScrFlgRes       ; q. PERUSE already resident
            jz      Init05                  ; a. no .. ok to INIT.

            mov     dx, offset UpAlrdy      ; dx -> up already message
            xor     al, al                  ; al = no help needed
            call    Die                     ; ..Die .. with honor

Init05:     mov     MemType, al             ; save type requested

            push    es                      ; save registers
            push    bx

            les     bx, KbFlag                   ; es:bx -> KB Flag byte
            and     byte ptr es:[bx], KbFlagC    ; .. clear scroll lock

            mov     es, ds:[002ch]          ; es -> environment
            mov     ah, 49h                 ; ax = release it
            int     21h                     ; ..tell dos to make it so

            mov     ah, 34h                 ; ah = get InDosFlag address
            int     21h                     ; ..es:bx -> InDosFlag

            mov     word ptr InDosFlg, bx   ; save offset
            mov     word ptr InDosFlg+2, es ; ..and segment

            pop     bx                      ; ..restore regs
            pop     es

            call    GetMemZ                 ; get memory size
            call    InitMem                 ; initialize memory rtn

            ret                             ; return to caller

Init        endp



; ----------------------------------------------------------------------
;
;   Process the uninstall command
;
;   Entry: ResSeg =  segment of active copy
;
;   Exit: Returns to DOS
;
; ----------------------------------------------------------------------

UnInstall   proc
            test    ScrFlg, ScrFlgRes       ; q. PERUSE already resident
            jnz     UnIns05                 ; a. yes.. continue

            mov     dx, offset NotUp        ; dx -> message
            jmp     Die                     ; ..die now, sucker

UnIns05:    mov     di, IntBlk1             ; bx -> first interrupt blk
            mov     cx, IntBlkCnt           ; cx = number of ints used

UnIns10:    mov     ah, 35h                 ; ah = get interrupt
            mov     al, [di+4]              ; al = interrupt number
            int     21h                     ; es:bx -> interrupt

            mov     ax, es                  ; ax = int segment
            cmp     ax, cs:ResSeg           ; q. resident segment?
            jne     UnIns80                 ; a. no .. can't UnInstall

            add     di, IntBlkLen           ; si -> next block
            loop    UnIns10                 ; ..check next block

            mov     di, IntBlk1             ; si -> first interrupt blk
            mov     cx, IntBlkCnt           ; cx = number of ints used

            cli                             ; no ints ..
UnIns20:    lds     dx, dword ptr es:[di]   ; ds:dx -> old interrupt rtn
            mov     ah, 25h                 ; ah = set interrupt
            mov     al, es:[di+4]           ; al = int to set
            int     21h                     ; ..set the interrupt

            add     di, IntBlkLen           ; si -> next block
            loop    UnIns20                 ; ..reset next interrupt
            sti                             ; ..allow interrupts again

            mov     ax, 4                   ; ax = free memory buffer
            call    es:MemRtn               ; ..using the memory routine

            mov     ah, 49h                 ; ah = free resident mem
            int     21h                     ; ..do it

            mov     dx, offset FreeOK       ; dx -> freed ok.
            jmp     short UnIns90           ; ..end the job, painlessly

UnIns80:    mov     dx, offset CantFree     ; dx -> can't free message

UnIns90:    xor     al, al                  ; al = no help

            push    cs                      ; restore our ..
            pop     ds                      ; ..own segment

            call    Die                     ; die .. hard & fast

UnInstall   endp



; ----------------------------------------------------------------------
;
;   Get memory buffer size reqested;  bx -> memory type argument
;
;                      return:  NumRecs  is set to size requested
;
; ----------------------------------------------------------------------

GetMemZ     proc
            push    bx                      ; save registers

            mov     al, 1[bx]               ; al = next char

            cmp     al, 0dh                 ; q. end of line?
            je      GetMemZ70               ; a. yes .. declare an error

            cmp     al, ' '                 ; q. white space?
            ja      GetMemZ10               ; a. no .. parse the operand

            inc     bx                      ; bx -> white space
            call    NxtOp                   ; .. get to next operand
            jc      GetMemZ70               ; .. leave if end of line

            dec     bx                      ; bx -> char before size

GetMemZ10:  lea     si, 1[bx]               ; si = byte after type

            call    AtoI                    ; ax = size specified

            cmp     ax, 16                  ; q. at least 16k specified?
            jb      GetMemZ70               ; a. no .. tell 'em & die

            cmp     ax, 8192                ; q. more than max?
            jna     GetMemZ80               ; a. no .. continue

GetMemZ70:  mov     dx, offset InvMemz      ; dx -> invalid memory size
            mov     al, 1                   ; al = give 'em help
            call    Die                     ; ..ice 'im

GetMemZ80:  mov     NumRecs, ax             ; set record count

            pop     bx                      ; restore registers
            ret                             ; return to caller

GetMemZ     endp



; ----------------------------------------------------------------------
;
;   AtoI - character to integer
;
;   entry: si -> string
;
;    exit: ax = number
;          si -> 1st non-numeric character
;
; ----------------------------------------------------------------------

AtoI        proc

            push    bx                      ; save registers
            push    cx
            push    dx
            xor     ax, ax                  ; ax = zero for accumulation
            xor     bh, bh                  ; bh = zero for addition
            mov     cx, 10                  ; cx = multiplier

AtoI10:     mov     bl, [si]                ; bl = character from string

            cmp     bl, '0'                 ; q. less than a zero?
            jb      AtoI90                  ; a. yes .. done

            cmp     bl, '9'                 ; q. greater than a nine?
            ja      AtoI90                  ; a. yes .. done here too

            sub     bl, '0'                 ; bl = AtoI of character
            mul     cx                      ; ax = ax * 10
            add     ax, bx                  ; ax = new accumulator
            inc     si                      ; si -> next string character
            jmp     short AtoI10            ; ..then loop till end of string

AtoI90:     pop     dx                      ; restore registers
            pop     cx
            pop     bx
            ret                             ; ..and return to caller

AtoI        endp



; ----------------------------------------------------------------------
;
;   Init the buffer handler;   bx -> memory type operand 
;
; ----------------------------------------------------------------------

InitMem     proc

            call    FindRtn                 ; find the memory routine

            mov     word ptr MemRtn, 3      ; setup the offset for the call
            xor     ax, ax                  ; ax = 0 = Get Memory
            mov     cx, NumRecs             ; cx = number of records req'd
            call    MemRtn                  ; call the handler
            jnc     InitMem90               ; ..return if all ok

            mov     dx, offset CantInit     ; dx -> initialization error
            xor     al, al                  ; al = no help
            call    Die                     ; ..die in flames

InitMem90:  ret

InitMem     endp



; ----------------------------------------------------------------------
;
;   Find the memory handler; bx -> memory type
;
;                       Return: MemRtnSeg = segment of routine
;                               MemRtnLen = length of routine (in bytes)
;
; ----------------------------------------------------------------------

FindRtn     proc                            ; find abs address of rtn

            mov     al, [bx]                ; al = first char of switch
            and     al, NOT 20h             ; upper case it.

            mov     si, offset MainLen      ; si -> start of 1st qrtn

FindRtn10:  cmp     [si+2], al              ; q. this the routine?
            je      FindRtn50               ; a. yes .. process the addr

            add     si, [si]                ; si -> next routine
            jmp     FindRtn10               ; ..find the next routine

FindRtn50:  mov     dx, [si]                ; dx = length of routine
            mov     MemRtnLen, dx           ; ..save the routine length

            mov     cl, 4                   ; cl = shift amt
            shr     si, cl                  ; ..div by 16
            mov     ax, cs                  ; ax = current segment
            add     ax, si                  ; ax = driver's segment

            mov     MemRtnSeg, ax           ; save segment of routine

            ret                             ; return to caller


FindRtn     endp



; ----------------------------------------------------------------------
;
;   terminate w/help message; dx -> intermediate message
;                             al not zero if help wanted
;
; ----------------------------------------------------------------------

Die         proc
            push    ax                      ; save help request

            mov     ah, 9                   ; ah = print ASCII$ message
            int     21h                     ; print the error message

            pop     ax                      ; restore help request
            or      al, al                  ; q. help wanted?
            jz      Die10                   ; a. no .. exit

            mov     dx, offset Help         ; dx -> help message
            mov     ah, 9                   ; ah = print ASCII$ message
            int     21h                     ; ... print the help

Die10:      mov     ax, 4c01h               ; die w/some notice
            int     21h                     ; ... return to dos
Die         endp

            align   16                      ; line up on paragraph
MainLen     label   byte                    ; ..length of main
MainLine    ends                            ; End of mainline code

            IFDEF   Borland                 ; if assembled with TASM
            
ENDSTMT     macro                           ; No end statement in modules
                                            ; .. make this a null macro
            endm                                 

            include PERUSEE.ASM             ; include EMS 
            include PERUSEX.ASM             ; .. and XMS

            ENDIF

            end     begin
